from awacs.model.errors import ValidationError
from awacs.wrappers.l7macro import L7Macro, VERSION_0_3_0
from infra.awacs.proto import api_pb2


xffy_behavior = api_pb2.GetBalancerXFFYBehaviorResponse


def get_xffy_header_index(l7_macro):
    for i, header_action in enumerate(l7_macro.headers):
        if header_action.target == u'x-forwarded-for-y':
            return i
    return None


def get_current_xffy_behavior(l7_macro, xffy_header_index):
    if l7_macro.get_version() >= VERSION_0_3_0:
        if l7_macro.pb.core.trust_x_forwarded_for_y:
            return xffy_behavior.TRUST
        else:
            return xffy_behavior.OVERRIDE
    if xffy_header_index is None:
        return xffy_behavior.NONE
    if xffy_header_index == 0:
        header_action = l7_macro.headers[xffy_header_index]
        if header_action.create and header_action.func == u'realip':
            if header_action.create.pb.keep_existing:
                return xffy_behavior.TRUST
            else:
                return xffy_behavior.OVERRIDE
    return xffy_behavior.UNKNOWN


def update_xffy_header_actions(l7_macro_pb, trust_xffy):
    l7_macro = L7Macro(l7_macro_pb)
    xffy_header_index = get_xffy_header_index(l7_macro)
    current_xffy_behavior = get_current_xffy_behavior(l7_macro, xffy_header_index)

    if current_xffy_behavior is xffy_behavior.UNKNOWN:
        raise ValidationError(
            u'This balancer has an unrecognized header action related to "X-Forwarded-For-Y" in its config, '
            u'see l7_macro.headers[{}]. Automatic upgrade is only possible if the header "X-Forwarded-For-Y" '
            u'is set to "realip" as the very first header action in the config (or not modified at all).'.format(
                xffy_header_index))

    if current_xffy_behavior is xffy_behavior.NONE:
        return  # no changes in config are needed

    if ((current_xffy_behavior is xffy_behavior.TRUST and trust_xffy)
            or (current_xffy_behavior is xffy_behavior.OVERRIDE and not trust_xffy)):
        # config matches the desired behavior, so we can remove the obsolete header action from config
        del l7_macro_pb.headers[xffy_header_index]
        return

    if trust_xffy:
        raise ValidationError(
            u'You selected an option to trust the "X-Forwarded-For-Y" header in incoming requests, '
            u'but this balancer is currently configured to overwrite it with request\'s IP address instead.'
        )
    else:
        raise ValidationError(
            u'You selected an option to overwrite the "X-Forwarded-For-Y" header in incoming requests '
            u'with the request\'s IP address, but this balancer is currently configured to keep this header '
            u'if it already exists.'
        )
