# coding: utf-8
import os
import flask
import ujson
from sepelib.core import config as appconfig
from cachetools.func import ttl_cache

from awacs.model.balancer.order.util import get_available_balancer_locations
from awacs.model.components import COMPONENT_CONFIGS_BY_TYPE
from awacs.model.external_clusters import AZURE_CLUSTERS_BY_NAME
from awacs.wrappers import rps_limiter_settings
from awacs.wrappers.l7upstreammacro import L7UpstreamMacroDc
from infra.awacs.proto import model_pb2
from infra.swatlib.auth import util as flask_util


index_bp = flask.Blueprint('ui_index', __name__, url_prefix='/ui')


def _make_config_json():
    available_balancer_locations = get_available_balancer_locations()
    l7_fast_namespaces = appconfig.get_value('run.l7_fast_namespaces')
    json_constants = ujson.dumps({
        'l7_fast_namespaces': l7_fast_namespaces,
        'certificator_ca': {k.title(): v for k, v in appconfig.get_value('certificator.ca').items()},
        'available_balancer_locations': sorted([loc.upper() for loc in available_balancer_locations]),
        'allowed_dc_names': [dc.upper() for dc in L7UpstreamMacroDc.ALLOWED_DC_NAMES],
        'component_types': [model_pb2.ComponentMeta.Type.Name(type_value) for type_value in COMPONENT_CONFIGS_BY_TYPE],
        'available_rpslimiter_installations': sorted(rps_limiter_settings.get_non_public_installation_names()),
        'available_azure_clusters': sorted(AZURE_CLUSTERS_BY_NAME.keys()),
        'unavailable_clusters': appconfig.get_value('run.unavailable_clusters', {})
    })
    return flask.Response(json_constants, content_type=b'application/javascript')


@index_bp.route('/config.json')
@flask_util.login_exempt
@ttl_cache(maxsize=1, ttl=300)
def get_config_js():
    return _make_config_json()


@index_bp.route('/', defaults={'path': 'index.html'})
@index_bp.route('/<path:path>')
def index(path):
    dir = os.path.join(os.path.abspath(os.path.dirname(__file__)), 'static')
    resp = flask.send_from_directory(dir, path, conditional=True)
    return resp
