# coding: utf-8
import re
from sepelib.core import config as appconfig

from awacs.lib import rpc
from awacs.lib.strutils import flatten_full_id2
from awacs.model import db, apicache
from awacs.model.balancer.generator import get_included_full_backend_ids,\
    get_rev_index_pb
from awacs.model.cache import IAwacsCache
from awacs.model.dao import IDao
from awacs.model.util import clone_pb, omit_duplicate_items_from_auth
from awacs.model.validation import validate_and_parse_yaml_upstream_config
from awacs.model.zk import IZkStorage
from awacs.model.balancer.state_handler import L7BalancerStateHandler
from awacs.web.auth.core import authorize_update, authorize_remove, get_acl, authorize_create
from awacs.web.validation import upstream
from awacs.wrappers import linter
from awacs.wrappers.base import ValidationCtx, Holder
from infra.awacs.proto import api_pb2, model_pb2
from infra.swatlib.gutils import idle_iter as gevent_idle_iter, get_periodic_gevent_idle
from .util import AwacsBlueprint, forbid_action_during_namespace_order, validate_namespace_total_objects_count
from .validation.upstream import validate_upstream_yaml_size

upstream_service_bp = AwacsBlueprint('rpc_upstream_service', __name__, '/api')


@upstream_service_bp.method('CreateUpstream',
                            request_type=api_pb2.CreateUpstreamRequest,
                            response_type=api_pb2.CreateUpstreamResponse)
def create_upstream(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.CreateUpstreamRequest
    :rtype: api_pb2.CreateUpstreamResponse
    """
    upstream.validate_request(req_pb)
    omit_duplicate_items_from_auth(req_pb.meta.auth)
    namespace_id = req_pb.meta.namespace_id
    upstream_id = req_pb.meta.id

    cache = IAwacsCache.instance()
    zk = IZkStorage.instance()

    namespace_pb = zk.must_get_namespace(namespace_id=namespace_id)
    validate_namespace_total_objects_count('upstream', len(cache.list_all_upstreams(namespace_id)), namespace_pb)
    authorize_create(namespace_pb, auth_subject)
    forbid_action_during_namespace_order(namespace_pb, auth_subject)

    if zk.does_upstream_exist(namespace_id, upstream_id):
        raise rpc.exceptions.ConflictError(
            'Upstream "{}" already exists in namespace "{}".'.format(upstream_id, namespace_id))

    if (namespace_pb.spec.easy_mode_settings.l7_upstream_macro_only.value and
            req_pb.spec.yandex_balancer.mode != req_pb.spec.yandex_balancer.EASY_MODE2):
        raise rpc.exceptions.ForbiddenError('Only easy mode upstreams balancers can be created in this namespace')

    if req_pb.spec.yandex_balancer.yaml:
        # check yaml size before validating it
        validate_upstream_yaml_size(req_pb.spec.yandex_balancer.yaml, field_name=u'spec.yandex_balancer.yaml')
    ctx = ValidationCtx.create_ctx_with_config_type_upstream(namespace_pb=namespace_pb,
                                                             full_upstream_id=(namespace_id, upstream_id),
                                                             upstream_spec_pb=req_pb.spec)
    holder = validate_and_parse_yaml_upstream_config(namespace_id=namespace_id,
                                                     upstream_id=upstream_id,
                                                     spec_pb=req_pb.spec,
                                                     ctx=ctx,
                                                     comment=req_pb.meta.comment)

    if not req_pb.spec.yandex_balancer.yaml:
        # validate yaml size after calling validate_and_parse_yaml_upstream_config so now it has 'yaml'
        validate_upstream_yaml_size(req_pb.spec.yandex_balancer.yaml,
                                    field_name=u'spec.yandex_balancer.yaml')

    rev_index_pb = get_rev_index_pb(namespace_id, req_pb.spec, holder)

    upstream_pb = IDao.instance().create_upstream(
        meta_pb=req_pb.meta,
        spec_pb=req_pb.spec,
        login=auth_subject.login,
        rev_index_pb=rev_index_pb)

    # no need to fill the statuses because the upstream has just been created and
    # there is very little chance it has already been processed by ctls
    return api_pb2.CreateUpstreamResponse(upstream=upstream_pb)


@upstream_service_bp.method('ListUpstreams',
                            request_type=api_pb2.ListUpstreamsRequest,
                            response_type=api_pb2.ListUpstreamsResponse,
                            max_in_flight=5)
def list_upstreams(req_pb, _):
    upstream.validate_request(req_pb)
    if not req_pb.HasField('field_mask'):
        req_pb.field_mask.AllFieldsFromDescriptor(model_pb2.Upstream.DESCRIPTOR)

    c = apicache.IAwacsApiCache.instance()

    query = {}
    if req_pb.query.id_regexp:
        query[c.UpstreamsQueryTarget.ID_REGEXP] = re.compile(req_pb.query.id_regexp)
    if req_pb.query.validated_status_in:
        query[c.UpstreamsQueryTarget.VALIDATED_STATUS_IN] = req_pb.query.validated_status_in
    if req_pb.query.in_progress_status_in:
        query[c.UpstreamsQueryTarget.IN_PROGRESS_STATUS_IN] = req_pb.query.in_progress_status_in
    if req_pb.query.active_status_in:
        query[c.UpstreamsQueryTarget.ACTIVE_STATUS_IN] = req_pb.query.active_status_in
    if req_pb.query.type_in:
        query[c.UpstreamsQueryTarget.TYPE_IN] = req_pb.query.type_in

    sort = []
    if req_pb.sort_target == req_pb.ID:
        sort.append(c.UpstreamsSortTarget.ID)
    elif req_pb.sort_target == req_pb.ORDER_LABEL:
        sort.append(c.UpstreamsSortTarget.ORDER_LABEL)
    elif req_pb.sort_target == req_pb.MTIME:
        sort.append(c.UpstreamsSortTarget.MTIME)
    sort.append(-1 if req_pb.sort_order == api_pb2.DESCEND else 1)

    kwargs = dict(sort=sort,
                  query=query,
                  skip=req_pb.skip or None,
                  limit=req_pb.limit or None)
    if req_pb.balancer_id:
        upstream_pbs, total = c.list_balancer_upstreams(namespace_id=req_pb.namespace_id,
                                                        balancer_id=req_pb.balancer_id, **kwargs)
    else:
        upstream_pbs, total = c.list_namespace_upstreams(namespace_id=req_pb.namespace_id, **kwargs)
    resp_pb = api_pb2.ListUpstreamsResponse(total=total)
    for upstream_pb in gevent_idle_iter(upstream_pbs, idle_period=30):
        req_pb.field_mask.MergeMessage(upstream_pb, resp_pb.upstreams.add())
    return resp_pb


@upstream_service_bp.method('GetUpstream',
                            request_type=api_pb2.GetUpstreamRequest,
                            response_type=api_pb2.GetUpstreamResponse)
def get_upstream(req_pb, _):
    upstream.validate_request(req_pb)
    namespace_id = req_pb.namespace_id
    upstream_id = req_pb.id

    if req_pb.consistency == api_pb2.STRONG:
        zk = IZkStorage.instance()
        zk.sync_balancer_states(namespace_id)
        upstream_pb = zk.must_get_upstream(namespace_id, upstream_id, sync=True)
        balancer_state_pbs = IAwacsCache.instance().list_all_balancer_states(namespace_id=namespace_id)
        upstream_pb = apicache.copy_per_balancer_statuses(entity_pb=upstream_pb,
                                                          balancer_state_pbs=balancer_state_pbs)
    else:
        c = apicache.IAwacsApiCache.instance()
        upstream_pb = c.must_get_upstream(namespace_id=namespace_id, upstream_id=upstream_id)

    if req_pb.annotations:
        annotated_upstream_pb = api_pb2.AnnotatedUpstream(upstream=upstream_pb)
        for annotation in req_pb.annotations:
            if annotation == api_pb2.AnnotatedUpstream.UA_INCLUDED_FULL_BACKEND_IDS:
                full_backend_ids = get_included_full_backend_ids(
                    upstream_pb, ticker=get_periodic_gevent_idle(5))
                for full_backend_id in sorted(full_backend_ids):
                    annotated_upstream_pb.included_full_backend_ids.append(flatten_full_id2(full_backend_id))
        return api_pb2.GetUpstreamResponse(annotated_upstream=annotated_upstream_pb)
    else:
        return api_pb2.GetUpstreamResponse(upstream=upstream_pb)


@upstream_service_bp.method('LintUpstream',
                            request_type=api_pb2.LintUpstreamRequest,
                            response_type=api_pb2.LintUpstreamResponse)
def lint_upstream(req_pb, auth_subject):
    upstream.validate_request(req_pb)
    namespace_id = req_pb.meta.namespace_id
    upstream_id = req_pb.meta.id

    enable_linter = appconfig.get_value('run.enable_linter', default=True)
    if not enable_linter:
        return api_pb2.LintUpstreamResponse()

    ctx = ValidationCtx.create_ctx_with_config_type_upstream(
        namespace_pb=IAwacsCache.instance().must_get_namespace(namespace_id),
        full_upstream_id=(namespace_id, upstream_id),
        upstream_spec_pb=req_pb.spec)
    holder = validate_and_parse_yaml_upstream_config(
        namespace_id=namespace_id,
        upstream_id=upstream_id,
        spec_pb=req_pb.spec,
        ctx=ctx)

    linters_to_run = [linter.arl.UpstreamChecker, linter.unsafe_outer_balancer2.UpstreamChecker]

    resp_pb = api_pb2.LintUpstreamResponse()
    for linter_to_run in linters_to_run:
        checker = linter_to_run(full_id=linter.model.FullId('upstream', namespace_id, upstream_id))
        linter.model.visit(holder, checker)

        for warning in checker.warnings:  # type: linter.model.Warning
            warning_pb = resp_pb.warnings.add()
            warning_pb.rule = warning.rule
            warning_pb.message = warning.message
            if warning.severity == warning.Severity.CRIT:
                warning_pb.severity = warning_pb.CRIT
            else:
                raise AssertionError()
            warning_pb.path.extend(warning.path)
            warning_pb.tags.extend(warning.tags)
    return resp_pb


@upstream_service_bp.method('UpdateUpstream',
                            request_type=api_pb2.UpdateUpstreamRequest,
                            response_type=api_pb2.UpdateUpstreamResponse)
def update_upstream(req_pb, auth_subject):
    upstream.validate_request(req_pb)
    omit_duplicate_items_from_auth(req_pb.meta.auth)
    namespace_id = req_pb.meta.namespace_id
    upstream_id = req_pb.meta.id

    zk = IZkStorage.instance()

    namespace_pb = zk.must_get_namespace(namespace_id)
    upstream_pb = zk.must_get_upstream(namespace_id, upstream_id)
    try:
        authorize_update(upstream_pb, req_pb, auth_subject)
    except rpc.exceptions.ForbiddenError:
        authorize_update(upstream_pb, req_pb, auth_subject, acl=get_acl(namespace_pb))

    rev_index_pbs = None
    auth_has_changed = req_pb.meta.HasField('auth') and upstream_pb.meta.auth != req_pb.meta.auth
    if req_pb.HasField('spec'):
        if req_pb.spec.yandex_balancer.yaml:
            # check yaml size before validating it
            validate_upstream_yaml_size(req_pb.spec.yandex_balancer.yaml, field_name=u'spec.yandex_balancer.yaml')

        ctx = ValidationCtx.create_ctx_with_config_type_upstream(namespace_pb=namespace_pb,
                                                                 full_upstream_id=(namespace_id, upstream_id),
                                                                 upstream_spec_pb=req_pb.spec)
        holder = validate_and_parse_yaml_upstream_config(
            namespace_id=namespace_id,
            upstream_id=upstream_id,
            spec_pb=req_pb.spec,
            curr_spec_pb=upstream_pb.spec,
            ctx=ctx)
        spec_has_changed = upstream_pb.spec != req_pb.spec
        if spec_has_changed:
            if (namespace_pb.spec.easy_mode_settings.l7_upstream_macro_only.value and
                    upstream_pb.spec.yandex_balancer.mode != req_pb.spec.yandex_balancer.mode and
                    req_pb.spec.yandex_balancer.mode != req_pb.spec.yandex_balancer.EASY_MODE2):
                raise rpc.exceptions.ForbiddenError('Only easy mode upstreams are allowed in this namespace')
            if req_pb.spec.yandex_balancer.type != upstream_pb.spec.yandex_balancer.type:
                raise rpc.exceptions.BadRequestError('"type": can not be changed')
            if not req_pb.spec.yandex_balancer.yaml:
                # validate yaml size after calling validate_and_parse_yaml_upstream_config so now it has 'yaml'
                validate_upstream_yaml_size(req_pb.spec.yandex_balancer.yaml,
                                            field_name=u'spec.yandex_balancer.yaml')

            rev_index_pbs = list(upstream_pb.meta.indices)
            rev_index_pbs.append(get_rev_index_pb(namespace_id, upstream_pb.spec, holder))
    else:
        spec_has_changed = False

    if not spec_has_changed and not auth_has_changed:
        return api_pb2.UpdateUpstreamResponse(upstream=upstream_pb)

    upstream_pb = IDao.instance().update_upstream(
        namespace_id=namespace_id,
        upstream_id=upstream_id,
        version=req_pb.meta.version,
        comment=req_pb.meta.comment,
        login=auth_subject.login,
        updated_spec_pb=req_pb.spec if spec_has_changed else None,
        updated_auth_pb=req_pb.meta.auth if auth_has_changed else None,
        rev_index_pbs=rev_index_pbs,
    )
    balancer_state_pbs = IAwacsCache.instance().list_all_balancer_states(namespace_id=namespace_id)
    upstream_pb = apicache.copy_per_balancer_statuses(entity_pb=upstream_pb,
                                                      balancer_state_pbs=balancer_state_pbs)
    return api_pb2.UpdateUpstreamResponse(upstream=upstream_pb)


@upstream_service_bp.method('RemoveUpstream',
                            request_type=api_pb2.RemoveUpstreamRequest,
                            response_type=api_pb2.RemoveUpstreamResponse,
                            is_destructive=True)
def remove_upstream(req_pb, auth_subject):
    upstream.validate_request(req_pb)
    namespace_id = req_pb.namespace_id
    upstream_id = req_pb.id

    zk = IZkStorage.instance()

    upstream_pb = zk.must_get_upstream(namespace_id, upstream_id)
    namespace_pb = zk.must_get_namespace(namespace_id)
    try:
        authorize_remove(upstream_pb, auth_subject)
    except rpc.exceptions.ForbiddenError:
        authorize_remove(upstream_pb, auth_subject, acl=get_acl(namespace_pb))
    forbid_action_during_namespace_order(namespace_pb, auth_subject)

    updated_spec_pb = clone_pb(upstream_pb.spec)
    updated_spec_pb.deleted = True

    IDao.instance().update_upstream(
        namespace_id=namespace_id,
        upstream_id=upstream_id,
        version=req_pb.version,
        comment='Marked as deleted by {}'.format(auth_subject.login),
        login=auth_subject.login,
        updated_spec_pb=updated_spec_pb
    )

    return api_pb2.RemoveUpstreamResponse()


@upstream_service_bp.method('GetUpstreamRevision',
                            request_type=api_pb2.GetUpstreamRevisionRequest,
                            response_type=api_pb2.GetUpstreamRevisionResponse)
def get_upstream_revision(req_pb, _):
    upstream.validate_request(req_pb)
    rev_id = req_pb.id

    rev_pb = db.IMongoStorage.instance().must_get_upstream_rev(rev_id)
    return api_pb2.GetUpstreamRevisionResponse(revision=rev_pb)


@upstream_service_bp.method('ListUpstreamRevisions',
                            request_type=api_pb2.ListUpstreamRevisionsRequest,
                            response_type=api_pb2.ListUpstreamRevisionsResponse,
                            max_in_flight=5)
def list_upstream_revisions(req_pb, _):
    upstream.validate_request(req_pb)
    upstream_id = req_pb.id
    namespace_id = req_pb.namespace_id

    mongo_storage = db.IMongoStorage.instance()
    zk = IZkStorage.instance()
    zk.must_get_upstream(namespace_id, upstream_id)

    skip = req_pb.skip or None
    limit = req_pb.limit or None
    revs = mongo_storage.list_upstream_revs(namespace_id, upstream_id, skip=skip, limit=limit)
    return api_pb2.ListUpstreamRevisionsResponse(revisions=revs.items, total=revs.total)


@upstream_service_bp.method('GetUpstreamRemovalChecks',
                            request_type=api_pb2.GetUpstreamRemovalChecksRequest,
                            response_type=api_pb2.GetUpstreamRemovalChecksResponse)
def get_upstream_removal_checks(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.GetUpstreamRemovalChecksRequest
    """
    upstream.validate_request(req_pb)
    namespace_id = req_pb.namespace_id
    upstream_id = req_pb.id

    zk = IZkStorage.instance()
    upstream_pb = zk.must_get_upstream(namespace_id, upstream_id)

    resp_pb = api_pb2.GetUpstreamRemovalChecksResponse()
    resp_pb.checks.extend(upstream.get_upstream_removal_checks(upstream_pb))
    return resp_pb


# AWACS-1326: for first initialization only
@upstream_service_bp.method('UpdateUpstreamIndices',
                            request_type=api_pb2.UpdateUpstreamIndicesRequest,
                            response_type=api_pb2.UpdateUpstreamIndicesResponse)
def update_upstream_indices(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.UpdateUpstreamIndicesRequest
    :rtype api_pb2.UpdateUpstreamIndicesResponse
    """
    if auth_subject.login not in appconfig.get_value('run.root_users', default=()):
        raise rpc.exceptions.ForbiddenError('Method is only allowed for roots.')

    if not req_pb.id:
        raise rpc.exceptions.BadRequestError('No "id" specified.')
    if not req_pb.namespace_id:
        raise rpc.exceptions.BadRequestError('No "namespace_id" specified.')

    namespace_id = req_pb.namespace_id
    upstream_id = req_pb.id
    full_up_id = (namespace_id, upstream_id)

    zk = IZkStorage.instance()
    mongo = db.IMongoStorage.instance()
    cache = IAwacsCache.instance()

    upstream_pb = zk.must_get_upstream(namespace_id, upstream_id)

    balancer_active_rev_ctimes = []
    balancer_state_pbs = cache.list_all_balancer_states(namespace_id)
    for state_pb in balancer_state_pbs:
        active_rev_ctime = None
        entity = L7BalancerStateHandler(state_pb).select_upstream(full_up_id)
        if entity:
            rev_status = entity.select_active_rev()
            if rev_status is not None and (
                active_rev_ctime is None or rev_status.pb.ctime.ToMicroseconds() > active_rev_ctime
            ):
                active_rev_ctime = rev_status.pb.ctime.ToMicroseconds()

        if active_rev_ctime is not None:
            balancer_active_rev_ctimes.append(active_rev_ctime)

    active_rev_ctime = None
    if balancer_active_rev_ctimes:
        active_rev_ctime = min(balancer_active_rev_ctimes)
    rev_pbs = []
    rev_ids = set()

    for state_pb in balancer_state_pbs:
        entity = L7BalancerStateHandler(state_pb).select_upstream(full_up_id)
        if entity:
            for rev_status in entity.list_revs():
                if rev_status.pb.revision_id in rev_ids:
                    continue
                if active_rev_ctime is None or rev_status.pb.ctime.ToMicroseconds() >= active_rev_ctime:
                    rev_pb = mongo.must_get_upstream_rev(rev_status.pb.revision_id)
                    rev_pbs.append(rev_pb)
                    rev_ids.add(rev_status.pb.revision_id)

    if not rev_pbs:
        raise rpc.exceptions.InternalError('No upstream revisions present in state')

    rev_index_pbs = []
    for rev_pb in rev_pbs:
        if active_rev_ctime is None or rev_pb.meta.ctime.ToMicroseconds() >= active_rev_ctime:
            ind_pb = get_rev_index_pb(namespace_id, rev_pb.spec)
            ind_pb.id = rev_pb.meta.id
            ind_pb.ctime.CopyFrom(rev_pb.meta.ctime)
            rev_index_pbs.append(ind_pb)

    IDao.instance().update_indices(
        object_name='upstream',
        id=upstream_id,
        namespace_id=namespace_id,
        version=upstream_pb.meta.version,
        rev_index_pbs=rev_index_pbs)

    return api_pb2.UpdateUpstreamIndicesResponse()
