# coding: utf-8
import contextlib
import random

import datetime
import flask
import gevent
import six
from google.protobuf import json_format
from sepelib.core import config
from six.moves import http_client as httplib

from awacs.lib import rpc, nannyclient
from awacs.lib.order_processor.model import is_order_in_progress
from awacs.model import errors as modelerrors
from awacs.model.l3_balancer.errors import L3ConfigValidationError
from infra.awacs.proto import api_pb2, model_pb2
from infra.swatlib.rpc.blueprint import make_response


class GeventIdler(object):
    __slots__ = ('data',)

    def __init__(self, data):
        self.data = data

    def toDict(self):
        """
        ujson supports toDict hook:
        https://github.com/ultrajson/ultrajson/blob/13e2ac7eeaae453d26c1adf732ab7cfc38e7cb30/python/objToJSON.c#L595
        It's called during JSON encoding; we use it to give up CPU and switch gevent contexts from time to time.
        """
        gevent.idle()
        return self.data


class GeventFriendlyPrinter(json_format._Printer):
    MIN_IDLE_PERIOD = 200
    MAX_IDLE_PERIOD = 500

    def __init__(self, *args, **kwargs):
        super(GeventFriendlyPrinter, self).__init__(*args, **kwargs)
        self._tick = 0
        self._period = random.randint(self.MIN_IDLE_PERIOD, self.MAX_IDLE_PERIOD)

    def _MessageToJsonObject(self, message):
        rv = super(GeventFriendlyPrinter, self)._MessageToJsonObject(message)
        if not isinstance(rv, dict):
            return rv
        
        self._tick += 1
        if self._tick % self._period == 0:
            return GeventIdler(rv)
        else:
            return rv


class AwacsBlueprint(rpc.blueprint.HttpRpcBlueprint):
    status_msg = api_pb2.Status

    def __init__(self, name, import_name, url_prefix):
        super(AwacsBlueprint, self).__init__(name, import_name, url_prefix=url_prefix, status_msg=self.status_msg)
        self.REJECTED_METHOD_STATUS = self.status_msg(status=u'Service Unavailable',
                                                      code=httplib.SERVICE_UNAVAILABLE,
                                                      message=u'This method is temporarily not available',
                                                      reason=u'Please contact support for details')
        self.REJECTED_USER_STATUS = self.status_msg(status=u'Service Unavailable',
                                                    code=httplib.SERVICE_UNAVAILABLE,
                                                    message=u'You are temporarily banned from calling this method',
                                                    reason=u'Please contact support for details')

    @classmethod
    def _get_pb_json_printer_cls(cls):
        return GeventFriendlyPrinter

    def _maybe_reject_method(self, method):
        """
        :type method: six.text_type
        :rtype: Optional[flask.Response]
        """
        if method in config.get_value(u'run.banned_methods', default=[]):
            return make_response(self.REJECTED_METHOD_STATUS,
                                 accept_mimetypes=flask.request.accept_mimetypes,
                                 status_msg=self.status_msg,
                                 status=httplib.SERVICE_UNAVAILABLE,
                                 headers={b'Retry-After': 60})
        return None

    def _maybe_reject_user(self, auth_subject):
        """
        :type auth_subject: swatlib.rpc.authentication.AuthSubject
        :rtype: Optional[flask.Response]
        """
        if auth_subject.login in config.get_value(u'run.banned_users', default=[]):
            return make_response(self.REJECTED_USER_STATUS,
                                 accept_mimetypes=flask.request.accept_mimetypes,
                                 status_msg=self.status_msg,
                                 status=httplib.SERVICE_UNAVAILABLE,
                                 headers={b'Retry-After': 60})
        return None

    def _maybe_write_extended_log(self, started_at, finished_at, protobuf_request, auth_subject, method_name,
                                  is_method_destructive=False, sent_at=None):
        """
        :type started_at: float
        :type finished_at: float
        :param protobuf_request: Protobuf message
        :type auth_subject: swatlib.rpc.authentication.AuthSubject
        :type method_name: six.text_type
        :type is_method_destructive: bool
        :type sent_at: float | None
        """
        enabled = config.get_value(u'web.extended_access_log.enabled', False)
        if enabled:
            extended_log_slow_answer_ms = config.get_value(u'web.extended_access_log.slow_answers_ms', 300)
            always_log_destructive_methods = config.get_value(u'web.extended_access_log.always_log_destructive_methods',
                                                              False)
            duration_ms = int((finished_at - started_at) * 1000)
            need_to_log = duration_ms >= extended_log_slow_answer_ms or (always_log_destructive_methods and
                                                                         is_method_destructive)
            if need_to_log:
                login = auth_subject.login if auth_subject else 'anonymous'
                message = json_format.MessageToJson(protobuf_request)
                request_body_max_length = config.get_value(u'web.extended_access_log.request_body_max_length', 2 << 10)
                if len(message) > request_body_max_length:
                    message = message[:request_body_max_length] + '...'
                started = datetime.datetime.utcfromtimestamp(started_at).time().replace(microsecond=0)
                finished = datetime.datetime.utcfromtimestamp(finished_at).time().replace(microsecond=0)
                if sent_at is None:
                    self.log.info(u'started: %s finished: %s duration: %dms %s by %s@\n%s',
                                  started, finished, duration_ms,
                                  method_name, login, message)
                else:
                    wait_ms = int((started_at - sent_at) * 1000)
                    sent = datetime.datetime.utcfromtimestamp(sent_at).time().replace(microsecond=0)
                    self.log.info(u'sent: %s started: %s finished: %s wait: %dms duration: %dms %s by %s@\n%s',
                                  sent, started, finished, wait_ms, duration_ms,
                                  method_name, login, message)

    @classmethod
    @contextlib.contextmanager
    def translate_errors(cls):
        try:
            yield
        except modelerrors.NotFoundError as e:
            raise rpc.exceptions.NotFoundError(six.text_type(e))
        except modelerrors.ConflictError as e:
            raise rpc.exceptions.ConflictError(six.text_type(e))
        except modelerrors.IntegrityError as e:
            raise rpc.exceptions.BadRequestError(six.text_type(e))
        except modelerrors.ValidationError as e:
            raise rpc.exceptions.BadRequestError(six.text_type(e))
        except L3ConfigValidationError as e:
            raise rpc.exceptions.BadRequestError(six.text_type(e))
        except modelerrors.InternalError as e:
            raise rpc.exceptions.InternalError(six.text_type(e))


def validate_nanny_service_nonexistence(nanny_service_id):
    """
    :type nanny_service_id: six.text_type
    """
    try:
        nannyclient.INannyClient.instance().get_service_auth_attrs(nanny_service_id)
    except nannyclient.NannyApiRequestException as e:
        if e.response is not None and e.response.status_code == httplib.NOT_FOUND:
            return
        else:
            raise rpc.exceptions.InternalError(
                u'Something went wrong during communicating with Nanny API, '
                u'please try again later')
    raise rpc.exceptions.BadRequestError(
        u'Nanny service "{}" already exists. If it was created to fulfill previous cancelled namespace order, '
        u'please consider removing it manually.'.format(nanny_service_id))


def forbid_action_during_namespace_order(namespace_pb, auth_subject):
    """
    :param auth_subject:
    :type namespace_pb: model_pb2.Namespace
    :type auth_subject: awacs.lib.rpc.authentication.AuthSubject
    :raises rpc.exceptions.ForbiddenError
    """
    if not namespace_pb:
        return
    login = auth_subject.login
    if not config.get_value(u'run.auth', default=True) or login in config.get_value(u'run.root_users', default=()):
        return
    if is_order_in_progress(namespace_pb):
        raise rpc.exceptions.ForbiddenError(u'Cannot do this while namespace order is in progress')


def validate_namespace_total_objects_count(object_name, current_count, namespace_pb):
    """
    :type object_name: six.text_type
    :type current_count: int
    :type namespace_pb: model_pb2.Namespace
    :raises: rpc.exceptions.BadRequestError
    """
    max_count = None
    if namespace_pb.spec.object_upper_limits.HasField(object_name):
        max_count = getattr(namespace_pb.spec.object_upper_limits, object_name).value
    if max_count is None:
        max_count = config.get_value('common_objects_limits.' + object_name, None)
    if max_count is None:
        return
    if current_count >= max_count:
        raise rpc.exceptions.BadRequestError(
            'Exceeded limit of {}s in the namespace: {}'.format(object_name, max_count))
