# coding: utf-8
import re

import nanny_rpc_client
from awacs.lib import rpc
from awacs.lib.rpc import exceptions
from awacs.lib.ypliterpcclient import IYpLiteRpcClient
from awacs.model.util import make_system_endpoint_set_id
from awacs.model.cache import IAwacsCache
from awacs.model.zk import IZkStorage
from sepelib.core import config
from infra.awacs.proto import api_pb2, model_pb2
from yp_lite_ui_repo import endpoint_sets_api_pb2
from .util import ID_RE, COMPAT_ID_RE, NAMESPACE_ID_RE, validate_auth_pb


GENCFG_GROUPS_LIMIT = 20
NANNY_SNAPSHOTS_LIMIT = 20
YP_ENDPOINT_SETS_LIMIT = 20
BALANCERS_LIMIT = 20

ALLOWED_YP_CLUSTERS = frozenset(('sas', 'man', 'vla', 'myt', 'iva', 'test_sas', 'sas_test', 'man_pre'))


def validate_port(port_pb, field_name='port'):
    """
    :param port_pb: awacs.proto.model_pb2.Port
    :type field_name: six.text_type
    :raises: exceptions.BadRequestError
    """
    if port_pb.policy == port_pb.SHIFT:
        if not (0 < port_pb.shift < 65535):
            raise exceptions.BadRequestError('if "SHIFT" policy selected, "{}.shift" is required and must be between '
                                             '0 and 65535'.format(field_name))
    if port_pb.policy == port_pb.OVERRIDE:
        if not (0 < port_pb.override < 65535):
            raise exceptions.BadRequestError('if "OVERRIDE" policy selected, "{}.override" is required and '
                                             'must be between 0 and 65535'.format(field_name))


def validate_selector(selector_pb, namespace_id, field_name='spec.selector'):
    """
    :type selector_pb: model_pb2.BackendSelector
    :type namespace_id: six.text_type
    :type field_name: six.text_type
    :raises: exceptions.BadRequestError
    """
    if selector_pb.type == selector_pb.NONE:
        raise exceptions.BadRequestError('"{0}.type" must be specified'.format(field_name))
    elif selector_pb.type == selector_pb.MANUAL:
        pass
    elif selector_pb.type == selector_pb.NANNY_SNAPSHOTS:
        if not selector_pb.nanny_snapshots:
            raise exceptions.BadRequestError('"{0}.nanny_snapshots" must be specified if '
                                             '"{0}.type" is set to NANNY_SNAPSHOTS'.format(field_name))
    elif selector_pb.type == selector_pb.GENCFG_GROUPS:
        if not selector_pb.gencfg_groups:
            raise exceptions.BadRequestError('"{0}.gencfg_groups" must be specified if '
                                             '"{0}.type" is set to GENCFG_GROUPS'.format(field_name))
    elif selector_pb.type in (selector_pb.YP_ENDPOINT_SETS, selector_pb.YP_ENDPOINT_SETS_SD):
        if not selector_pb.yp_endpoint_sets:
            raise exceptions.BadRequestError(
                '"{0}.yp_endpoint_sets" must be specified if "{0}.type" '
                'is set to YP_ENDPOINT_SETS or YP_ENDPOINT_SETS_SD'.format(field_name))
    elif selector_pb.type == selector_pb.BALANCERS:
        if not selector_pb.balancers:
            raise exceptions.BadRequestError('"{0}.balancers" must be specified if '
                                             '"{0}.type" is set to BALANCERS'.format(field_name))
        zk = IZkStorage.instance()
        yp_lite_rpc_client = IYpLiteRpcClient.instance()
        for i, selector_balancer_pb in enumerate(selector_pb.balancers):
            balancer_id = selector_balancer_pb.id
            balancer_pb = zk.must_get_balancer(namespace_id, balancer_id)
            if not balancer_pb.meta.location.yp_cluster:
                raise exceptions.BadRequestError(
                    u'"{}.balancers[{}]": BALANCERS selectors can be used only with '
                    u'YP.lite-powered balancer with defined location'.format(field_name, i))
            if balancer_pb.spec.deleted:
                raise exceptions.BadRequestError(
                    u'"{}.balancers[{}]": cannot create backend for deleted balancer'.format(field_name, i))
            if not does_system_endpoint_set_exist(balancer_pb, yp_lite_rpc_client):
                raise exceptions.BadRequestError(
                    u'"{}.balancers[{}]": balancer doesn\'t have system endpoint set'.format(field_name, i))

    if len(selector_pb.nanny_snapshots) > NANNY_SNAPSHOTS_LIMIT:
        raise exceptions.BadRequestError(
            'number of items in "{0}.nanny_snapshots" ({1}) exceeds allowed limit of {2}'.format(
                field_name, len(selector_pb.nanny_snapshots), NANNY_SNAPSHOTS_LIMIT))

    if len(selector_pb.gencfg_groups) > GENCFG_GROUPS_LIMIT:
        raise exceptions.BadRequestError(
            'number of items in "{0}.gencfg_groups" ({1}) exceeds allowed limit of {2}'.format(
                field_name, len(selector_pb.gencfg_groups), GENCFG_GROUPS_LIMIT))

    if len(selector_pb.yp_endpoint_sets) > YP_ENDPOINT_SETS_LIMIT:
        raise exceptions.BadRequestError(
            'number of items in "{0}.yp_endpoint_sets" ({1}) exceeds allowed limit of {2}'.format(
                field_name, len(selector_pb.yp_endpoint_sets), YP_ENDPOINT_SETS_LIMIT))

    if len(selector_pb.balancers) > BALANCERS_LIMIT:
        raise exceptions.BadRequestError(
            'number of items in "{0}.balancers" ({1}) exceeds allowed limit of {2}'.format(
                field_name, len(selector_pb.balancers), BALANCERS_LIMIT))

    if selector_pb.type == selector_pb.YP_ENDPOINT_SETS_SD:
        if selector_pb.HasField('port') and selector_pb.port.policy != selector_pb.port.KEEP:
            raise exceptions.BadRequestError(
                '"{}.port": must be set to KEEP -- '
                'YP_ENDPOINT_SETS_SD do not support port overriding'.format(field_name))

    for i, gencfg_group_pb in enumerate(selector_pb.gencfg_groups):  # type: int, model_pb2.BackendSelector.GencfgGroup
        item_field_name = '{0}.gencfg_groups[{1}]'.format(field_name, i)
        if not gencfg_group_pb.name:
            raise exceptions.BadRequestError('"{}.name": is required'.format(item_field_name))
        if not gencfg_group_pb.version:
            raise exceptions.BadRequestError('"{}.version": is required'.format(item_field_name))
        if gencfg_group_pb.HasField('port'):
            validate_port(gencfg_group_pb.port, field_name=item_field_name + '.port')
        if gencfg_group_pb.version == 'trunk':
            raise exceptions.BadRequestError(
                '"{}.version": "trunk" is not supported yet'.format(item_field_name))

    for i, nanny_snapshot_pb in enumerate(
            selector_pb.nanny_snapshots):  # type: int, model_pb2.BackendSelector.NannySnapshot
        item_field_name = '{0}.nanny_snapshots[{1}]'.format(field_name, i)
        if not nanny_snapshot_pb.service_id:
            raise exceptions.BadRequestError('"{}.service_id": is required'.format(item_field_name))
        if nanny_snapshot_pb.HasField('port'):
            validate_port(nanny_snapshot_pb.port, field_name=item_field_name + '.port')

    for i, yp_endpoint_set_pb in enumerate(
            selector_pb.yp_endpoint_sets):  # type: int, model_pb2.BackendSelector.YpEndpointSet
        item_field_name = '{0}.yp_endpoint_sets[{1}]'.format(field_name, i)
        if not yp_endpoint_set_pb.cluster:
            raise exceptions.BadRequestError('"{}.cluster": is required'.format(item_field_name))
        if yp_endpoint_set_pb.cluster not in ALLOWED_YP_CLUSTERS:
            raise exceptions.BadRequestError('"{}.cluster" must be one of the following: "{}"'.format(
                item_field_name, '", "'.join(ALLOWED_YP_CLUSTERS)))
        if not yp_endpoint_set_pb.endpoint_set_id:
            raise exceptions.BadRequestError('"{}.endpoint_set_id": is required'.format(item_field_name))

        if selector_pb.type == selector_pb.YP_ENDPOINT_SETS_SD:
            if (yp_endpoint_set_pb.HasField('port') and
                    yp_endpoint_set_pb.port.policy != yp_endpoint_set_pb.port.KEEP):
                raise exceptions.BadRequestError(
                    '"{}.port": must be set to KEEP -- '
                    'YP_ENDPOINT_SETS_SD do not support port overriding'.format(item_field_name))
            if (yp_endpoint_set_pb.HasField('weight') and
                    yp_endpoint_set_pb.weight.policy != yp_endpoint_set_pb.weight.KEEP):
                raise exceptions.BadRequestError(
                    '"{}.weight": must be set to KEEP -- '
                    'YP_ENDPOINT_SETS_SD do not support weight overriding'.format(item_field_name))
        else:
            if yp_endpoint_set_pb.HasField('port'):
                validate_port(yp_endpoint_set_pb.port, field_name=item_field_name + '.port')

    if selector_pb.type == selector_pb.NANNY_SNAPSHOTS:
        for i, snapshot_pb in enumerate(selector_pb.nanny_snapshots):
            if snapshot_pb.snapshot_id:
                raise exceptions.BadRequestError(
                    u'"{0}.nanny_snapshots[{1}].snapshot_id" must be not specified, '
                    u'see SWAT-6604 for details'.format(field_name, i))


def does_system_endpoint_set_exist(balancer_pb, yp_lite_rpc_client):
    es_id = make_system_endpoint_set_id(balancer_pb.spec.config_transport.nanny_static_file.service_id)
    req_pb = endpoint_sets_api_pb2.GetEndpointSetRequest(
        id=es_id,
        cluster=balancer_pb.meta.location.yp_cluster
    )
    try:
        yp_lite_rpc_client.get_endpoint_set(req_pb)
        return True
    except nanny_rpc_client.exceptions.NotFoundError:
        return False


def validate_spec(spec_pb, namespace_id, field_name='spec'):
    """
    :type spec_pb: model_pb2.BackendSpec
    :type namespace_id: six.text_type
    :type field_name: six.text_type
    :raises: exceptions.BadRequestError
    """
    if not spec_pb.HasField('selector'):
        raise exceptions.BadRequestError('"{}.selector" must be set'.format(field_name))

    selector_pb = spec_pb.selector
    validate_selector(selector_pb, namespace_id=namespace_id, field_name=field_name + '.selector')


def validate_create_or_update_request(req_pb, id_re=ID_RE):
    """
    :type req_pb: api_pb2.CreateBackendRequest | api_pb2.UpdateBackendRequest
    :type id_re: re.Pattern
    :raises: exceptions.BadRequestError
    """
    if not req_pb.HasField('meta'):
        raise exceptions.BadRequestError('"meta" must be set')
    if not req_pb.meta.id:
        raise exceptions.BadRequestError('"meta.id" must be set')
    if not id_re.match(req_pb.meta.id):
        raise exceptions.BadRequestError('"meta.id" is not valid')
    if not req_pb.meta.namespace_id:
        raise exceptions.BadRequestError('"meta.namespace_id" must be set')
    if not NAMESPACE_ID_RE.match(req_pb.meta.namespace_id):
        raise exceptions.BadRequestError('"meta.namespace_id" is not valid')


def validate_create_backend_request(req_pb):
    """
    :type req_pb: api_pb2.CreateBackendRequest
    :raises: exceptions.BadRequestError
    """
    validate_create_or_update_request(req_pb)
    if not req_pb.meta.HasField('auth'):
        raise exceptions.BadRequestError('"meta.auth" must be set')
    validate_auth_pb(req_pb.meta.auth, field_name='meta.auth')
    if not req_pb.HasField('spec'):
        raise exceptions.BadRequestError('"spec" must be set')
    if req_pb.spec.selector.type == model_pb2.BackendSelector.BALANCERS and not req_pb.meta.is_system.value:
        raise exceptions.BadRequestError('"meta.is_system.value" must be set to "true" for BALANCERS backend')
    validate_spec(req_pb.spec, namespace_id=req_pb.meta.namespace_id)
    selector_pb = req_pb.spec.selector
    if selector_pb.type in (selector_pb.YP_ENDPOINT_SETS, selector_pb.YP_ENDPOINT_SETS_SD):
        for yp_es_pb in selector_pb.yp_endpoint_sets:
            if config.get_value('run.unavailable_clusters.{}.disable_backends_creation'
                                .format(yp_es_pb.cluster.lower()), False):
                raise exceptions.BadRequestError('Creating backends in {} is disabled'.format(yp_es_pb.cluster))


def validate_update_backend_request(req_pb):
    """
    :type req_pb: api_pb2.UpdateBackendRequest
    :raises: exceptions.BadRequestError
    """
    validate_create_or_update_request(req_pb, id_re=COMPAT_ID_RE)
    if not req_pb.meta.version:
        raise exceptions.BadRequestError('"meta.version" must be set')
    if not req_pb.HasField('spec') and not req_pb.meta.HasField('auth') and not req_pb.meta.HasField('is_system'):
        raise exceptions.BadRequestError('at least one of the "spec", or "meta.auth", '
                                         'or "meta.is_system" fields must be present')
    if req_pb.HasField('spec'):
        validate_spec(req_pb.spec, namespace_id=req_pb.meta.namespace_id)
    if req_pb.meta.HasField('auth'):
        validate_auth_pb(req_pb.meta.auth, field_name='meta.auth')


def validate_list_backends_request(req_pb):
    """
    :type req_pb: api_pb2.ListBackendsRequest
    :raises: exceptions.BadRequestError
    """
    if not req_pb.namespace_id and not req_pb.HasField('query'):
        raise exceptions.BadRequestError('Either "namespace_id" or "query" must be specified.')
    if req_pb.balancer_id and not req_pb.namespace_id:
        raise exceptions.BadRequestError('"namespace_id" must specified together with "balancer_id".')
    if req_pb.HasField('field_mask'):
        if not req_pb.field_mask.IsValidForDescriptor(model_pb2.Backend.DESCRIPTOR):
            raise exceptions.BadRequestError('"field_mask" is not valid')
    if req_pb.HasField('query') and req_pb.query.only_system and req_pb.query.exclude_system:
        raise exceptions.BadRequestError('"query": options "only_system" and "exclude_system" '
                                         'cannot be used at the same time')
    if req_pb.HasField('query') and req_pb.query.id_regexp:
        try:
            re.compile(req_pb.query.id_regexp)
        except Exception as e:
            raise exceptions.BadRequestError('"query.id_regexp" contains invalid regular expression: {}'.format(e))


def validate_request(req_pb):
    """
    :raises: exceptions.BadRequestError
    """
    if isinstance(req_pb, api_pb2.ListBackendsRequest):
        validate_list_backends_request(req_pb)
    elif isinstance(req_pb, (api_pb2.RemoveBackendRequest,
                             api_pb2.GetBackendRequest,
                             api_pb2.GetBackendRemovalChecksRequest,
                             api_pb2.ListBackendRevisionsRequest)):
        if not req_pb.id:
            raise exceptions.BadRequestError('No "id" specified.')
        if not req_pb.namespace_id:
            raise exceptions.BadRequestError('No "namespace_id" specified.')
        if isinstance(req_pb, api_pb2.RemoveBackendRequest):
            if not req_pb.version:
                raise exceptions.BadRequestError('No "version" specified.')
    elif isinstance(req_pb, api_pb2.GetBackendRevisionRequest):
        if not req_pb.id:
            raise exceptions.BadRequestError('No "id" specified.')
    elif isinstance(req_pb, api_pb2.UpdateBackendRequest):
        validate_update_backend_request(req_pb)
    elif isinstance(req_pb, api_pb2.CreateBackendRequest):
        validate_create_backend_request(req_pb)
    else:
        raise RuntimeError('Incorrect `req_pb` type: {}'.format(req_pb.__class__.__name__))


def validate_selector_type(namespace_id, backend_id, curr_selector_type, updated_selector_type, comment):
    sd_suffix_has_been_removed = (
            curr_selector_type == model_pb2.BackendSelector.YP_ENDPOINT_SETS_SD and
            updated_selector_type == model_pb2.BackendSelector.YP_ENDPOINT_SETS)
    if sd_suffix_has_been_removed and u'changed using awacsctl' in comment:
        raise rpc.exceptions.BadRequestError(
            u'awacsctl would change YP_ENDPOINT_SETS_SD to YP_ENDPOINT_SETS, '
            u'please see SWAT-6425 for details')

    sd_suffix_has_been_added = (
            curr_selector_type == model_pb2.BackendSelector.YP_ENDPOINT_SETS and
            updated_selector_type == model_pb2.BackendSelector.YP_ENDPOINT_SETS_SD)
    dns_records_for_backend = IAwacsCache.instance().list_full_dns_record_ids_for_backend(namespace_id, backend_id)
    if sd_suffix_has_been_added and dns_records_for_backend:
        raise rpc.exceptions.BadRequestError(
            u'Cannot set YP_ENDPOINT_SETS_SD type for backends that are used in DNS records: "{}"'.format(
                u', '.join(u'{}:{}'.format(ns_id, d_id) for ns_id, d_id in dns_records_for_backend)
            ))


def balancer_usage_removal_check(backend_pb):
    check_pb = model_pb2.RemovalCheck()
    check_pb.type = check_pb.REFERENCES
    c = IAwacsCache().instance()
    used_in_external_balancers = False
    for namespace_id, balancer_id in c.list_full_balancer_ids_for_backend(namespace_id=backend_pb.meta.namespace_id,
                                                                          backend_id=backend_pb.meta.id):
        if namespace_id != backend_pb.meta.namespace_id:
            used_in_external_balancers = True
        check_pb.state = check_pb.FAILED
        check_pb.message = 'Backend is used in balancers'
        ref_pb = check_pb.references.references.add(namespace_id=namespace_id, id=balancer_id)
        ref_pb.object_type = ref_pb.REMOVAL_REF_BALANCER
        check_pb.references.show_map_link = True
    if used_in_external_balancers:
        check_pb.references.show_map_link = False
    if check_pb.state == check_pb.FAILED:
        return check_pb

    check_pb.state = check_pb.PASSED
    check_pb.message = 'Backend is not used in balancers'
    return check_pb


def l3_balancer_usage_removal_check(backend_pb):
    check_pb = model_pb2.RemovalCheck()
    check_pb.type = check_pb.REFERENCES
    c = IAwacsCache().instance()
    for namespace_id, l3_balancer_id in c.list_full_l3_balancer_ids_for_backend(namespace_id=backend_pb.meta.namespace_id,
                                                                                backend_id=backend_pb.meta.id):
        check_pb.state = check_pb.FAILED
        check_pb.message = 'Backend is used in L3 balancers'
        ref_pb = check_pb.references.references.add(namespace_id=namespace_id, id=l3_balancer_id)
        ref_pb.object_type = ref_pb.REMOVAL_REF_L3_BALANCER
    if check_pb.state == check_pb.FAILED:
        return check_pb

    check_pb.state = check_pb.PASSED
    check_pb.message = 'Backend is not used in L3 balancers'
    return check_pb


def dns_record_usage_removal_check(backend_pb):
    check_pb = model_pb2.RemovalCheck()
    check_pb.type = check_pb.REFERENCES
    c = IAwacsCache().instance()
    for namespace_id, dns_record_id in c.list_full_dns_record_ids_for_backend(namespace_id=backend_pb.meta.namespace_id,
                                                                              backend_id=backend_pb.meta.id):
        check_pb.state = check_pb.FAILED
        check_pb.message = 'Backend is used in DNS records'
        ref_pb = check_pb.references.references.add(namespace_id=namespace_id, id=dns_record_id)
        ref_pb.object_type = ref_pb.REMOVAL_REF_DNS_RECORD
    if check_pb.state == check_pb.FAILED:
        return check_pb

    check_pb.state = check_pb.PASSED
    check_pb.message = 'Backend is not used in DNS records'
    return check_pb


def get_backend_removal_checks(backend_pb):
    return [
        balancer_usage_removal_check(backend_pb),
        l3_balancer_usage_removal_check(backend_pb),
        dns_record_usage_removal_check(backend_pb)
    ]
