# coding: utf-8
import logging

import ipaddr
import re
import six
from yp_lite_ui_repo import endpoint_sets_api_pb2

from awacs.lib import nannyclient, yasm_client
from awacs.lib import validators
from awacs.lib.order_processor.model import is_order_in_progress, can_be_cancelled
from awacs.lib.rpc import exceptions
from awacs.lib.ypliterpcclient import IYpLiteRpcClient
from awacs.model import errors, zk, cache, components, apicache, external_clusters
from awacs.model.balancer.container_spec import L3_DECAPSULATOR_TUNNEL_ID
from awacs.model.balancer.operations.create_system_backend import get_create_system_backend_processors
from awacs.model.balancer.operations.migrate_from_gencfg_to_yp_lite import (
    GencfgMigration,
    get_gencfg_migration_processors,
)
from awacs.model.balancer.order import util
from awacs.model.balancer.order.processors import BalancerOrder, get_balancer_order_processors
from awacs.model.balancer.removal.processors import BalancerRemoval
from awacs.model.components import is_removed, is_set, iter_changed_balancer_components
from awacs.model.util import YP_CLUSTERS, GENCFG_DATA_CENTERS
from awacs.web.validation.util import NAMESPACE_ID_RE, validate_auth_pb, validate_nanny_service_slug
from awacs.wrappers.base import Holder
from awacs.wrappers.errors import ValidationError
from infra.awacs.proto import api_pb2, model_pb2


BALANCER_ID_RE = re.compile('^[a-z][a-z0-9-._]{1,69}$')

LOCATION_AFFIX_TEMPLATE = r'(-{0}$)|(-{0}-yp$)|(_{0}$)|(_{0}_yp$)|(^{0}\.)|(^yp\.{0}\.)|(^{0}-yp\.)'

# essentially LOCATION_AFFIX_TEMPLATE but matches in any position of the string, bounded by punctuation,
# using negative lookahead (?!) and lookbehind (?<!)
LOCATION_PART_TEMPLATE = (r'(-{0}(?![a-zA-Z\d]))|(-{0}-yp)|(_{0}_yp)|(_{0}(?![a-zA-Z\d]))|((?<![a-zA-Z\d]){0}\.)|'
                          r'(yp\.{0}\.)|({0}-yp\.)')

log = logging.getLogger(__name__)

MAX_BALANCER_LINES_COUNT = 1000
MAX_BALANCER_CHARS_COUNT = 80000
VALID_BALANCER_CTL_VERSIONS = (
    0,  # base
    3,  # previous allowed
    4,  # current
    5,  # being introduced for st/AWACS-756
    6,  # AWACS-502, AWACS-483
)


def validate_balancer_yaml_size(yml, field_name):
    """
    :type yml: six.text_type
    :type field_name: six.text_type
    """
    chars_count = len(yml)
    if chars_count > MAX_BALANCER_CHARS_COUNT:
        raise exceptions.BadRequestError(
            u'"{}" contains too many characters ({}, allowed limit is {})'.format(
                field_name, chars_count, MAX_BALANCER_CHARS_COUNT))

    lines_count = len(yml.splitlines())
    if lines_count > MAX_BALANCER_LINES_COUNT:
        raise exceptions.BadRequestError(
            u'"{}" contains too many lines ({}, allowed limit is {})'.format(
                field_name, lines_count, MAX_BALANCER_LINES_COUNT))


def _validate(holder_pb):
    """
    :param holder_pb: awacs.proto.model_pb2.Holder
    :raises: ValidationError
    """
    h = Holder(holder_pb)
    h.validate()


def validate_yandex_balancer_config(holder_pb, field_name='spec.yandex_balancer.config'):
    """
    :param holder_pb: model_pb2.Holder
    :param field_name: six.text_type
    :raises: exceptions.BadRequestError
    """
    try:
        _validate(holder_pb)
    except ValidationError as e:
        raise exceptions.BadRequestError('{}: {}'.format(field_name, six.text_type(e)))


def validate_yandex_balancer_spec_pb(spec_pb, field_name='yandex_balancer'):
    """
    :param spec_pb: model_pb2.YandexBalancerSpec
    :param field_name: six.text_type
    :raises: exceptions.BadRequestError
    """
    if not spec_pb.HasField('config') and not spec_pb.yaml:
        raise exceptions.BadRequestError(
            'At least one of the "{0}.config" or "{0}.yaml" fields must be set'.format(field_name))


OUTBOUND_TUNNEL_ID_RE = r'^[a-z][a-z0-9\-]+$'
OUTBOUND_TUNNEL_MIN_ID_LEN = 3
OUTBOUND_TUNNEL_MAX_ID_LEN = 16

OUTBOUND_TUNNEL_MIN_RULES_COUNT = 1
OUTBOUND_TUNNEL_MAX_RULES_COUNT = 25

OUTBOUND_TUNNELS_MAX_COUNT = 16
INBOUND_TUNNELS_MAX_COUNT = 5
OUTBOUND_TUNNELS_FROM_IP_NETWORK = '5.45.202.0/24'


def validate_balancer_container_spec(container_spec_pb, field_name):
    """
    :type container_spec_pb: model_pb2.BalancerContainerSpec
    :type field_name: six.text_type
    :raises: exceptions.BadRequestError
    """
    if container_spec_pb.requirements:
        raise exceptions.BadRequestError('{}.requirements: must not be set'.format(field_name))

    if len(container_spec_pb.outbound_tunnels) > OUTBOUND_TUNNELS_MAX_COUNT:
        raise exceptions.BadRequestError('{}.outbound_tunnels: exceeds the limit '
                                         'of {} tunnels'.format(field_name, OUTBOUND_TUNNELS_MAX_COUNT))

    if len(container_spec_pb.inbound_tunnels) > INBOUND_TUNNELS_MAX_COUNT:
        raise exceptions.BadRequestError('{}.inbound_tunnels: exceeds the limit '
                                         'of {} tunnels'.format(field_name, INBOUND_TUNNELS_MAX_COUNT))

    for i, tunnel_pb in enumerate(container_spec_pb.outbound_tunnels):
        tunnel_field_name = '{}.outbound_tunnels[{}]'.format(field_name, i)
        if not tunnel_pb.id:
            raise exceptions.BadRequestError('{}.id: must be set')
        if not re.match(OUTBOUND_TUNNEL_ID_RE, tunnel_pb.id):
            raise exceptions.BadRequestError('{}.id: must match {}'.format(tunnel_field_name, OUTBOUND_TUNNEL_ID_RE))
        if not (OUTBOUND_TUNNEL_MIN_ID_LEN <= len(tunnel_pb.id) <= OUTBOUND_TUNNEL_MAX_ID_LEN):
            raise exceptions.BadRequestError(
                '{}.id: must be at least {} and at most {} '
                'characters long'.format(tunnel_field_name, OUTBOUND_TUNNEL_MIN_ID_LEN, OUTBOUND_TUNNEL_MAX_ID_LEN))

        if tunnel_pb.mode == tunnel_pb.NONE:
            raise exceptions.BadRequestError('{}.mode: must be set'.format(tunnel_field_name))

        if not tunnel_pb.remote_ip:
            raise exceptions.BadRequestError('{}.remote_ip: must be set'.format(tunnel_field_name))
        if not validators.ipv6(tunnel_pb.remote_ip):
            raise exceptions.BadRequestError('"{}.remote_ip": is not a valid IPv6 address'.format(tunnel_field_name))

        if not tunnel_pb.rules:
            raise exceptions.BadRequestError('{}.rules: must be set'.format(tunnel_field_name))

        if not (OUTBOUND_TUNNEL_MIN_RULES_COUNT <= len(tunnel_pb.rules) <= OUTBOUND_TUNNEL_MAX_RULES_COUNT):
            raise exceptions.BadRequestError('{}.rules: must contain at least {} and at most {} rules'.format(
                tunnel_field_name, OUTBOUND_TUNNEL_MIN_RULES_COUNT, OUTBOUND_TUNNEL_MAX_RULES_COUNT))

        for j, rule_pb in enumerate(tunnel_pb.rules):
            rule_field_name = '{}.rules[{}]'.format(tunnel_field_name, j)

            if not rule_pb.from_ip:
                raise exceptions.BadRequestError('{}.from_ip: must be set'.format(rule_field_name))

            if tunnel_pb.mode == tunnel_pb.IPIP6:
                if not validators.ipv4(rule_pb.from_ip):
                    raise exceptions.BadRequestError(
                        '"{}.from_ip": is not a valid IPv4 address'.format(rule_field_name))
                if (tunnel_pb.id != L3_DECAPSULATOR_TUNNEL_ID
                        and (ipaddr.IPAddress(rule_pb.from_ip)
                             not in ipaddr.IPv4Network(OUTBOUND_TUNNELS_FROM_IP_NETWORK))):
                    raise exceptions.BadRequestError(
                        '"{}.from_ip": must be in {}'.format(rule_field_name, OUTBOUND_TUNNELS_FROM_IP_NETWORK))

            if tunnel_pb.mode == tunnel_pb.IP6IP6:
                if not validators.ipv6(rule_pb.from_ip):
                    raise exceptions.BadRequestError(
                        '"{}.from_ip": is not a valid IPv6 address'.format(rule_field_name))

            if rule_pb.to_ip and not validators.ipv6(rule_pb.to_ip):
                raise exceptions.BadRequestError(
                    '"{}.to_ip": is not a valid IPv6 address'.format(rule_field_name))

    if container_spec_pb.virtual_ips:
        if not container_spec_pb.inbound_tunnels:
            raise exceptions.BadRequestError(
                '"{}.inbound_tunnels": must be configured to use virtual_ips'.format(field_name))

    seen_tunnel_types = set()
    for i, tunnel_pb in enumerate(container_spec_pb.inbound_tunnels):
        tunnel_field_name = '{}.inbound_tunnels[{}]'.format(field_name, i)
        if tunnel_pb.HasField('fallback_ip6'):
            if 'fallback_ip6' in seen_tunnel_types:
                raise exceptions.BadRequestError('"{}": duplicate tunnel type "fallback_ip6"'.format(tunnel_field_name))
            seen_tunnel_types.add('fallback_ip6')
        else:
            raise exceptions.BadRequestError('"{}": must not be empty'.format(tunnel_field_name))

    seen_virtual_ips = set()
    for i, virtual_ip_pb in enumerate(container_spec_pb.virtual_ips):
        ip_field_name = '{}.virtual_ips[{}]'.format(field_name, i)

        if not virtual_ip_pb.ip:
            raise exceptions.BadRequestError('"{}.ip": must be present'.format(ip_field_name))

        if not virtual_ip_pb.description:
            raise exceptions.BadRequestError('"{}.description": must be present'.format(ip_field_name))

        try:
            ipaddr.IPAddress(virtual_ip_pb.ip)
        except ValueError:
            raise exceptions.BadRequestError('"{}.ip": "{}" is not a valid IP address'.format(ip_field_name,
                                                                                              virtual_ip_pb.ip, ))

        if virtual_ip_pb.ip in seen_virtual_ips:
            raise exceptions.BadRequestError('"{}": duplicate virtual IP: "{}"'.format(ip_field_name, virtual_ip_pb.ip))
        seen_virtual_ips.add(virtual_ip_pb.ip)


def validate_balancer_component(component, balancer_component_pb, field_name):
    """
    :type component: components.ComponentConfig
    :type balancer_component_pb: model_pb2.BalancerSpec.ComponentsSpec.Component
    :type field_name: six.text_type
    :raises: exceptions.BadRequestError
    """
    state = balancer_component_pb.state
    version = balancer_component_pb.version
    if state == balancer_component_pb.UNKNOWN:
        if version:
            raise exceptions.BadRequestError(u'"{}.version" must not be set if state is UNKNOWN'.format(field_name))
    elif state == balancer_component_pb.REMOVED:
        # REMOVED state is valid for transport for components except INSTANCECTL.
        # But now all of them are required for every balancer, so we will forbid it
        if version:
            raise exceptions.BadRequestError(u'"{}.version" must not be set if state is REMOVED'.format(field_name))
        if not component.can_be_removed_from_spec:
            raise exceptions.BadRequestError(u'"{}.state": '
                                             u'{} component can not be removed'.format(field_name,
                                                                                       component.str_type))
    elif state == balancer_component_pb.SET:
        if not version:
            raise exceptions.BadRequestError(u'"{}.version": is required'.format(field_name))
        component_pb = component.get_component_pb(cache.IAwacsCache.instance(), version)
        if component_pb is None:
            raise exceptions.NotFoundError(u'"{}": component {} with version {} does not exist'
                                           .format(field_name, component.str_type, version))
        if component_pb.status.status != component_pb.status.PUBLISHED:
            raise exceptions.BadRequestError(
                u'"{}": only published components can be used in balancers'.format(field_name))


def validate_balancer_components(components_spec_pb, field_name):
    """
    :type components_spec_pb: model_pb2.BalancerSpec.ComponentsSpec
    :type field_name: six.text_type
    :raises: exceptions.BadRequestError
    """
    for component in components.COMPONENT_CONFIGS:
        c_pb = getattr(components_spec_pb, component.pb_field_name)
        validate_balancer_component(component, c_pb, field_name=field_name + '.' + component.pb_field_name)

    if is_removed(components_spec_pb.instancectl_conf) and not is_set(components_spec_pb.awacslet):
        raise exceptions.BadRequestError(
            u'"{}": AWACSLET must be set if INSTANCECTL_CONF is removed'.format(field_name))

    if is_removed(components_spec_pb.awacslet) and not is_set(components_spec_pb.instancectl_conf):
        raise exceptions.BadRequestError(
            u'"{}": INSTANCECTL_CONF must be set if AWACSLET is removed'.format(field_name))

    if is_set(components_spec_pb.awacslet):
        if not is_set(components_spec_pb.awacslet_get_workers_provider):
            raise exceptions.BadRequestError(
                u'"{}": AWACSLET_GET_WORKERS_PROVIDER must be set if AWACSLET is set'.format(field_name))

        if not is_set(components_spec_pb.instancectl):
            raise exceptions.BadRequestError(
                u'"{}": INSTANCECTL must be set if AWACSLET is set'.format(field_name))

        if not is_removed(components_spec_pb.instancectl_conf):
            raise exceptions.BadRequestError(
                u'"{}": INSTANCECTL_CONF must be removed if AWACSLET is set'.format(field_name))

        if not is_removed(components_spec_pb.get_workers_provider):
            raise exceptions.BadRequestError(
                u'"{}": GET_WORKERS_PROVIDER must be removed if AWACSLET is set'.format(field_name))

        instancectl_config = components.get_component_config(model_pb2.ComponentMeta.INSTANCECTL)
        actual_instancectl_version = instancectl_config.parse_version(components_spec_pb.instancectl.version)
        min_instancectl_version = instancectl_config.parse_version(u'2.8')
        if actual_instancectl_version < min_instancectl_version:
            raise exceptions.BadRequestError(
                u'"{}": INSTANCECTL must be >= 2.8 if AWACSLET is set'.format(field_name))

    if is_set(components_spec_pb.get_workers_provider) and is_set(components_spec_pb.awacslet_get_workers_provider):
        raise exceptions.BadRequestError(
            u'"{}": GET_WORKERS_PROVIDER and AWACSLET_GET_WORKERS_PROVIDER '
            u'must not be set at the same time'.format(field_name))


def validate_location_balancer_id(yp_cluster, _id):
    """
    :param yp_cluster: six.text_type
    :param _id: six.text_type
    :raises: exceptions.BadRequestError
    """
    yp_cluster_l = yp_cluster.lower()
    if re.search(LOCATION_AFFIX_TEMPLATE.format(yp_cluster_l), _id) is None:
        raise errors.ValidationError(
            '"meta.id": balancer in location {0} must start with "{1}.", "yp.{1}." or "{1}-yp." '
            'or end with "-{1}", "_{1}", "-{1}-yp" or "_{1}_yp"'.format(yp_cluster, yp_cluster_l))
    if len(re.findall(LOCATION_PART_TEMPLATE.format('({})'.format('|'.join(YP_CLUSTERS))), _id, re.IGNORECASE)) == 1:
        return  # only one cluster in balancer name
    if len(re.findall(LOCATION_PART_TEMPLATE.format(yp_cluster_l), _id)) > 1:
        raise errors.ValidationError(('"meta.id": duplicate location affix "{}"'.format(yp_cluster_l)))
    other_clusters = set(YP_CLUSTERS) - {yp_cluster, }
    # XXX romanovich@: sorry, sorting and key is just to keep incorrect test_validate_location_balancer_id happy for now
    for cluster in sorted(other_clusters, key=lambda cluster: {'SAS': 1, 'MAN': 2, 'VLA': 3}.get(cluster, 0)):
        cluster = cluster.lower()
        if re.search(LOCATION_PART_TEMPLATE.format(cluster), _id) is not None:
            raise errors.ValidationError('"meta.id": wrong location affix "{}" for balancer in location "{}"'.format(
                cluster, yp_cluster))


def validate_balancer_location(location_pb, _id):
    """
    :param location_pb: model_pb2.BalancerMeta.Location
    :type _id: six.text_type
    :raises: exceptions.BadRequestError
    """
    loc_type = location_pb.type
    yp_cluster = location_pb.yp_cluster
    gencfg_dc = location_pb.gencfg_dc
    azure_cluster = location_pb.azure_cluster
    if not loc_type:
        raise errors.ValidationError('"meta.location" must be specified')
    if loc_type == model_pb2.BalancerMeta.Location.YP_CLUSTER:
        if not yp_cluster:
            raise errors.ValidationError('"meta.location": "yp_cluster" must be specified if "type" == YP_CLUSTER')
        if not yp_cluster.isupper():
            raise errors.ValidationError('"meta.location.yp_cluster": YP cluster must be uppercase'.format(yp_cluster))
        if yp_cluster not in YP_CLUSTERS:
            raise errors.ValidationError('"meta.location.yp_cluster": unknown YP cluster: {}'.format(yp_cluster))
        validate_location_balancer_id(yp_cluster, _id)
    if loc_type == model_pb2.BalancerMeta.Location.GENCFG_DC:
        if not gencfg_dc:
            raise errors.ValidationError('"meta.location": "gencfg_dc" must be specified if "type" == GENCFG_DC')
        if not gencfg_dc.isupper():
            raise errors.ValidationError(
                '"meta.location.gencfg_dc": GENCFG data center must be uppercase'.format(gencfg_dc))
        if gencfg_dc not in GENCFG_DATA_CENTERS:
            raise errors.ValidationError('"meta.location.gencfg_dc": unknown GENCFG data center: {}'.format(gencfg_dc))
    if loc_type == model_pb2.BalancerMeta.Location.AZURE_CLUSTER:
        if not azure_cluster:
            raise errors.ValidationError('"meta.location": "azure_cluster" must be specified if "type" == AZURE_CLUSTER')
        if not azure_cluster.isupper():
            raise errors.ValidationError(
                '"meta.location.azure_cluster": AZURE cluster must be uppercase'.format(azure_cluster))
        if azure_cluster not in external_clusters.AZURE_CLUSTERS_BY_NAME:
            raise errors.ValidationError('"meta.location.azure_cluster": unknown AZURE cluster: {}'.format(azure_cluster))


def validate_spec_pb(spec_pb, field_name='spec'):
    """
    :param spec_pb: model_pb2.BalancerSpec
    :type field_name: six.text_type
    :raises: exceptions.BadRequestError
    """
    spec_type_name = model_pb2.BalancerType.Name(spec_pb.type)
    if spec_pb.type == model_pb2.YANDEX_BALANCER:
        if spec_pb.WhichOneof('spec') != 'yandex_balancer':
            raise exceptions.BadRequestError(
                '"{0}.type" is set to "{1}", but "{0}.yandex_balancer" '
                'field is missing'.format(field_name, spec_type_name))
        validate_yandex_balancer_spec_pb(spec_pb.yandex_balancer, field_name='{}.yandex_balancer'.format(field_name))
    else:
        raise exceptions.BadRequestError('{0}.type: unsupported config type "{1}"'.format(field_name, spec_type_name))

    if spec_pb.ctl_version not in VALID_BALANCER_CTL_VERSIONS:
        raise exceptions.BadRequestError('"{}.ctl_version" must be one of the {}'.format(field_name,
                                                                                         VALID_BALANCER_CTL_VERSIONS))

    if not spec_pb.HasField('config_transport'):
        raise exceptions.BadRequestError('"{}.config_transport" must be set'.format(field_name))
    transport_type_name = model_pb2.ConfigTransportType.Name(spec_pb.config_transport.type)
    if spec_pb.type == model_pb2.NANNY_STATIC_FILE:
        if not spec_pb.config_transport.HasField('nanny_static_file'):
            raise exceptions.BadRequestError('"{}.config_transport.nanny_static_file" must be set'.format(field_name))
        if not spec_pb.config_transport.nanny_static_file.service_id:
            raise exceptions.BadRequestError(
                '"{}.config_transport.nanny_static_file.service_id" must be set'.format(field_name))
    else:
        raise exceptions.BadRequestError(
            '{0}.type: unsupported config type "{1}"'.format(field_name, transport_type_name))

    if spec_pb.HasField('components'):
        validate_balancer_components(spec_pb.components, field_name=field_name + '.components')

    if spec_pb.HasField('container_spec'):
        validate_balancer_container_spec(spec_pb.container_spec, field_name=field_name + '.container_spec')


def validate_create_balancer_request(req_pb):
    """
    :type req_pb: api_pb2.CreateBalancerRequest
    :raises: exceptions.BadRequestError
    """
    if not req_pb.HasField('meta'):
        raise exceptions.BadRequestError('"meta" must be set')
    if not req_pb.meta.id:
        raise exceptions.BadRequestError('"meta.id" must be set')
    if not BALANCER_ID_RE.match(req_pb.meta.id):
        raise exceptions.BadRequestError('"meta.id" is not valid')
    if not req_pb.meta.namespace_id:
        raise exceptions.BadRequestError('"meta.namespace_id" must be set')
    if not NAMESPACE_ID_RE.match(req_pb.meta.namespace_id):
        raise exceptions.BadRequestError('"meta.namespace_id" is not valid')
    if not req_pb.meta.HasField('auth'):
        raise exceptions.BadRequestError('"meta.auth" must be set')
    validate_balancer_location(req_pb.meta.location, req_pb.meta.id)
    validate_auth_pb(req_pb.meta.auth, field_name='meta.auth')

    if req_pb.spec.HasField('components'):
        raise exceptions.BadRequestError('"spec.components" must not be set')

    if req_pb.HasField('spec') and req_pb.spec != model_pb2.BalancerSpec():
        validate_spec_pb(req_pb.spec)
    elif req_pb.HasField('order') and req_pb.order != model_pb2.BalancerOrder.Content():
        validate_order_pb(req_pb.meta.namespace_id, req_pb.order, location=get_location(req_pb))
    else:
        raise exceptions.BadRequestError('One of "order" and "spec" must be set')


def validate_order_pb(namespace_id, order_content_pb, location, field_name='order'):
    """
    :type namespace_id: six.text_type
    :type order_content_pb: model_pb2.BalancerOrder.Content
    :type location: six.text_type
    :type field_name: six.text_type
    :raises: exceptions.BadRequestError
    """
    mode = order_content_pb.mode
    if not order_content_pb.HasField('allocation_request'):
        raise exceptions.BadRequestError('"{}.allocation_request" must be set'.format(field_name))
    validate_allocation_request(order_content_pb.allocation_request, order_content_pb.cloud_type,
                                field_name='order.allocation_request')
    if not order_content_pb.abc_service_id:
        raise exceptions.BadRequestError('"{}.abc_service_id" must be set'.format(field_name))
    if mode != order_content_pb.YP_LITE:
        raise AssertionError('unexpected balancer order mode')
    if location != order_content_pb.allocation_request.location:
        raise exceptions.BadRequestError('"{}.allocation_request.location" must match "meta.location"'.format(
            field_name))
    c = cache.IAwacsCache.instance()
    copy_spec_from_balancer_id = order_content_pb.copy_spec_from_balancer_id
    if copy_spec_from_balancer_id:
        balancer_pb = c.must_get_balancer(namespace_id=namespace_id, balancer_id=copy_spec_from_balancer_id)
        if balancer_pb.spec.incomplete or balancer_pb.spec.deleted or balancer_pb.order.cancelled.value:
            raise exceptions.BadRequestError(
                '"{}.copy_spec_from_balancer_id": cannot copy an incomplete or removed balancer "{}"'.format(
                    field_name, copy_spec_from_balancer_id))
    if order_content_pb.HasField('copy_nanny_service'):
        copy_from_balancer_id = order_content_pb.copy_nanny_service.balancer_id
        if not copy_from_balancer_id:
            raise exceptions.BadRequestError('"{}.copy_nanny_service.balancer_id" must be set'.format(field_name))
        else:
            balancer_pb = c.must_get_balancer(namespace_id=namespace_id, balancer_id=copy_from_balancer_id)
            if balancer_pb.spec.incomplete or balancer_pb.spec.deleted or balancer_pb.order.cancelled.value:
                raise exceptions.BadRequestError(
                    '"{}.copy_nanny_service.balancer_id": cannot copy an incomplete or removed balancer "{}"'.format(
                        field_name, copy_from_balancer_id))


def validate_allocation_request(allocation_request_pb, cloud_type,
                                field_name='order.yp_lite_allocation_request'):
    """
    :type allocation_request_pb: model_pb2.BalancerOrder.Content.AllocationRequest
    :type cloud_type: model_pb2.CloudType
    :type field_name: six.text_type
    :raises: exceptions.BadRequestError
    """
    if not allocation_request_pb.nanny_service_id_slug:
        raise exceptions.BadRequestError('"{}.nanny_service_id_slug" must be set'.format(field_name))
    validate_nanny_service_slug(allocation_request_pb.nanny_service_id_slug,
                                field_name='{}.nanny_service_id_slug'.format(field_name))

    if not allocation_request_pb.location:
        raise exceptions.BadRequestError('"{}.location" must be set'.format(field_name))
    if cloud_type == model_pb2.CT_AZURE:
        available_balancer_locations = [name.lower() for name in external_clusters.AZURE_CLUSTERS_BY_NAME]
    else:
        available_balancer_locations = util.get_available_balancer_locations()
    if allocation_request_pb.location.lower() not in available_balancer_locations:
        raise exceptions.BadRequestError(
            '"{}.location" must be '
            'one of the following: "{}"'.format(field_name, '", "'.join(
                util.get_available_balancer_locations())))

    if not allocation_request_pb.network_macro:
        raise exceptions.BadRequestError('"{}.network_macro" must be set'.format(field_name))

    if allocation_request_pb.type != allocation_request_pb.PRESET:
        raise exceptions.BadRequestError('"{}.type" must be PRESET'.format(field_name))
    if not allocation_request_pb.HasField('preset'):
        raise exceptions.BadRequestError('"{}.preset" must be set'.format(field_name))
    if allocation_request_pb.preset.type == allocation_request_pb.preset.NONE:
        raise exceptions.BadRequestError('"{}.preset.type" must be set'.format(field_name))
    if allocation_request_pb.preset.instances_count < 1:
        raise exceptions.BadRequestError(
            '"{}.preset.instances_count" must be equal or greater than 1'.format(field_name))


def validate_cancel_balancer_order(balancer_pb):
    if not is_order_in_progress(balancer_pb):
        raise exceptions.BadRequestError('Cannot cancel order that is not in progress')
    if not can_be_cancelled(balancer_pb, get_balancer_order_processors()):
        raise exceptions.BadRequestError('Cannot cancel balancer order at this stage')


def validate_change_balancer_order_abc_service_id(req_pb, balancer_pb):
    if not BalancerOrder(balancer_pb).can_change_abc_group():
        raise exceptions.BadRequestError('Cannot change balancer ABC service at this time')
    if not req_pb.change_abc_service_id.abc_service_id:
        raise exceptions.BadRequestError('"change_abc_service_id.abc_service_id" must be set')


def validate_update_balancer_request(req_pb):
    """
    :type req_pb: api_pb2.UpdateBalancerRequest
    :raises: exceptions.BadRequestError
    """
    if not req_pb.HasField('meta'):
        raise exceptions.BadRequestError('"meta" must be set')
    if not req_pb.meta.id:
        raise exceptions.BadRequestError('"meta.id" must be set')
    if not BALANCER_ID_RE.match(req_pb.meta.id):
        raise exceptions.BadRequestError('"meta.id" is not valid')
    if not req_pb.meta.namespace_id:
        raise exceptions.BadRequestError('"meta.namespace_id" must be set')
    if not NAMESPACE_ID_RE.match(req_pb.meta.namespace_id):
        raise exceptions.BadRequestError('"meta.namespace_id" is not valid')
    if not req_pb.meta.version:
        raise exceptions.BadRequestError('"meta.version" must be set')
    if req_pb.meta.HasField('location'):
        validate_balancer_location(req_pb.meta.location, req_pb.meta.id)
    if (not req_pb.HasField('spec') and
            not req_pb.meta.HasField('auth') and
            not req_pb.meta.HasField('transport_paused') and
            not req_pb.meta.HasField('flags') and
            not req_pb.meta.HasField('location')):
        raise exceptions.BadRequestError('at least one of the "spec", "meta.auth", "meta.location", "meta.flags" or '
                                         '"meta.transport_paused" fields must be present')
    if req_pb.meta.HasField('auth'):
        validate_auth_pb(req_pb.meta.auth, field_name='meta.auth')
    if req_pb.HasField('spec'):
        validate_spec_pb(req_pb.spec)


def validate_list_balancers_request_query(query_pb, field_name=u'query'):
    """
    :type query_pb: api_pb2.ListBalancersQuery
    :type field_name: six.text_type
    :raises: exceptions.BadRequestError
    """
    if not query_pb.nanny_service_id_in and not query_pb.component_in:
        msg = (u'"{0}.nanny_service_id_in" or "{0}.component_in" must be set'.format(field_name))
        log.warn(u'Would raise an exception: {}'.format(msg))
        # raise exceptions.BadRequestError(msg)

    for i, cmp_pb in enumerate(query_pb.component_in):
        if not cmp_pb.type:
            raise exceptions.BadRequestError(u'"{}.component_in[{}].type": '
                                             u'must be set'.format(field_name, i))


def validate_list_balancers_request(req_pb):
    """
    :type req_pb: api_pb2.ListBalancersRequest
    :raises: exceptions.BadRequestError
    """
    if req_pb.HasField('field_mask'):
        if not req_pb.field_mask.IsValidForDescriptor(model_pb2.Balancer.DESCRIPTOR):
            raise exceptions.BadRequestError('"field_mask" is not valid')
    if req_pb.HasField('query'):
        validate_list_balancers_request_query(req_pb.query)


def validate_request(req_pb):
    """
    :raises: exceptions.BadRequestError
    """
    if isinstance(req_pb, api_pb2.CreateBalancerOperationRequest):
        validate_create_balancer_operation_request(req_pb)
    elif isinstance(req_pb, api_pb2.GetBalancerRevisionRequest):
        if not req_pb.id:
            raise exceptions.BadRequestError('No "id" specified.')
    elif isinstance(req_pb, (api_pb2.RemoveBalancerRequest,
                             api_pb2.GetBalancerRequest,
                             api_pb2.GetBalancerRemovalChecksRequest,
                             api_pb2.GetBalancerOperationRequest,
                             api_pb2.GetBalancerAspectsSetRequest,
                             api_pb2.GetBalancerStateRequest,
                             api_pb2.ListBalancerRevisionsRequest)):
        if not req_pb.id:
            raise exceptions.BadRequestError('No "id" specified.')
        if not req_pb.namespace_id:
            raise exceptions.BadRequestError('No "namespace_id" specified.')
    elif isinstance(req_pb, (api_pb2.EnableBalancerPushclientRequest,
                             api_pb2.DisableBalancerPushclientRequest)):
        if not req_pb.id:
            raise exceptions.BadRequestError('No "id" specified.')
        if not req_pb.namespace_id:
            raise exceptions.BadRequestError('No "namespace_id" specified.')
        if not req_pb.version:
            raise exceptions.BadRequestError('No "version" specified.')
    elif isinstance(req_pb, (api_pb2.CancelBalancerOrderRequest,
                             api_pb2.CancelBalancerOperationRequest,
                             api_pb2.ChangeBalancerOrderRequest,
                             api_pb2.ChangeBalancerOperationOrderRequest,
                             api_pb2.ChangeBalancerRemovalRequest,
                             api_pb2.CancelBalancerRemovalRequest,
                             )):
        if not req_pb.id:
            raise exceptions.BadRequestError('No "id" specified.')
        if not req_pb.namespace_id:
            raise exceptions.BadRequestError('No "namespace_id" specified.')
    elif isinstance(req_pb, api_pb2.ListBalancerAspectsSetsRequest):
        if not req_pb.namespace_id:
            raise exceptions.BadRequestError('No "namespace_id" specified.')
    elif isinstance(req_pb, api_pb2.UpdateBalancerRequest):
        validate_update_balancer_request(req_pb)
    elif isinstance(req_pb, api_pb2.CreateBalancerRequest):
        validate_create_balancer_request(req_pb)
    elif isinstance(req_pb, api_pb2.ListBalancersRequest):
        validate_list_balancers_request(req_pb)
    elif isinstance(req_pb, api_pb2.ListBalancerOperationsRequest):
        if not req_pb.namespace_id:
            raise exceptions.BadRequestError('No "namespace_id" specified.')
    elif isinstance(req_pb, api_pb2.UpdateBalancerOperationOrderContextRequest):
        if not req_pb.id:
            raise exceptions.BadRequestError('"id" must be set')
        if not req_pb.namespace_id:
            raise exceptions.BadRequestError('"namespace_id" must be set')
        if not req_pb.fields:
            raise exceptions.BadRequestError('"fields" must be set')
        for field_pb in req_pb.fields:
            if not field_pb.key:
                raise exceptions.BadRequestError('"key" must be set')
    elif isinstance(req_pb, api_pb2.UpdateBalancerTransportPausedRequest):
        if not req_pb.id:
            raise exceptions.BadRequestError('"id" must be set')
        if not req_pb.namespace_id:
            raise exceptions.BadRequestError('"namespace_id" must be set')
        if not req_pb.HasField('transport_paused'):
            raise exceptions.BadRequestError('"transport_paused" must be set')
    elif isinstance(req_pb, api_pb2.UpdateBalancerL7MacroTo03xRequest):
        required_fields = ['id', 'namespace_id', 'version']
        errors = []
        for field in required_fields:
            if not getattr(req_pb, field):
                errors.append('"{}" must be set'.format(field))
        if errors:
            raise exceptions.BadRequestError('; '.join(errors))
    elif isinstance(req_pb, api_pb2.GetBalancerXFFYBehaviorRequest):
        required_fields = ['id', 'namespace_id']
        errors = []
        for field in required_fields:
            if not getattr(req_pb, field):
                errors.append('"{}" must be set'.format(field))
        if errors:
            raise exceptions.BadRequestError('; '.join(errors))
    else:
        raise RuntimeError('Incorrect `req_pb` type: {}'.format(req_pb.__class__.__name__))


def validate_balancer_service_and_set_location_if_needed(meta_pb, spec_pb):
    nanny_service_id = spec_pb.config_transport.nanny_static_file.service_id
    try:
        nanny_service = nannyclient.INannyClient.instance().get_service(nanny_service_id)
    except nannyclient.NannyApiRequestException:
        raise exceptions.InternalError('Something went wrong during communicating with Nanny API, '
                                       'please try again later')
    if nanny_service['info_attrs']['content'].get('type') != "AWACS_BALANCER":
        msg = ('Nanny service must have type AWACS_BALANCER. See '
               'https://wiki.yandex-team.ru/cplb/awacs/awacs-balancers-from-existing-nanny-service/ for details')
        raise exceptions.BadRequestError(msg)

    if nanny_service['runtime_attrs']['content']['engines']['engine_type'] == 'YP_LITE':
        yp_cluster = nanny_service['info_attrs']['content'].get('yp_cluster')
        if not yp_cluster:
            msg = ('Balancer can be created only from single location Nanny service. See '
                   'https://wiki.yandex-team.ru/cplb/awacs/awacs-balancers-from-existing-nanny-service/ for details')
            raise exceptions.BadRequestError(msg)
        if not meta_pb.HasField('location') or meta_pb.location.type != meta_pb.location.YP_CLUSTER:
            location_pb = model_pb2.BalancerMeta.Location()
            location_pb.type = location_pb.YP_CLUSTER
            location_pb.yp_cluster = yp_cluster
            validate_balancer_location(location_pb, meta_pb.id)
            meta_pb.location.CopyFrom(location_pb)
        elif meta_pb.location.yp_cluster != yp_cluster:
            msg = ('Balancer location must be the same as Nanny service location. See '
                   'https://wiki.yandex-team.ru/cplb/awacs/awacs-balancers-from-existing-nanny-service/ for details')
            raise exceptions.BadRequestError(msg)


def validate_cancel_balancer_operation(balancer_op_pb):
    if not is_order_in_progress(balancer_op_pb):
        raise exceptions.BadRequestError('Cannot cancel operation that is not in progress')
    if balancer_op_pb.order.content.HasField('migrate_from_gencfg_to_yp_lite'):
        processors = get_gencfg_migration_processors()
    elif balancer_op_pb.order.content.HasField('create_system_backend'):
        processors = get_create_system_backend_processors()
    else:
        raise exceptions.BadRequestError('Unknown balancer operation')
    if not can_be_cancelled(balancer_op_pb, processors):
        raise exceptions.BadRequestError('Cannot cancel balancer operation at this stage')


def validate_cancel_balancer_removal(balancer_pb):
    if not is_order_in_progress(balancer_pb, field='removal'):
        raise exceptions.BadRequestError('Cannot cancel removal that is not in progress')
    if not can_be_cancelled(balancer_pb, BalancerRemoval.get_processors(), field='removal'):
        raise exceptions.BadRequestError('Cannot cancel balancer removal at this stage')


def validate_change_balancer_operation_order_instance_tags(balancer_op_pb):
    if not GencfgMigration(balancer_op_pb).can_change_instance_tags():
        raise exceptions.BadRequestError('Cannot change instance tags at this time')


def validate_create_balancer_operation_request(req_pb):
    """
    :type req_pb: api_pb2.CreateBalancerOperationRequest
    :raises: exceptions.BadRequestError
    """
    if not req_pb.HasField('meta'):
        raise exceptions.BadRequestError('"meta" must be set')
    if not req_pb.meta.id:
        raise exceptions.BadRequestError('"meta.id" must be set')
    if not BALANCER_ID_RE.match(req_pb.meta.id):
        raise exceptions.BadRequestError('"meta.id" is not valid')
    if not req_pb.meta.namespace_id:
        raise exceptions.BadRequestError('"meta.namespace_id" must be set')
    if not NAMESPACE_ID_RE.match(req_pb.meta.namespace_id):
        raise exceptions.BadRequestError('"meta.namespace_id" is not valid')

    if not req_pb.HasField('order'):
        raise exceptions.BadRequestError('"order" must be set')

    validate_balancer_operation_order_content(req_pb.meta.namespace_id, req_pb.meta.id, req_pb.order)


def validate_balancer_operation_order_content(namespace_id, balancer_id, content_pb):
    """
    :type content_pb: model_pb2.BalancerOperationOrder.Content
    :type namespace_id: six.text_type
    :type balancer_id: six.text_type
    :raises: exceptions.BadRequestError
    """
    balancer_pb = zk.IZkStorage.instance().must_get_balancer(namespace_id, balancer_id)
    if content_pb.HasField('migrate_from_gencfg_to_yp_lite'):
        new_balancer_id = content_pb.migrate_from_gencfg_to_yp_lite.new_balancer_id
        if not new_balancer_id:
            raise exceptions.BadRequestError('"order.content.migrate_from_gencfg_to_yp_lite.new_balancer_id" '
                                             'must be set')
        if not BALANCER_ID_RE.match(new_balancer_id):
            raise exceptions.BadRequestError('"order.content.migrate_from_gencfg_to_yp_lite.new_balancer_id" '
                                             'is not valid')
        if not content_pb.migrate_from_gencfg_to_yp_lite.abc_service_id:
            raise exceptions.BadRequestError('"order.content.migrate_from_gencfg_to_yp_lite.abc_service_id" '
                                             'must be specified')
        if not content_pb.migrate_from_gencfg_to_yp_lite.HasField('allocation_request'):
            raise exceptions.BadRequestError('"order.content.migrate_from_gencfg_to_yp_lite.allocation_request" '
                                             'must be specified')
        if balancer_pb.meta.location.type != balancer_pb.meta.location.GENCFG_DC:
            raise exceptions.BadRequestError('Target balancer must have location.type=GENCFG_DC')
        validate_allocation_request(content_pb.migrate_from_gencfg_to_yp_lite.allocation_request,
                                    cloud_type=model_pb2.CT_RTC,
                                    field_name='order.content.migrate_from_gencfg_to_yp_lite.allocation_request')
        validate_location_balancer_id(content_pb.migrate_from_gencfg_to_yp_lite.allocation_request.location,
                                      new_balancer_id)
    elif content_pb.HasField('create_system_backend'):
        if balancer_pb.meta.location.type != balancer_pb.meta.location.YP_CLUSTER:
            raise exceptions.BadRequestError('Target balancer must have location.type=YP_CLUSTER')
        c = cache.IAwacsCache.instance()
        maybe_system_backend = c.get_backend(balancer_pb.meta.namespace_id, balancer_pb.meta.id)
        if maybe_system_backend is not None:
            if maybe_system_backend.meta.is_system.value:
                raise exceptions.BadRequestError('System backend for balancer already exists')
            raise exceptions.BadRequestError(
                'System backend can not be created for the balancer: non-system backend "{}" '
                'already exists'.format(balancer_pb.meta.id))
    else:
        raise exceptions.BadRequestError('"order.content": migrate_from_gencfg_to_yp_lite or '
                                         'create_system_backend must be specified')


def validate_nanny_service_for_copy(namespace_id, balancer_id):
    """
    :type namespace_id: six.text_type
    :type balancer_id: six.text_type
    """
    b_pb = zk.IZkStorage.instance().must_get_balancer(namespace_id, balancer_id)
    nanny_service_id = b_pb.spec.config_transport.nanny_static_file.service_id
    runtime_attrs = nannyclient.INannyClient.instance().get_service_runtime_attrs(nanny_service_id)
    if runtime_attrs['content']['engines']['engine_type'] != 'YP_LITE':
        raise exceptions.BadRequestError(
            "Cannot copy service \"{}\" because it's not in YP_LITE".format(nanny_service_id))


def validate_balancer_location_for_copy(namespace_id, location):
    """
    :type namespace_id: six.text_type
    :type location: six.text_type
    """
    for balancer_pb in cache.IAwacsCache.instance().list_all_balancers(namespace_id):
        if location == get_location(balancer_pb):
            raise exceptions.BadRequestError(
                'Cannot create copied balancer in the same location as another balancer: "{}"'.format(
                    balancer_pb.meta.id))


def rps_removal_check(balancer_pb):
    check_pb = model_pb2.RemovalCheck()
    check_pb.type = check_pb.IMAGE

    yc = yasm_client.IYasmClient.instance()
    if (balancer_pb.meta.HasField('location') and
            balancer_pb.spec.config_transport.nanny_static_file.HasField('instance_tags')):
        check_pb.image.href = yc.get_last_balancer_rps_link(balancer_pb.meta.location,
                                                            balancer_pb.spec.config_transport.nanny_static_file.instance_tags)
        check_pb.image.src = yc.get_screenshot_link(check_pb.image.href)

    skip_rps_check_during_removal = balancer_pb.meta.flags.skip_rps_check_during_removal
    if skip_rps_check_during_removal.value:
        check_pb.state = check_pb.PASSED
        check_pb.message = ('Forced by skip_rps_check_during_removal flag (set by {} at {} UTC)'
                            .format(skip_rps_check_during_removal.author,
                                    skip_rps_check_during_removal.mtime.ToDatetime().replace(microsecond=0)))
        return check_pb

    if balancer_pb.spec.incomplete:
        check_pb.state = check_pb.PASSED
        check_pb.message = 'Balancer has not been created'
        return check_pb

    if (not balancer_pb.meta.HasField('location') or
            not balancer_pb.spec.config_transport.nanny_static_file.HasField('instance_tags')):
        check_pb.state = check_pb.UNKNOWN
        check_pb.message = ('Could not check if balancer has any traffic, please check it manually and contact support '
                            'for approval of balancer removal')
        return check_pb

    last_known_rps = yc.get_last_balancer_rps(balancer_pb.meta.location,
                                              balancer_pb.spec.config_transport.nanny_static_file.instance_tags)
    if last_known_rps > 1:
        check_pb.state = check_pb.FAILED
        check_pb.message = (
            'Seems like balancer still receives traffic. This exception can occur if balancer has migrated'
            ' from gencfg to YP. Please contact support if you are sure that balancer should be removed.')
        return check_pb

    check_pb.state = check_pb.PASSED
    check_pb.message = 'No or almost no traffic on the balancer'
    return check_pb


def nanny_service_usage_check(balancer_pb):
    check_pb = model_pb2.RemovalCheck()
    check_pb.type = check_pb.REFERENCES
    service_id = balancer_pb.spec.config_transport.nanny_static_file.service_id
    if not service_id:
        check_pb.state = check_pb.PASSED
        check_pb.message = 'Balancer service has not been created'
    ac = apicache.IAwacsApiCache.instance()
    query = {ac.BackendsQueryTarget.NANNY_SERVICE_ID_IN: [service_id]}
    backends = ac.list_backends(query)
    if not backends.total:
        check_pb.state = check_pb.PASSED
        check_pb.message = 'Nanny service is not used in backends'
        return check_pb

    check_pb.state = check_pb.FAILED
    check_pb.message = 'Nanny service is used in backends'
    for backend_pb in backends.items:
        ref_pb = check_pb.references.references.add(namespace_id=backend_pb.meta.namespace_id,
                                                    id=backend_pb.meta.id)
        ref_pb.object_type = ref_pb.REMOVAL_REF_BACKEND
    return check_pb


def gencfg_groups_usage_check(balancer_pb):
    check_pb = model_pb2.RemovalCheck()
    check_pb.type = check_pb.REFERENCES
    service_id = balancer_pb.spec.config_transport.nanny_static_file.service_id
    if not service_id:
        check_pb.state = check_pb.PASSED
        check_pb.message = 'Balancer service has not been created'
        return check_pb
    if balancer_pb.meta.location.type in (balancer_pb.meta.location.YP_CLUSTER, balancer_pb.meta.location.AZURE_CLUSTER):
        check_pb.state = check_pb.PASSED
        check_pb.message = 'Balancer is Yp.Lite-powered'
        return check_pb

    _, instances = nannyclient.INannyClient.instance().get_service_instances_section(service_id)
    gencfg_group_names = [group.name for group in instances.extended_gencfg_groups.groups]
    if not gencfg_group_names:
        check_pb.state = check_pb.PASSED
        check_pb.message = 'No gencfg groups in service'
        return check_pb

    ac = apicache.IAwacsApiCache.instance()
    query = {ac.BackendsQueryTarget.GENCFG_GROUP_NAME_IN: gencfg_group_names}
    backends = ac.list_backends(query)
    if not backends.total:
        check_pb.state = check_pb.PASSED
        check_pb.message = 'Gencfg groups of service are not used in backends'
        return check_pb

    check_pb.state = check_pb.FAILED
    check_pb.message = 'Gencfg groups of service are used in backends'
    for backend_pb in backends.items:
        ref_pb = check_pb.references.references.add(namespace_id=backend_pb.meta.namespace_id,
                                                    id=backend_pb.meta.id)
        ref_pb.object_type = ref_pb.REMOVAL_REF_BACKEND
    return check_pb


def endpoint_sets_usage_check(balancer_pb):
    check_pb = model_pb2.RemovalCheck()
    check_pb.type = check_pb.REFERENCES
    service_id = balancer_pb.spec.config_transport.nanny_static_file.service_id
    yp_cluster = balancer_pb.meta.location.yp_cluster
    if not service_id:
        check_pb.state = check_pb.PASSED
        check_pb.message = 'Balancer service has not been created'
        return check_pb
    if not yp_cluster:
        check_pb.state = check_pb.PASSED
        check_pb.message = 'Balancer is gencfg-powered'
        return check_pb

    list_es_req_pb = endpoint_sets_api_pb2.ListEndpointSetsRequest(service_id=service_id, cluster=yp_cluster)
    endpoint_sets_pb = IYpLiteRpcClient.instance().list_endpoint_sets(list_es_req_pb).endpoint_sets
    if not endpoint_sets_pb:
        check_pb.state = check_pb.PASSED
        check_pb.message = 'Balancer has no endpoint sets'
        return check_pb
    ac = apicache.IAwacsApiCache.instance()
    query = {ac.BackendsQueryTarget.YP_ENDPOINT_SET_FULL_ID_IN: [
        api_pb2.YpEndpointSetRef(cluster=yp_cluster.lower(), id=endpoint_set_pb.meta.id)
        for endpoint_set_pb in endpoint_sets_pb
    ]}
    backends = ac.list_backends(query)
    system_backend_pb = None
    for backend_pb in backends.items:
        if not backend_pb.meta.is_system.value:
            check_pb.state = check_pb.FAILED
            check_pb.message = 'Balancer endpoint sets are used in backends'
            ref_pb = check_pb.references.references.add(namespace_id=backend_pb.meta.namespace_id,
                                                        id=backend_pb.meta.id)
            ref_pb.object_type = ref_pb.REMOVAL_REF_BACKEND
        else:
            system_backend_pb = backend_pb
    if check_pb.state == check_pb.FAILED:
        return check_pb

    if system_backend_pb:
        c = cache.IAwacsCache().instance()
        full_l3_balancer_ids = c.list_full_l3_balancer_ids_for_backend(balancer_pb.meta.namespace_id,
                                                                       system_backend_pb.meta.id)
        if full_l3_balancer_ids:
            check_pb.state = check_pb.FAILED
            check_pb.message = 'Balancer is used in L3 balancers'
            for n_id, l3_id in full_l3_balancer_ids:
                ref_pb = check_pb.references.references.add(namespace_id=n_id, id=l3_id)
                ref_pb.object_type = ref_pb.REMOVAL_REF_L3_BALANCER
            return check_pb

        full_dns_record_ids = c.list_full_dns_record_ids_for_backend(balancer_pb.meta.namespace_id,
                                                                     system_backend_pb.meta.id)
        if full_dns_record_ids:
            check_pb.state = check_pb.FAILED
            check_pb.message = 'Balancer is used in DNS records'
            for n_id, dns_id in full_dns_record_ids:
                ref_pb = check_pb.references.references.add(namespace_id=n_id, id=dns_id)
                ref_pb.object_type = ref_pb.REMOVAL_REF_DNS_RECORD
            return check_pb

    check_pb.state = check_pb.PASSED
    check_pb.message = 'Balancer endpoint sets are not used in backends'
    return check_pb


def different_balancers_removal_check(balancer_pb):
    check_pb = model_pb2.RemovalCheck()
    check_pb.type = check_pb.REFERENCES
    c = cache.IAwacsCache().instance()
    balancer_pbs = c.list_balancers(namespace_id=balancer_pb.meta.namespace_id).items
    for other_balancer_pb in balancer_pbs:
        if other_balancer_pb.meta.id != balancer_pb.meta.id and other_balancer_pb.spec.deleted:
            check_pb.state = check_pb.FAILED
            check_pb.message = (u'Different balancers can not be removed simultaneously in one '
                                u'namespace, other balancer is removing now')
            ref_pb = check_pb.references.references.add(namespace_id=balancer_pb.meta.namespace_id,
                                                        id=other_balancer_pb.meta.id)
            ref_pb.object_type = ref_pb.REMOVAL_REF_BALANCER
            return check_pb

    check_pb.state = check_pb.PASSED
    check_pb.message = 'No other balancers in the namespace are being deleted'
    return check_pb


def no_active_operation_check(balancer_pb):
    check_pb = model_pb2.RemovalCheck()
    balancer_op_pb = zk.IZkStorage.instance().get_balancer_operation(balancer_pb.meta.namespace_id, balancer_pb.meta.id)
    if balancer_op_pb:
        check_pb.state = check_pb.FAILED
        check_pb.message = u'Cannot remove balancer with an active operation'
        return check_pb

    check_pb.state = check_pb.PASSED
    check_pb.message = 'No active operation for balancer'
    return check_pb


def get_balancer_removal_checks(balancer_pb):
    return [
        rps_removal_check(balancer_pb),
        endpoint_sets_usage_check(balancer_pb),
        nanny_service_usage_check(balancer_pb),
        gencfg_groups_usage_check(balancer_pb),
        different_balancers_removal_check(balancer_pb),
        no_active_operation_check(balancer_pb),
    ]


def get_location(pb):
    if pb.meta.location.type == model_pb2.BalancerMeta.Location.YP_CLUSTER:
        return pb.meta.location.yp_cluster
    elif pb.meta.location.type == model_pb2.BalancerMeta.Location.GENCFG_DC:
        return pb.meta.location.gencfg_dc
    elif pb.meta.location.type == model_pb2.BalancerMeta.Location.AZURE_CLUSTER:
        return pb.meta.location.azure_cluster
    raise AssertionError('unknown location type {}'.format(pb.meta.location.type))


def validate_changed_components(namespace_pb, balancer_pb, updated_spec_pb, is_author_root):
    """
    :type namespace_pb: model_pb2.Namespace
    :type balancer_pb: model_pb2.Balancer
    :type updated_spec_pb: model_pb2.BalancerSpec
    :type is_author_root: bool
    :raises: exceptions.BadRequestError
    """
    if updated_spec_pb.components == balancer_pb.spec.components:
        return

    updated_awacslet_pb = updated_spec_pb.components.awacslet
    if (updated_awacslet_pb.state == updated_awacslet_pb.SET and
            balancer_pb.meta.location.type != balancer_pb.meta.location.YP_CLUSTER):
        raise exceptions.BadRequestError(
            u'"spec.components.awacslet" can only be set for YP.lite-powered balancers')

    if is_author_root:
        return

    changed_components = iter_changed_balancer_components(balancer_pb.spec.components,
                                                          updated_spec_pb.components)
    for component_config, prev_component_pb, updated_component_pb in changed_components:
        if component_config.type not in namespace_pb.spec.customizable_components:
            raise exceptions.BadRequestError(
                u'"spec.components": {} in this namespace can only be changed by '
                u'someone with root privileges. Please contact support if you would '
                u'like to manage it.'.format(component_config.str_type))


def validate_spec_invariants(req_pb, balancer_pb, updated_spec_pb, is_author_root):
    # transport has type NANNY_STATIC_FILE -- guaranteed by previously called balancer.validate_request
    updated_service_id = updated_spec_pb.config_transport.nanny_static_file.service_id
    current_service_id = balancer_pb.spec.config_transport.nanny_static_file.service_id
    if current_service_id != updated_service_id:
        raise exceptions.BadRequestError(
            '"spec.config_transport.nanny_static_file.service_id" can not be changed')

    updated_mode = updated_spec_pb.yandex_balancer.mode
    current_mode = balancer_pb.spec.yandex_balancer.mode

    if (current_mode == model_pb2.YandexBalancerSpec.EASY_MODE and
            updated_mode != model_pb2.YandexBalancerSpec.EASY_MODE and
            not is_author_root):
        raise exceptions.BadRequestError(
            '"spec.yandex_balancer.mode": EASY_MODE can only be changed by someone with root privileges. '
            'Please contact support if you really need to convert your balancer to another mode.')

    old_instance_tags = balancer_pb.spec.config_transport.nanny_static_file.instance_tags
    new_instance_tags = updated_spec_pb.config_transport.nanny_static_file.instance_tags

    if old_instance_tags != new_instance_tags and not is_author_root:
        raise exceptions.BadRequestError('"spec.config_transport.nanny_static_file.instance_tags": can not be changed')

    container_spec_has_been_removed = (balancer_pb.spec.HasField('container_spec') and
                                       not updated_spec_pb.HasField('container_spec'))
    if container_spec_has_been_removed and u'changed using awacsctl' in req_pb.meta.comment:
        raise exceptions.BadRequestError(
            u'"spec.container_spec" can not be removed using awacsctl (see SWAT-6307 for details)')


def validate_balancer_normalized_prj(namespace_id, prj, field_name="order.instance_tags.prj"):
    _cache = cache.IAwacsCache.instance()
    namespace_balancers_pb = _cache.list_all_balancers(namespace_id=namespace_id)
    expected_prjs = set()
    for pb in namespace_balancers_pb:
        prj = pb.spec.config_transport.nanny_static_file.instance_tags.prj or pb.order.content.instance_tags.prj
        if prj:
            expected_prjs.add(prj)
    if len(expected_prjs) > 1:
        raise AssertionError('Found different prjs in one namespace "{}"'.format(namespace_id))
    elif len(expected_prjs) == 1:
        expected_prj = expected_prjs.pop()
        if prj != expected_prj:
            raise exceptions.BadRequestError('"{}": Other balancers in this namespace already have prj tag set to "{}",'
                                             ' this new balancer must have the same value instead of "{}"'
                                             .format(field_name, expected_prj, prj))
        return
    else:
        _namespace_id = _cache.get_namespace_id_by_normalised_prj(prj)
        if _namespace_id is not None and namespace_id != _namespace_id:
            raise exceptions.BadRequestError('"{}": normalized prj tag "{}" is already used by another namespace: {}'
                                             .format(field_name, _cache.normalise_prj_tag(prj), _namespace_id))
