import logging

import gevent
import re
import time
from sepelib.core import config as appconfig

from awacs.lib import certs
from awacs.lib.order_processor.model import is_order_in_progress, can_be_cancelled
from awacs.lib.rpc import exceptions
from awacs.lib.strutils import quote_join_sorted
from awacs.model import cache
from awacs.model.certs.processors.processors import get_cert_state_processors
from awacs.model.util import BALANCER_LOCATIONS
from awacs.web.validation.util import (CERT_ID_RE, NAMESPACE_ID_RE, validate_san, validate_auth_pb, is_root,
                                       get_issuable_tls_hostnames, validate_tls_hostname_issuability)
from infra.awacs.proto import api_pb2, model_pb2


log = logging.getLogger(__name__)

MAX_NUMBER_OF_SAN = 100

INTERNAL_CA_NAMES = frozenset([u'InternalCA', u'InternalTestCA'])


def validate_request(req_pb, auth_subject):
    """
    :raises: exceptions.BadRequestError
    """
    if isinstance(req_pb, api_pb2.GetCertificateRevisionRequest):
        if not req_pb.id:
            raise exceptions.BadRequestError(u'No "id" specified.')

    elif isinstance(req_pb, api_pb2.OrderCertificateRequest):
        validate_order_cert_request(req_pb, auth_subject)

    elif isinstance(req_pb, api_pb2.CreateCertificateRequest):
        validate_create_cert_request(req_pb)

    elif isinstance(req_pb, api_pb2.ListCertificateRevisionsRequest):
        if not req_pb.namespace_id:
            raise exceptions.BadRequestError(u'No "namespace_id" specified.')

    elif isinstance(req_pb, api_pb2.ListCertificatesRequest):
        validate_list_certs_request(req_pb)

    elif isinstance(req_pb, (api_pb2.GetCertificateRequest,
                             api_pb2.GetCertificateRenewalRequest)):
        if not req_pb.namespace_id:
            raise exceptions.BadRequestError(u'No "namespace_id" specified.')
        if not req_pb.id:
            raise exceptions.BadRequestError(u'No "id" specified.')

    elif isinstance(req_pb, (api_pb2.RemoveCertificateRequest,
                             api_pb2.RestoreCertificateFromBackupRequest,
                             api_pb2.UnpauseCertificateRenewalRequest,
                             api_pb2.CancelCertificateOrderRequest,
                             api_pb2.ForceCertificateRenewalRequest,
                             )):
        if not req_pb.namespace_id:
            raise exceptions.BadRequestError(u'No "namespace_id" specified.')
        if not req_pb.id:
            raise exceptions.BadRequestError(u'No "id" specified.')
        if not req_pb.version:
            raise exceptions.BadRequestError(u'No "version" specified.')
        if isinstance(req_pb, api_pb2.RemoveCertificateRequest) and not req_pb.state:
            raise exceptions.BadRequestError(u'No "state" specified.')
        if isinstance(req_pb, api_pb2.UnpauseCertificateRenewalRequest) and req_pb.HasField('target_discoverability'):
            validate_cert_discoverability(req_pb.target_discoverability, field_name=u'target_discoverability')

    elif isinstance(req_pb, (api_pb2.UpdateCertificateRequest,
                             api_pb2.UpdateCertificateRenewalRequest)):
        validate_update_cert_request(req_pb)
    elif isinstance(req_pb, api_pb2.ListCertificateRenewalsRequest):
        pass
    else:
        raise RuntimeError(u'Unsupported `req_pb` type: {}'.format(req_pb.__class__.__name__))
    forbid_actions_during_transfer(req_pb)


def forbid_actions_during_transfer(req_pb):
    c = cache.IAwacsCache.instance()
    if isinstance(req_pb, (api_pb2.RemoveCertificateRequest,
                           api_pb2.RestoreCertificateFromBackupRequest,
                           api_pb2.UnpauseCertificateRenewalRequest,
                           api_pb2.ForceCertificateRenewalRequest,
                           )):
        cert_pb = c.must_get_cert(req_pb.namespace_id, req_pb.id)
    elif isinstance(req_pb, api_pb2.UpdateCertificateRequest):
        cert_pb = c.must_get_cert(req_pb.meta.namespace_id, req_pb.meta.id)
    else:
        return
    if cert_pb.meta.is_being_transferred.value:
        raise exceptions.ForbiddenError(u"Cannot modify certificate while it's being transferred")


def validate_cert_order_content(content_pb, login, field_name='order'):
    """
    :type content_pb: model_pb2.CertificateOrder.Content
    :type field_name: six.text_type
    :type login: six.text_type
    :raises: exceptions.BadRequestError
    """
    if not content_pb.abc_service_id:
        raise exceptions.BadRequestError(u'"{}.abc_service_id" must be set'.format(field_name))
    cn = content_pb.common_name
    if not cn:
        raise exceptions.BadRequestError(u'"{}.common_name" must be set'.format(field_name))
    if not content_pb.ca_name:
        raise exceptions.BadRequestError(u'"{}.ca_name" must be set'.format(field_name))

    if appconfig.get_value('run.disable_tls_hostname_issuability_validation', default=False):
        issuable_tls_hostnames = None
    else:
        try:
            with gevent.Timeout(5):
                issuable_tls_hostnames = get_issuable_tls_hostnames()
        except (gevent.Timeout, Exception):
            log.exception(u'Failed to get issuable TLS hostnames from certificator during cert order validation')
            # we've failed to communicate with certificator, so let's just skip synchronous validation and fail later,
            # during order processing
            issuable_tls_hostnames = None

    try:
        validate_san(cn)
        if issuable_tls_hostnames is not None:
            validate_tls_hostname_issuability(cn, issuable_tls_hostnames=issuable_tls_hostnames)
    except Exception as e:
        raise exceptions.BadRequestError(u'"{}" in "{}.common_name" is not valid: {}'.format(cn, field_name, e))
    if cn in content_pb.subject_alternative_names:
        raise exceptions.BadRequestError(
            u'"{0}.common_name" must not be included in "{0}.subject_alternative_names": "{1}"'.format(field_name, cn))
    san_amount = len(content_pb.subject_alternative_names)
    if san_amount > MAX_NUMBER_OF_SAN:
        raise exceptions.BadRequestError(
            u'"{0}.subject_alternative_names" cannot contain more than 100 domains. Current amount: {1}'.format(
                field_name, san_amount))
    for domain in content_pb.subject_alternative_names:
        try:
            validate_san(domain)
            if issuable_tls_hostnames is not None:
                validate_tls_hostname_issuability(domain, issuable_tls_hostnames=issuable_tls_hostnames)
        except Exception as e:
            raise exceptions.BadRequestError(u'"{}" in "{}.subject_alternative_names" is not valid: {}'.format(
                domain, field_name, e))
    if content_pb.ttl:
        if not is_root(login):
            raise exceptions.ForbiddenError(u'"{}.ttl" can be set only by admins'.format(field_name))
        if content_pb.ttl > 182 or content_pb.ttl < 14:
            raise exceptions.BadRequestError(u'"{}.ttl": incorrect value "{}", must be between 14 and 182'.format(
                field_name, content_pb.ttl))


def validate_order_cert_request(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.OrderCertificateRequest
    :raises: exceptions.BadRequestError
    """
    validate_meta(req_pb)
    if not req_pb.HasField('order'):
        raise exceptions.BadRequestError(u'"order" must be set')
    validate_cert_order_content(req_pb.order, auth_subject, "order")


def validate_normalized_cert_id(namespace_id, cert_id):
    normalized_cert_id = certs.normalize_cert_id(cert_id)
    for existing_cert_id in cache.IAwacsCache.instance().list_all_cert_ids(namespace_id=namespace_id):
        if normalized_cert_id == certs.normalize_cert_id(existing_cert_id):
            raise exceptions.ConflictError(
                u'"meta.id": normalized id matches existing certificate "{}". '
                u'To avoid this, change or add any alphanumeric characters in this id'.format(existing_cert_id))


def validate_meta(req_pb):
    if not req_pb.HasField('meta'):
        raise exceptions.BadRequestError(u'"meta" must be set')
    if not req_pb.meta.id:
        raise exceptions.BadRequestError(u'"meta.id" must be set')
    if not CERT_ID_RE.match(req_pb.meta.id):
        raise exceptions.BadRequestError(u'"meta.id" is not valid')
    if not req_pb.meta.namespace_id:
        raise exceptions.BadRequestError(u'"meta.namespace_id" must be set')
    if not NAMESPACE_ID_RE.match(req_pb.meta.namespace_id):
        raise exceptions.BadRequestError(u'"meta.namespace_id" is not valid')
    if not req_pb.meta.HasField('auth'):
        raise exceptions.BadRequestError(u'"meta.auth" must be set')
    validate_auth_pb(req_pb.meta.auth, field_name=u'meta.auth')
    validate_normalized_cert_id(req_pb.meta.namespace_id, req_pb.meta.id)


def validate_create_cert_request(req_pb):
    """
    :type req_pb: api_pb2.CreateCertificateRequest
    :raises: exceptions.BadRequestError
    """
    validate_meta(req_pb)
    if not req_pb.HasField('spec'):
        raise exceptions.BadRequestError(u'"spec" must be set')
    if req_pb.spec.incomplete:
        raise exceptions.BadRequestError(u'"spec.incomplete" cannot be set on import')
    validate_create_cert_spec(req_pb.spec, u"spec")


def validate_list_certs_request(req_pb):
    """
    :type req_pb: api_pb2.ListCertificateRequest
    :raises: exceptions.BadRequestError
    """
    if not req_pb.namespace_id:
        raise exceptions.BadRequestError(u'No "namespace_id" specified.')
    if req_pb.HasField('query') and req_pb.query.id_regexp:
        try:
            re.compile(req_pb.query.id_regexp)
        except Exception as e:
            raise exceptions.BadRequestError(u'"query.id_regexp" contains invalid regular expression: {}'.format(e))


def validate_cert_discoverability(discoverability_pb, field_name='meta.discoverability'):
    """
    :type discoverability_pb: model_pb2.DiscoverabilityCondition
    :type field_name: six.text_type
    :raises: exceptions.BadRequestError
    """
    if not discoverability_pb.HasField('default'):
        raise exceptions.BadRequestError(u'"{}.default" must be set'.format(field_name))

    for location in discoverability_pb.per_location.values:
        if location not in BALANCER_LOCATIONS:
            raise exceptions.BadRequestError(
                u'"{}.per_location.values" must only contain '
                u'balancer locations: "{}"'.format(field_name, quote_join_sorted(BALANCER_LOCATIONS)))


def validate_update_cert_request(req_pb):
    """
    :type req_pb: api_pb2.UpdateCertificateRequest or api_pb2.UpdateCertificateRenewalRequest
    :raises: exceptions.BadRequestError
    """
    if not req_pb.HasField('meta'):
        raise exceptions.BadRequestError(u'"meta" must be set')
    if not req_pb.meta.id:
        raise exceptions.BadRequestError(u'"meta.id" must be set')
    # Hotfix for https://st.yandex-team.ru/AWACS-983
    # if not ID_RE.match(req_pb.meta.id):
    #     raise exceptions.BadRequestError(u'"meta.id" is not valid')
    if not req_pb.meta.namespace_id:
        raise exceptions.BadRequestError(u'"meta.namespace_id" must be set')
    if not NAMESPACE_ID_RE.match(req_pb.meta.namespace_id):
        raise exceptions.BadRequestError(u'"meta.namespace_id" is not valid')

    if isinstance(req_pb, api_pb2.UpdateCertificateRequest) and not req_pb.meta.HasField('auth'):
        raise exceptions.BadRequestError(u'"meta.auth" must be set')

    if not req_pb.meta.version:
        raise exceptions.BadRequestError(u'"meta.version" must be set')
    if req_pb.meta.HasField('auth'):
        validate_auth_pb(req_pb.meta.auth, field_name='meta.auth')

    if isinstance(req_pb, api_pb2.UpdateCertificateRequest):
        if req_pb.meta.HasField('unrevokable'):
            if not req_pb.meta.unrevokable.comment:
                raise exceptions.BadRequestError(u'"meta.unrevokable.comment" must be set')

        if req_pb.meta.HasField('discoverability'):
            validate_cert_discoverability(req_pb.meta.discoverability, field_name=u'meta.discoverability')


def validate_create_cert_spec(spec_pb, field_name=u'spec'):
    """
    :type spec_pb: model_pb2.CertificateSpec
    :type field_name: six.text_type
    :raises: exceptions.BadRequestError
    """
    if spec_pb.storage.type == model_pb2.CertificateSpec.Storage.YA_VAULT:
        secret = spec_pb.storage.ya_vault_secret
        if not secret.secret_id:
            raise exceptions.BadRequestError(u'"{}.storage.ya_vault_secret.secret_id" must be set'.format(field_name))
        if not secret.secret_ver:
            raise exceptions.BadRequestError(u'"{}.storage.ya_vault_secret.secret_ver" must be set'.format(field_name))
    elif spec_pb.storage.type == model_pb2.CertificateSpec.Storage.NANNY_VAULT:
        secret = spec_pb.storage.nanny_vault_secret
        if not secret.keychain_id:
            raise exceptions.BadRequestError(
                u'"{}.storage.nanny_vault_secret.keychain_id" must be set'.format(field_name))
        if not secret.secret_id:
            raise exceptions.BadRequestError(
                u'"{}.storage.nanny_vault_secret.secret_id" must be set'.format(field_name))
        if not secret.secret_revision_id:
            raise exceptions.BadRequestError(
                u'"{}.storage.nanny_vault_secret.secret_revision_id" must be set'.format(field_name))
    else:
        raise exceptions.BadRequestError(u'"{}.storage.type" value "{}" is not supported'.format(field_name,
                                                                                                 spec_pb.storage.type))
    if spec_pb.source == spec_pb.IMPORTED:
        if not spec_pb.HasField('imported'):
            raise exceptions.BadRequestError(u'"{}.imported" must be set if source is IMPORTED'.format(field_name))
        if not spec_pb.imported.abc_service_id:
            raise exceptions.BadRequestError(u'"{}.imported.abc_service_id" must be set'.format(field_name))
    elif spec_pb.source in (spec_pb.CERTIFICATOR, spec_pb.CERTIFICATOR_TESTING):
        if not spec_pb.HasField('certificator'):
            raise exceptions.BadRequestError(u'"{}.certificator" must be set if source is {}'.format(
                field_name, model_pb2.CertificateSpec.Source.Name(spec_pb.source)))
        if not spec_pb.certificator.order_id:
            raise exceptions.BadRequestError(u'"{}.certificator.order_id" must be set'.format(field_name))
        if not spec_pb.certificator.abc_service_id:
            raise exceptions.BadRequestError(u'"{}.certificator.abc_service_id" must be set'.format(field_name))


def validate_remove_cert(req_pb, cert_pb):
    if is_order_in_progress(cert_pb):
        raise exceptions.BadRequestError(
            u"Cannot remove certificate while it's being ordered. "
            u"Please either cancel the order or wait until it's finished.")
    if req_pb.state == model_pb2.CertificateSpec.PRESENT:
        raise exceptions.BadRequestError(u'Use "state" other than "PRESENT" to remove certificate')
    if req_pb.state == model_pb2.CertificateSpec.REVOKED_AND_REMOVED_FROM_AWACS_AND_STORAGE:
        if not cert_pb.spec.certificator.order_id:
            raise exceptions.BadRequestError(u'Cannot revoke certificate: no "certificator.order_id" in cert spec')
        if cert_pb.meta.unrevokable.value:
            raise exceptions.BadRequestError(
                u'Cannot revoke certificate: "{}"'.format(cert_pb.meta.unrevokable.comment))
    if (req_pb.state in (model_pb2.CertificateSpec.REVOKED_AND_REMOVED_FROM_AWACS_AND_STORAGE,
                         model_pb2.CertificateSpec.REMOVED_FROM_AWACS_AND_STORAGE) and
            cert_pb.spec.storage.type != model_pb2.CertificateSpec.Storage.YA_VAULT):
        raise exceptions.BadRequestError(u'Cannot remove certificate from storage: {} storage is not supported'.format(
            model_pb2.CertificateSpec.Storage.StorageType.Name(cert_pb.spec.storage.type)
        ))


def validate_cancel_cert_order(cert_pb):
    if not is_order_in_progress(cert_pb):
        raise exceptions.BadRequestError(u'Cannot cancel order that is not in progress')
    if not can_be_cancelled(cert_pb, get_cert_state_processors()):
        raise exceptions.BadRequestError(u'Cannot cancel cert order at this stage')


def validate_restore_cert_from_backup(cert_spec_pb):
    """
    :type cert_spec_pb: model_pb2.CertificateSpec
    :raises: exceptions.BadRequestError
    """
    if cert_spec_pb.fields.validity.not_after.ToSeconds() <= time.time():
        raise exceptions.BadRequestError(u'Certificate backup has already expired')
    if (cert_spec_pb.fields.validity.not_after.ToSeconds() + 3600) <= time.time():
        raise exceptions.BadRequestError(u'Certificate backup expires in less than an hour, backup is disabled')
