# coding: utf-8
import re

from awacs.lib.rpc import exceptions
from awacs.lib.strutils import quote_join_sorted
from awacs.model import components
from awacs.model.balancer.order.util import get_available_balancer_locations
from infra.awacs.proto import api_pb2, model_pb2

RBTORRENT_PATTERN = r'^rbtorrent:([a-z0-9]){40}$'
RBTORRENT_RE = re.compile(RBTORRENT_PATTERN)
REASON_PATTERN = r'^[A-Z]+-[0-9]+$'
REASON_RE = re.compile(REASON_PATTERN)

ALLOWED_TASK_TYPES = frozenset(['NANNY_REMOTE_COPY_RESOURCE', 'YA_MAKE', 'YA_MAKE_2', 'MDS_UPLOAD', 'HTTP_UPLOAD',
                                'HTTP_UPLOAD2', 'YA_PACKAGE', 'YA_MAKE_TGZ'])


def validate_sandbox_resource(resource_pb, component, field_name='spec.source.sandbox_resource'):
    """
    :type resource_pb: model_pb2.ComponentSpec.Source.SandboxResource
    :type component: components.ComponentConfig
    :type field_name: six.text_type
    """
    if not resource_pb.task_type:
        raise exceptions.BadRequestError('"{}.task_type" must be set'.format(field_name))
    if not resource_pb.task_id:
        raise exceptions.BadRequestError('"{}.task_id" must be set'.format(field_name))
    if not resource_pb.resource_type:
        raise exceptions.BadRequestError('"{}.resource_type" must be set'.format(field_name))
    if not resource_pb.resource_id:
        raise exceptions.BadRequestError('"{}.resource_id" must be set'.format(field_name))

    sandbox_resource = component.sandbox_resource
    if resource_pb.task_type not in sandbox_resource.task_types and resource_pb.task_type not in ALLOWED_TASK_TYPES:
        task_types_str = quote_join_sorted(sandbox_resource.task_types)
        raise exceptions.BadRequestError('"{}.task_type" must be one of '
                                         '"{}" for selected component type'.format(field_name, task_types_str))
    if resource_pb.resource_type not in sandbox_resource.resource_types:
        resource_types_str = quote_join_sorted(sandbox_resource.resource_types)
        raise exceptions.BadRequestError('"{}.resource_type" must be one of '
                                         '"{}" for selected component type'.format(field_name, resource_types_str))

    if resource_pb.rbtorrent and not RBTORRENT_RE.match(resource_pb.rbtorrent):
        raise exceptions.BadRequestError('"{}.rbtorrent" must match pattern "{}"'
                                         ''.format(field_name, RBTORRENT_PATTERN))


def validate_draft_component_request(req_pb):
    """
    :type req_pb: api_pb2.DraftComponentRequest
    :raises: exceptions.BadRequestError
    """
    if not req_pb.type:
        raise exceptions.BadRequestError('"type" must be set')
    if not req_pb.version:
        raise exceptions.BadRequestError('"version" must be set')
    if req_pb.type not in components.COMPONENT_CONFIGS_BY_TYPE:
        raise exceptions.BadRequestError('"type": is not supported')
    if not req_pb.startrek_issue_key:
        raise exceptions.BadRequestError('"startrek_issue_key" must be set')
    if req_pb.startrek_issue_key and not REASON_RE.match(req_pb.startrek_issue_key):
        raise exceptions.BadRequestError('"startrek_issue_key" must match pattern "{}"'.format(REASON_PATTERN))
    component_config = components.get_component_config(req_pb.type)
    try:
        component_config.validate_version(req_pb.version)
    except ValueError as e:
        raise exceptions.BadRequestError(u'"version": {}'.format(e))
    source_pb = req_pb.spec.source
    component_config.validate_source_type(source_pb.WhichOneof('sources'), field_name='spec.source')
    if source_pb.HasField('sandbox_resource'):
        validate_sandbox_resource(source_pb.sandbox_resource, component_config)


def validate_publish_component_request(req_pb):
    """
    :type req_pb: api_pb2.PublishComponentRequest
    :raises: exceptions.BadRequestError
    """
    if not req_pb.type:
        raise exceptions.BadRequestError('"type" must be set')
    if not req_pb.version:
        raise exceptions.BadRequestError('"version" must be set')
    if req_pb.startrek_issue_key and not REASON_RE.match(req_pb.startrek_issue_key):
        raise exceptions.BadRequestError('"startrek_issue_key" must match pattern "{}"'.format(REASON_PATTERN))


def validate_retire_component_request(req_pb):
    """
    :type req_pb: api_pb2.RetireComponentRequest
    :raises: exceptions.BadRequestError
    """
    if not req_pb.type:
        raise exceptions.BadRequestError('"type" must be set')
    if not req_pb.version:
        raise exceptions.BadRequestError('"version" must be set')
    if not req_pb.superseded_by:
        raise exceptions.BadRequestError('"superseded_by" must be set')
    try:
        components.get_component_config(req_pb.type).validate_version(req_pb.superseded_by)
    except ValueError as e:
        raise exceptions.BadRequestError(u'"superseded_by": {}'.format(e))


def validate_list_components_request(req_pb):
    """
    :type req_pb: api_pb2.ListComponentsRequest
    :raises: exceptions.BadRequestError
    """
    if req_pb.HasField('field_mask'):
        if not req_pb.field_mask.IsValidForDescriptor(model_pb2.Component.DESCRIPTOR):
            raise exceptions.BadRequestError('"field_mask" is not valid')


def validate_request(req_pb):
    """
    :raises: exceptions.BadRequestError
    """
    if isinstance(req_pb, (api_pb2.GetComponentRequest,
                           api_pb2.RemoveComponentRequest,
                           )):
        if not req_pb.type:
            raise exceptions.BadRequestError('"type" must be set')
        if not req_pb.version:
            raise exceptions.BadRequestError('"version" must be set')
    elif isinstance(req_pb, api_pb2.GetComponentUsageRequest):
        if not req_pb.type:
            raise exceptions.BadRequestError('"type" must be set')
    elif isinstance(req_pb, api_pb2.SetComponentAsDefaultRequest):
        if not req_pb.type:
            raise exceptions.BadRequestError('"type" must be set')
        if not req_pb.version:
            raise exceptions.BadRequestError('"version" must be set')
        if not req_pb.cluster:
            raise exceptions.BadRequestError('"cluster" must be set')
        locations = [dc.upper() for dc in get_available_balancer_locations()]
        if req_pb.cluster not in locations:
            raise exceptions.BadRequestError('"cluster" must be one of "{}"'.format('", "'.join(locations)))
    elif isinstance(req_pb, api_pb2.ListComponentsRequest):
        validate_list_components_request(req_pb)
    elif isinstance(req_pb, api_pb2.DraftComponentRequest):
        validate_draft_component_request(req_pb)
    elif isinstance(req_pb, api_pb2.PublishComponentRequest):
        validate_publish_component_request(req_pb)
    elif isinstance(req_pb, api_pb2.RetireComponentRequest):
        validate_retire_component_request(req_pb)
    else:
        raise RuntimeError('Incorrect `req_pb` type: {}'.format(req_pb.__class__.__name__))
