import ipaddr
import six

from awacs.lib.order_processor.model import is_order_in_progress, can_be_cancelled
from awacs.lib.rpc import exceptions
from awacs.model import cache, zk
from awacs.model.l3_balancer.l3mgr import is_fully_managed
from awacs.model.dns_records.operations.modify_addresses import get_modify_address_processors
from awacs.model.dns_records.order.processors import get_dns_record_order_processors
from infra.awacs.proto import api_pb2, model_pb2
from awacs.web.validation.util import ID_RE, NAMESPACE_ID_RE, validate_auth_pb, validate_domain_name, is_root, get_first_item


CTL_VERSIONS = (0, 1)


def validate_request(req_pb, auth_subject):
    """
    :raises: exceptions.BadRequestError
    """
    if isinstance(req_pb, api_pb2.ListNameServersRequest):
        pass

    elif isinstance(req_pb, api_pb2.GetDnsRecordRevisionRequest):
        if not req_pb.id:
            raise exceptions.BadRequestError('No "id" specified.')

    elif isinstance(req_pb, api_pb2.CreateDnsRecordRequest):
        validate_create_dns_record_request(req_pb, auth_subject)

    elif isinstance(req_pb, api_pb2.CreateDnsRecordOperationRequest):
        validate_create_dns_record_op_request(req_pb)

    elif isinstance(req_pb, (api_pb2.ListDnsRecordsRequest,
                             api_pb2.ListDnsRecordOperationsRequest)):
        if not req_pb.namespace_id:
            raise exceptions.BadRequestError('No "namespace_id" specified.')

    elif isinstance(req_pb, (api_pb2.GetDnsRecordRequest,
                             api_pb2.GetDnsRecordOperationRequest,
                             api_pb2.GetDnsRecordStateRequest,
                             api_pb2.ListDnsRecordRevisionsRequest,
                             api_pb2.GetNameServerConfigRequest,
                             api_pb2.RemoveDnsRecordRequest,
                             api_pb2.CancelDnsRecordOrderRequest,
                             api_pb2.CancelDnsRecordOperationRequest,
                             api_pb2.ResolveDnsRecordRequest,
                             api_pb2.GetNameServerRequest,
                             )):
        if not req_pb.namespace_id:
            raise exceptions.BadRequestError('No "namespace_id" specified.')
        if not req_pb.id:
            raise exceptions.BadRequestError('No "id" specified.')

    elif isinstance(req_pb, api_pb2.UpdateDnsRecordRequest):
        validate_update_dns_record_request(req_pb)

    else:
        raise RuntimeError('Unsupported `req_pb` type: {}'.format(req_pb.__class__.__name__))


def validate_ctl_version(spec_pb, field_name):
    if spec_pb.ctl_version not in CTL_VERSIONS:
        raise exceptions.BadRequestError('"{}.ctl_version" is not valid: supported versions are {}'.format(
            field_name, ', '.join(six.text_type(v) for v in CTL_VERSIONS)))


def validate_backend(backend_type, namespace_id, backend_id, i, prev_backend_ids, field_name):
    c = cache.IAwacsCache.instance()
    if backend_type == model_pb2.DnsBackendsSelector.EXPLICIT:
        backend_pb = c.must_get_backend(namespace_id, backend_id)
    elif backend_type == model_pb2.DnsBackendsSelector.BALANCERS:
        balancer_pb = c.must_get_balancer(namespace_id, backend_id)
        if backend_id not in prev_backend_ids and balancer_pb.spec.deleted:
            raise exceptions.BadRequestError(
                '"{0}[{1}]": balancer "{2}" is marked as removed and cannot be used'.format(field_name, i, backend_id))
        backend_pb = c.get_system_backend_for_balancer(namespace_id, backend_id)
        if not backend_pb:
            raise exceptions.BadRequestError(
                '"{0}[{1}]": backend for balancer "{2}" not found'.format(field_name, i, backend_id))
    else:
        raise exceptions.BadRequestError('"{0}[{1}]": unknown backend type {2}'.format(field_name, i, backend_type))

    if backend_id not in prev_backend_ids and backend_pb.spec.deleted:
        raise exceptions.BadRequestError(
            '"{0}[{1}]": backend "{2}" is marked as removed and cannot be used'.format(
                field_name, i, backend_id))
    if backend_pb.spec.selector.type == model_pb2.BackendSelector.YP_ENDPOINT_SETS_SD:
        raise exceptions.BadRequestError(
            '"{0}[{1}]": backend "{2}" has type '
            'YP_ENDPOINT_SETS_SD, which cannot be used in DNS record'.format(
                field_name, i, backend_id))


def validate_backends(namespace_id, dns_record_id, content_pb, name_server_order_pb, field_name):
    backends_selector_pb = content_pb.address.backends
    field_name = '{}.address.backends'.format(field_name)
    c = cache.IAwacsCache.instance()
    name_server_pb = c.must_get_name_server(name_server_order_pb.namespace_id, name_server_order_pb.id)

    if backends_selector_pb.type == model_pb2.DnsBackendsSelector.L3_BALANCERS:
        if not backends_selector_pb.l3_balancers:
            raise exceptions.BadRequestError('"{}.l3_balancers" must be set'.format(field_name))
        for i, l3_balancer in enumerate(backends_selector_pb.l3_balancers):
            l3_balancer_pb = c.must_get_l3_balancer(namespace_id, l3_balancer.id)
            if l3_balancer_pb.spec.incomplete:
                raise exceptions.BadRequestError(
                    '"{}.l3_balancers[{}]": cannot use L3 balancer "{}" that is not created yet'.format(
                        field_name, i, l3_balancer.id))
            if not is_fully_managed(l3_balancer_pb.spec):
                raise exceptions.BadRequestError(
                    '"{}.l3_balancers[{}]": cannot use L3 balancer "{}" that is not fully managed by awacs'.format(
                        field_name, i, l3_balancer.id))
        return

    if (name_server_pb.spec.type == model_pb2.NameServerSpec.DNS_MANAGER or
            name_server_pb.spec.selector == model_pb2.NameServerSpec.L3_BALANCERS_ONLY):
        raise exceptions.BadRequestError(
            '"{}": must use L3 balancers as a backend for name server "{}"'.format(
                field_name, name_server_order_pb.id))

    if backends_selector_pb.type == model_pb2.DnsBackendsSelector.EXPLICIT:
        field_name = '{}.backends'.format(field_name)
        if not backends_selector_pb.backends:
            raise exceptions.BadRequestError('"{}" must be set'.format(field_name))
        backends_pb = backends_selector_pb.backends
    elif backends_selector_pb.type == model_pb2.DnsBackendsSelector.BALANCERS:
        field_name = '{}.balancers'.format(field_name)
        if not backends_selector_pb.balancers:
            raise exceptions.BadRequestError('"{}" must be set'.format(field_name))
        backends_pb = backends_selector_pb.balancers
    else:
        raise exceptions.BadRequestError('{}.type: "{}" is not supported yet'.format(
            field_name, model_pb2.DnsBackendsSelector.Type.Name(backends_selector_pb.type)))

    prev_backend_ids = set()
    prev_dns_record_pb = c.get_dns_record(namespace_id, dns_record_id)
    if prev_dns_record_pb and prev_dns_record_pb.spec.address.backends.type == backends_selector_pb.type:
        if backends_selector_pb.type == model_pb2.DnsBackendsSelector.EXPLICIT:
            for b_pb in prev_dns_record_pb.spec.address.backends.backends:
                prev_backend_ids.add(b_pb.id)
        else:
            for b_pb in prev_dns_record_pb.spec.address.backends.balancers:
                prev_backend_ids.add(b_pb.id)
    seen_backend_ids = set()
    for i, b_pb in enumerate(backends_pb):
        if b_pb.id in seen_backend_ids:
            raise exceptions.BadRequestError(
                '"{0}[{1}]" contains duplicate id: "{2}"'.format(field_name, i, b_pb.id))
        seen_backend_ids.add(b_pb.id)
        if backends_selector_pb.type == model_pb2.DnsBackendsSelector.EXPLICIT:
            if b_pb.namespace_id and b_pb.namespace_id != namespace_id:
                raise exceptions.BadRequestError(
                    '"{0}[{1}]": cannot use backend "{2}" from another namespace "{3}"'.format(
                        field_name, i, b_pb.id, b_pb.namespace_id))

        validate_backend(backends_selector_pb.type, namespace_id, b_pb.id, i, prev_backend_ids, field_name)


def validate_fqdn_uniqness(name_server_namespace_id, name_server_id, zone, namespace_id, dns_record_id, field_name):
    c = cache.IAwacsCache.instance()
    existing_fqdns = c.list_dns_record_fqdns()
    name_server_pb = c.must_get_name_server(name_server_namespace_id, name_server_id)
    existing_zone = existing_fqdns.get(name_server_pb.spec.zone)
    if not existing_zone:
        return
    existing_fqdn_info = existing_zone.get(zone)
    if not existing_fqdn_info:
        return
    ns_id = existing_fqdn_info.namespace_id
    d_id = existing_fqdn_info.id
    if ns_id == namespace_id and d_id == dns_record_id:
        return
    raise exceptions.BadRequestError(
        '"{0}": DNS record for "{1}.{2}" already exists: "{3}:{4}"'.format(
            field_name, zone, name_server_pb.spec.zone, ns_id, d_id))


def validate_ids(req_pb):
    if not req_pb.HasField('meta'):
        raise exceptions.BadRequestError('"meta" must be set')
    if not req_pb.meta.id:
        raise exceptions.BadRequestError('"meta.id" must be set')
    if not ID_RE.match(req_pb.meta.id):
        raise exceptions.BadRequestError('"meta.id" is not valid')
    if not req_pb.meta.namespace_id:
        raise exceptions.BadRequestError('"meta.namespace_id" must be set')
    if not NAMESPACE_ID_RE.match(req_pb.meta.namespace_id):
        raise exceptions.BadRequestError('"meta.namespace_id" is not valid')


def validate_create_dns_record_request(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.CreateDnsRecordRequest
    :raises: exceptions.BadRequestError
    """
    validate_ids(req_pb)

    if not req_pb.meta.HasField('auth'):
        raise exceptions.BadRequestError('"meta.auth" must be set')
    validate_auth_pb(req_pb.meta.auth, field_name='meta.auth')

    if req_pb.HasField('order'):
        content_pb = req_pb.order
        field_name = 'order'
    elif req_pb.HasField('spec'):
        content_pb = req_pb.spec
        field_name = 'spec'
    else:
        raise exceptions.BadRequestError('"spec" or "order" must be set')
    validate_dns_record_order(content_pb, auth_subject.login, req_pb.meta.namespace_id, req_pb.meta.id, field_name)


def validate_fqdn(name_server_namespace_id, name_server_id, zone, namespace_id, dns_record_id, field_name):
    """
    :type name_server_namespace_id: six.text_type
    :type name_server_id: six.text_type
    :type zone: six.text_type
    :type namespace_id: six.text_type
    :type dns_record_id: six.text_type
    :type field_name: six.text_type
    :raises exceptions.BadRequestError:
    """
    c = cache.IAwacsCache.instance()
    name_server_pb = c.must_get_name_server(name_server_namespace_id, name_server_id)
    fqdn = zone + "." + name_server_pb.spec.zone
    # NOTE in future turn in this validation not only for rtc.* zones.
    if name_server_pb.spec.type == name_server_pb.spec.AWACS_MANAGED:
        validate_wildcard_overlapping(fqdn, namespace_id, field_name)
    validate_fqdn_uniqness(name_server_namespace_id, name_server_id, zone, namespace_id, dns_record_id, field_name)
    if zone == "*":
        raise exceptions.BadRequestError('{}: wildcard={} is forbidden'.format(field_name, fqdn))


def validate_wildcard_overlapping(fqdn, namespace_id, field_name):
    c = cache.IAwacsCache.instance()
    splitted_fqdn = fqdn.split(".")
    if splitted_fqdn[0] == "*":
        wildcard_base = ".".join(splitted_fqdn[1:])
        namespaces_covered_fqdns_count = c.namespaces_covered_fqdns_count_by_dns_record_potential_wildcard.get(wildcard_base, None)
        if namespaces_covered_fqdns_count:
            namespaces_with_covered_fqdns = set(namespaces_covered_fqdns_count)
            if {namespace_id} != namespaces_with_covered_fqdns:
                conflictable_ns = get_first_item(namespaces_with_covered_fqdns)
                raise exceptions.BadRequestError('"{}": There are DNS record(s) with fqdn(s) in external namespace "{}" which are covered by wildcard {}'.format(
                    field_name, conflictable_ns, fqdn))
    for i in range(1, len(splitted_fqdn)):
        fqdn_part = ".".join(splitted_fqdn[i:])
        ns_wildcard_owner = c._namespace_id_by_dns_record_wildcard.get(fqdn_part, None)
        if ns_wildcard_owner is not None and ns_wildcard_owner != namespace_id:
            raise exceptions.BadRequestError('"{}": There is DNS record in external namespace "{}" with wildcard "*.{}" which coveres fqdn {}'.format(
                field_name, ns_wildcard_owner, fqdn_part, fqdn))


def validate_dns_record_order(content_pb, login, namespace_id, dns_record_id, field_name):
    """
    :type content_pb: model_pb2.DnsRecordSpec | model_pb2.DnsRecordOrder.Content
    :type login: six.text_type
    :type namespace_id: six.text_type
    :type dns_record_id: six.text_type
    :type field_name: six.text_type
    :raises: exceptions.BadRequestError
    """
    try:
        validate_domain_name(content_pb.address.zone)
    except Exception as e:
        raise exceptions.BadRequestError('"{}.address.zone" is not valid: {}'.format(field_name, e))
    if content_pb.address.backends.type == model_pb2.DnsBackendsSelector.L3_BALANCERS:
        if isinstance(content_pb, model_pb2.DnsRecordSpec):
            raise exceptions.BadRequestError('Cannot create DNS record from spec with L3 balancers backend, '
                                             'use "order" instead'.format(field_name))
    if not content_pb.name_server.namespace_id:
        raise exceptions.BadRequestError('"{}.name_server.namespace_id" must be set')
    if not content_pb.name_server.id:
        raise exceptions.BadRequestError('"{}.name_server.id" must be set')
    name_server_pb = cache.IAwacsCache.instance().must_get_name_server(content_pb.name_server.namespace_id,
                                                                       content_pb.name_server.id)
    if name_server_pb.spec.type == name_server_pb.spec.DNS_MANAGER and not is_root(login):
        raise exceptions.BadRequestError('"{}.name_server.id": Only awacs admins can use name server "{}"'.format(
            field_name, content_pb.name_server.id))
    validate_backends(namespace_id, dns_record_id, content_pb, content_pb.name_server, field_name)
    validate_fqdn(content_pb.name_server.namespace_id, content_pb.name_server.id, content_pb.address.zone,
                  namespace_id, dns_record_id, field_name)
    validate_ctl_version(content_pb, field_name)


def validate_create_dns_record_op_request(req_pb):
    """
    :type req_pb: api_pb2.CreateDnsRecordOperationRequest
    :raises: exceptions.BadRequestError
    """
    validate_ids(req_pb)
    if not req_pb.HasField('order'):
        raise exceptions.BadRequestError('"order" must be set')
    if not req_pb.order.content.HasField('modify_addresses'):
        raise exceptions.BadRequestError('"order.content.modify_addresses" must be set')
    if len(req_pb.order.content.modify_addresses.requests) == 0:
        raise exceptions.BadRequestError('"order.content.modify_addresses.requests" must be set')
    if req_pb.meta.dns_record_id != req_pb.meta.id:
        raise exceptions.BadRequestError('"meta.dns_record_id" must be equal to "meta.id"')
    dns_record_pb = zk.IZkStorage.instance().get_dns_record(req_pb.meta.namespace_id, req_pb.meta.dns_record_id)
    if dns_record_pb is None:
        raise exceptions.NotFoundError("Cannot create operation for DNS record that doesn't exist: \"{}:{}\"".format(
            req_pb.meta.namespace_id, req_pb.meta.dns_record_id
        ))
    ips = set()
    for i, request_pb in enumerate(req_pb.order.content.modify_addresses.requests):
        if request_pb.action not in (request_pb.CREATE, request_pb.REMOVE):
            raise exceptions.BadRequestError(
                u'"order.content.modify_addresses.requests[{}].action": '
                u'unsupported address modification action: "{}"'.format(
                    i, model_pb2.DnsRecordOperationOrder.Content.ModifyAddresses.Request.Action.Name(request_pb.action)
                ))
        address = request_pb.address
        try:
            ipaddr.IPAddress(address)
        except ValueError:
            raise exceptions.BadRequestError('"order.content.modify_addresses.requests[{}].address": '
                                             '"{}" is not a valid IP address'.format(i, address))
        if address in ips:
            raise exceptions.ConflictError(
                '"order.content.modify_addresses.requests[{}].address": '
                'used more than once in this operation'.format(i, address))
        ips.add(address)


def validate_update_dns_record_request(req_pb):
    """
    :type req_pb: api_pb2.UpdateDnsRecordRequest
    :raises: exceptions.BadRequestError
    """
    validate_ids(req_pb)
    c = cache.IAwacsCache.instance()
    dns_record_pb = c.must_get_dns_record(req_pb.meta.namespace_id, req_pb.meta.id)
    if is_order_in_progress(dns_record_pb):
        raise exceptions.BadRequestError('Cannot update DNS record while its order is in progress')

    if not req_pb.meta.version:
        raise exceptions.BadRequestError('"meta.version" must be set')

    if not req_pb.HasField('spec') and not req_pb.meta.HasField('auth'):
        raise exceptions.BadRequestError('at least one of the "spec" or "meta.auth" fields must be present')

    if req_pb.meta.HasField('auth'):
        validate_auth_pb(req_pb.meta.auth, field_name='meta.auth')

    if req_pb.HasField('spec'):
        validate_backends(req_pb.meta.namespace_id, req_pb.meta.id, req_pb.spec, dns_record_pb.spec.name_server, "spec")
        validate_ctl_version(req_pb.spec, "spec")


def validate_cancel_dns_record_order(dns_record_pb):
    if not is_order_in_progress(dns_record_pb):
        raise exceptions.BadRequestError('Cannot cancel order that is not in progress')
    if not can_be_cancelled(dns_record_pb, get_dns_record_order_processors()):
        raise exceptions.BadRequestError('Cannot cancel DNS record order at this stage')


def validate_cancel_dns_record_operation(dns_record_op_pb):
    if not is_order_in_progress(dns_record_op_pb):
        raise exceptions.BadRequestError('Cannot cancel operation that is not in progress')
    if not can_be_cancelled(dns_record_op_pb, get_modify_address_processors()):
        raise exceptions.BadRequestError('Cannot cancel DNS record operation at this stage')


def validate_remove_dns_record(dns_record_pb):
    if is_order_in_progress(dns_record_pb):
        raise exceptions.BadRequestError('Cannot remove DNS record while its order is in progress')


def validate_name_server_and_zone(old_spec, new_spec):
    if old_spec.name_server != new_spec.name_server:
        raise exceptions.BadRequestError('"spec.name_server": cannot change name server for an existing DNS record')
    if old_spec.address.zone != new_spec.address.zone:
        raise exceptions.BadRequestError('"spec.address.zone": cannot change zone for an existing DNS record')
