import copy
from collections import defaultdict

import re
import six

from awacs.lib.order_processor.model import is_order_in_progress, can_be_cancelled
from awacs.lib.rpc import exceptions
from awacs.lib.strutils import quote_join_sorted
from awacs.model import zk, cache
from awacs.model.certs.processors.processors import SIGNATURE_ALGO_EC, SIGNATURE_ALGO_RSA
from awacs.model.domain.operations.set_cert import get_set_cert_processors
from awacs.model.domain.operations.set_fqdns import get_set_fqdns_processors
from awacs.model.domain.operations.set_protocol import get_set_protocol_processors
from awacs.model.domain.operations.set_upstreams import get_set_upstreams_processors
from awacs.model.domain.operations.transfer import get_transfer_processors
from awacs.model.domain.order.processors import get_domain_state_processors
from awacs.model.errors import NotFoundError
from awacs.web.auth.core import authorize_create
from awacs.web.validation.certificate import validate_cert_order_content
from awacs.web.validation.util import NAMESPACE_ID_RE, validate_san, DOMAIN_ID_RE, get_first_item
from awacs.wrappers import l7macro
from awacs.wrappers.errors import ValidationError
from awacs.wrappers.main import IncludeUpstreams
from infra.awacs.proto import api_pb2, model_pb2, modules_pb2


MAX_NUMBER_OF_SAN = 100


def validate_request(req_pb, auth_subject):
    """
    :raises: exceptions.BadRequestError
    """
    if isinstance(req_pb, api_pb2.GetDomainRevisionRequest):
        if not req_pb.id:
            raise exceptions.BadRequestError(u'No "id" specified.')

    elif isinstance(req_pb, api_pb2.CreateDomainOperationRequest):
        validate_create_domain_operation_request(req_pb, auth_subject)

    elif isinstance(req_pb, api_pb2.CreateDomainRequest):
        validate_create_domain_request(req_pb, auth_subject)

    elif isinstance(req_pb, api_pb2.ListDomainOperationsRequest):
        if not req_pb.namespace_id:
            raise exceptions.BadRequestError(u'No "namespace_id" specified.')

    elif isinstance(req_pb, api_pb2.ListDomainsRequest):
        validate_list_domains_request(req_pb)

    elif isinstance(req_pb, (api_pb2.GetDomainRequest,
                             api_pb2.GetDomainOperationRequest,
                             api_pb2.ListDomainRevisionsRequest,
                             api_pb2.ResolveDomainFqdnsRequest,
                             )):
        if not req_pb.namespace_id:
            raise exceptions.BadRequestError(u'No "namespace_id" specified.')
        if not req_pb.id:
            raise exceptions.BadRequestError(u'No "id" specified.')

    elif isinstance(req_pb, (api_pb2.RemoveDomainRequest,
                             )):
        if not req_pb.namespace_id:
            raise exceptions.BadRequestError(u'No "namespace_id" specified.')
        if not req_pb.id:
            raise exceptions.BadRequestError(u'No "id" specified.')
        if not req_pb.version:
            raise exceptions.BadRequestError(u'No "version" specified.')
    elif isinstance(req_pb, (api_pb2.CancelDomainOrderRequest,
                             api_pb2.CancelDomainOperationRequest,
                             api_pb2.ChangeDomainOperationRequest,)):
        if not req_pb.namespace_id:
            raise exceptions.BadRequestError(u'No "namespace_id" specified.')
        if not req_pb.id:
            raise exceptions.BadRequestError(u'No "id" specified.')
    else:
        raise RuntimeError(u'Unsupported `req_pb` type: {}'.format(req_pb.__class__.__name__))
    forbid_actions_during_transfer(req_pb)


def validate_ids(req_pb):
    """
    :type req_pb: api_pb2.CreateDomainOperationRequest | api_pb2.CreateDomainRequest
    :raises: exceptions.BadRequestError
    """
    if not req_pb.HasField('meta'):
        raise exceptions.BadRequestError(u'"meta" must be set')
    if not req_pb.meta.id:
        raise exceptions.BadRequestError(u'"meta.id" must be set')
    if not DOMAIN_ID_RE.match(req_pb.meta.id) and not isinstance(req_pb, api_pb2.CreateDomainOperationRequest):  # BALANCERSUPPORT-3892
        raise exceptions.BadRequestError(u'"meta.id" is not valid')
    if not req_pb.meta.namespace_id:
        raise exceptions.BadRequestError(u'"meta.namespace_id" must be set')
    if not NAMESPACE_ID_RE.match(req_pb.meta.namespace_id):
        raise exceptions.BadRequestError(u'"meta.namespace_id" is not valid')


def forbid_actions_during_transfer(req_pb):
    c = cache.IAwacsCache.instance()
    if isinstance(req_pb, api_pb2.RemoveDomainRequest):
        domain_pb = c.must_get_domain(req_pb.namespace_id, req_pb.id)
    elif isinstance(req_pb, api_pb2.CreateDomainOperationRequest):
        domain_pb = c.must_get_domain(req_pb.meta.namespace_id, req_pb.meta.id)
    else:
        return
    if domain_pb.meta.is_being_transferred.value:
        raise exceptions.ForbiddenError(u"Cannot modify domain while it's being transferred")


def validate_create_domain_operation_request(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.CreateDomainOperationRequest
    :type auth_subject: awacs.lib.rpc.authentication.AuthSubject
    :raises: exceptions.BadRequestError
    """
    validate_ids(req_pb)

    if not req_pb.HasField('order'):
        raise exceptions.BadRequestError(u'"order" must be set')

    domain_pb = zk.IZkStorage.instance().must_get_domain(req_pb.meta.namespace_id, req_pb.meta.id, sync=True)
    if domain_pb.spec.deleted:
        raise exceptions.BadRequestError(u'Cannot create operation for removed domain')
    if domain_pb.spec.incomplete:
        raise exceptions.BadRequestError(u'Cannot create operation for incomplete domain')
    validate_domain_operation_order_content(domain_pb, req_pb.meta.namespace_id, req_pb.meta.id, req_pb.order,
                                            auth_subject)


def validate_create_domain_request(req_pb, auth_subject):
    """
    :type req_pb: api_pb2.CreateDomainRequest
    :type auth_subject: awacs.lib.rpc.authentication.AuthSubject
    :raises: exceptions.BadRequestError
    """
    validate_ids(req_pb)
    if req_pb.meta.id == l7macro.DOMAIN_NOT_FOUND_SECTION_NAME:
        raise exceptions.BadRequestError(u'id "{}" is reserved and cannot be used'.format(
            l7macro.DOMAIN_NOT_FOUND_SECTION_NAME))

    if not req_pb.HasField('order'):
        raise exceptions.BadRequestError(u'"order" must be set')

    validate_domain_order_content(req_pb.meta.namespace_id, req_pb.meta.id, req_pb.order, auth_subject,
                                  field_name=u'order.content')


def validate_list_domains_request(req_pb):
    """
    :type req_pb: api_pb2.ListDomainsRequest
    :raises: exceptions.BadRequestError
    """
    if not req_pb.namespace_id:
        raise exceptions.BadRequestError(u'No "namespace_id" specified.')
    if req_pb.HasField('query') and req_pb.query.id_regexp:
        try:
            re.compile(req_pb.query.id_regexp)
        except Exception as e:
            raise exceptions.BadRequestError(u'"query.id_regexp" contains invalid regular expression: {}'.format(e))


def validate_domain_names(fqdns, field_name):
    for i, fqdn in enumerate(fqdns):
        try:
            validate_san(fqdn)
        except ValueError as e:
            raise exceptions.BadRequestError(u'"{}[{}]": {} is not a valid FQDN: {}'.format(
                field_name, i, fqdn, six.text_type(e)))


def validate_cert_orders_or_refs(namespace_id, pb, login, field_name):
    """
    :type namespace_id: six.text_type
    :type pb: (model_pb2.DomainOrder.Content or
               model_pb2.DomainOperationOrder.Content.SetFqdns or
               model_pb2.DomainOperationOrder.Content.SetCert or
               model_pb2.DomainOperationOrder.Content.SetProtocol)
    :type field_name: six.text_type
    :type login: six.text_type
    :return:
    """
    c = cache.IAwacsCache.instance()
    cert_order_content_pb = cert_pb = secondary_cert_order_content_pb = secondary_cert_pb = None
    if pb.HasField('cert_order'):
        cert_order_content_pb = pb.cert_order.content
        validate_cert_order_content(cert_order_content_pb, login,
                                    field_name=u'{}.cert_order.content'.format(field_name))
    elif pb.HasField('cert_ref'):
        try:
            cert_pb = c.must_get_cert(namespace_id, pb.cert_ref.id)
        except NotFoundError:
            raise exceptions.BadRequestError(u'"{}.cert_ref.id": cert "{}" not found'.format(field_name,
                                                                                             pb.cert_ref.id))
    else:
        raise exceptions.BadRequestError(u'"{}": "cert_order" or "cert_ref" must be set'.format(field_name))

    if pb.HasField('secondary_cert_order'):
        secondary_cert_order_content_pb = pb.secondary_cert_order.content
        validate_cert_order_content(secondary_cert_order_content_pb, login,
                                    field_name=u'{}.secondary_cert_order.content'.format(field_name))
    elif pb.HasField('secondary_cert_ref'):
        if pb.secondary_cert_ref.id == pb.cert_ref.id:
            raise exceptions.BadRequestError(
                u'"{0}.secondary_cert_ref.id": must be different from "{0}.cert_ref.id"'.format(field_name))
        try:
            secondary_cert_pb = c.must_get_cert(namespace_id, pb.secondary_cert_ref.id)
        except NotFoundError:
            raise exceptions.BadRequestError(u'"{}.secondary_cert_ref.id": cert "{}" not found'.format(
                field_name, pb.secondary_cert_ref.id))
    else:
        return
    validate_primary_and_secondary_cert_compat(cert_order_content_pb, cert_pb,
                                               secondary_cert_order_content_pb, secondary_cert_pb,
                                               field_name)


def extract_cert_info(order_pb, cert_pb):
    if order_pb:
        return {
            'common_name': order_pb.common_name,
            'public_key_algorithm_id': order_pb.public_key_algorithm_id,
            'subject_alternative_names': sorted(order_pb.subject_alternative_names)
        }
    else:
        common_name = cert_pb.spec.fields.subject_common_name
        return {
            'common_name': common_name,
            'public_key_algorithm_id': cert_pb.spec.fields.public_key_info.algorithm_id,
            'subject_alternative_names': sorted(cn for cn in cert_pb.spec.fields.subject_alternative_names
                                                if cn != common_name)
        }


def validate_primary_and_secondary_cert_compat(cert_order_content_pb, cert_pb,
                                               secondary_cert_order_content_pb, secondary_cert_pb, field_name):
    primary_fields = extract_cert_info(cert_order_content_pb, cert_pb)
    secondary_fields = extract_cert_info(secondary_cert_order_content_pb, secondary_cert_pb)

    for cert_field_name, primary_value in sorted(six.iteritems(primary_fields)):
        secondary_value = secondary_fields[cert_field_name]
        if cert_field_name == u'public_key_algorithm_id':
            if primary_value == secondary_value:
                raise exceptions.BadRequestError(
                    u'"{}": field "public_key_algorithm_id" in primary and secondary certs must be different'.format(
                        field_name))
            if primary_value != SIGNATURE_ALGO_EC:
                raise exceptions.BadRequestError(
                    u'"{}": field "public_key_algorithm_id" in primary cert must have value "ec", not "{}"'.format(
                        field_name, primary_value))
            if secondary_value != SIGNATURE_ALGO_RSA:
                raise exceptions.BadRequestError(
                    u'"{}": field "public_key_algorithm_id" in secondary cert must have value "rsa", not "{}"'.format(
                        field_name, primary_value))
        elif primary_value != secondary_value:
            raise exceptions.BadRequestError(
                u'"{}": field "{}" in primary and secondary certs must match: "{}" != "{}"'.format(
                    field_name, cert_field_name, primary_value, secondary_value))


def validate_protocol(protocol, field_name):
    if protocol == model_pb2.DomainSpec.Config.NONE:
        raise exceptions.BadRequestError(u'"{}": "protocol" cannot be NONE'.format(field_name))


def validate_redirect_to_https(content_pb, protocol, field_name):
    if content_pb.HasField('redirect_to_https') and protocol != model_pb2.DomainSpec.Config.HTTP_AND_HTTPS:
        raise exceptions.BadRequestError(u'"{}": "redirect_to_https" can be enabled only for HTTP_AND_HTTPS'.format(
            field_name))


def validate_verify_client_cert(content_pb, protocol, field_name):
    if content_pb.HasField('verify_client_cert') and protocol == model_pb2.DomainSpec.Config.HTTP_ONLY:
        raise exceptions.BadRequestError(u'"{}": "verify_client_cert" can not be enabled only for HTTP_ONLY'.format(
            field_name))


def validate_domain_operation_order_content(domain_pb, namespace_id, domain_id, content_pb, auth_subject):
    """
    :type domain_pb: model_pb2.Domain
    :type content_pb: model_pb2.DomainOperationOrder.Content
    :type namespace_id: six.text_type
    :type domain_id: six.text_type
    :type auth_subject: awacs.lib.rpc.authentication.AuthSubject
    :raises: exceptions.BadRequestError
    """
    login = auth_subject.login
    config_type = domain_pb.spec.yandex_balancer.config.type
    if config_type == model_pb2.DomainSpec.Config.YANDEX_TLD:
        if content_pb.WhichOneof('operations') != 'set_upstreams':
            raise exceptions.BadRequestError(u'You can only change upstreams for YANDEX_TLD domain')
    if config_type == model_pb2.DomainSpec.Config.WILDCARD:
        if content_pb.WhichOneof('operations') not in ('set_upstreams', 'set_protocol', 'set_cert'):
            raise exceptions.BadRequestError(u'You can only change upstreams, protocol and cert for WILDCARD domain')
    if content_pb.HasField('set_fqdns'):
        if not content_pb.set_fqdns.fqdns and not content_pb.set_fqdns.shadow_fqdns:
            raise exceptions.BadRequestError(u'"order.set_fqdns": "fqdns" or "shadow_fqdns" must be set')
        validate_domain_names(content_pb.set_fqdns.fqdns, field_name=u'order.set_fqdns.fqdns')
        validate_domain_names(content_pb.set_fqdns.shadow_fqdns, field_name=u'order.set_fqdns.shadow_fqdns')
        validate_shadow_fqdns(content_pb.set_fqdns.fqdns, content_pb.set_fqdns.shadow_fqdns,
                              field_name=u'order.set_fqdns.shadow_fqdns')
        validate_domain_fqdn_uniqueness(namespace_id, domain_id, content_pb.set_fqdns.fqdns,
                                        field_name=u'order.set_fqdns',
                                        is_domain_op=True)
        if domain_pb.spec.yandex_balancer.config.protocol == model_pb2.DomainSpec.Config.HTTP_ONLY:
            return
        validate_cert_orders_or_refs(namespace_id, content_pb.set_fqdns, login, field_name=u'order.set_fqdns')
        all_fqdns = list(content_pb.set_fqdns.fqdns) + list(content_pb.set_fqdns.shadow_fqdns)
        validate_cert_fqdns(namespace_id, content_pb.set_fqdns, all_fqdns, field_name=u'order.set_fqdns')
    elif content_pb.HasField('set_cert'):
        if domain_pb.spec.yandex_balancer.config.protocol == model_pb2.DomainSpec.Config.HTTP_ONLY:
            return
        validate_cert_orders_or_refs(namespace_id, content_pb.set_cert, login, field_name=u'order.set_cert')
        all_fqdns = (list(domain_pb.spec.yandex_balancer.config.fqdns) +
                     list(domain_pb.spec.yandex_balancer.config.shadow_fqdns))
        validate_cert_fqdns(namespace_id, content_pb.set_cert, all_fqdns, field_name=u'order.set_cert')
    elif content_pb.HasField('set_protocol'):
        validate_protocol(content_pb.set_protocol.protocol, field_name=u'order.set_protocol')
        validate_redirect_to_https(content_pb.set_protocol.set_redirect_to_https, content_pb.set_protocol.protocol,
                                   field_name=u'order.set_protocol.set_redirect_to_https')
        validate_verify_client_cert(content_pb.set_protocol.set_verify_client_cert, content_pb.set_protocol.protocol,
                                    field_name=u'order.set_protocol.set_redirect_to_https')
        if content_pb.set_protocol.protocol != model_pb2.DomainSpec.Config.HTTP_ONLY:
            validate_cert_orders_or_refs(namespace_id, content_pb.set_protocol, login, field_name=u'order.set_protocol')
            all_fqdns = (list(domain_pb.spec.yandex_balancer.config.fqdns) +
                         list(domain_pb.spec.yandex_balancer.config.shadow_fqdns))
            validate_cert_fqdns(namespace_id, content_pb.set_protocol, all_fqdns, field_name=u'order.set_protocol')
    elif content_pb.HasField('set_upstreams'):
        validate_upstreams(content_pb.set_upstreams, field_name=u'order.set_upstreams')
    elif content_pb.HasField('transfer'):
        validate_upstreams(content_pb.transfer, field_name=u'order.transfer')
        target_namespace_id = content_pb.transfer.target_namespace_id
        if not target_namespace_id:
            raise exceptions.BadRequestError(u'"order.transfer.target_namespace_id": is required')
        validate_transfer(namespace_id=namespace_id,
                          target_namespace_id=target_namespace_id,
                          domain_id=domain_id,
                          domain_pb=domain_pb)
        authorize_create(namespace_pb=cache.IAwacsCache.instance().must_get_namespace(target_namespace_id),
                         auth_subject=auth_subject)
    else:
        raise exceptions.BadRequestError(u'"order": Operation field not specified')


def validate_transfer(namespace_id, target_namespace_id, domain_id, domain_pb):
    c = cache.IAwacsCache.instance()
    if c.get_domain(target_namespace_id, domain_id) is not None:
        raise exceptions.ConflictError(u'Domain "{}" already exists in target namespace "{}"'.format(
            domain_id, target_namespace_id
        ))
    for cert_id in (domain_pb.spec.yandex_balancer.config.cert.id,
                    domain_pb.spec.yandex_balancer.config.secondary_cert.id):
        if not cert_id:
            continue
        if c.get_cert(target_namespace_id, cert_id) is not None:
            raise exceptions.ConflictError(u'Certificate "{}" already exists in target namespace "{}"'.format(
                cert_id, target_namespace_id
            ))
        cert_pb = c.must_get_cert(namespace_id, cert_id)
        if not cert_pb.spec.storage.ya_vault_secret.secret_id:
            raise exceptions.BadRequestError(u'Cannot transfer certificate stored in Nanny Vault: "{}:{}"'.format(
                namespace_id, cert_id))
        cert_renewal_pb = c.get_cert_renewal(namespace_id, cert_id)
        if cert_renewal_pb is not None and cert_renewal_pb.meta.target_rev == cert_pb.meta.version:
            raise exceptions.BadRequestError(
                u'Certificate "{}:{}" must be renewed before transferring the domain'.format(namespace_id, cert_id))
        cert_usages = set()
        for d_pb in c.list_all_domains(namespace_id):
            if d_pb.meta.id == domain_id:
                continue
            if d_pb.spec.incomplete:
                cert_ref_id = d_pb.order.content.cert_ref.id
            else:
                cert_ref_id = d_pb.spec.yandex_balancer.config.cert.id
            if cert_ref_id == cert_id:
                cert_usages.add(d_pb.meta.id)
        for domain_op_pb in c.list_all_domain_operations(namespace_id):
            content_pb = domain_op_pb.order.content
            if any(cert_ref_id == cert_id for cert_ref_id in (content_pb.set_fqdns.cert_ref.id,
                                                              content_pb.set_protocol.cert_ref.id,
                                                              content_pb.set_cert.cert_ref.id)):
                cert_usages.add(domain_op_pb.meta.id)
        if cert_usages:
            raise exceptions.BadRequestError(
                u'Cannot transfer certificate "{}:{}" used in multiple domains: "{}"'.format(
                    namespace_id, cert_id, quote_join_sorted(cert_usages)))


def validate_upstreams(pb, field_name):
    if not pb.HasField('include_upstreams'):
        raise exceptions.BadRequestError(u'"{}.include_upstreams": must be set'.format(field_name))
    # We rely on these checks in awacs2d. Code below means that for domains type NONE is equivalent to type ALL
    if pb.include_upstreams.type == modules_pb2.NONE:
        if not pb.include_upstreams.filter.any:
            raise exceptions.BadRequestError(u'"{0}.include_upstreams.filter.any": must be true if '
                                             u'"{0}.include_upstreams.type is NONE"'.format(field_name))
        if pb.include_upstreams.order.label.name != 'order':
            raise exceptions.BadRequestError(u'"{0}.include_upstreams.filter.order.label.name": must be "order" if '
                                             u'"{0}.include_upstreams.type is NONE"'.format(field_name))
    include_upstreams = IncludeUpstreams(pb.include_upstreams)
    try:
        include_upstreams.validate()
    except ValidationError as e:
        raise exceptions.BadRequestError(u'"{}.include_upstreams": {}'.format(field_name, six.text_type(e)))


def validate_domain_order_content(namespace_id, domain_id, content_pb, auth_subject, field_name):
    """
    :type namespace_id: six.text_type
    :type domain_id: six.text_type
    :type content_pb: model_pb2.DomainOrder.Content
    :type field_name: six.text_type
    :type auth_subject: awacs.lib.rpc.authentication.AuthSubject
    :raises: exceptions.BadRequestError
    """
    if content_pb.type in (model_pb2.DomainSpec.Config.YANDEX_TLD, model_pb2.DomainSpec.Config.WILDCARD):
        for domain_pb in cache.IAwacsCache.instance().list_all_domains(namespace_id=namespace_id):
            if content_pb.type in (domain_pb.spec.yandex_balancer.config.type, domain_pb.order.content.type):
                raise exceptions.BadRequestError(u'Only one {} domain can exist in a namespace. Existing: {}'.format(
                    model_pb2.DomainSpec.Config.Type.Name(content_pb.type),
                    domain_pb.meta.id
                ))
    if content_pb.type == model_pb2.DomainSpec.Config.YANDEX_TLD:
        validate_yandex_tld_domain_order(content_pb, field_name)
    elif content_pb.type == model_pb2.DomainSpec.Config.WILDCARD:
        validate_wildcard_domain_order(content_pb, field_name)
    else:
        if not content_pb.fqdns and not content_pb.shadow_fqdns:
            raise exceptions.BadRequestError(u'"order.content": "fqdns" or "shadow_fqdns" must be set')
        validate_domain_names(content_pb.fqdns, field_name + u'.fqdns')
        validate_domain_names(content_pb.shadow_fqdns, field_name + u'.shadow_fqdns')
        validate_shadow_fqdns(content_pb.fqdns, content_pb.shadow_fqdns, field_name + u'.shadow_fqdns')
        validate_domain_fqdn_uniqueness(namespace_id, domain_id, content_pb.fqdns,
                                        field_name=field_name + u'.fqdns',
                                        is_domain_op=False)
    validate_protocol(content_pb.protocol, field_name)
    validate_redirect_to_https(content_pb, content_pb.protocol, field_name)
    validate_verify_client_cert(content_pb, content_pb.protocol, field_name)
    validate_upstreams(content_pb, field_name=field_name)
    if content_pb.protocol != model_pb2.DomainSpec.Config.HTTP_ONLY:
        validate_cert_orders_or_refs(namespace_id, content_pb, auth_subject.login, field_name=field_name)
        all_fqdns = list(content_pb.fqdns) + list(content_pb.shadow_fqdns)
        validate_cert_fqdns(namespace_id, content_pb, all_fqdns, field_name=field_name)


def validate_shadow_fqdns(fqdns, shadow_fqdns, field_name):
    uniq_shadow_fqdns = set()
    dupes = set()
    for fqdn in shadow_fqdns:
        if fqdn in uniq_shadow_fqdns:
            dupes.add(fqdn)
        else:
            uniq_shadow_fqdns.add(fqdn)
    if dupes:
        raise exceptions.BadRequestError(u'"{}": FQDNs "{}" are used multiple times'.format(field_name,
                                                                                            u'", "'.join(sorted(dupes))))
    uniq_fqdns = set(fqdns)
    dupes = uniq_shadow_fqdns.intersection(uniq_fqdns)
    if dupes:
        raise exceptions.BadRequestError(u'"{}": FQDNs "{}" '
                                         u'cannot be used in "fqdns" field at the same time'.format(field_name,
                                                                                                    u'", "'.join(
                                                                                                        sorted(dupes))))


def validate_yandex_tld_domain_order(content_pb, field_name):
    if content_pb.protocol != model_pb2.DomainSpec.Config.HTTP_ONLY:
        raise exceptions.BadRequestError(u'"{}": YANDEX_TLD domain must have protocol HTTP_ONLY'.format(field_name))
    if content_pb.fqdns:
        raise exceptions.BadRequestError(u'"{}": YANDEX_TLD domain cannot have custom fqdns'.format(field_name))


def validate_wildcard_domain_order(content_pb, field_name):
    if content_pb.fqdns:
        raise exceptions.BadRequestError(u'"{}": WILDCARD domain cannot have custom fqdns'.format(field_name))


def validate_cert_fqdns(namespace_id, content_pb, fqdns, field_name):
    """
    :type content_pb: (model_pb2.DomainOrder.Content or
                       model_pb2.DomainOperationOrder.Content.SetFqdns or
                       model_pb2.DomainOperationOrder.Content.SetCert or
                       model_pb2.DomainOperationOrder.Content.SetProtocol)
    :type namespace_id: six.text_type
    :type field_name: six.text_type
    :type fqdns: Iterable[six.text_type]
    :raises: exceptions.BadRequestError
    """
    if content_pb.HasField('cert_order'):
        # CN and SANs are stored separately in order
        cert_order_pb = content_pb.cert_order.content
        cert_fqdns = {cert_order_pb.common_name}.union(cert_order_pb.subject_alternative_names)
        cert_name = u'order'
    elif content_pb.HasField('cert_ref'):
        cert_pb = cache.IAwacsCache.instance().must_get_cert(namespace_id, content_pb.cert_ref.id)
        # SAN list includes CN as the first element in the final spec
        cert_fqdns = set(cert_pb.spec.fields.subject_alternative_names)
        cert_name = u'"{}"'.format(content_pb.cert_ref.id)
    else:
        raise exceptions.BadRequestError(u'"{}": Certificate reference or order are missing'.format(field_name))
    unmatched = get_fqdns_unmatched_by_cert(domain_fqdns=fqdns, cert_fqdns=cert_fqdns)
    if unmatched:
        raise exceptions.BadRequestError(
            u'"{}": FQDNs "{}" are not covered by certificate {}'.format(field_name, quote_join_sorted(unmatched),
                                                                         cert_name))


def validate_cancel_domain_operation(domain_op_pb):
    if not is_order_in_progress(domain_op_pb):
        raise exceptions.BadRequestError(u'Cannot cancel operation that is not in progress')
    if domain_op_pb.order.content.HasField('set_fqdns'):
        processors = get_set_fqdns_processors()
    elif domain_op_pb.order.content.HasField('set_protocol'):
        processors = get_set_protocol_processors()
    elif domain_op_pb.order.content.HasField('set_cert'):
        processors = get_set_cert_processors()
    elif domain_op_pb.order.content.HasField('set_upstreams'):
        processors = get_set_upstreams_processors()
    elif domain_op_pb.order.content.HasField('transfer'):
        processors = get_transfer_processors()
    else:
        raise exceptions.BadRequestError(u'Unknown domain operation')
    if not can_be_cancelled(domain_op_pb, processors):
        raise exceptions.BadRequestError(u'Cannot cancel domain operation at this stage')


def validate_cancel_domain_order(domain_pb):
    if not is_order_in_progress(domain_pb):
        raise exceptions.BadRequestError(u'Cannot cancel order that is not in progress')
    if not can_be_cancelled(domain_pb, get_domain_state_processors()):
        raise exceptions.BadRequestError(u'Cannot cancel domain order at this stage')


def get_prepared_domain_fqdns(fqdns, field_name):
    """
    :type fqdns: Iterable[six.text_type]
    :type field_name: six.text_type
    :rtype dict[six.text_type, set[six.text_type]]

    Map higher-level domain parts to their respective last-level domain parts, guaranteeing uniqness
    """
    rv = defaultdict(set)
    for fqdn in fqdns:
        last_level_part, higher_level_part = fqdn.split(u'.', 1)
        existing = rv.get(higher_level_part)
        if existing:
            if last_level_part == u'*':
                raise exceptions.BadRequestError(
                    u'"{0}": FQDN "{1}.{2}" is already represented by FQDN "*.{2}" in this domain'.format(
                        field_name, existing.pop(), higher_level_part))
            if u'*' in existing:
                raise exceptions.BadRequestError(
                    u'"{0}": FQDN "{1}.{2}" is already represented by FQDN "*.{2}" in this domain'.format(
                        field_name, last_level_part, higher_level_part))
            if last_level_part in existing:
                raise exceptions.BadRequestError(
                    u'"{}": FQDN "{}" is used more than once in this domain'.format(field_name, fqdn))
        rv[higher_level_part].add(last_level_part)
    return rv


def validate_domain_fqdn_uniqueness(namespace_id, domain_id, fqdns, field_name, is_domain_op):
    """
    Separate each fqdn into last-level part (before the first dot), and the higher-level part (the rest of the fqdn).
    For each existing higher-level fqdn part,
    check that new fqdns with that same part don't have overlapping last-level parts.
    Error messages don't show full duplicate list to make them more readable.

    :type namespace_id: six.text_type
    :type domain_id: six.text_type
    :type fqdns: Iterable[six.text_type]
    :type field_name: six.text_type
    :type is_domain_op: bool
    """
    prepared_fqdns = get_prepared_domain_fqdns(fqdns, field_name)
    existing_domain_fqdns = copy.deepcopy(cache.IAwacsCache.instance().list_domain_fqdns())  # so we don't corrupt cache
    for new_higher_level_part, new_last_level_parts in six.iteritems(prepared_fqdns):
        existing_last_level_parts = existing_domain_fqdns.get(new_higher_level_part)
        if not existing_last_level_parts:
            continue
        if is_domain_op:  # filter out its own domain, since it will conflict with the operation's fqdns
            for last_level_name, records in list(six.iteritems(existing_last_level_parts)):
                fqdn_info = cache.fqdn_info_tuple(u'domain', namespace_id, domain_id)
                records.discard(fqdn_info)
                if len(records) == 0:
                    existing_last_level_parts.pop(last_level_name)
        if not existing_last_level_parts:
            continue
        if u'*' in new_last_level_parts:
            duplicate = get_first_item(existing_last_level_parts)
            entity, e_ns_id, e_id = get_first_item(existing_last_level_parts[duplicate])
            raise exceptions.BadRequestError(
                u'"{0}": FQDN "*.{1}" would overlap FQDN "{2}.{1}" that already exists in {3} "{4}:{5}"'.format(
                    field_name, new_higher_level_part, duplicate, entity, e_ns_id, e_id))
        if u'*' in existing_last_level_parts:
            entity, e_ns_id, e_id = get_first_item(existing_last_level_parts[u'*'])
            duplicate = get_first_item(new_last_level_parts)
            raise exceptions.BadRequestError(
                u'"{0}": FQDN "{1}.{2}" is already represented by FQDN "*.{2}" in {3} "{4}:{5}"'.format(
                    field_name, duplicate, new_higher_level_part, entity, e_ns_id, e_id))
        duplicates = new_last_level_parts & set(existing_last_level_parts.keys())
        if duplicates:
            duplicate = get_first_item(duplicates)
            entity, e_ns_id, e_id = get_first_item(existing_last_level_parts[duplicate])
            raise exceptions.BadRequestError(
                u'"{0}": FQDN "{1}.{2}" already exists in {3} "{4}:{5}"'.format(
                    field_name, duplicate, new_higher_level_part, entity, e_ns_id, e_id))


def get_fqdns_unmatched_by_cert(domain_fqdns, cert_fqdns):
    """
    :type domain_fqdns: Iterable[six.text_type]
    :type cert_fqdns: Iterable[six.text_type]
    """
    cert_fqdn_parts = defaultdict(set)
    for fqdn in cert_fqdns:
        last_level_part, higher_level_part = fqdn.split(u'.', 1)
        cert_fqdn_parts[higher_level_part].add(last_level_part)
    matches = set()
    for fqdn in domain_fqdns:
        last_level_part, higher_level_part = fqdn.split(u'.', 1)
        cert_last_level_parts = cert_fqdn_parts.get(higher_level_part, frozenset())
        if u'*' in cert_last_level_parts or last_level_part in cert_last_level_parts:
            matches.add(fqdn)
    unmatched = set(domain_fqdns) - matches
    return unmatched
