# coding: utf-8
from awacs.lib import validators

from awacs.lib.rpc import exceptions
from infra.awacs.proto import api_pb2, model_pb2
from .util import ID_RE, COMPAT_ID_RE, NAMESPACE_ID_RE, validate_auth_pb


def validate_spec(spec_pb, field_name='spec'):
    """
    :type spec_pb: model_pb2.EndpointSetSpec
    :type field_name: six.text_type
    :raises: exceptions.BadRequestError
    """
    if spec_pb.deleted:
        raise exceptions.BadRequestError('"{}.deleted" must not be set'.format(field_name))
    if not spec_pb.instances and not spec_pb.deleted:
        raise exceptions.BadRequestError('"{}.instances" must be set'.format(field_name))
    seen_instances = set()
    for i, instance_pb in enumerate(spec_pb.instances):
        item_name = '{}.instances[{}]'.format(field_name, i)
        if not instance_pb.host:
            raise exceptions.BadRequestError('"{}.host" must be set'.format(item_name))
        if not instance_pb.port:
            raise exceptions.BadRequestError('"{}.port" must be set'.format(item_name))
        if not instance_pb.weight:
            raise exceptions.BadRequestError(
                '"{}.weight" must be set and must not be 0. '
                'Please use -1 instead of 0 to assign zero weight to the instance'.format(item_name))
        if instance_pb.ipv4_addr and not validators.ipv4(instance_pb.ipv4_addr):
            raise exceptions.BadRequestError('"{}.ipv4_addr": is not a valid IPv4 address'.format(item_name))
        if instance_pb.ipv6_addr:
            if not validators.ipv6(instance_pb.ipv6_addr):
                raise exceptions.BadRequestError('"{}.ipv6_addr": is not a valid IPv6 address'.format(item_name))
        else:
            raise exceptions.BadRequestError('"{}.ipv6_addr" must be set'.format(item_name))
        if (instance_pb.host, instance_pb.port) in seen_instances:
            raise exceptions.BadRequestError(('"{}.instances[{}]": duplicate host and port "{}:{}"'
                                              ''.format(field_name, i, instance_pb.host, instance_pb.port)))
        seen_instances.add((instance_pb.host, instance_pb.port))


def validate_create_endpoint_set_request(req_pb):
    """
    :type req_pb: api_pb2.CreateEndpointSetRequest
    :raises: exceptions.BadRequestError
    """
    if not req_pb.HasField('meta'):
        raise exceptions.BadRequestError('"meta" must be set')
    if not req_pb.meta.id:
        raise exceptions.BadRequestError('"meta.id" must be set')
    if not ID_RE.match(req_pb.meta.id):
        raise exceptions.BadRequestError('"meta.id" is not valid')
    if not req_pb.meta.namespace_id:
        raise exceptions.BadRequestError('"meta.namespace_id" must be set')
    if not NAMESPACE_ID_RE.match(req_pb.meta.namespace_id):
        raise exceptions.BadRequestError('"meta.namespace_id" is not valid')
    if not req_pb.meta.HasField('auth'):
        raise exceptions.BadRequestError('"meta.auth" must be set')
    if not req_pb.backend_version:
        raise exceptions.BadRequestError('"backend_version" must be set')
    validate_auth_pb(req_pb.meta.auth, field_name='meta.auth')
    if not req_pb.HasField('spec'):
        raise exceptions.BadRequestError('"spec" must be set')
    validate_spec(req_pb.spec)


def validate_update_endpoint_set_request(req_pb):
    """
    :type req_pb: api_pb2.UpdateEndpointSetRequest
    :raises: exceptions.BadRequestError
    """
    if not req_pb.HasField('meta'):
        raise exceptions.BadRequestError('"meta" must be set')
    if not req_pb.meta.id:
        raise exceptions.BadRequestError('"meta.id" must be set')
    if not COMPAT_ID_RE.match(req_pb.meta.id):
        raise exceptions.BadRequestError('"meta.id" is not valid')
    if not req_pb.meta.namespace_id:
        raise exceptions.BadRequestError('"meta.namespace_id" must be set')
    if not NAMESPACE_ID_RE.match(req_pb.meta.namespace_id):
        raise exceptions.BadRequestError('"meta.namespace_id" is not valid')
    if not req_pb.meta.version:
        raise exceptions.BadRequestError('"meta.version" must be set')
    if not req_pb.backend_version:
        raise exceptions.BadRequestError('"backend_version" must be set')
    if not req_pb.HasField('spec') and not req_pb.meta.HasField('auth') and not req_pb.meta.HasField('is_system'):
        raise exceptions.BadRequestError('at least one of the fields "spec", or "meta.auth", '
                                         'or "meta.is_system" must be present')
    if req_pb.HasField('spec'):
        validate_spec(req_pb.spec)
    if req_pb.meta.HasField('auth'):
        validate_auth_pb(req_pb.meta.auth, field_name='meta.auth')


def validate_list_endpoint_sets_request(req_pb):
    """
    :type req_pb: api_pb2.ListEndpointSetsRequest
    :raises: exceptions.BadRequestError
    """
    if not req_pb.namespace_id:
        raise exceptions.BadRequestError('"namespace_id" must be set')
    if req_pb.HasField('field_mask'):
        if not req_pb.field_mask.IsValidForDescriptor(model_pb2.EndpointSet.DESCRIPTOR):
            raise exceptions.BadRequestError('"field_mask" is not valid')


def validate_request(req_pb):
    """
    :raises: exceptions.BadRequestError
    """
    if isinstance(req_pb, api_pb2.ListEndpointSetsRequest):
        validate_list_endpoint_sets_request(req_pb)
    elif isinstance(req_pb, (api_pb2.GetEndpointSetRequest,
                             api_pb2.ListEndpointSetRevisionsRequest)):
        if not req_pb.id:
            raise exceptions.BadRequestError('No "id" specified')
        if not req_pb.namespace_id:
            raise exceptions.BadRequestError('No "namespace_id" specified')
    elif isinstance(req_pb, api_pb2.GetEndpointSetRevisionRequest):
        if not req_pb.id:
            raise exceptions.BadRequestError('No "id" specified')
    elif isinstance(req_pb, api_pb2.UpdateEndpointSetRequest):
        validate_update_endpoint_set_request(req_pb)
    elif isinstance(req_pb, api_pb2.CreateEndpointSetRequest):
        validate_create_endpoint_set_request(req_pb)
    else:
        raise RuntimeError('Incorrect `req_pb` type: {}'.format(req_pb.__class__.__name__))
