# coding: utf-8
from awacs.lib.rpc import exceptions
from infra.awacs.proto import api_pb2, model_pb2
from awacs.web.validation.util import ID_RE, NAMESPACE_ID_RE, validate_auth_pb


def validate_knob_spec(spec_pb, field_name='spec'):
    """
    :type spec_pb: model_pb2.KnobSpec
    :type field_name: six.text_type
    :raises: exceptions.BadRequestError
    """
    if spec_pb.mode != spec_pb.MANAGED:
        raise exceptions.BadRequestError('"{}.mode" must be MANAGED'.format(field_name))
    if not spec_pb.type:
        raise exceptions.BadRequestError('"{}.type" must be set'.format(field_name))


def validate_create_knob_request(req_pb):
    """
    :type req_pb: api_pb2.CreateKnobRequest
    :raises: exceptions.BadRequestError
    """
    if not req_pb.HasField('meta'):
        raise exceptions.BadRequestError('"meta" must be set')
    if not req_pb.meta.id:
        raise exceptions.BadRequestError('"meta.id" must be set')
    if not ID_RE.match(req_pb.meta.id):
        raise exceptions.BadRequestError('"meta.id" is not valid')
    if not req_pb.meta.namespace_id:
        raise exceptions.BadRequestError('"meta.namespace_id" must be set')
    if not NAMESPACE_ID_RE.match(req_pb.meta.namespace_id):
        raise exceptions.BadRequestError('"meta.namespace_id" is not valid')
    if not req_pb.meta.HasField('auth'):
        raise exceptions.BadRequestError('"meta.auth" must be set')
    validate_auth_pb(req_pb.meta.auth, field_name='meta.auth')
    if not req_pb.HasField('spec'):
        raise exceptions.BadRequestError('"spec" must be set')
    validate_knob_spec(req_pb.spec, 'spec')


def validate_update_knob_request(req_pb):
    """
    :type req_pb: api_pb2.UpdateKnobRequest
    :raises: exceptions.BadRequestError
    """
    if not req_pb.HasField('meta'):
        raise exceptions.BadRequestError('"meta" must be set')
    if not req_pb.meta.id:
        raise exceptions.BadRequestError('"meta.id" must be set')
    if not ID_RE.match(req_pb.meta.id):
        raise exceptions.BadRequestError('"meta.id" is not valid')
    if not req_pb.meta.namespace_id:
        raise exceptions.BadRequestError('"meta.namespace_id" must be set')
    if not NAMESPACE_ID_RE.match(req_pb.meta.namespace_id):
        raise exceptions.BadRequestError('"meta.namespace_id" is not valid')
    if not req_pb.meta.version:
        raise exceptions.BadRequestError('"meta.version" must be set')
    if not req_pb.HasField('spec') and not req_pb.meta.HasField('auth'):
        raise exceptions.BadRequestError('at least one of the "spec" or "meta.auth" fields must be present')
    if req_pb.HasField('spec'):
        validate_knob_spec(req_pb.spec)
    if req_pb.meta.HasField('auth'):
        validate_auth_pb(req_pb.meta.auth, field_name='meta.auth')


def validate_update_knob_value_request(req_pb):
    """
    :type req_pb: api_pb2.UpdateKnobValueRequest
    :raises: exceptions.BadRequestError
    """
    if not req_pb.id:
        raise exceptions.BadRequestError('"id" must be set')
    if not ID_RE.match(req_pb.id):
        raise exceptions.BadRequestError('"id" is not valid')
    if not req_pb.namespace_id:
        raise exceptions.BadRequestError('"namespace_id" must be set')
    if not NAMESPACE_ID_RE.match(req_pb.namespace_id):
        raise exceptions.BadRequestError('"namespace_id" is not valid')
    if not req_pb.namespace_id:
        raise exceptions.BadRequestError('"namespace_id" must be set')


def validate_remove_knob_value_request(req_pb):
    """
    :type req_pb: api_pb2.RemoveKnobValueRequest
    :raises: exceptions.BadRequestError
    """
    if not req_pb.id:
        raise exceptions.BadRequestError('"id" must be set')
    if not ID_RE.match(req_pb.id):
        raise exceptions.BadRequestError('"id" is not valid')
    if not req_pb.namespace_id:
        raise exceptions.BadRequestError('"namespace_id" must be set')
    if not NAMESPACE_ID_RE.match(req_pb.namespace_id):
        raise exceptions.BadRequestError('"namespace_id" is not valid')
    if not req_pb.version:
        raise exceptions.BadRequestError('"version" must be set')


def validate_list_knobs_request(req_pb):
    """
    :type req_pb: api_pb2.ListKnobsRequest
    :raises: exceptions.BadRequestError
    """
    if not req_pb.namespace_id:
        raise exceptions.BadRequestError('"namespace_id" must be specified.')
    if not req_pb.namespace_id and not req_pb.HasField('query'):
        raise exceptions.BadRequestError('Either "namespace_id" or "query" must be specified.')
    if req_pb.balancer_id and not req_pb.namespace_id:
        raise exceptions.BadRequestError('"namespace_id" must be specified together with "balancer_id".')
    if req_pb.HasField('field_mask'):
        if not req_pb.field_mask.IsValidForDescriptor(model_pb2.Knob.DESCRIPTOR):
            raise exceptions.BadRequestError('"field_mask" is not valid')


def validate_list_annotated_knobs_request(req_pb):
    """
    :type req_pb: api_pb2.ListAnnotatedKnobsRequest
    :raises: exceptions.BadRequestError
    """
    if not req_pb.namespace_id:
        raise exceptions.BadRequestError('"namespace_id" must be specified.')


def validate_request(req_pb):
    """
    :raises: exceptions.BadRequestError
    """
    if isinstance(req_pb, api_pb2.ListKnobsRequest):
        validate_list_knobs_request(req_pb)
    elif isinstance(req_pb, api_pb2.ListAnnotatedKnobsRequest):
        validate_list_annotated_knobs_request(req_pb)
    elif isinstance(req_pb, (api_pb2.RemoveKnobRequest,
                             api_pb2.GetKnobRequest,
                             api_pb2.ListKnobRevisionsRequest)):
        if not req_pb.id:
            raise exceptions.BadRequestError('No "id" specified.')
        if not req_pb.namespace_id:
            raise exceptions.BadRequestError('No "namespace_id" specified.')
        if isinstance(req_pb, api_pb2.RemoveKnobRequest):
            if not req_pb.version:
                raise exceptions.BadRequestError('No "version" specified.')
    elif isinstance(req_pb, api_pb2.GetKnobRevisionRequest):
        if not req_pb.id:
            raise exceptions.BadRequestError('No "id" specified.')
    elif isinstance(req_pb, api_pb2.UpdateKnobRequest):
        validate_update_knob_request(req_pb)
    elif isinstance(req_pb, api_pb2.UpdateKnobValueRequest):
        validate_update_knob_value_request(req_pb)
    elif isinstance(req_pb, api_pb2.RemoveKnobValueRequest):
        validate_remove_knob_value_request(req_pb)
    elif isinstance(req_pb, api_pb2.CreateKnobRequest):
        validate_create_knob_request(req_pb)
    else:
        raise RuntimeError('Incorrect `req_pb` type: {}'.format(req_pb.__class__.__name__))
