# coding: utf-8
import re
import six

from awacs.lib.strutils import flatten_full_id2
from awacs.lib import validators
from awacs.lib.order_processor.model import is_order_in_progress, can_be_cancelled
from awacs.lib.rpc import exceptions
from awacs.model import cache
from awacs.model.errors import ConflictError
from awacs.model.l3_balancer.l3mgr import is_fully_managed
from awacs.model.l3_balancer.order.processors import UpdatingL3mgrServicePermissions, L3BalancerOrder
from awacs.model.util import NANNY_ROBOT_LOGIN
from awacs.web.validation import util
from awacs.web.validation.util import ID_RE, NAMESPACE_ID_RE, validate_auth_pb, is_root, logger
from infra.awacs.proto import api_pb2, model_pb2
from infra.swatlib.auth.abc import IAbcClient, AbcError


ALLOWED_CTL_VERSIONS = (0, 1, 2)

L3MGR_FQDN_RE = r"^[a-z0-9]([a-z0-9-]*[a-z0-9]+)?(\.[a-z]([a-z0-9-]*[a-z0-9]+)?)*$"


def is_valid_l3mgr_fqdn(fqdn):
    # https://st.yandex-team.ru/AWACS-857
    return re.match(L3MGR_FQDN_RE, fqdn) is not None


def validate_real_servers_backends(namespace_id, l3_balancer_id, real_servers_pb, field_name):
    if not real_servers_pb.backends:
        raise exceptions.BadRequestError(
            '"{0}.type" is set to "BACKENDS", but "{0}.backends" field is empty'.format(field_name))
    c = cache.IAwacsCache.instance()
    prev_backend_ids = set()
    prev_l3_balancer_pb = c.get_l3_balancer(namespace_id, l3_balancer_id)
    if prev_l3_balancer_pb and prev_l3_balancer_pb.spec.real_servers.type == real_servers_pb.type:
        for b_pb in prev_l3_balancer_pb.spec.real_servers.backends:
            prev_backend_ids.add(b_pb.id)
    seen_backend_ids = set()
    for i, b_pb in enumerate(real_servers_pb.backends):
        backend_id = b_pb.id
        if backend_id in seen_backend_ids:
            raise exceptions.BadRequestError(
                '"{0}.backends[{1}]" contains duplicate id: "{2}"'.format(field_name, i, backend_id))
        seen_backend_ids.add(backend_id)
        backend_pb = c.must_get_backend(namespace_id, backend_id)
        if backend_id not in prev_backend_ids and backend_pb.spec.deleted:
            # allow keeping removed backends if they were already present in spec
            raise exceptions.BadRequestError(
                '"{0}.backends[{1}]": backend "{2}" is marked as removed and cannot be used'.format(
                    field_name, i, backend_id))
        if backend_pb.spec.selector.type == model_pb2.BackendSelector.YP_ENDPOINT_SETS_SD:
            raise exceptions.BadRequestError(
                '"{0}.backends[{1}]": backend "{2}" has type '
                'YP_ENDPOINT_SETS_SD, which cannot be used in L3 balancer, change its type to YP_ENDPOINT_SETS'.format(
                    field_name, i, backend_id))


def validate_real_servers_balancers(namespace_id, l3_balancer_id, real_servers_pb, field_name):
    if not real_servers_pb.balancers:
        raise exceptions.BadRequestError(
            '"{0}.type" is set to "BALANCERS", but "{0}.balancers" field is empty'.format(field_name))
    c = cache.IAwacsCache.instance()
    prev_balancer_ids = set()
    prev_l3_balancer_pb = c.get_l3_balancer(namespace_id, l3_balancer_id)
    if prev_l3_balancer_pb and prev_l3_balancer_pb.spec.real_servers.type == real_servers_pb.type:
        for b_pb in prev_l3_balancer_pb.spec.real_servers.balancers:
            prev_balancer_ids.add(b_pb.id)
    seen_balancer_ids = set()
    for i, b_pb in enumerate(real_servers_pb.balancers):
        balancer_id = b_pb.id
        balancer_pb = c.must_get_balancer(namespace_id, balancer_id)
        if balancer_pb.meta.location.type != balancer_pb.meta.location.YP_CLUSTER:
            raise exceptions.BadRequestError(
                ('"{0}.balancers[{1}]": balancer "{2}" cannot be used as a backend because it\'s not located in YP. '
                 'Use BACKENDS type for this L3 balancer instead, with backends that point to your L7 balancers.'
                 .format(field_name, i, balancer_id)))
        if balancer_id not in prev_balancer_ids and balancer_pb.spec.deleted:
            # allow keeping removed balancers if they were already present in spec
            raise exceptions.BadRequestError(
                '"{0}.balancers[{1}]": balancer "{2}" is marked as removed and cannot be used'.format(
                    field_name, i, balancer_id))
        if balancer_id in seen_balancer_ids:
            raise exceptions.BadRequestError(
                '"{0}.balancers[{1}]" contains duplicate id: "{2}"'.format(field_name, i, balancer_id))
        seen_balancer_ids.add(balancer_id)
        backend_pb = c.get_system_backend_for_balancer(namespace_id, balancer_id)
        if not backend_pb:
            raise exceptions.NotFoundError(
                '"{0}.balancers[{1}]": backend for balancer "{2}" not found'.format(field_name, i, balancer_id))
        if balancer_id not in prev_balancer_ids and backend_pb.spec.deleted:
            # allow keeping removed backends if they were already present in spec
            raise exceptions.BadRequestError(
                '"{0}.balancers[{1}]": backend for balancer "{2}" is marked as removed and cannot be used'.format(
                    field_name, i, balancer_id))


def validate_real_servers_pb(namespace_id, l3_balancer_id, real_servers_pb, field_name=u'real_servers'):
    """
    :type namespace_id: six.text_type
    :type l3_balancer_id: six.text_type
    :type real_servers_pb: model_pb2.L3BalancerRealServersSelector
    :type field_name: six.text_type
    """
    spec_type_name = model_pb2.L3BalancerRealServersSelector.Type.Name(real_servers_pb.type)
    if real_servers_pb.type not in (model_pb2.L3BalancerRealServersSelector.BACKENDS,
                                    model_pb2.L3BalancerRealServersSelector.BALANCERS):
        raise exceptions.BadRequestError(u'"{0}.type": "{1}" is not supported yet'.format(field_name, spec_type_name))

    if real_servers_pb.type == model_pb2.L3BalancerRealServersSelector.BACKENDS:
        validate_real_servers_backends(namespace_id, l3_balancer_id, real_servers_pb, field_name)
    elif real_servers_pb.type == model_pb2.L3BalancerRealServersSelector.BALANCERS:
        validate_real_servers_balancers(namespace_id, l3_balancer_id, real_servers_pb, field_name)


def validate_update_l3_with_dns_records(namespace_id, l3_balancer_id, old_spec_pb, new_spec_pb):
    """l3_with_dns_records means there are DnsRecord(s) in awacs which point(s) to this l3 
    :type namespace_id: six.text_type
    :type l3_balancer_id: six.text_type
    :type spec_pb: model_pb2.L3BalancerSpec
    :raises exceptions.BadRequestError: 
    """
    c = cache.IAwacsCache.instance()
    dns_record_for_l3_balancer_pbs = c.list_dns_records_for_l3_balancer(namespace_id, l3_balancer_id)
    if not dns_record_for_l3_balancer_pbs:
        return
    dns_record_for_l3_balancer_full_ids = [flatten_full_id2((r.meta.namespace_id, r.meta.id)) for r in dns_record_for_l3_balancer_pbs]
    if is_fully_managed(old_spec_pb) and not is_fully_managed(new_spec_pb):
        raise exceptions.BadRequestError('can not disable fully-managed by awacs mode (spec.config_management_mode=MODE_REAL_AND_VIRTUAL_SERVERS) '
                                         'cause there are dns records which point to this l3 balancer: {}'.format(
                                             ",".join(sorted(dns_record_for_l3_balancer_full_ids))
                                         ))
    if old_spec_pb.virtual_servers != new_spec_pb.virtual_servers:
        raise exceptions.BadRequestError('"spec.virtual_servers" can not be changed '
                                         'cause there are dns records which point to this l3 balancer: {}'.format(
                                             ",".join(sorted(dns_record_for_l3_balancer_full_ids))
                                         ))


def validate_spec_pb(namespace_id, l3_balancer_id, spec_pb, field_name='spec'):
    """
    :type namespace_id: six.text_type
    :type l3_balancer_id: six.text_type
    :type spec_pb: model_pb2.L3BalancerSpec
    :type field_name: six.text_type
    :raises: exceptions.BadRequestError
    """
    if spec_pb.ctl_version not in ALLOWED_CTL_VERSIONS:
        raise exceptions.BadRequestError(u'"{}.ctl_version" must be one of: {}'.format(
            field_name, u', '.join(six.text_type(v) for v in ALLOWED_CTL_VERSIONS)))
    validate_real_servers_pb(namespace_id, l3_balancer_id, spec_pb.real_servers,
                             field_name=field_name + u'.real_servers')
    validate_virtual_servers(spec_pb)


def validate_l3_balancer_order(namespace_id, l3_balancer_id, order_content_pb, login, field_name=u'order'):
    """
    :type namespace_id: six.text_type
    :type l3_balancer_id: six.text_type
    :type order_content_pb: api_pb2.L3BalancerOrder.Content
    :type login: six.text_type
    :type field_name: six.text_type
    :raises: exceptions.BadRequestError
    """
    if order_content_pb.ctl_version not in ALLOWED_CTL_VERSIONS:
        raise exceptions.BadRequestError(u'"{}.ctl_version" must be one of: {}'.format(
            field_name, u', '.join(six.text_type(v) for v in ALLOWED_CTL_VERSIONS)))
    if not order_content_pb.fqdn:
        raise exceptions.BadRequestError(u'"{}.fqdn" must be set'.format(field_name))
    if not is_valid_l3mgr_fqdn(order_content_pb.fqdn):
        raise exceptions.BadRequestError(u'"{}.fqdn" must match "{}"'.format(field_name, L3MGR_FQDN_RE))
    if not order_content_pb.HasField('real_servers'):
        raise exceptions.BadRequestError(u'"{}.real_servers" must be set'.format(field_name))
    validate_real_servers_pb(namespace_id, l3_balancer_id, order_content_pb.real_servers,
                             field_name=field_name + u'.real_servers')
    if is_fully_managed(order_content_pb):
        if order_content_pb.preserve_foreign_real_servers:
            raise exceptions.BadRequestError(u'"{}.preserve_foreign_real_servers" is not supported in fully-managed '
                                             u'L3 balancers'.format(field_name))
        if order_content_pb.real_servers.type != model_pb2.L3BalancerRealServersSelector.BALANCERS:
            raise exceptions.BadRequestError(u'"{}.real_servers" must point to L7 balancers in fully-managed '
                                             u'L3 balancers'.format(field_name))


def validate_meta(req_pb):
    if not req_pb.HasField('meta'):
        raise exceptions.BadRequestError('"meta" must be set')
    if not req_pb.meta.id:
        raise exceptions.BadRequestError('"meta.id" must be set')
    if not ID_RE.match(req_pb.meta.id):
        raise exceptions.BadRequestError('"meta.id" is not valid')
    if not req_pb.meta.namespace_id:
        raise exceptions.BadRequestError('"meta.namespace_id" must be set')
    if not NAMESPACE_ID_RE.match(req_pb.meta.namespace_id):
        raise exceptions.BadRequestError('"meta.namespace_id" is not valid')


def validate_create_l3_balancer_request(req_pb, login):
    """
    :type login: six.text_type
    :type req_pb: infra.awacs.proto.api_pb2.CreateL3BalancerRequest
    :raises: exceptions.BadRequestError
    """
    validate_meta(req_pb)
    if not req_pb.meta.HasField('auth'):
        raise exceptions.BadRequestError(u'"meta.auth" must be set')
    validate_auth_pb(req_pb.meta.auth, field_name=u'meta.auth')

    if req_pb.HasField('order'):
        validate_l3_balancer_order(req_pb.meta.namespace_id, req_pb.meta.id, req_pb.order, login)
        validate_preserve_foreign_rs(False, req_pb.order.preserve_foreign_real_servers, login, u'order')
        if not req_pb.order.abc_service_id and is_fully_managed(req_pb.spec):
            raise exceptions.BadRequestError(u'"order.abc_service_id" must be set')
        return

    if req_pb.HasField('spec'):
        if is_fully_managed(req_pb.spec):
            raise exceptions.BadRequestError(u'"order" must be set')
        validate_spec_pb(req_pb.meta.namespace_id, req_pb.meta.id, req_pb.spec)
        validate_preserve_foreign_rs(False, req_pb.spec.preserve_foreign_real_servers, login, u'spec')
        validate_virtual_servers(req_pb.spec)
        return

    raise exceptions.BadRequestError(u'"order" must be set')


def validate_preserve_foreign_rs(current_value, new_value, login, field_name):
    if is_root(login):
        return
    if current_value != new_value:
        raise exceptions.ForbiddenError(u'"{}.preserve_foreign_real_servers": '
                                        u'Only awacs admins can change this setting. '
                                        u'Please create a ticket in st/BALANCERSUPPORT '
                                        u'if you need to change it'.format(field_name))


def validate_virtual_servers(spec_pb):
    """
    :type spec_pb: model_pb2.L3BalancerSpec
    :return:
    """
    if spec_pb.virtual_servers and not is_fully_managed(spec_pb):
        raise exceptions.BadRequestError(u'"spec.virtual_servers" must not be set for RS-only L3 balancer')
    if not spec_pb.virtual_servers:
        if is_fully_managed(spec_pb):
            raise exceptions.BadRequestError(u'"spec.virtual_servers" must be set for fully-managed L3 balancer')
        return
    seen_vs = set()
    for i, vs_pb in enumerate(spec_pb.virtual_servers):  # type: int, model_pb2.L3BalancerSpec.VirtualServer
        field = u'spec.virtual_servers[{}]'.format(i)
        if not vs_pb.ip:
            raise exceptions.BadRequestError(u'"{}.ip" must be set'.format(field))
        if not validators.is_ip_address(vs_pb.ip):
            raise exceptions.BadRequestError(u'"{}.ip": "{}" is not a valid IP address'.format(field, vs_pb.ip))
        if not vs_pb.port:
            raise exceptions.BadRequestError(u'"{}.port" must be set'.format(field))
        if (vs_pb.ip, vs_pb.port) in seen_vs:
            raise exceptions.BadRequestError(u'"{}": IP+port "{}:{}" is already used in another VS'.format(
                field, vs_pb.ip, vs_pb.port))
        seen_vs.add((vs_pb.ip, vs_pb.port))
        if not vs_pb.traffic_type:
            raise exceptions.BadRequestError(u'"{}.traffic_type" must be set'.format(field))
        if not vs_pb.health_check_settings.url:
            raise exceptions.BadRequestError(u'"{}.health_check_settings.url" must be set'.format(field))
        if not vs_pb.health_check_settings.check_type:
            raise exceptions.BadRequestError(u'"{}.health_check_settings.check_type" must be set'.format(field))


def validate_import_l3_balancer_request(req_pb, login):
    """
    :type login: six.text_type
    :type req_pb: api_pb2.ImportL3BalancerRequest
    :raises: exceptions.BadRequestError
    """
    validate_meta(req_pb)
    if not req_pb.meta.HasField('auth'):
        raise exceptions.BadRequestError('"meta.auth" must be set')
    validate_auth_pb(req_pb.meta.auth, field_name='meta.auth')
    if is_fully_managed(req_pb.spec):
        raise exceptions.BadRequestError(u'"spec.config_management_mode" must be MODE_REAL_SERVERS_ONLY. '
                                         u'You can migrate to other modes after import')
    validate_spec_pb(req_pb.meta.namespace_id, req_pb.meta.id, req_pb.spec)
    validate_preserve_foreign_rs(False, req_pb.spec.preserve_foreign_real_servers, login, 'spec')
    if req_pb.spec.incomplete:
        raise exceptions.BadRequestError(u'"spec.incomplete" cannot be set on import')
    if not req_pb.spec.l3mgr_service_id:
        raise exceptions.BadRequestError(u'"spec.l3mgr_service_id" must be set')


def validate_update_l3_balancer_request(req_pb):
    """
    :type req_pb: api_pb2.UpdateL3BalancerRequest
    :raises: exceptions.BadRequestError
    """
    validate_meta(req_pb)
    if not req_pb.meta.version:
        raise exceptions.BadRequestError('"meta.version" must be set')
    if (not req_pb.HasField('spec') and
            not req_pb.meta.HasField('auth') and
            not req_pb.meta.HasField('transport_paused')):
        raise exceptions.BadRequestError('at least one of the "spec", "meta.auth" or '
                                         '"meta.transport_paused" fields must be present')
    if req_pb.HasField('spec'):
        if req_pb.spec.incomplete:
            raise exceptions.BadRequestError('"spec.incomplete" can only be set on creation')
        if not req_pb.spec.HasField('real_servers'):
            raise exceptions.BadRequestError('"spec.real_servers" must be set')
        validate_spec_pb(req_pb.meta.namespace_id, req_pb.meta.id, req_pb.spec)


def validate_remove_l3_balancer_request(req_pb):
    """
    :type req_pb: api_pb2.RemoveL3BalancerRequest
    :raises exceptions.BadRequestError: 
    """
    if not req_pb.id:
        raise exceptions.BadRequestError('No "id" specified.')
    if not req_pb.namespace_id:
        raise exceptions.BadRequestError('No "namespace_id" specified.')
    c = cache.IAwacsCache.instance()
    dns_record_for_l3_balancer_pbs = c.list_dns_records_for_l3_balancer(req_pb.namespace_id, req_pb.id)
    if dns_record_for_l3_balancer_pbs:
        dns_record_for_l3_balancer_full_ids = [r.meta.namespace_id + "/" + r.meta.id for r in dns_record_for_l3_balancer_pbs]
        raise exceptions.BadRequestError('There are dns records which point to this l3 balancer: {}'.format(
            ",".join(sorted(dns_record_for_l3_balancer_full_ids))
        ))


def validate_update_l3_balancer_state_request(req_pb):
    """
    :type req_pb: api_pb2.UpdateL3BalancerStateRequest
    :raises: exceptions.BadRequestError
    """
    if not req_pb.id:
        raise exceptions.BadRequestError('"id" must be set')
    if not req_pb.namespace_id:
        raise exceptions.BadRequestError('"namespace_id" must be set')
    if not req_pb.type:
        raise exceptions.BadRequestError('"type" must be set')
    if req_pb.type == req_pb.SKIP_IN_PROGRESS_L3MGR_CONFIG:
        if not req_pb.HasField('skip_in_progress_l3mgr_config'):
            raise exceptions.BadRequestError('"skip_in_progress_l3mgr_config" must be set if "type" is '
                                             'SKIP_IN_PROGRESS_L3MGR_CONFIG')
        if not req_pb.skip_in_progress_l3mgr_config.service_id:
            raise exceptions.BadRequestError('"skip_in_progress_l3mgr_config.service_id" must be set')
        if not req_pb.skip_in_progress_l3mgr_config.config_id:
            raise exceptions.BadRequestError('"skip_in_progress_l3mgr_config.config_id" must be set')
    else:
        raise AssertionError('unknown UpdateL3BalancerStateRequest.type: {}'.format(req_pb.type))


def validate_list_l3_balancers_request(req_pb):
    """
    :type req_pb: api_pb2.ListL3BalancersRequest
    :raises: exceptions.BadRequestError
    """
    if req_pb.HasField('field_mask'):
        if not req_pb.field_mask.IsValidForDescriptor(model_pb2.L3Balancer.DESCRIPTOR):
            raise exceptions.BadRequestError('"field_mask" is not valid')


def validate_request(req_pb, auth_subject):
    """
    :raises: exceptions.BadRequestError
    """
    if isinstance(req_pb, api_pb2.GetL3BalancerRevisionRequest):
        if not req_pb.id:
            raise exceptions.BadRequestError('No "id" specified.')
    elif isinstance(req_pb, (api_pb2.CancelL3BalancerOrderRequest,
                             api_pb2.GetL3BalancerRequest,
                             api_pb2.GetL3BalancerStateRequest,
                             api_pb2.ListL3BalancerRevisionsRequest)):
        if not req_pb.id:
            raise exceptions.BadRequestError('No "id" specified.')
        if not req_pb.namespace_id:
            raise exceptions.BadRequestError('No "namespace_id" specified.')
    elif isinstance(req_pb, api_pb2.UpdateL3BalancerRequest):
        validate_update_l3_balancer_request(req_pb)
    elif isinstance(req_pb, api_pb2.UpdateL3BalancerStateRequest):
        validate_update_l3_balancer_state_request(req_pb)
    elif isinstance(req_pb, api_pb2.RemoveL3BalancerRequest):
        validate_remove_l3_balancer_request(req_pb)
    elif isinstance(req_pb, api_pb2.CreateL3BalancerRequest):
        validate_create_l3_balancer_request(req_pb, auth_subject.login)
    elif isinstance(req_pb, api_pb2.ImportL3BalancerRequest):
        validate_import_l3_balancer_request(req_pb, auth_subject.login)
    elif isinstance(req_pb, api_pb2.ListL3BalancersRequest):
        if not req_pb.namespace_id:
            raise exceptions.BadRequestError('No "namespace_id" specified.')
        validate_list_l3_balancers_request(req_pb)
    elif isinstance(req_pb, api_pb2.CheckL3BalancerMigrationRequest):
        if not req_pb.l3mgr_service_id:
            raise exceptions.BadRequestError('No "l3mgr_service_id" specified.')
    else:
        raise RuntimeError('Incorrect `req_pb` type: {}'.format(req_pb.__class__.__name__))


def validate_vs_order(vs_order_pb, field):
    if not vs_order_pb.ip_version:
        raise exceptions.BadRequestError(u'"{}.ip_version" must be set'.format(field))
    if not vs_order_pb.traffic_type:
        raise exceptions.BadRequestError(u'"{}.traffic_type" must be set'.format(field))
    if not vs_order_pb.health_check_url:
        raise exceptions.BadRequestError(u'"{}.health_check_url" must be set'.format(field))
    if not vs_order_pb.ports:
        raise exceptions.BadRequestError(u'"{}.ports" must be set'.format(field))
    seen = set()
    bad_ports = set()
    duplicates = set()
    for port in vs_order_pb.ports:
        if port in seen:
            duplicates.add(six.text_type(port))
        seen.add(port)
        if not (1 <= port <= 65535):
            bad_ports.add(six.text_type(port))
    if duplicates:
        raise exceptions.BadRequestError(u'"{}.ports": duplicate values: "{}"'.format(field, u', '.join(duplicates)))
    if bad_ports:
        raise exceptions.BadRequestError(
            u'"{}.ports": must belong to the interval [1, 65535]. Invalid values: "{}"'.format(field,
                                                                                               u', '.join(bad_ports)))


def validate_cancel_l3_balancer_order(req_pb, l3_balancer_pb, auth_subject):
    if req_pb.force:
        if not is_root(auth_subject.login):
            raise exceptions.BadRequestError('Must be awacs root to force cancel')
    else:
        if not is_order_in_progress(l3_balancer_pb):
            raise exceptions.BadRequestError('Cannot cancel order that is not in progress')
        if not can_be_cancelled(l3_balancer_pb, L3BalancerOrder.get_processors()):
            raise exceptions.BadRequestError('Cannot cancel order at this stage')


def validate_remove_l3_balancer(l3_balancer_pb):
    if is_order_in_progress(l3_balancer_pb):
        raise exceptions.BadRequestError('Cannot remove while order is in progress')


def validate_update_l3_balancer(l3_balancer_pb):
    if is_order_in_progress(l3_balancer_pb):
        raise exceptions.BadRequestError('Cannot update while order is in progress')


def validate_l3mgr_service_id_not_used_in_awacs(namespace_id, l3_balancer_id, l3mgr_service_id):
    """
    :type namespace_id: six.text_type
    :type l3_balancer_id: six.text_type
    :type l3mgr_service_id: six.text_type
    :raises: ConflictError
    """
    c = cache.IAwacsCache.instance()
    l3_balancer_pbs = c.list_all_l3_balancers(query={c.L3BalancersQueryTarget.L3MGR_SERVICE_ID_IN: [l3mgr_service_id]})
    for l3_balancer_pb in l3_balancer_pbs:
        if l3_balancer_pb.meta.namespace_id != namespace_id or l3_balancer_pb.meta.id != l3_balancer_id:
            raise ConflictError(
                u'L3 balancer with L3mgr id "{}" is already managed by another awacs L3 balancer: "{}:{}"'.format(
                    l3mgr_service_id, l3_balancer_pb.meta.namespace_id, l3_balancer_pb.meta.id))


def validate_nanny_robot_is_responsible_for_l3(abc_service_id, field_name):
    abc_client = IAbcClient.instance()
    abc_service_slug = util.get_abc_service_slug_by_id(abc_client, abc_service_id, field_name)
    try:
        roles = abc_client.get_service_member_roles(login=NANNY_ROBOT_LOGIN, service_id=abc_service_id)
    except AbcError as e:
        logger.exception(six.text_type(e))
        raise exceptions.InternalError(
            u'"{}": failed to check ABC membership info for service "{} ({})"'.format(
                field_name, abc_service_slug, abc_service_id))
    if not roles:
        raise exceptions.BadRequestError(
            u'"{}": "{}" must be a member of ABC service "{} ({})" and have role "Responsible for L3"'.format(
                field_name, NANNY_ROBOT_LOGIN, abc_service_slug, abc_service_id))
    for role in roles:
        if role[u'code'] == u'l3_responsible':
            return True
    raise exceptions.BadRequestError(
        u'"{}": "{}" must have role "Responsible for L3" in ABC service "{} ({})"'.format(
            field_name, NANNY_ROBOT_LOGIN, abc_service_slug, abc_service_id))


def validate_nanny_robot_has_l3_access(l3mgr_client, l3mgr_service_id):
    abc_service_slug = l3mgr_client.get_service(l3mgr_service_id)[u'abc']
    abc_service_id = util.get_abc_service_id_by_slug(abc_service_slug, u'abc_service')
    try:
        validate_nanny_robot_is_responsible_for_l3(abc_service_id, u'abc_service_id')
        return
    except exceptions.BadRequestError:
        # legacy fallback - ABC group "rclb" may be given permissions to edit balancer in L3Mgr
        missing_permissions = set(UpdatingL3mgrServicePermissions.USER_L3MGR_PERMISSIONS)
        for role in l3mgr_client.list_roles(svc_id=l3mgr_service_id)[u'objects']:
            if role[u'abc'] == u'rclb':
                missing_permissions.discard(role[u'permission'])
                if not missing_permissions:
                    return
        if missing_permissions:
            # don't mention permissions, instead ask user to make nanny-robot responsible for L3
            raise


def validate_abc_service_has_load_balancers_quota(l3mgr_client, abc_client, abc_service_id, field_name):
    abc_service_slug = util.get_abc_service_slug_by_id(abc_client, abc_service_id, field_name)
    if len(l3mgr_client.get_abc_service_info(abc_service_slug)[u'lb']) == 0:
        raise exceptions.BadRequestError(
            u'"{}": ABC service "{} ({})" has no available L3 load balancers. '
            u'Please create a ticket in st/TRAFFIC to allocate LBs for your ABC service.'.format(
                field_name, abc_service_slug, abc_service_id))
