# coding: utf-8
import re

from awacs.lib.order_processor.model import is_order_in_progress, can_be_cancelled
from awacs.lib.rpc import exceptions
from awacs.model.l7heavy_config.order.processors import L7HeavyConfigOrder
from infra.awacs.proto import api_pb2, model_pb2
from .util import WEIGHT_SECTION_ID_RE, NAMESPACE_ID_RE, is_root
from .weight_section import validate_spec as validate_weight_section_spec


def validate_meta(meta_pb, field_name='meta'):
    """
    :type meta_pb: model_pb2.L7HeavyConfigMeta
    :type field_name: six.text_type
    :raises: spec.BadRequestError
    """
    if not meta_pb.namespace_id:
        raise exceptions.BadRequestError('"{}.namespace_id" must be set'.format(field_name))
    if not NAMESPACE_ID_RE.match(meta_pb.namespace_id):
        raise exceptions.BadRequestError('"{}.namespace_id" is not valid'.format(field_name))
    if not meta_pb.id:
        raise exceptions.BadRequestError('"{}.id" must be set'.format(field_name))
    if meta_pb.namespace_id != meta_pb.id:
        raise exceptions.BadRequestError('"{0}.id": must be equal to "{0}.namespace_id"'.format(field_name))


def validate_spec(spec_pb, field_name='spec'):
    """
    :type spec_pb: model_pb2.L7HeavyConfigSpec
    :type field_name: six.text_type
    :type section_type: model_pb2.L7HeavyConfigMeta.Type
    :raises: spec.BadRequestError
    """
    if not spec_pb.l7_heavy_config_id:
        raise exceptions.BadRequestError('"{}.l7_heavy_config_id" must be set'.format(field_name))
    if not spec_pb.group_id:
        raise exceptions.BadRequestError('"{}.group_id" must be set'.format(field_name))


def validate_create_l7heavy_config_request(req_pb):
    """
    :type req_pb: api_pb2.CreateL7HeavyConfigRequest
    :raises: exceptions.BadRequestError
    """
    if not req_pb.HasField('meta'):
        raise exceptions.BadRequestError('"meta" must be set')
    validate_meta(req_pb.meta)

    if not req_pb.HasField('order'):
        raise exceptions.BadRequestError('"order" must be set')
    if req_pb.order.mode == req_pb.order.UNKNOWN:
        raise exceptions.BadRequestError('"order.mode" must be set')

    if req_pb.order.mode != req_pb.order.IMPORT:
        if req_pb.order.l7_heavy_config_id:
            raise exceptions.BadRequestError('"order.l7_heavy_config_id": must not be set')
        if not req_pb.order.group_id:
            raise exceptions.BadRequestError('"order.group_id": must be set')
    if req_pb.order.mode == req_pb.order.IMPORT:
        if not req_pb.order.l7_heavy_config_id:
            raise exceptions.BadRequestError('"order.l7_heavy_config_id": must be set')
        if req_pb.order.group_id:
            raise exceptions.BadRequestError('"order.group_id": must not be set')
    for i, ws_pb in enumerate(req_pb.order.weight_sections_to_create):
        if ws_pb.meta.namespace_id != req_pb.meta.namespace_id:
            raise exceptions.BadRequestError('"order.weight_sections_to_create[{}]": meta.namespace_id must be {}'
                                             .format(i, req_pb.meta.namespace_id))
        if not ws_pb.meta.id:
            raise exceptions.BadRequestError('"order.weight_sections_to_create[{}]": meta.id must be set'.format(i))
        if not WEIGHT_SECTION_ID_RE.match(ws_pb.meta.id):
            raise exceptions.BadRequestError('"order.weight_sections_to_create[{}]": meta.id is not valid'.format(i))
        if not ws_pb.meta.type:
            raise exceptions.BadRequestError('"order.weight_sections_to_create[{}]": meta.type must be set'.format(i))
        validate_weight_section_spec(ws_pb.spec, ws_pb.meta.type,
                                     field_name='"order.weight_sections_to_create[{}]".spec')


def validate_update_l7heavy_config_request(req_pb):
    """
    :type req_pb: api_pb2.UpdateL7HeavyConfigRequest
    :raises: exceptions.BadRequestError
    """
    if not req_pb.HasField('meta'):
        raise exceptions.BadRequestError('"meta" must be set')
    validate_meta(req_pb.meta)
    if not req_pb.meta.version:
        raise exceptions.BadRequestError('"meta.version" must be set')

    if req_pb.HasField('spec'):
        validate_spec(req_pb.spec)


def validate_remove_l7heavy_config(l7heavy_config_pb):
    if is_order_in_progress(l7heavy_config_pb):
        raise exceptions.BadRequestError('Cannot remove while order is in progress')


def validate_cancel_l7heavy_config_order(req_pb, l7heavy_config_pb, auth_subject):
    if req_pb.force_cancel:
        if not is_root(auth_subject.login):
            raise exceptions.BadRequestError('Must be awacs root to force cancel')
        if not can_be_cancelled(l7heavy_config_pb, L7HeavyConfigOrder.get_processors()):
            raise exceptions.BadRequestError('Cannot force cancel order at this stage')
    else:
        if not is_order_in_progress(l7heavy_config_pb):
            raise exceptions.BadRequestError('Cannot cancel order that is not in progress')
        if not can_be_cancelled(l7heavy_config_pb, L7HeavyConfigOrder.get_processors()):
            raise exceptions.BadRequestError('Cannot cancel order at this stage')


def validate_request(req_pb):
    """
    :raises: exceptions.BadRequestError
    """
    if isinstance(req_pb, (api_pb2.GetL7HeavyConfigRequest,
                           api_pb2.GetL7HeavyConfigStateRequest,
                           api_pb2.RemoveL7HeavyConfigRequest,
                           api_pb2.ListL7HeavyConfigRevisionsRequest,
                           api_pb2.CancelL7HeavyConfigOrderRequest)):
        if not req_pb.id:
            raise exceptions.BadRequestError('No "id" specified')
        if not req_pb.namespace_id:
            raise exceptions.BadRequestError('No "namespace_id" specified')
    elif isinstance(req_pb, api_pb2.GetL7HeavyConfigRevisionRequest):
        if not req_pb.id:
            raise exceptions.BadRequestError('No "id" specified')
    elif isinstance(req_pb, api_pb2.UpdateL7HeavyConfigRequest):
        validate_update_l7heavy_config_request(req_pb)
    elif isinstance(req_pb, api_pb2.CreateL7HeavyConfigRequest):
        validate_create_l7heavy_config_request(req_pb)
    else:
        raise RuntimeError('Incorrect `req_pb` type: {}'.format(req_pb.__class__.__name__))
