import logging
import re

import gevent
import semantic_version
import six
import yaml
from cachetools.func import ttl_cache
from sepelib.core import config
from yp_lite_ui_repo import pod_sets_api_pb2

from awacs.lib import staffclient, rpc, juggler_client, itsclient, l3mgrclient
from awacs.lib.order_processor.model import is_order_in_progress, can_be_cancelled
from awacs.lib.rpc import exceptions
from awacs.lib.ypliterpcclient import IYpLiteRpcClient
from awacs.model import alerting, cache, external_clusters, zk
from awacs.model import objects
from awacs.model.balancer.order import util as balancer_util
from awacs.model.balancer.order.util import make_nanny_service_id, make_awacs_balancer_id
from awacs.model.l3_balancer import l3mgr
from awacs.model.namespace.operations.ctl import NamespaceOperationCtl
from awacs.web.auth.core import authorize_create, is_root
from awacs.web.util import validate_nanny_service_nonexistence, forbid_action_during_namespace_order
from awacs.web.validation import util
from awacs.web.validation.backend import validate_selector
from awacs.web.validation.certificate import validate_cert_order_content
from awacs.web.validation import dns_record as dns_record_validation
from awacs.web.validation.util import ID_RE
from infra.awacs.proto import api_pb2, model_pb2


CATEGORY_RE = re.compile(r'^[a-z][a-z0-9_-]*(?:/[a-z][a-z0-9_.-]*)*$')
VALID_STAFF_GROUP_TYPES = ('service', 'servicerole')
logger = logging.getLogger(__name__)

MAX_ITS_CTL_VERSION = 2


@ttl_cache(maxsize=1, ttl=600)
def list_cached_its_ruchka_ids():
    its_client = itsclient.IItsClient.instance()
    return its_client.list_ruchka_ids()


def validate_category(name, field_name):
    if not CATEGORY_RE.match(name):
        raise exceptions.BadRequestError(
            '"{}" is not a valid category. '
            'Only letters, digits, underscores, dashes, slashes and dots are allowed'.format(field_name))


def validate_yp_lite_allocation_request(yp_lite_allocation_request_pb,
                                        allowed_location_names,
                                        field_name='order.yp_lite_allocation_request'):
    """
    :type yp_lite_allocation_request_pb: api_pb2.NamespaceOrder.Content.YpLiteAllocationRequest
    :type allowed_location_names: list[six.text_type]
    :type field_name: six.text_type
    :raises: exceptions.BadRequestError
    """
    if not yp_lite_allocation_request_pb.nanny_service_id_slug:
        raise exceptions.BadRequestError('"{}.nanny_service_id_slug" must be set'.format(field_name))
    util.validate_nanny_service_slug(
        yp_lite_allocation_request_pb.nanny_service_id_slug,
        field_name=field_name + '.nanny_service_id_slug')

    if not yp_lite_allocation_request_pb.locations:
        raise exceptions.BadRequestError('"{}.locations" must be set'.format(field_name))
    for i, location in enumerate(yp_lite_allocation_request_pb.locations):
        if location.lower() not in allowed_location_names:
            raise exceptions.BadRequestError(
                '"{}.locations[{}]" must be '
                'one of the following: "{}"'.format(field_name, i, '", "'.join(allowed_location_names)))
        if config.get_value('run.unavailable_clusters.{}.disable_balancers_creation'.format(location.lower()), False):
            raise exceptions.BadRequestError(
                '"{}.locations[{}]": creating balancers in {} is disabled'.format(field_name, i, location))

    if not yp_lite_allocation_request_pb.network_macro:
        raise exceptions.BadRequestError('"{}.network_macro" must be set'.format(field_name))

    if yp_lite_allocation_request_pb.type != yp_lite_allocation_request_pb.PRESET:
        raise exceptions.BadRequestError('"{}.type" must be PRESET'.format(field_name))
    if not yp_lite_allocation_request_pb.HasField('preset'):
        raise exceptions.BadRequestError('"{}.preset" must be set'.format(field_name))
    if yp_lite_allocation_request_pb.preset.type == yp_lite_allocation_request_pb.preset.NONE:
        raise exceptions.BadRequestError('"{}.preset.type" must be set'.format(field_name))
    if yp_lite_allocation_request_pb.preset.instances_count < 1:
        raise exceptions.BadRequestError(
            '"{}.preset.instances_count" must be greater than 0'.format(field_name))


def validate_namespace_order(order_content_pb, namespace_id, field_name='order'):
    """
    :type order_content_pb: model_pb2.NamespaceOrder.Content
    :type namespace_id: six.text_type
    :type field_name: six.text_type
    :raises: exceptions.BadRequestError
    """
    flow_type = order_content_pb.flow_type
    if flow_type in (order_content_pb.YP_LITE, order_content_pb.QUICK_START):
        if not order_content_pb.HasField('yp_lite_allocation_request'):
            raise exceptions.BadRequestError('"{}.yp_lite_allocation_request" must be set '
                                             'if flow_type is YP_LITE or QUICK_START'.format(field_name))
        if order_content_pb.cloud_type == model_pb2.CT_AZURE:
            allowed_location_names = [name.lower() for name in external_clusters.AZURE_CLUSTERS_BY_NAME]
        else:
            allowed_location_names = balancer_util.get_available_balancer_locations()
        validate_yp_lite_allocation_request(order_content_pb.yp_lite_allocation_request,
                                            allowed_location_names=allowed_location_names,
                                            field_name='order.yp_lite_allocation_request')

        if flow_type == order_content_pb.YP_LITE and order_content_pb.HasField('alerting_simple_settings'):
            if not order_content_pb.alerting_simple_settings.notify_staff_group_id:
                raise exceptions.BadRequestError('"{}.alerting_simple_settings.notify_staff_group_id"'
                                                 ' must be set if flow_type is YP_LITE'.format(field_name))

    if flow_type == order_content_pb.QUICK_START:
        if not order_content_pb.HasField('certificate_order_content'):
            raise exceptions.BadRequestError('"{}.certificate_order_content" must be set'.format(field_name))
        if not order_content_pb.endpoint_sets:
            raise exceptions.BadRequestError('"{}.endpoint_sets" must be set'.format(field_name))
        if order_content_pb.backends:
            raise exceptions.BadRequestError('"{}.backends" must not be set'.format(field_name))
        for i, es_pb in enumerate(order_content_pb.endpoint_sets):
            if config.get_value('run.unavailable_clusters.{}.disable_backends_creation'.format(es_pb.cluster.lower()),
                                False):
                raise exceptions.BadRequestError('"{}.endpoint_sets[{}]": Creating backends in {} is disabled'
                                                 .format(field_name, i, es_pb.cluster))
    elif flow_type in (order_content_pb.GENCFG, order_content_pb.YP_LITE):
        if not order_content_pb.backends:
            raise exceptions.BadRequestError('"{}.backends" must be set'.format(field_name))
        if order_content_pb.endpoint_sets:
            raise exceptions.BadRequestError('"{}.endpoint_sets" must not be set'.format(field_name))

        for backend_id, backend_selector_pb in six.iteritems(order_content_pb.backends):
            item_field_name = 'order.backends[{}]'.format(backend_id)
            if not ID_RE.match(backend_id):
                raise exceptions.BadRequestError('"{}": backend_id "{}" is not valid'.format(item_field_name,
                                                                                             backend_id))
            if backend_selector_pb.type == backend_selector_pb.YP_ENDPOINT_SETS:
                raise exceptions.BadRequestError(
                    '"{}": backend_id "{}" has type YP_ENDPOINT_SETS which is allowed only for L3-only namespaces, '
                    'use YP_ENDPOINT_SETS_SD instead.'.format(item_field_name, backend_id)
                )
            validate_selector(backend_selector_pb, namespace_id=namespace_id, field_name=item_field_name)
            if backend_selector_pb.type == backend_selector_pb.YP_ENDPOINT_SETS_SD:
                for yp_es_pb in backend_selector_pb.yp_endpoint_sets:
                    if config.get_value('run.unavailable_clusters.{}.disable_backends_creation'
                                        .format(yp_es_pb.cluster.lower()), False):
                        raise exceptions.BadRequestError('Creating backends in {} is disabled'.format(yp_es_pb.cluster))
    elif flow_type != order_content_pb.EMPTY:
        raise AssertionError('unexpected flow_type')


def validate_create_namespace_request(req_pb):
    """
    :type req_pb: api_pb2.CreateNamespaceRequest
    :raises: exceptions.BadRequestError
    """
    if not req_pb.meta.id:
        raise exceptions.BadRequestError('"meta.id" must be set')
    # we forbid trailing dot only for new namespaces, as some old ones already have it
    # (thus we can't change NAMESPACE_ID_RE)
    if not util.NAMESPACE_ID_RE.match(req_pb.meta.id) or req_pb.meta.id.endswith('.'):
        raise exceptions.BadRequestError('"meta.id" is not valid')
    if not req_pb.meta.category:
        raise exceptions.BadRequestError('"meta.category" must be set')
    validate_category(req_pb.meta.category, 'meta.category')
    if not req_pb.meta.HasField('auth'):
        raise exceptions.BadRequestError('"meta.auth" field must be present')
    if not req_pb.meta.abc_service_id:
        raise exceptions.BadRequestError('"meta.abc_service_id" field must be present')
    util.validate_auth_pb(req_pb.meta.auth, field_name='meta.auth')
    util.validate_staff_logins_list(req_pb.meta.webauth.responsible.logins,
                                    field_name='meta.webauth.responsible.logins')
    if req_pb.HasField('order'):
        validate_namespace_order(req_pb.order, namespace_id=req_pb.meta.id)


def validate_create_namespace_on_allocated_resources_request(req_pb):
    """
    :type req_pb: api_pb2.CreateNamespaceOnAllocatedResourcesRequest
    :raises: exceptions.BadRequestError
    """
    if not req_pb.meta.id:
        raise exceptions.BadRequestError('"meta.id" must be set')
    if not util.NAMESPACE_ID_RE.match(req_pb.meta.id):
        raise exceptions.BadRequestError('"meta.id" is not valid')
    if not req_pb.meta.category:
        raise exceptions.BadRequestError('"meta.category" must be set')
    validate_category(req_pb.meta.category, 'meta.category')
    if not req_pb.meta.HasField('auth'):
        raise exceptions.BadRequestError('"meta.auth" field must be present')
    if not req_pb.meta.abc_service_id:
        raise exceptions.BadRequestError('"meta.abc_service_id" field must be present')
    util.validate_auth_pb(req_pb.meta.auth, field_name='meta.auth')
    util.validate_staff_logins_list(req_pb.meta.webauth.responsible.logins,
                                    field_name='meta.webauth.responsible.logins')

    if not req_pb.HasField('order'):
        raise exceptions.BadRequestError('"order" field must be present')
    if not req_pb.order.HasField('allocation'):
        raise exceptions.BadRequestError('"order.allocation" field must be present')
    if req_pb.order.allocation.type != req_pb.order.allocation.GENCFG_GROUPS:
        raise exceptions.BadRequestError('"order.allocation.type" must be GENCFG_GROUPS')
    if not req_pb.order.allocation.gencfg_groups:
        raise exceptions.BadRequestError('"order.allocation.gencfg_groups" must be specified')
    for location, gencfg_group_pb in six.iteritems(req_pb.order.allocation.gencfg_groups):
        if location not in balancer_util.get_available_balancer_locations():
            raise exceptions.BadRequestError('"order.allocation.gencfg_groups": '
                                             'location {} is not allowed'.format(location))
        item_field_name = 'order.allocation.gencfg_groups[{0}]'.format(location)
        if not gencfg_group_pb.name:
            raise exceptions.BadRequestError('"{}.name" is required'.format(item_field_name))
        if not gencfg_group_pb.version:
            raise exceptions.BadRequestError('"{}.version" is required'.format(item_field_name))
    if not req_pb.order.backends:
        raise exceptions.BadRequestError('"order.backends" must be specified')
    for backend_id, backend_selector_pb in six.iteritems(req_pb.order.backends):
        item_field_name = 'order.backends[{0}]'.format(backend_id)
        validate_selector(backend_selector_pb, namespace_id=req_pb.meta.id, field_name=item_field_name)


def validate_staff_group_ids(staff_group_ids, field_name=u'staff_group_ids'):
    staff_client = staffclient.IStaffClient.instance()
    groups_info = staff_client.get_groups_by_ids(staff_group_ids, fields=('id', 'url', 'type'))
    if len(staff_group_ids) != len(groups_info):
        raise exceptions.BadRequestError(u'"{}": not found groups: {}'.format(
            field_name, [g_id for g_id in staff_group_ids if groups_info.get(g_id) is None]))
    for group in groups_info.values():
        if group.get('type') not in VALID_STAFF_GROUP_TYPES:
            raise exceptions.BadRequestError(u'"{}": group "{}" is not a valid ABC group or role'.format(
                field_name, group.get('id')))


def validate_namespace_spec_pb(spec_pb, namespace_id, field_name=u'spec'):
    """
    :type spec_pb: model_pb2.NamespaceSpec
    :type field_name: six.text_type
    """
    c = cache.IAwacsCache.instance()
    if spec_pb.easy_mode_settings.l7_macro_only.value:
        not_l7_macro_balancer_ids = []
        for balancer_pb in c.list_balancers(namespace_id=namespace_id).items:
            if balancer_pb.spec.yandex_balancer.mode != balancer_pb.spec.yandex_balancer.EASY_MODE:
                not_l7_macro_balancer_ids.append(balancer_pb.meta.id)
        if not_l7_macro_balancer_ids:
            raise exceptions.BadRequestError('"{}.easy_mode_settings.l7_macro_only": can not be set, not easy mode '
                                             'balancers found: "{}"'.format(
                                                 field_name,
                                                 '", "'.join(not_l7_macro_balancer_ids)
                                             ))

    if spec_pb.easy_mode_settings.l7_upstream_macro_only.value:
        not_l7_upstream_macro_upstream_ids = []
        for upstream_pb in c.list_upstreams(namespace_id=namespace_id).items:
            if upstream_pb.spec.yandex_balancer.mode != upstream_pb.spec.yandex_balancer.EASY_MODE2:
                not_l7_upstream_macro_upstream_ids.append(upstream_pb.meta.id)
        if not_l7_upstream_macro_upstream_ids:
            raise exceptions.BadRequestError('"{}.easy_mode_settings.l7_upstream_macro_only": can not be set, '
                                             'not easy mode upstreams found: "{}"'.format(
                                                 field_name,
                                                 '", "'.join(not_l7_upstream_macro_upstream_ids)
                                             ))

    if spec_pb.easy_mode_settings.prohibit_explicit_l3_selector.value:
        not_easy_l3_balancer_ids = []
        for l3_balancer_pb in c.list_l3_balancers(namespace_id=namespace_id).items:
            if l3_balancer_pb.spec.real_servers.type != l3_balancer_pb.spec.real_servers.BALANCERS:
                not_easy_l3_balancer_ids.append(l3_balancer_pb.meta.id)
        if not_easy_l3_balancer_ids:
            raise exceptions.BadRequestError('"{}.easy_mode_settings.prohibit_explicit_l3_selector": can not be set, '
                                             'l3 balancers with explicit selectors found: "{}"'.format(
                                                 field_name,
                                                 '", "'.join(not_easy_l3_balancer_ids)
                                             ))

    if spec_pb.easy_mode_settings.prohibit_explicit_dns_selector.value:
        not_easy_dns_record_ids = []
        for dns_record_pb in c.list_all_dns_records(namespace_id=namespace_id):
            if dns_record_pb.spec.address.backends.type not in (dns_record_pb.spec.address.backends.BALANCERS,
                                                                dns_record_pb.spec.address.backends.L3_BALANCERS):
                not_easy_dns_record_ids.append(dns_record_pb.meta.id)
        if not_easy_dns_record_ids:
            raise exceptions.BadRequestError('"{}.easy_mode_settings.prohibit_explicit_dns_selector": can not be set, '
                                             'DNS records with explicit selectors found: "{}"'.format(
                                                 field_name,
                                                 '", "'.join(not_easy_dns_record_ids)
                                             ))

    for object_name, count_func in (
            ('upstream', c.count_upstreams),
            ('backend', c.count_backends),
            ('balancer', c.count_balancers),
            ('l3_balancer', c.count_l3_balancers),
            ('dns_record', c.count_dns_records),
            ('domain', c.count_domains),
            ('knob', c.count_knobs),
            ('certificate', c.count_certs),
            ('weight_section', objects.WeightSection.cache.count),
            ('l7heavy_config', objects.L7HeavyConfig.cache.count),
    ):
        if spec_pb.object_upper_limits.HasField(object_name):
            max_value = getattr(spec_pb.object_upper_limits, object_name).value
            count = count_func(namespace_id)
            if count > max_value:
                raise exceptions.BadRequestError('"{}.object_upper_limits.{}": can not be {}, {} {}s found'
                                                 .format(field_name, object_name, max_value, count, object_name))

    if spec_pb.HasField('modules_whitelist'):
        seen = set()
        for m in spec_pb.modules_whitelist.modules:
            if m in seen:
                raise exceptions.BadRequestError('"{}.modules_whitelist.modules": duplicate module "{}"'
                                                 .format(field_name, m))
            seen.add(m)

    if spec_pb.HasField('its'):
        if not spec_pb.balancer_constraints.instance_tags.prj:
            raise rpc.exceptions.BadRequestError(
                '"{}.its": can not be enabled if "balancer_constraints.instance_tags.prj"'
                ' is empty'.format(field_name))

        if spec_pb.its.ctl_version > MAX_ITS_CTL_VERSION:
            raise rpc.exceptions.BadRequestError('"{}.its.ctl_version": must not be greater than {}'
                                                 .format(field_name, MAX_ITS_CTL_VERSION))

        if spec_pb.its.acl.staff_group_ids:
            validate_staff_group_ids(
                staff_group_ids=spec_pb.its.acl.staff_group_ids,
                field_name='spec.its.acl.staff_group_ids'
            )
        persons = staffclient.IStaffClient.instance().list_persons(spec_pb.its.acl.logins,
                                                                   ('login', 'official.is_robot'))
        logins = []
        for person in persons:
            if not person['official']['is_robot']:
                raise exceptions.BadRequestError('"spec.its.acl.logins": "{}" is not a robot'.format(person['login']))
            logins.append(person['login'])
        if len(logins) != len(spec_pb.its.acl.logins):
            spec_pb.its.acl.ClearField('logins')
            spec_pb.its.acl.logins.extend(logins)

        try:
            ruchka_ids = list_cached_its_ruchka_ids()
        except:
            raise exceptions.ServiceUnavailable('Failed to communicate with ITS API, please try again later')

        seen_ruchka_ids = set()
        for knob_pb in spec_pb.its.knobs.by_balancer_knobs:
            if knob_pb.its_ruchka_id not in ruchka_ids:
                raise exceptions.BadRequestError('"spec.its.knobs.by_balancer_knobs":'
                                                 'ruchka "{}" does not exist'.format(knob_pb.its_ruchka_id))
            if knob_pb.its_ruchka_id in seen_ruchka_ids:
                raise exceptions.BadRequestError('"spec.its.knobs.by_balancer_knobs":'
                                                 'duplicate ruchka "{}"'.format(knob_pb.its_ruchka_id))
            seen_ruchka_ids.add(knob_pb.its_ruchka_id)
        spec_pb.its.knobs.by_balancer_knobs.sort(key=lambda x: x.its_ruchka_id)

        seen_ruchka_ids = set()
        for knob_pb in spec_pb.its.knobs.common_knobs:
            if knob_pb.its_ruchka_id not in ruchka_ids:
                raise exceptions.BadRequestError('"spec.its.knobs.common_knobs":'
                                                 'ruchka "{}" does not exist'.format(knob_pb.its_ruchka_id))
            if knob_pb.its_ruchka_id in seen_ruchka_ids:
                raise exceptions.BadRequestError('"spec.its.knobs.common_knobs":'
                                                 'duplicate ruchka "{}"'.format(knob_pb.its_ruchka_id))
            seen_ruchka_ids.add(knob_pb.its_ruchka_id)
        spec_pb.its.knobs.common_knobs.sort(key=lambda x: x.its_ruchka_id)

    if spec_pb.HasField('alerting'):
        if not spec_pb.alerting.version:
            raise exceptions.BadRequestError('"{}.alerting.version" must be set'.format(field_name))
        try:
            alerting_version = semantic_version.Version(spec_pb.alerting.version)
        except ValueError:
            raise exceptions.BadRequestError('"{}.alerting.version" is not valid, should be semantic'.format(
                field_name))

        valid_versions = alerting.get_versions()
        if alerting_version not in valid_versions:
            raise exceptions.BadRequestError('"{}.alerting.version" is not valid, should be in {}'.format(
                field_name, [str(v) for v in valid_versions]))

        juggler_raw_downtimers = alerting.AlertingConfig.get_juggler_raw_downtimers(spec_pb.alerting)

        if juggler_raw_downtimers.staff_group_ids:
            # https://bb.yandex-team.ru/projects/JUGGLER/repos/juggler/browse/juggler/validators/system.py#140
            # _valid_login_or_group
            validate_staff_group_ids(
                staff_group_ids=juggler_raw_downtimers.staff_group_ids,
                field_name='spec.alerting.juggler_raw_downtimers.staff_group_ids'
            )

        if spec_pb.alerting.HasField('notify_rules_disabled'):
            if not spec_pb.alerting.notify_rules_disabled:
                raise exceptions.BadRequestError('"spec.alerting.notify_rules_disabled" '
                                                 'cannot be set to False')
            return

        juggler_notify_rules = alerting.AlertingConfig.get_juggler_raw_notify_rules(spec_pb.alerting)
        balancer_juggler_notify_rules = [n for group, n in juggler_notify_rules if group == alerting.BALANCER_GROUP]
        platform_juggler_notify_rules = [n for group, n in juggler_notify_rules if group == alerting.PLATFORM_GROUP]
        if not balancer_juggler_notify_rules or len(balancer_juggler_notify_rules) > 5:
            raise exceptions.BadRequestError('"spec.alerting.juggler_raw_notify_rules.balancer[]" '
                                             'length should be in range [1, 5]')

        if len(platform_juggler_notify_rules) > 5:
            raise exceptions.BadRequestError('"spec.alerting.juggler_raw_notify_rules.platform[]" '
                                             'length should be in range [0, 5]')

        has_notify_crit_status = False
        for i, (group, juggler_raw_notify_rule) in enumerate(juggler_notify_rules):
            group_field = 'balancer'
            if group == alerting.PLATFORM_GROUP:
                group_field = 'platform'
                i -= len(balancer_juggler_notify_rules)
            if not juggler_raw_notify_rule.template_name:
                raise exceptions.BadRequestError('"spec.alerting.juggler_raw_notify_rules.{}[{}].template_name" '
                                                 'must be set'.format(group_field, i))

            if not juggler_raw_notify_rule.template_kwargs:
                raise exceptions.BadRequestError('"spec.alerting.juggler_raw_notify_rules.{}[{}].template_kwargs" '
                                                 'must be set'.format(group_field, i))
            try:
                data = yaml.safe_load(juggler_raw_notify_rule.template_kwargs)
                status_change = data.get('status')
                if (isinstance(status_change, list) and
                        group == alerting.BALANCER_GROUP and
                        juggler_raw_notify_rule.template_name == alerting.REQUIRED_BALANCER_NOTIFY_RULE_NAME and
                        sorted(status_change, key=alerting.notify_rule_status_change_key) == alerting.REQUIRED_CRIT_BALANCER_NOTIFY_RULE_STATUS_CHANGE and
                        data.get('login') and
                        data.get('method')):
                    has_notify_crit_status = True

            except Exception as e:
                raise exceptions.BadRequestError('"spec.alerting.juggler_raw_notify_rules.{}[{}].template_kwargs" '
                                                 'should be valid yaml: {}'.format(group_field, i, e))

        if not has_notify_crit_status:
            _msg_tpl = ('"spec.alerting.juggler_raw_notify_rules.balancer[0]" should be valid juggler notify rule '
                        'with template_name: {} and template_kwargs: {}')
            raise exceptions.BadRequestError(_msg_tpl.format(
                alerting.REQUIRED_BALANCER_NOTIFY_RULE_NAME,
                alerting.REQUIRED_CRIT_BALANCER_NOTIFY_RULE_STATUS_CHANGE
            ))


def validate_update_namespace_request(req_pb):
    """
    :type req_pb: api_pb2.UpdateNamespaceRequest
    :raises: exceptions.BadRequestError
    """
    if not req_pb.meta.id:
        raise exceptions.BadRequestError(u'"meta.id" must be set')
    if not util.NAMESPACE_ID_RE.match(req_pb.meta.id):
        raise exceptions.BadRequestError(u'"meta.id" is not valid')
    if (not req_pb.meta.category and
            not req_pb.meta.HasField('auth') and
            not req_pb.meta.abc_service_id and
            not req_pb.HasField('spec')):
        raise exceptions.BadRequestError(
            u'at least one of the "spec", "meta.category", "meta.auth" or "meta.abc_service_id" '
            u'fields must be present')
    if req_pb.meta.category:
        validate_category(req_pb.meta.category, u'meta.category')
    util.validate_staff_logins_list(req_pb.meta.webauth.responsible.logins,
                                    field_name='meta.webauth.responsible.logins')
    if req_pb.meta.HasField('auth'):
        util.validate_auth_pb(req_pb.meta.auth, field_name=u'meta.auth')
    if req_pb.HasField('spec'):
        validate_namespace_spec_pb(req_pb.spec, req_pb.meta.id, field_name=u'spec')


def validate_list_namespaces_request(req_pb):
    """
    :type req_pb: api_pb2.ListNamespacesRequest
    :raises: exceptions.BadRequestError
    """
    if req_pb.HasField('field_mask'):
        if not req_pb.field_mask.IsValidForDescriptor(model_pb2.Namespace.DESCRIPTOR):
            raise exceptions.BadRequestError('"field_mask" is not valid')


def validate_request(req_pb):
    """
    :raises: exceptions.BadRequestError
    """
    if isinstance(req_pb, api_pb2.GetNamespaceKnobsRequest):
        if not req_pb.namespace_id:
            raise exceptions.BadRequestError('"namespace_id" must be set')
    elif isinstance(req_pb, api_pb2.GetNamespaceAlertingConfigRequest):
        if not req_pb.namespace_id:
            raise exceptions.BadRequestError('"namespace_id" must be set')
    elif isinstance(req_pb, api_pb2.GetNamespaceRevisionRequest):
        if not req_pb.id:
            raise exceptions.BadRequestError('"id" must be set')
    elif isinstance(req_pb, api_pb2.ListNamespacesRequest):
        validate_list_namespaces_request(req_pb)
    elif isinstance(req_pb, api_pb2.ListNamespaceSummariesRequest):
        pass
    elif isinstance(req_pb, (api_pb2.GetNamespaceRequest,
                             api_pb2.GetNamespaceAspectsSetRequest,
                             api_pb2.ListNamespaceRevisionsRequest,
                             api_pb2.RemoveNamespaceRequest,
                             )):
        if not req_pb.id:
            raise exceptions.BadRequestError('"id" must be set')
        if not util.NAMESPACE_ID_RE.match(req_pb.id):
            raise exceptions.BadRequestError('"id" is not valid')
    elif isinstance(req_pb, api_pb2.CancelNamespaceOrderRequest):
        if not req_pb.id:
            raise exceptions.BadRequestError('"id" must be set')
        if not util.NAMESPACE_ID_RE.match(req_pb.id):
            raise exceptions.BadRequestError('"id" is not valid')
    elif isinstance(req_pb, api_pb2.CreateNamespaceRequest):
        validate_create_namespace_request(req_pb)
    elif isinstance(req_pb, api_pb2.CreateNamespaceOnAllocatedResourcesRequest):
        validate_create_namespace_on_allocated_resources_request(req_pb)
    elif isinstance(req_pb, api_pb2.UpdateNamespaceRequest):
        validate_update_namespace_request(req_pb)
    elif isinstance(req_pb, api_pb2.UpdateNamespaceOrderContextRequest):
        if not req_pb.id:
            raise exceptions.BadRequestError('"id" must be set')
        if not req_pb.fields:
            raise exceptions.BadRequestError('"fields" must be set')
        for field_pb in req_pb.fields:
            if not field_pb.key:
                raise exceptions.BadRequestError('"key" must be set')
    elif isinstance(req_pb, api_pb2.ApplyNamespaceAlertingPresetRequest):
        if not req_pb.namespace_id:
            raise exceptions.BadRequestError('"namespace_id" must be set')
        if req_pb.preset == model_pb2.NamespaceSpec.PR_UNKNOWN:
            raise exceptions.BadRequestError('"preset" must be set')
    elif isinstance(req_pb, api_pb2.GetNamespaceOperationRequest):
        if not req_pb.namespace_id:
            raise exceptions.BadRequestError('"namespace_id" must be set')
        if not req_pb.id:
            raise exceptions.BadRequestError('"id" must be set')
    elif isinstance(req_pb, api_pb2.ListNamespaceOperationsRequest):
        if not req_pb.namespace_id:
            raise exceptions.BadRequestError('"namespace_id" must be set')
    else:
        raise RuntimeError('Incorrect `req_pb` type: {}'.format(req_pb.__class__.__name__))


def validate_cancel_namespace_order(namespace_pb):
    if not is_order_in_progress(namespace_pb):
        raise exceptions.BadRequestError('Cannot cancel order that is not in progress')
    from awacs.model.namespace.order.processors import get_namespace_order_processors
    if not can_be_cancelled(namespace_pb, get_namespace_order_processors()):
        raise exceptions.BadRequestError('Cannot cancel namespace order at this stage')


def validate_network_macro(location, login, network_macro):
    """
    :param six.text_type location:
    :param six.text_typ login:
    :param six.text_typ network_macro:
    """
    yp_lite = IYpLiteRpcClient.instance()
    req_pb = pod_sets_api_pb2.ValidateNetworkMacroRequest(
        cluster=location.upper(),
        login=login,
        network_macro=network_macro)
    resp_pb = yp_lite.validate_network_macro(req_pb)
    if resp_pb.WhichOneof('outcome') == 'failure':
        message = resp_pb.failure.message
        if resp_pb.failure.status == 'False':
            raise rpc.exceptions.BadRequestError(message)
        else:
            raise rpc.exceptions.InternalError(message)


def validate_quota_settings(location, login, abc_service_id):
    """
    :param six.text_typ location:
    :param six.text_typ login:
    :param int abc_service_id:
    """
    yp_lite = IYpLiteRpcClient.instance()
    req_pb = pod_sets_api_pb2.ValidateQuotaSettingsRequest(
        cluster=location.upper(),
        login=login)
    req_pb.quota_settings.mode = req_pb.quota_settings.ABC_SERVICE
    req_pb.quota_settings.abc_service_id = abc_service_id
    resp_pb = yp_lite.validate_quota_settings(req_pb)
    if resp_pb.WhichOneof('outcome') == 'failure':
        message = resp_pb.failure.message
        if resp_pb.failure.status == 'False':
            raise rpc.exceptions.BadRequestError(message)
        else:
            raise rpc.exceptions.InternalError(message)


def validate_dns_record_request(dns_record_req_pb):
    """
    :type dns_record_req_pb: model_pb2.NamespaceOrder.Content.DnsRecordRequest
    """
    if dns_record_req_pb.type != model_pb2.NamespaceOrder.Content.DnsRecordRequest.DEFAULT:
        raise rpc.exceptions.BadRequestError(
            u'DNS record request of type "{}" is not supported'.format(dns_record_req_pb.type))
    zone = dns_record_req_pb.default.zone
    try:
        util.validate_domain_name(zone)
    except Exception as e:
        raise rpc.exceptions.BadRequestError(u'DNS zone "{}" has incorrect format: {}'.format(zone, e))
    dns_record_validation.validate_fqdn(dns_record_req_pb.default.name_server.namespace_id, dns_record_req_pb.default.name_server.id,
                                        zone, '', '', 'order.dns_record_request')


def validate_order_content(order_content_pb, namespace_id, auth_subject, abc_service_id, validate_yp_endpoint_sets):
    """
    :type order_content_pb: model_pb2.NamespaceOrder.Content
    :type namespace_id: six.text_type
    :type abc_service_id: int
    :type validate_yp_endpoint_sets: bool
    """
    if order_content_pb.flow_type == order_content_pb.GENCFG:
        raise rpc.exceptions.BadRequestError('order.content.flow_type: GENCFG balancers are not supported')

    validate_system_backend_ids(order_content_pb, namespace_id)
    if order_content_pb.HasField('yp_lite_allocation_request'):
        validate_yp_acl(namespace_id, order_content_pb, abc_service_id, auth_subject.login)
    if order_content_pb.HasField('dns_record_request'):
        validate_dns_record_request(order_content_pb.dns_record_request)
    if order_content_pb.HasField('certificate_order_content'):
        validate_cert_order_content(order_content_pb.certificate_order_content, auth_subject, 'certificate_order')

    if order_content_pb.flow_type == order_content_pb.YP_LITE:
        validate_yp_lite_backends(order_content_pb, validate_yp_endpoint_sets=validate_yp_endpoint_sets)

    if not order_content_pb.alerting_simple_settings.notify_staff_group_id:
        raise rpc.exceptions.BadRequestError(
            'order.content.alerting_simple_settings.notify_staff_group_id is mandatory')

    validate_staff_group_ids(
        staff_group_ids=[order_content_pb.alerting_simple_settings.notify_staff_group_id],
        field_name='order.alerting_simple_settings.notify_staff_group_id',
    )


def validate_system_backend_ids(order_content_pb, namespace_id):
    """
    :type order_content_pb: model_pb2.NamespaceOrder.Content
    :type namespace_id: six.text_type
    """
    user_backend_ids = make_backend_ids(order_content_pb)
    system_backend_ids = make_system_backend_ids(order_content_pb, namespace_id)

    intersecting_ids = user_backend_ids.intersection(system_backend_ids)
    if intersecting_ids:
        raise rpc.exceptions.BadRequestError(
            u'The following backend ids are reserved for awacs use, please rename them: "{}"'.format(
                ', '.join(sorted(intersecting_ids))
            ))


def make_backend_ids(order_content_pb):
    if order_content_pb.flow_type == order_content_pb.QUICK_START:
        return {u'{}_{}'.format(es.id, es.cluster).replace('.', '_').lower() for es in order_content_pb.endpoint_sets}
    else:
        return set(order_content_pb.backends.keys())


def make_system_backend_ids(order_content_pb, namespace_id):
    balancer_ids = set()
    for location in order_content_pb.yp_lite_allocation_request.locations:
        balancer_ids.add(balancer_util.make_awacs_balancer_id(namespace_id, location))
    return balancer_ids


def validate_yp_lite_backends(order_content_pb, validate_yp_endpoint_sets):
    """
    :type order_content_pb: model_pb2.NamespaceOrder.Content
    :type validate_yp_endpoint_sets: bool
    """
    jobs = []
    service_ids = []
    for backend_id, backend_pb in six.iteritems(order_content_pb.backends):
        if validate_yp_endpoint_sets and backend_pb.type in (model_pb2.BackendSelector.YP_ENDPOINT_SETS,
                                                             model_pb2.BackendSelector.YP_ENDPOINT_SETS_SD):
            jobs.append(gevent.spawn(util.validate_yp_endpoint_sets,
                                     backend_id, backend_pb.yp_endpoint_sets))
        elif backend_pb.type == model_pb2.BackendSelector.NANNY_SNAPSHOTS:
            service_ids.extend(snapshot_pb.service_id for snapshot_pb in backend_pb.nanny_snapshots)
    if service_ids:
        util.validate_services_policy(service_ids)
    if jobs:
        gevent.wait(jobs)
        for job in jobs:
            job.get()


def validate_yp_acl(namespace_id, order_pb, abc_service_id, login):
    allocation_request_pb = order_pb.yp_lite_allocation_request
    locations = allocation_request_pb.locations
    network_macro = allocation_request_pb.network_macro
    jobs = []
    for location in locations:
        if order_pb.cloud_type == model_pb2.CT_AZURE:
            location = external_clusters.AZURE_CLUSTERS_BY_NAME[location].yp_cluster
            location_type = model_pb2.BalancerMeta.Location.AZURE_CLUSTER
        else:
            location_type = model_pb2.BalancerMeta.Location.YP_CLUSTER
        jobs.append(gevent.spawn(validate_nanny_service_nonexistence,
                                 make_nanny_service_id(make_awacs_balancer_id(namespace_id, location), location_type)))
        jobs.append(gevent.spawn(validate_network_macro, location, login, network_macro))
        jobs.append(gevent.spawn(validate_quota_settings, location, login, abc_service_id))
    gevent.wait(jobs)
    for job in jobs:
        job.get()


def validate_alerting(namespace_id, old_spec_pb, new_spec_pb):
    alerting_is_enabled = old_spec_pb.HasField('alerting')
    alerting_should_be_enabled = new_spec_pb.HasField('alerting')

    if not alerting_is_enabled and not alerting_should_be_enabled:
        return  # noop
    elif alerting_is_enabled and not alerting_should_be_enabled:
        raise exceptions.BadRequestError('Alerting can not be disabled')
    elif not alerting_is_enabled and alerting_should_be_enabled:
        if semantic_version.Version(new_spec_pb.alerting.version) < alerting.CURRENT_VERSION:
            raise exceptions.BadRequestError('"spec.alerting.version": {} is not supported anymore '
                                             '(use "{}" or later)'.format(new_spec_pb.alerting.version,
                                                                          alerting.CURRENT_VERSION))
    elif alerting_is_enabled and alerting_should_be_enabled:
        # change alerting settings
        alerting_config = alerting.get_config(new_spec_pb.alerting.version)
        alerting_prefix = config.get_value('alerting.name_prefix')
        if not alerting_prefix:
            raise RuntimeError('"alerting.name_prefix" awacs config value not set')
        if new_spec_pb.alerting.version != old_spec_pb.alerting.version:
            if semantic_version.Version(new_spec_pb.alerting.version) < alerting.CURRENT_VERSION:
                raise exceptions.BadRequestError('"spec.alerting.version": {} is not supported anymore '
                                                 '(use {} or later)'.format(new_spec_pb.alerting.version,
                                                                            alerting.CURRENT_VERSION))

        current_notify_rules = alerting_config.gen_juggler_notify_rules(namespace_id, alerting_prefix,
                                                                        old_spec_pb.alerting)
        new_notify_rules = alerting_config.gen_juggler_notify_rules(namespace_id, alerting_prefix,
                                                                    new_spec_pb.alerting)
        if new_notify_rules == current_notify_rules:
            return

        _juggler_client = juggler_client.IJugglerClient.instance()
        juggler_namespace = alerting_config.build_juggler_namespace(alerting_prefix, namespace_id)
        try:
            _juggler_client.sync_notify_rules(juggler_namespace, [r[0] for r in new_notify_rules])
        except Exception as e:
            notify_rule_field, notify_rule_index = 'none', 'none'
            if hasattr(e, 'notify_rule'):
                for notify_rule, notify_rule_group, juggler_raw_rule_pb in new_notify_rules:
                    if e.notify_rule != notify_rule:
                        continue
                    notify_rule_field = alerting.NOTIFY_GROUP_TO_FIELD_NAME[notify_rule_group]
                    rule_pbs = getattr(new_spec_pb.alerting.juggler_raw_notify_rules, notify_rule_field)
                    for i, value_pb in enumerate(rule_pbs):
                        if value_pb == juggler_raw_rule_pb:
                            notify_rule_index = i

            raise exceptions.BadRequestError('"spec.alerting.juggler_raw_notify_rules.{}[{}].template_kwargs" '
                                             'has error: {}'.format(notify_rule_field, notify_rule_index, e))


def validate_preset(project, preset):
    """
    :type project: six.text_type
    :type preset: model_pb2.NamespaceSpec.NamespaceSettingsPreset
    """
    if preset == model_pb2.NamespaceSpec.PR_WITHOUT_NOTIFICATIONS and project != 'taxi':
        raise rpc.exceptions.BadRequestError(
            u'Project {!r} is not allowed to use preset {!r}'.format(
                project,
                model_pb2.NamespaceSpec.NamespaceSettingsPreset.Name(preset),
            )
        )
    elif preset == model_pb2.NamespaceSpec.PR_PLATFORM_CHECKS_ONLY and project != 'maps':
        raise rpc.exceptions.BadRequestError(
            u'Project {!r} is not allowed to use preset {!r}'.format(
                project,
                model_pb2.NamespaceSpec.NamespaceSettingsPreset.Name(preset),
            )
        )


def validate_namespace_normalized_prj(namespace_id, prj, field_name="order.instance_tags.prj"):
    _cache = cache.IAwacsCache.instance()
    _namespace_id = _cache.get_namespace_id_by_normalised_prj(prj)
    if _namespace_id is not None and namespace_id != _namespace_id:
        raise exceptions.BadRequestError(
            '"{}": normalized prj tag "{}" '
            'is already used by another namespace: {}'.format(field_name,
                                                              _cache.normalise_prj_tag(prj),
                                                              _namespace_id)
        )


def validate_namespaces_with_enabled_its_count():
    _cache = cache.IAwacsCache.instance()
    namespaces_with_enabled_its_max_count = config.get_value('run.namespaces_with_enabled_its_max_count', None)
    if not namespaces_with_enabled_its_max_count:
        return
    if _cache.count_namespaces_with_enabled_its() >= namespaces_with_enabled_its_max_count:
        raise rpc.exceptions.ForbiddenError('"spec.its": max count of namespaces with enabled ITS reached, '
                                            'please contact st/BALANCERSUPPORT.')


def apply_l3_only_layout_constraints(namespace_spec_pb):
    namespace_spec_pb.object_upper_limits.upstream.value = 0
    namespace_spec_pb.object_upper_limits.balancer.value = 0
    namespace_spec_pb.object_upper_limits.domain.value = 0
    namespace_spec_pb.object_upper_limits.knob.value = 0
    namespace_spec_pb.object_upper_limits.certificate.value = 0
    namespace_spec_pb.object_upper_limits.weight_section.value = 0
    namespace_spec_pb.object_upper_limits.l7heavy_config.value = 0


def apply_global_layout_constraints(namespace_spec_pb):
    namespace_spec_pb.object_upper_limits.upstream.value = 0
    namespace_spec_pb.object_upper_limits.balancer.value = 0
    namespace_spec_pb.object_upper_limits.l3_balancer.value = 0
    namespace_spec_pb.object_upper_limits.dns_record.value = 0
    namespace_spec_pb.object_upper_limits.domain.value = 0
    namespace_spec_pb.object_upper_limits.knob.value = 0
    namespace_spec_pb.object_upper_limits.certificate.value = 0
    namespace_spec_pb.object_upper_limits.weight_section.value = 0
    namespace_spec_pb.object_upper_limits.l7heavy_config.value = 0


def apply_external_layout_constraints(namespace_spec_pb):
    namespace_spec_pb.object_upper_limits.knob.value = 0
    namespace_spec_pb.object_upper_limits.l3_balancer.value = 0
    namespace_spec_pb.object_upper_limits.dns_record.value = 0
    namespace_spec_pb.object_upper_limits.weight_section.value = 0
    namespace_spec_pb.object_upper_limits.l7heavy_config.value = 0


def validate_create(req_pb, auth_pb):
    namespace_id = req_pb.meta.namespace_id
    c = cache.IAwacsCache.instance()

    namespace_pb = c.must_get_namespace(namespace_id=namespace_id)
    authorize_create(namespace_pb, auth_pb)
    forbid_action_during_namespace_order(namespace_pb, auth_pb)


def validate_namespace_op_create(req_pb, auth_pb):
    """
    :type req_pb: api_pb2.CreateNamespaceOperationRequest
    """
    if not req_pb.meta.namespace_id:
        raise exceptions.BadRequestError(u'"meta.namespace_id" must be set')
    validate_create(req_pb, auth_pb)

    op_name = objects.NamespaceOperation.get_operation_name(req_pb.order)
    op_pb = objects.NamespaceOperation.get_operation(req_pb.order)
    field = u'req_pb.meta.parent_versions.l3_versions'
    l3_balancer_id = op_pb.l3_balancer_id

    if l3_balancer_id not in req_pb.meta.parent_versions.l3_versions:
        raise exceptions.BadRequestError(
            u'"{}": must contain the latest revision of L3 balancer "{}"'.format(field, l3_balancer_id))
    l3_balancer_pb = zk.IZkStorage.instance().must_get_l3_balancer(req_pb.meta.namespace_id, l3_balancer_id)
    given_ver = req_pb.meta.parent_versions.l3_versions[l3_balancer_id]
    if given_ver != l3_balancer_pb.meta.version:
        raise exceptions.ConflictError(
            u'"{}[{}]": latest revision is "{}", not "{}"'.format(field, l3_balancer_id, l3_balancer_pb.meta.version,
                                                                  given_ver))

    if l3_balancer_pb.spec.incomplete:
        raise exceptions.BadRequestError(u'Cannot create an operation for an incomplete L3 balancer')
    if op_name is objects.NamespaceOperation.IMPORT_VS_FROM_L3MGR:
        if l3mgr.is_fully_managed(l3_balancer_pb.spec):
            raise exceptions.BadRequestError(u'"spec.config_management_mode" cannot be MODE_REAL_AND_VIRTUAL_SERVERS '
                                             u'to use operation "import_virtual_servers_from_l3mgr"')
        if l3_balancer_pb.spec.preserve_foreign_real_servers:
            raise exceptions.BadRequestError(u'"spec.preserve_foreign_real_servers" is not supported in fully-managed '
                                             u'L3 balancers')
        if not op_pb.enforce_configs:
            # check consistency of all VS configs
            l3mgr_client = l3mgrclient.IL3MgrClient.instance()
            svc_id = l3_balancer_pb.spec.l3mgr_service_id
            latest_config = l3mgr.ServiceConfig.latest_from_api(l3mgr_client, svc_id)
            l3mgr.VirtualServers.from_api(l3mgr_client, svc_id, latest_config.vs_ids,
                                          allow_l3mgr_configs_mismatch=False)


def validate_namespace_op_cancel(req_pb, auth_pb):
    """
    :type req_pb: api_pb2.CancelNamespaceOperationRequest
    :type auth_pb:
    """
    if not req_pb.namespace_id:
        raise exceptions.BadRequestError(u'"meta.namespace_id" must be set')
    if not req_pb.id:
        raise exceptions.BadRequestError(u'"meta.id" must be set')
    namespace_pb = cache.IAwacsCache.instance().must_get_namespace(namespace_id=req_pb.namespace_id)
    authorize_create(namespace_pb, auth_pb)
    forbid_action_during_namespace_order(namespace_pb, auth_pb)

    ns_op_pb = objects.NamespaceOperation.cache.must_get(req_pb.namespace_id, req_pb.id)
    if not is_order_in_progress(ns_op_pb):
        raise exceptions.BadRequestError(u'Cannot cancel operation that is not in progress')
    if req_pb.force:
        if is_root(auth_pb):
            return
        raise exceptions.ForbiddenError(u'Only awacs admins can force cancel')
    operation = objects.NamespaceOperation.get_operation_name(ns_op_pb.order.content)
    processors = NamespaceOperationCtl.runners[operation].processors
    if not can_be_cancelled(ns_op_pb, processors):
        raise exceptions.BadRequestError(u'Cannot cancel operation at this stage')
