# coding: utf-8
import itertools

import six

from awacs.lib.rpc import exceptions

from six.moves import zip_longest
from google.protobuf import descriptor
from google.protobuf.pyext._message import RepeatedCompositeContainer, MessageMapContainer


__all__ = ['validate_readonly_fields']


def field_is_readonly(field):
    return any(
        option_type.full_name == 'awacs.model.awacs' and option_value.readonly is True
        for option_type, option_value in field.GetOptions().ListFields()
    )


def field_is_map_entry(field):
    return (
        field.type == descriptor.FieldDescriptor.TYPE_MESSAGE
        and field.message_type.has_options
        and field.message_type.GetOptions().map_entry
    )



def validate_readonly_default_values(pb):
    for field in pb.DESCRIPTOR.fields:
        is_readonly = field_is_readonly(field)

        try:
            default_value = field.default_value
        except NotImplementedError:  # proto3 + complex message
            default_value = None

        val = getattr(pb, field.name)

        if field_is_map_entry(field):
            if is_readonly and len(val):
                raise exceptions.BadRequestError(
                    '{!r} is readonly field and must be empty'.format(field.full_name)
                )

            if isinstance(val, RepeatedCompositeContainer):
                for k in val:
                    validate_readonly_default_values(val[k])

        elif field.label == descriptor.FieldDescriptor.LABEL_REPEATED:
            if is_readonly and len(val):
                raise exceptions.BadRequestError(
                    '{!r} is readonly field and must be empty'.format(field.full_name)
                )

            if field.message_type is not None:
                for item in val:
                    validate_readonly_default_values(item)

        elif field.message_type is None:
            if is_readonly and val != default_value:
                raise exceptions.BadRequestError(
                    '{!r} is readonly field and must not be changed'.format(field.full_name)
                )

        else:
            if is_readonly and pb.HasField(field.name):
                raise exceptions.BadRequestError(
                    '{!r} is readonly field and must not be set'.format(field.full_name)
                )

            validate_readonly_default_values(val)


def validate_readonly_fields(old_pb, new_pb):
    assert old_pb.DESCRIPTOR == new_pb.DESCRIPTOR, "Cannot compare {} with {}".format(
        old_pb.DESCRIPTOR.full_name,
        new_pb.DESCRIPTOR.full_name,
    )

    for field in old_pb.DESCRIPTOR.fields:
        is_readonly = field_is_readonly(field)
        try:
            default_value = field.default_value
            if default_value is None:
                assert six.PY3
                raise NotImplementedError()
        except NotImplementedError:  # proto3 + complex message
            default_value = None

            try:
                old_has_attr = old_pb.HasField(field.name)
            except ValueError as e:  # that's what proto does for repeated fields
                old_has_attr = True

            try:
                new_has_attr = new_pb.HasField(field.name)
            except ValueError as e:  # same here
                new_has_attr = True

        else:
            old_has_attr = new_has_attr = True

        old_val = getattr(old_pb, field.name)
        new_val = getattr(new_pb, field.name)

        if old_has_attr and not new_has_attr:
            if is_readonly:
                raise exceptions.BadRequestError(
                    '{!r} is readonly field and must not be changed'.format(field.full_name)
                )
            else:
                continue
        elif not old_has_attr and new_has_attr:
            validate_readonly_default_values(new_val)
            continue
        elif not old_has_attr and not new_has_attr:
            continue

        if is_readonly and old_val != new_val:
            raise exceptions.BadRequestError(
                '{!r} is readonly field and cannot be changed by user'.format(field.full_name)
            )

        if field.message_type is None:
            continue

        if isinstance(old_val, RepeatedCompositeContainer):
            for pair in zip_longest(old_val, new_val):
                if pair[0] is None:
                    # if user is adding complex message previously being unset, its readonly fields must be default
                    validate_readonly_default_values(pair[1])
                elif pair[1] is None:
                    # but we consider it safe to remove complex message even if it contains any readonly fields
                    continue
                else:
                    validate_readonly_fields(pair[0], pair[1])

        elif isinstance(old_val, MessageMapContainer):
            keys = set(old_val.keys()) | set(new_val.keys())
            value_type = field.message_type.fields_by_name['value'].message_type
            assert value_type is not None, "Probably broken protobuf, {!r} value type should not be None".format(
                field.full_name,
            )

            for k in keys:
                old_val_dict_value = old_val.get(k)
                new_val_dict_value = new_val.get(k)

                if new_val_dict_value is None:
                    # we consider it safe to remove complex message even if it contains any readonly fields
                    continue
                elif old_val_dict_value is None:
                    # but if user is adding complex message previously being unset, its readonly fields must be default
                    validate_readonly_default_values(new_val_dict_value)
                else:
                    validate_readonly_fields(old_val_dict_value, new_val_dict_value)
        else:
            validate_readonly_fields(old_val, new_val)
