# coding: utf-8
import re

from awacs.lib.rpc import exceptions
from awacs.model.cache import IAwacsCache
from infra.awacs.proto import api_pb2, model_pb2, modules_pb2
from awacs.web.validation.util import UPSTREAM_ID_RE, COMPAT_ID_RE, NAMESPACE_ID_RE, validate_auth_pb


MAX_UPSTREAM_LINES_COUNT = 1500
MAX_UPSTREAM_CHARS_COUNT = 80000


def validate_upstream_yaml_size(yml, field_name):
    """
    :type yml: six.text_type
    :type field_name: six.text_type
    """
    chars_count = len(yml)
    if chars_count > MAX_UPSTREAM_CHARS_COUNT:
        raise exceptions.BadRequestError(
            u'"{}" contains too many characters ({}, allowed limit is {})'.format(
                field_name, chars_count, MAX_UPSTREAM_CHARS_COUNT))

    lines_count = len(yml.splitlines())
    if lines_count > MAX_UPSTREAM_LINES_COUNT:
        raise exceptions.BadRequestError(
            u'"{}" contains too many lines ({}, allowed limit is {})'.format(
                field_name, lines_count, MAX_UPSTREAM_LINES_COUNT))


def validate_yandex_balancer_upstream_spec(spec_pb, field_name='yandex_balancer'):
    """
    :type spec_pb: model_pb2.YandexBalancerUpstreamSpec
    :type field_name: six.text_type
    :raises: exceptions.BadRequestError
    """
    if not spec_pb.HasField('config') and not spec_pb.yaml:
        raise exceptions.BadRequestError(
            'Either "{0}.config" or "{0}.yaml" must be set'.format(field_name))
    if spec_pb.type == spec_pb.INTERNAL and spec_pb.mode != spec_pb.EASY_MODE2:
        raise exceptions.BadRequestError('"{}.mode": must be EASY_MODE2 for internal upstreams'.format(field_name))



def validate_id(spec_pb, upstream_id, field_name='meta.id'):
    """
    :type spec_pb: model_pb2.UpstreamSpec
    :type upstream_id: six.text_type
    :type field_name: six.text_type
    :raises: spec.BadRequestError
    """
    if spec_pb.yandex_balancer.mode == spec_pb.yandex_balancer.EASY_MODE2 and '.' in upstream_id:
        raise exceptions.BadRequestError("\"{}\": easy-mode upstream id cannot contain '.'".format(field_name))


def validate_spec(spec_pb, field_name='spec'):
    """
    :type spec_pb: model_pb2.UpstreamSpec
    :type field_name: six.text_type
    :raises: spec.BadRequestError
    """
    spec_type_name = model_pb2.BalancerType.Name(spec_pb.type)
    if spec_pb.type == model_pb2.YANDEX_BALANCER:
        if spec_pb.WhichOneof('spec') != 'yandex_balancer':
            raise exceptions.BadRequestError(
                '"{0}.type" is set to "{1}", but "{0}.yandex_balancer" '
                'field is missing'.format(field_name, spec_type_name))
        validate_yandex_balancer_upstream_spec(spec_pb.yandex_balancer, field_name=field_name + '.yandex_balancer')
    else:
        raise exceptions.BadRequestError('"{0}.type": unsupported config type "{1}"'.format(field_name, spec_type_name))


def validate_create_upstream_request(req_pb):
    """
    :type req_pb: api_pb2.CreateUpstreamRequest
    :raises: exceptions.BadRequestError
    """
    if not req_pb.HasField('meta'):
        raise exceptions.BadRequestError('"meta" must be set')
    if not req_pb.meta.id:
        raise exceptions.BadRequestError('"meta.id" must be set')
    if not UPSTREAM_ID_RE.match(req_pb.meta.id):
        raise exceptions.BadRequestError('"meta.id" is not valid')
    if (req_pb.spec.yandex_balancer.type == model_pb2.YandexBalancerUpstreamSpec.INTERNAL and
            not req_pb.meta.id.startswith('_')):
        raise exceptions.BadRequestError('"meta.id": must be started with "_" for internal upstreams')
    elif (req_pb.spec.yandex_balancer.type != model_pb2.YandexBalancerUpstreamSpec.INTERNAL and
            req_pb.meta.id.startswith('_')):
        raise exceptions.BadRequestError('"meta.id": must not be started with "_" for non-internal upstreams')
    if not req_pb.meta.namespace_id:
        raise exceptions.BadRequestError('"meta.namespace_id" must be set')
    if not NAMESPACE_ID_RE.match(req_pb.meta.namespace_id):
        raise exceptions.BadRequestError('"meta.namespace_id" is not valid')
    if not req_pb.meta.HasField('auth'):
        raise exceptions.BadRequestError('"meta.auth" must be set')
    validate_auth_pb(req_pb.meta.auth, field_name='meta.auth')
    if not req_pb.HasField('spec'):
        raise exceptions.BadRequestError('"spec" must be set')
    validate_id(req_pb.spec, req_pb.meta.id)
    validate_spec(req_pb.spec)


def validate_lint_upstream_request(req_pb):
    """
    :type req_pb: api_pb2.LintUpstreamRequest
    :raises: exceptions.BadRequestError
    """
    if not req_pb.HasField('meta'):
        raise exceptions.BadRequestError('"meta" must be set')
    if not req_pb.meta.id:
        raise exceptions.BadRequestError('"meta.id" must be set')
    if not COMPAT_ID_RE.match(req_pb.meta.id):
        raise exceptions.BadRequestError('"meta.id" is not valid')
    if not req_pb.meta.namespace_id:
        raise exceptions.BadRequestError('"meta.namespace_id" must be set')
    if not NAMESPACE_ID_RE.match(req_pb.meta.namespace_id):
        raise exceptions.BadRequestError('"meta.namespace_id" is not valid')
    if req_pb.HasField('spec'):
        validate_spec(req_pb.spec)


def validate_update_upstream_request(req_pb):
    """
    :type req_pb: api_pb2.UpdateUpstreamRequest
    :raises: exceptions.BadRequestError
    """
    if not req_pb.HasField('meta'):
        raise exceptions.BadRequestError('"meta" must be set')
    if not req_pb.meta.id:
        raise exceptions.BadRequestError('"meta.id" must be set')
    if not COMPAT_ID_RE.match(req_pb.meta.id):
        raise exceptions.BadRequestError('"meta.id" is not valid')
    if not req_pb.meta.namespace_id:
        raise exceptions.BadRequestError('"meta.namespace_id" must be set')
    if not NAMESPACE_ID_RE.match(req_pb.meta.namespace_id):
        raise exceptions.BadRequestError('"meta.namespace_id" is not valid')
    if not req_pb.meta.version:
        raise exceptions.BadRequestError('"meta.version" must be set')
    if not req_pb.HasField('spec') and not req_pb.meta.HasField('auth'):
        raise exceptions.BadRequestError('at least one of the "spec" or "meta.auth" fields must be present')
    if req_pb.HasField('spec'):
        validate_id(req_pb.spec, req_pb.meta.id)
        validate_spec(req_pb.spec)
    if req_pb.meta.HasField('auth'):
        validate_auth_pb(req_pb.meta.auth, field_name='meta.auth')


def validate_list_upstreams_request(req_pb):
    """
    :type req_pb: api_pb2.ListUpstreamsRequest
    :raises: exceptions.BadRequestError
    """
    if not req_pb.namespace_id:
        raise exceptions.BadRequestError('No "namespace_id" specified.')
    if req_pb.HasField('query') and req_pb.query.id_regexp:
        try:
            re.compile(req_pb.query.id_regexp)
        except Exception as e:
            raise exceptions.BadRequestError('"query.id_regexp" contains invalid regular expression: {}'.format(e))
    if req_pb.HasField('field_mask'):
        if not req_pb.field_mask.IsValidForDescriptor(model_pb2.Upstream.DESCRIPTOR):
            raise exceptions.BadRequestError('"field_mask" is not valid')


def validate_request(req_pb):
    """
    :raises: exceptions.BadRequestError
    """
    if isinstance(req_pb, api_pb2.ListUpstreamsRequest):
        validate_list_upstreams_request(req_pb)
    elif isinstance(req_pb, (api_pb2.GetUpstreamRequest,
                             api_pb2.RemoveUpstreamRequest,
                             api_pb2.GetUpstreamRemovalChecksRequest,
                             api_pb2.ListUpstreamRevisionsRequest)):
        if not req_pb.id:
            raise exceptions.BadRequestError('No "id" specified.')
        if not req_pb.namespace_id:
            raise exceptions.BadRequestError('No "namespace_id" specified.')
        if isinstance(req_pb, api_pb2.RemoveUpstreamRequest):
            if not req_pb.version:
                raise exceptions.BadRequestError('No "version" specified.')
    elif isinstance(req_pb, api_pb2.GetUpstreamRevisionRequest):
        if not req_pb.id:
            raise exceptions.BadRequestError('No "id" specified.')
    elif isinstance(req_pb, api_pb2.LintUpstreamRequest):
        validate_lint_upstream_request(req_pb)
    elif isinstance(req_pb, api_pb2.UpdateUpstreamRequest):
        validate_update_upstream_request(req_pb)
    elif isinstance(req_pb, api_pb2.CreateUpstreamRequest):
        validate_create_upstream_request(req_pb)
    else:
        raise RuntimeError('Incorrect `req_pb` type: {}'.format(req_pb.__class__.__name__))


def use_in_domains_removal_check(upstream_pb):
    check_pb = model_pb2.RemovalCheck()
    check_pb.type = check_pb.REFERENCES
    c = IAwacsCache().instance()
    for domain_pb in c.list_domains(upstream_pb.meta.namespace_id).items:
        if domain_pb.spec.yandex_balancer.config.include_upstreams.type == modules_pb2.BY_ID:
            if upstream_pb.meta.id in domain_pb.spec.yandex_balancer.config.include_upstreams.ids:
                check_pb.state = check_pb.FAILED
                check_pb.message = 'Upstream is required by one or more domains'
                ref_pb = check_pb.references.references.add(namespace_id=upstream_pb.meta.namespace_id,
                                                             id=domain_pb.meta.id)
                ref_pb.object_type = ref_pb.REMOVAL_REF_DOMAIN
    if check_pb.state == check_pb.FAILED:
        return check_pb

    check_pb.state = check_pb.PASSED
    check_pb.message = 'Upstream is not required by domains'
    return check_pb


def get_upstream_removal_checks(upstream_pb):
    return [
        use_in_domains_removal_check(upstream_pb)
    ]
