# coding: utf-8
from __future__ import unicode_literals
import logging
import re

import gevent
import inject
import regex
import six
from sepelib.core import config as appconfig
from six.moves import http_client as httplib
from cachetools.func import ttl_cache

from awacs.lib.certificator import ICertificatorClient
from awacs.lib import rpc, yp_service_discovery
from awacs.lib.nannyrpcclient import INannyRpcClient
from awacs.lib.rpc import exceptions
from awacs.lib.rpc.exceptions import BadRequestError
from awacs.model import cache
from infra.awacs.proto import internals_pb2
from nanny_repo import repo_pb2
from nanny_rpc_client import exceptions as nanny_rpc_exceptions
from infra.swatlib.auth.abc import IAbcClient, AbcError


logger = logging.getLogger(__name__)

YP_LITE_NANNY_SERVICE_SLUG_RE = re.compile('^[a-z][a-z0-9-]+$')
YP_LITE_NANNY_SERVICE_SLUG_MAX_LENGTH = 19
YP_LITE_NANNY_SERVICE_SLUG_MIN_LENGTH = 3

ID_RE = re.compile('^[a-z][a-z0-9_.-]{,100}$')
WEIGHT_SECTION_ID_RE = re.compile('^[a-z][a-z0-9]{,60}$')
UPSTREAM_ID_RE = re.compile('^[a-z_][a-z0-9_.-]{,100}$')
COMPAT_ID_RE = re.compile('^[A-Za-z_][A-Za-z0-9_.-]{,100}$')
DOMAIN_ID_RE = re.compile('^[a-z0-9][a-z0-9_.-]{,100}$')
CERT_ID_RE = re.compile('^[a-z0-9*_][a-z0-9_.-]{,100}$')
NAMESPACE_ID_RE = re.compile('^[a-z][a-z0-9-._]{1,60}$')

# negative lookbehind (?<!) and lookahead (?!-) to reject preceding/trailing hyphens
# \p{Ll} matches lowercase six.text_type letters (allows us to accept cyrillic domains, see SWAT-7427)
DOMAIN_PATTERN = r'^(?!-)[\p{Ll}0-9][\p{Ll}0-9-.]*?(?<!-)$'  # noqa
# we use regex (https://pypi.org/project/regex/) here because it supports \p{Ll} matchers
DOMAIN_NAME_RE = regex.compile(DOMAIN_PATTERN)
DOMAIN_NAME_I_RE = regex.compile(DOMAIN_PATTERN, re.IGNORECASE)

STAFF_LOGIN_RE = '^[a-z0-9-_.]+$'

MAX_AUTH_STAFF_LENGTH = 100

NANNY_WAIT_TIMEOUT = 5
YP_SD_TIMEOUT = 5

def get_first_item(iterable):
    return next(iter(sorted(iterable)))


@ttl_cache(maxsize=1, ttl=600)
def get_issuable_tls_hostnames():
    """
    :raises: CertificatorApiRequestException
    """
    # https://st.yandex-team.ru/AWACS-818
    return set(ICertificatorClient.instance().list_auto_managed_hosts())


def validate_staff_logins_list(logins, field_name):
    for login in logins:
        if not re.match(STAFF_LOGIN_RE, login):
            raise BadRequestError('"{}": login "{}" is not valid'.format(field_name, login))


def validate_auth_pb(auth_pb, field_name='meta.auth'):
    if not auth_pb.type:
        raise BadRequestError('"{}.type" must be set'.format(field_name))
    elif auth_pb.type != auth_pb.STAFF:
        auth_type_name = auth_pb.AuthType.Name(auth_pb.type)
        raise BadRequestError('"{}.type": auth type "{}" is not supported'.format(field_name, auth_type_name))
    if len(set(auth_pb.staff.owners.logins)) > MAX_AUTH_STAFF_LENGTH:
        raise BadRequestError(('"{}.staff.owners.logins": number of logins can not be '
                               'more than {}'.format(field_name, MAX_AUTH_STAFF_LENGTH)))
    validate_staff_logins_list(auth_pb.staff.owners.logins, field_name='{}.staff.owners.logins'.format(field_name))
    if len(set(auth_pb.staff.owners.group_ids)) > MAX_AUTH_STAFF_LENGTH:
        raise BadRequestError(('"{}.staff.owners.group_ids": number of group ids can not be '
                               'more than {}'.format(field_name, MAX_AUTH_STAFF_LENGTH)))


def validate_san(hostname):
    """
    :type hostname: six.text_type
    """
    if hostname.count('.') == 0:
        raise ValueError('hostname must contain at least one "."'.format(hostname))
    if len(hostname) > 64:
        raise ValueError('hostname length must be less than 65')
    if hostname.count('*') > 1:
        raise ValueError(u"hostname wildcard can't contain more than 1 asterisk")
    if hostname.find('*') > 0:
        raise ValueError(u"hostname wildcard can't contain asterisk in a non-leading position")
    if hostname.startswith('*'):
        if not hostname.startswith('*.'):
            raise ValueError('hostname wildcard must start with *.')
        hostname = hostname[2:]
    for part in hostname.split('.'):
        if not DOMAIN_NAME_I_RE.match(part):
            raise ValueError('string "{}" does not match pattern: {}'.format(part, DOMAIN_PATTERN))


AWACS_818_THIRD_PARTY_TLS_DOMAINS = {
    'yandex', 'mk-beta.ru', 'ruscorpora.ru', 'sportyandex.ru', 'ydf-conference.com',
    'moikrug.ru', 'themegalosaur.com', 'mk-prod.ru', 'mk-test.ru', 'mk-dev.ru', 'bem.info', 'mk-stress.ru',
}


def validate_tls_hostname_issuability(hostname, issuable_tls_hostnames):
    """
    :type hostname: six.text_type
    :type issuable_tls_hostnames: set[six.text_type]
    """
    if hostname in issuable_tls_hostnames:
        return
    # Please see https://st.yandex-team.ru/AWACS-818#60b7d08748fd44111db68f02 for details
    for issuable_tls_hostname in set(issuable_tls_hostnames) - AWACS_818_THIRD_PARTY_TLS_DOMAINS:
        if hostname.endswith('.' + issuable_tls_hostname):
            return
    raise ValueError('TLS certificate for FQDN "{}" can not be issued automatically, '
                     'please use the question form located on https://wiki.yandex-team.ru/security/ssl/'.format(hostname))


def validate_domain_name(domain_name):
    if not domain_name:
        raise ValueError('domain name cannot be empty')
    if len(domain_name) > 64:
        raise ValueError('domain name length must be less than 65')
    if domain_name[-1] == '.':
        raise ValueError('domain name must not end in .')
    if not DOMAIN_NAME_RE.match(domain_name):
        raise ValueError('domain name "{}" does not match pattern: {}'.format(domain_name, DOMAIN_PATTERN))


def get_endpoint_set_len(yp_sd_resolver, cluster, yp_endpoint_set_id, treat_not_exists_as_empty):
    """
    :type yp_sd_resolver: yp_service_discovery.Resolver
    :type cluster: six.text_type
    :type yp_endpoint_set_id: six.text_type
    :type treat_not_exists_as_empty: bool
    :rtype int
    :return number of endpoints in endpoint set
    """
    req_id = yp_service_discovery.sd_resolver.generate_reqid()
    req_pb = internals_pb2.TReqResolveEndpoints(
        cluster_name=cluster,
        endpoint_set_id=yp_endpoint_set_id)
    try:
        resp_pb = yp_sd_resolver.resolve_endpoints(req_pb, req_id=req_id)
    except:  # noqa
        logger.exception(
            'Failed to communicate with YP Service Discovery, '
            'skipped validation of YP endpoint set "%s:%s"', cluster, yp_endpoint_set_id)
        return 1

    if resp_pb.resolve_status == internals_pb2.NOT_EXISTS:
        if treat_not_exists_as_empty:
            return 0
        else:
            raise rpc.exceptions.BadRequestError('YP endpoint set "{}:{}" does not exist'.format(
                cluster, yp_endpoint_set_id))
    return len(resp_pb.endpoint_set.endpoints)


def validate_yp_endpoint_sets(backend_id, yp_endpoint_set_pbs):
    """
    :type backend_id: six.text_type
    :type yp_endpoint_set_pbs: list[model_pb2.BackendSelector.YpEndpointSet]
    :raises: exceptions.BadRequestError
    """
    yp_sd_resolver = yp_service_discovery.IResolver.instance()
    jobs = []
    treat_not_exists_as_empty = appconfig.get_value('run.enable_sd.treat_not_exists_as_empty', default=False)
    for yp_es_pb in yp_endpoint_set_pbs:
        jobs.append(gevent.spawn(get_endpoint_set_len,
                                 yp_sd_resolver, yp_es_pb.cluster, yp_es_pb.endpoint_set_id,
                                 treat_not_exists_as_empty))
    finished_jobs = gevent.wait(jobs, timeout=YP_SD_TIMEOUT)
    if len(finished_jobs) != len(jobs):
        logger.exception('Timeout on YP ES validation for backend "%s"', backend_id)
        gevent.killall(jobs)
        # to raise any exceptions occurred in finished jobs
        for job in finished_jobs:
            job.get()
    else:
        total_endpoints = 0
        for job in finished_jobs:
            total_endpoints += job.get()
        if total_endpoints == 0:
            raise rpc.exceptions.BadRequestError('All endpoint sets in backend "{}" are empty'.format(backend_id))


def validate_service_policy(nanny_rpc_client, service_id):
    try:
        p = nanny_rpc_client.get_replication_policy(service_id).policy
    except nanny_rpc_exceptions.NotFoundError:
        # Policy does not exist
        return
    except Exception as e:
        logger.exception(('Failed to communicate with Nanny API, '
                          'skipped validation of {}`s replication policy: {}'.format(service_id, e)))
        return
    if p.spec.replication_method == repo_pb2.ReplicationPolicySpec.MOVE:
        msg = (
            "Service {} has MOVE replication policy and can not be served by a backend with "
            "NANNY_SNAPSHOTS selector type. Please see "
            "https://wiki.yandex-team.ru/cplb/awacs/awacs-backends-restrictions/ "
            "for details.").format(service_id)
        raise exceptions.BadRequestError(msg)


def validate_services_policy(service_ids):
    """
    :type service_ids: Iterable[six.text_type]
    :raises: exceptions.BadRequestError
    """
    nanny_rpc_client = inject.instance(INannyRpcClient)

    jobs = []
    for service_id in service_ids:
        jobs.append(gevent.spawn(validate_service_policy, nanny_rpc_client, service_id))
    finished_jobs = gevent.wait(jobs, timeout=NANNY_WAIT_TIMEOUT)
    if len(finished_jobs) != len(jobs):
        logger.exception('Timeout on service replication policy validation')
        gevent.killall(jobs)
    for job in finished_jobs:
        job.get()


def validate_nanny_service_slug(slug, field_name):
    if (not YP_LITE_NANNY_SERVICE_SLUG_RE.match(slug) or
        not (YP_LITE_NANNY_SERVICE_SLUG_MIN_LENGTH <= len(slug) <= YP_LITE_NANNY_SERVICE_SLUG_MAX_LENGTH)):
        raise exceptions.BadRequestError(
            '"{}" is not a valid Nanny service slug. '
            'It must start with "a-z", contain only "a-z", "0-9", "-" '
            'and be of length between {} to {} inclusive.'.format(
                field_name, YP_LITE_NANNY_SERVICE_SLUG_MIN_LENGTH, YP_LITE_NANNY_SERVICE_SLUG_MAX_LENGTH))


def get_abc_service_id_by_slug(slug, field_name):
    abc_client = IAbcClient.instance()
    resp = abc_client.list_services(spec={'slug': slug})
    if len(resp['results']) == 0:
        raise rpc.exceptions.BadRequestError(
            '"{}": ABC service with slug "{}" does not exist '
            '(please see https://wiki.yandex-team.ru/intranet/abc/vodstvo/create/#other)'.format(field_name, slug))
    return resp['results'][0]['id']


def get_abc_service_slug_by_id(abc_client, abc_service_id, field_name):
    try:
        return abc_client.get_service_slug(abc_service_id)
    except AbcError as e:
        logger.exception(six.text_type(e))
        if e.resp is not None and e.resp.status_code == httplib.NOT_FOUND:
            raise rpc.exceptions.BadRequestError(
                '"{}": ABC service "{}" does not exist '
                '(please see https://wiki.yandex-team.ru/intranet/abc/vodstvo/create/#other)'.format(field_name,
                                                                                                     abc_service_id))
        raise rpc.exceptions.InternalError(
            '"{}": failed to check ABC info for service "{}"'.format(field_name, abc_service_id))


def validate_user_belongs_to_abc_service(author_login, abc_service_id, field_name):
    """
    :type author_login: six.text_type
    :type abc_service_id: int
    :type field_name: six.text_type
    """
    auth_enabled = appconfig.get_value('run.auth', default=True)
    is_author_root = author_login in appconfig.get_value('run.root_users', default=())

    abc_client = IAbcClient.instance()
    abc_service_slug = get_abc_service_slug_by_id(abc_client, abc_service_id, field_name)

    if not auth_enabled or is_author_root:
        return

    try:
        member_logins = abc_client.list_service_members(service_id=abc_service_id)
    except AbcError as e:
        logger.exception(six.text_type(e))
        raise rpc.exceptions.InternalError(
            '"{}": failed to check ABC membership info for service "{} ({})"'.format(
                field_name, abc_service_slug, abc_service_id))

    if author_login not in member_logins:
        raise rpc.exceptions.ForbiddenError(
            '"{}": user "{}" does not belong to ABC service "{} ({})"'.format(
                field_name, author_login, abc_service_slug, abc_service_id))


def validate_unwhitelisted_modules_nonexistence(namespace_id, modules):
    """
    :type namespace_id: six.text_type
    :type modules: set[six.text_type]
    :raises: rpc.exceptions.ForbiddenError
    """
    if not modules:
        return

    from awacs.wrappers.base import Holder
    from awacs.wrappers.l7upstreammacro import L7UpstreamMacro
    from awacs.wrappers.l7macro import L7Macro

    _cache = cache.IAwacsCache.instance()
    for balancer_pb in _cache.list_all_balancers(namespace_id=namespace_id):
        h = Holder(balancer_pb.spec.yandex_balancer.config)
        for module in h.walk_chain(visit_branches=True):
            if module.MODULE_NAME in modules:
                raise rpc.exceptions.BadRequestError(
                    '"spec.modules_whitelist": "{}" can not be removed from whitelist, it is used in balancer {}:{}'
                        .format(module.MODULE_NAME, balancer_pb.meta.namespace_id, balancer_pb.meta.id)
                )
    for upstream_pb in _cache.list_all_upstreams(namespace_id=namespace_id):
        h = Holder(upstream_pb.spec.yandex_balancer.config)
        for module in h.walk_chain(visit_branches=True):
            if module.MODULE_NAME in modules:
                raise rpc.exceptions.BadRequestError(
                    '"spec.modules_whitelist": "{}" can not be removed from whitelist, it is used in upstream {}:{}'
                        .format(module.MODULE_NAME, upstream_pb.meta.namespace_id, upstream_pb.meta.id)
                )


def is_root(login):
    if not appconfig.get_value('run.auth', default=True):
        return True
    return login in appconfig.get_value('run.root_users', default=())


def restrict_to_admins(auth_subject):
    if not is_root(auth_subject.login):
        raise exceptions.ForbiddenError('This method is only available for awacs admins')
