# coding: utf-8
import re

from awacs.lib.rpc import exceptions
from awacs.wrappers.l7upstreammacro import L7UpstreamMacroDc
from infra.awacs.proto import api_pb2, model_pb2
from .util import WEIGHT_SECTION_ID_RE, NAMESPACE_ID_RE


def validate_spec(spec_pb, section_type, field_name='spec'):
    """
    :type spec_pb: model_pb2.WeightSectionSpec
    :type field_name: six.text_type
    :type section_type: model_pb2.WeightSectionMeta.Type
    :raises: spec.BadRequestError
    """
    if not spec_pb.locations:
        raise exceptions.BadRequestError('"{}.locations": must be set'.format(field_name))

    names = set()
    weights_sum = 0
    not_fallback_location_exist = False
    for location_pb in spec_pb.locations:
        if not location_pb.is_fallback:
            not_fallback_location_exist = True
        if location_pb.name in names:
            raise exceptions.BadRequestError('"{}.locations": duplicate name "{}"'.format(field_name, location_pb.name))
        names.add(location_pb.name)
        weights_sum += location_pb.default_weight
    if weights_sum != 100:
        raise exceptions.BadRequestError('"{}.locations": sum of weights must be equal to 100, got {}'
                                         .format(field_name, weights_sum))
    if not not_fallback_location_exist:
        raise exceptions.BadRequestError('"{}.locations": at least one location must be not fallback'.format(field_name))

    if section_type == model_pb2.WeightSectionMeta.ST_DC_WEIGHTS:
        for location_pb in spec_pb.locations:
            if not location_pb.name.isupper():
                raise exceptions.BadRequestError('"{}.locations": name "{}" is not valid, must be uppercase'.format(field_name, location_pb.name))
            if not location_pb.is_fallback and location_pb.name.lower() not in L7UpstreamMacroDc.ALLOWED_DC_NAMES:
                raise exceptions.BadRequestError('"{}.locations": name "{}" is not valid, expected one of ({})'
                        .format(field_name, location_pb.name,
                                ', '.join(sorted([dc.upper() for dc in L7UpstreamMacroDc.ALLOWED_DC_NAMES]))))
    elif section_type == model_pb2.WeightSectionMeta.ST_TRAFFIC_SPLIT:
        for location_pb in spec_pb.locations:
            if not location_pb.name.isupper():
                raise exceptions.BadRequestError('"{}.locations": name "{}" is not valid, must be uppercase'.format(field_name, location_pb.name))
    else:
        raise exceptions.BadRequestError('"{}.type": is not supported'.format(field_name))


def validate_create_or_update_weight_section_request(req_pb):
    """
    :type req_pb: api_pb2.CreateWeightSectionRequest | api_pb2.UpdateWeightSectionRequest
    :raises: exceptions.BadRequestError
    """
    if not req_pb.HasField('meta'):
        raise exceptions.BadRequestError('"meta" must be set')
    if not req_pb.meta.id:
        raise exceptions.BadRequestError('"meta.id" must be set')
    if not WEIGHT_SECTION_ID_RE.match(req_pb.meta.id):
        raise exceptions.BadRequestError('"meta.id" is not valid')
    if not req_pb.meta.namespace_id:
        raise exceptions.BadRequestError('"meta.namespace_id" must be set')
    if not NAMESPACE_ID_RE.match(req_pb.meta.namespace_id):
        raise exceptions.BadRequestError('"meta.namespace_id" is not valid')

    if not req_pb.HasField('spec'):
        raise exceptions.BadRequestError('"spec" must be set')
    validate_spec(req_pb.spec, req_pb.meta.type)


def validate_create_weight_section_request(req_pb):
    """
    :type req_pb: api_pb2.CreateWeightSectionRequest
    :raises: exceptions.BadRequestError
    """
    validate_create_or_update_weight_section_request(req_pb)
    if not req_pb.meta.type:
        raise exceptions.BadRequestError('"meta.type" must be set')


def validate_update_weight_section_request(req_pb):
    """
    :type req_pb: api_pb2.UpdateWeightSectionRequest
    :raises: exceptions.BadRequestError
    """
    validate_create_or_update_weight_section_request(req_pb)
    if not req_pb.meta.version:
        raise exceptions.BadRequestError('"meta.version" must be set')


def validate_list_weight_sections_request(req_pb):
    """
    :type req_pb: api_pb2.ListWeightSectionsRequest
    :raises: exceptions.BadRequestError
    """
    if not req_pb.namespace_id:
        raise exceptions.BadRequestError('No "namespace_id" specified.')
    if req_pb.HasField('query') and req_pb.query.id_regexp:
        try:
            re.compile(req_pb.query.id_regexp)
        except Exception as e:
            raise exceptions.BadRequestError('"query.id_regexp" contains invalid regular expression: {}'.format(e))
    if req_pb.HasField('field_mask'):
        if not req_pb.field_mask.IsValidForDescriptor(model_pb2.WeightSection.DESCRIPTOR):
            raise exceptions.BadRequestError('"field_mask" is not valid')
    

def validate_request(req_pb):
    """
    :raises: exceptions.BadRequestError
    """
    if isinstance(req_pb, api_pb2.ListWeightSectionsRequest):
        validate_list_weight_sections_request(req_pb)
    elif isinstance(req_pb, (api_pb2.GetWeightSectionRequest,
                             api_pb2.ListWeightSectionRevisionsRequest)):
        if not req_pb.id:
            raise exceptions.BadRequestError('No "id" specified')
        if not req_pb.namespace_id:
            raise exceptions.BadRequestError('No "namespace_id" specified')
    elif isinstance(req_pb, api_pb2.RemoveWeightSectionRequest):
        if not req_pb.id:
            raise exceptions.BadRequestError('No "id" specified')
        if not req_pb.namespace_id:
            raise exceptions.BadRequestError('No "namespace_id" specified')
        if not req_pb.version:
            raise exceptions.BadRequestError('No "version" specified')
    elif isinstance(req_pb, api_pb2.GetWeightSectionRevisionRequest):
        if not req_pb.id:
            raise exceptions.BadRequestError('No "id" specified')
    elif isinstance(req_pb, api_pb2.UpdateWeightSectionRequest):
        validate_update_weight_section_request(req_pb)
    elif isinstance(req_pb, api_pb2.CreateWeightSectionRequest):
        validate_create_weight_section_request(req_pb)
    else:
        raise RuntimeError('Incorrect `req_pb` type: {}'.format(req_pb.__class__.__name__))
