# coding: utf-8
import re

from awacs.lib import rpc
from awacs.model import apicache, objects
from awacs.model.cache import IAwacsCache
from awacs.model.util import clone_pb
from awacs.model.zk import IZkStorage
from infra.awacs.proto import api_pb2, model_pb2
from awacs.web.auth.core import authorize_update, get_acl, authorize_create, authorize_remove
from awacs.web.validation import weight_section
from .util import AwacsBlueprint, forbid_action_during_namespace_order

weight_section_service_bp = AwacsBlueprint('rpc_weight_section_service', __name__, '/api')


@weight_section_service_bp.method('CreateWeightSection',
                                   request_type=api_pb2.CreateWeightSectionRequest,
                                   response_type=api_pb2.CreateWeightSectionResponse)
def create_weight_section(req_pb, auth_subject):
    weight_section.validate_request(req_pb)
    ns_id = req_pb.meta.namespace_id
    ws_id = req_pb.meta.id

    zk = IZkStorage.instance()

    namespace_pb = zk.must_get_namespace(namespace_id=ns_id)
    authorize_create(namespace_pb, auth_subject)

    if objects.WeightSection.zk.get(ns_id, ws_id) is not None:
        raise rpc.exceptions.ConflictError(
            'WeightSection "{}" already exists in namespace "{}".'.format(ws_id, ns_id))

    weight_section_pb = objects.WeightSection.create(req_pb.meta, req_pb.spec, auth_subject.login)
    return api_pb2.CreateWeightSectionResponse(weight_section=weight_section_pb)


@weight_section_service_bp.method('UpdateWeightSection',
                                   request_type=api_pb2.UpdateWeightSectionRequest,
                                   response_type=api_pb2.UpdateWeightSectionResponse)
def update_weight_section(req_pb, auth_subject):
    weight_section.validate_request(req_pb)
    ns_id = req_pb.meta.namespace_id
    ws_id = req_pb.meta.id

    zk = IZkStorage.instance()

    ws_pb = objects.WeightSection.zk.must_get(ns_id, ws_id)
    if ws_pb.meta.type != req_pb.meta.type:
        raise rpc.exceptions.BadRequestError('"meta.type": can not be changed')
    namespace_pb = zk.must_get_namespace(ns_id)
    authorize_update(ws_pb, req_pb, auth_subject, acl=get_acl(namespace_pb))

    ws_pb = objects.WeightSection.update(ns_id, ws_id, req_pb.meta.version, req_pb.meta.comment, auth_subject.login,
                                          req_pb.spec)
    balancer_state_pbs = IAwacsCache.instance().list_all_balancer_states(namespace_id=ns_id)
    ws_pb = apicache.copy_per_balancer_statuses(entity_pb=ws_pb,
                                                balancer_state_pbs=balancer_state_pbs)
    return api_pb2.UpdateWeightSectionResponse(weight_section=ws_pb)


@weight_section_service_bp.method('RemoveWeightSection',
                                   request_type=api_pb2.RemoveWeightSectionRequest,
                                   response_type=api_pb2.RemoveWeightSectionResponse)
def remove_weight_section(req_pb, auth_subject):
    weight_section.validate_request(req_pb)
    ns_id = req_pb.namespace_id
    ws_id = req_pb.id

    zk = IZkStorage.instance()

    weight_section_pb = objects.WeightSection.zk.must_get(ns_id, ws_id, sync=True)
    namespace_pb = zk.must_get_namespace(ns_id)
    authorize_remove(weight_section_pb, auth_subject, acl=get_acl(namespace_pb))
    forbid_action_during_namespace_order(namespace_pb, auth_subject)

    updated_spec_pb = clone_pb(weight_section_pb.spec)
    updated_spec_pb.deleted = True

    objects.WeightSection.update(ns_id,
                                  ws_id,
                                  req_pb.version,
                                  'Marked as deleted by {}'.format(auth_subject.login),
                                  auth_subject.login,
                                  updated_spec_pb)

    return api_pb2.RemoveWeightSectionResponse()


@weight_section_service_bp.method('ListWeightSections',
                                   request_type=api_pb2.ListWeightSectionsRequest,
                                   response_type=api_pb2.ListWeightSectionsResponse,
                                   max_in_flight=5)
def list_weight_sections(req_pb, _):
    weight_section.validate_request(req_pb)
    ns_id = req_pb.namespace_id

    if not req_pb.HasField('field_mask'):
        req_pb.field_mask.AllFieldsFromDescriptor(model_pb2.WeightSection.DESCRIPTOR)
    c = apicache.IAwacsApiCache.instance()

    query = {}
    if req_pb.query.id_regexp:
        query[objects.WeightSection.cache.QueryTarget.ID_REGEXP] = re.compile(req_pb.query.id_regexp)
    if req_pb.query.validated_status_in:
        query[objects.WeightSection.cache.QueryTarget.VALIDATED_STATUS_IN] = req_pb.query.validated_status_in
    if req_pb.query.in_progress_status_in:
        query[objects.WeightSection.cache.QueryTarget.IN_PROGRESS_STATUS_IN] = req_pb.query.in_progress_status_in
    if req_pb.query.active_status_in:
        query[objects.WeightSection.cache.QueryTarget.ACTIVE_STATUS_IN] = req_pb.query.active_status_in

    sort = []
    if req_pb.sort_target == req_pb.ID:
        sort.append(objects.WeightSection.cache.SortTarget.ID)
    elif req_pb.sort_target == req_pb.MTIME:
        sort.append(objects.WeightSection.cache.SortTarget.MTIME)
    sort.append(-1 if req_pb.sort_order == api_pb2.DESCEND else 1)
    kwargs = dict(sort=sort,
                  query=query,
                  skip=req_pb.skip or None,
                  limit=req_pb.limit or None)

    weight_section_pbs, total = c.list_namespace_weight_sections(namespace_id=ns_id, **kwargs)
    return api_pb2.ListWeightSectionsResponse(weight_sections=weight_section_pbs, total=total)


@weight_section_service_bp.method('GetWeightSection',
                                   request_type=api_pb2.GetWeightSectionRequest,
                                   response_type=api_pb2.GetWeightSectionResponse)
def get_weight_section(req_pb, _):
    weight_section.validate_request(req_pb)
    ns_id = req_pb.namespace_id
    ws_id = req_pb.id

    if req_pb.consistency == api_pb2.STRONG:
        weight_section_pb = objects.WeightSection.zk.must_get(ns_id, ws_id, sync=True)
        zk = IZkStorage.instance()
        zk.sync_balancer_states(ns_id)
        balancer_state_pbs = IAwacsCache.instance().list_all_balancer_states(namespace_id=ns_id)
    else:
        weight_section_pb = objects.WeightSection.cache.must_get(ns_id, ws_id)
        balancer_state_pbs = IAwacsCache.instance().list_all_balancer_states(namespace_id=ns_id)
    weight_section_pb = apicache.copy_per_balancer_statuses(entity_pb=weight_section_pb,
                                                            balancer_state_pbs=balancer_state_pbs)
    weight_section_pb = apicache.copy_per_l7heavy_config_statuses(entity_pb=weight_section_pb)
    return api_pb2.GetWeightSectionResponse(weight_section=weight_section_pb)


@weight_section_service_bp.method('GetWeightSectionRevision',
                                   request_type=api_pb2.GetWeightSectionRevisionRequest,
                                   response_type=api_pb2.GetWeightSectionRevisionResponse)
def get_weight_section_revision(req_pb, _):
    weight_section.validate_request(req_pb)
    rev_pb = objects.WeightSection.mongo.must_get_rev(req_pb.id)
    return api_pb2.GetWeightSectionRevisionResponse(revision=rev_pb)


@weight_section_service_bp.method('ListWeightSectionRevisions',
                                   request_type=api_pb2.ListWeightSectionRevisionsRequest,
                                   response_type=api_pb2.ListWeightSectionRevisionsResponse,
                                   max_in_flight=5)
def list_weight_section_revisions(req_pb, _):
    weight_section.validate_request(req_pb)
    ws_id = req_pb.id
    ns_id = req_pb.namespace_id

    objects.WeightSection.zk.must_get(ns_id, ws_id)

    skip = req_pb.skip or None
    limit = req_pb.limit or None
    revs = objects.WeightSection.mongo.list_revs(ns_id, ws_id, skip=skip, limit=limit)
    return api_pb2.ListWeightSectionRevisionsResponse(revisions=revs.items, total=revs.total)
