# coding: utf-8
import collections
import itertools
import re

import enum
import semantic_version
import six
from awacs.wrappers import min_component_versions
from awacs.lib import OrderedDict
from awacs.lib.strutils import quote_join_sorted
from awacs.model.util import is_upstream_internal
from sepelib.core import config as appconfig
from infra.awacs.proto import modules_pb2, model_pb2
from . import tls_settings
from . import rps_limiter_settings
from .base import ConfigWrapperBase, ModuleWrapperBase, WrapperBase, MacroBase, DEFAULT_CTX, DomainsSet
from .errors import ValidationError
from .luautil import dump_string
from .util import (validate, validate_pire_regexp, validate_range, validate_header_name, hosts_to_regexp,
                   ports_to_regexp, validate_re2_regexp, validate_version, validate_header_func,
                   validate_timedelta_range, MEGABYTE, KILOBYTE, META_MODULE_ID)


VERSION_0_0_1 = semantic_version.Version(u'0.0.1')
VERSION_0_0_2 = semantic_version.Version(u'0.0.2')
VERSION_0_0_3 = semantic_version.Version(u'0.0.3')
VERSION_0_0_4 = semantic_version.Version(u'0.0.4')
VERSION_0_0_5 = semantic_version.Version(u'0.0.5')
VERSION_0_1_0 = semantic_version.Version(u'0.1.0')
VERSION_0_1_1 = semantic_version.Version(u'0.1.1')
# romanovich@:
# 0.1.2 was introduced for SWAT-5801 and "removed" shortly thereafter due to
# https://st.yandex-team.ru/SWAT-5801#5f85d53c2a032c58597f2116.
# Now 0.1.2 is synonym for 0.1.1.
# Let's keep 0.1.2 in VALID_VERSIONS for a while, just to let users roll back to previous configs.
VERSION_0_1_2 = semantic_version.Version(u'0.1.2')
VERSION_0_2_0 = semantic_version.Version(u'0.2.0')
VERSION_0_2_1 = semantic_version.Version(u'0.2.1')
VERSION_0_2_2 = semantic_version.Version(u'0.2.2')
VERSION_0_2_3 = semantic_version.Version(u'0.2.3')
VERSION_0_2_4 = semantic_version.Version(u'0.2.4')
VERSION_0_2_5 = semantic_version.Version(u'0.2.5')
VERSION_0_2_6 = semantic_version.Version(u'0.2.6')
VERSION_0_2_7 = semantic_version.Version(u'0.2.7')
VERSION_0_2_8 = semantic_version.Version(u'0.2.8')
VERSION_0_2_9 = semantic_version.Version(u'0.2.9')
VERSION_0_2_10 = semantic_version.Version(u'0.2.10')
VERSION_0_2_11 = semantic_version.Version(u'0.2.11')
VERSION_0_2_12 = semantic_version.Version(u'0.2.12')
VERSION_0_3_0 = semantic_version.Version(u'0.3.0')
VERSION_0_3_1 = semantic_version.Version(u'0.3.1')
VERSION_0_3_2 = semantic_version.Version(u'0.3.2')
VERSION_0_3_3 = semantic_version.Version(u'0.3.3')
VERSION_0_3_4 = semantic_version.Version(u'0.3.4')
VERSION_0_3_5 = semantic_version.Version(u'0.3.5')
VERSION_0_3_6 = semantic_version.Version(u'0.3.6')
VERSION_0_3_7 = semantic_version.Version(u'0.3.7')
VERSION_0_3_8 = semantic_version.Version(u'0.3.8')
VERSION_0_3_9 = semantic_version.Version(u'0.3.9')
VERSION_0_3_10 = semantic_version.Version(u'0.3.10')
VERSION_0_3_11 = semantic_version.Version(u'0.3.11')
VERSION_0_3_12 = semantic_version.Version(u'0.3.12')
VERSION_0_3_13 = semantic_version.Version(u'0.3.13')
VERSION_0_3_14 = semantic_version.Version(u'0.3.14')
VERSION_0_3_15 = semantic_version.Version(u'0.3.15')
VERSION_0_3_16 = semantic_version.Version(u'0.3.16')
VERSION_0_3_17 = semantic_version.Version(u'0.3.17')
VERSION_0_3_18 = semantic_version.Version(u'0.3.18')
VERSION_0_4_0 = semantic_version.Version(u'0.4.0')
VERSION_0_4_1 = semantic_version.Version(u'0.4.1')
VERSION_0_4_2 = semantic_version.Version(u'0.4.2')
VERSION_0_4_3 = semantic_version.Version(u'0.4.3')
VERSION_0_4_4 = semantic_version.Version(u'0.4.4')
VERSION_0_4_5 = semantic_version.Version(u'0.4.5')

VALID_VERSIONS = frozenset((VERSION_0_0_1, VERSION_0_0_2, VERSION_0_0_3, VERSION_0_0_4, VERSION_0_0_5,
                            VERSION_0_1_0, VERSION_0_1_1, VERSION_0_1_2,
                            VERSION_0_2_0, VERSION_0_2_1, VERSION_0_2_2, VERSION_0_2_3, VERSION_0_2_4, VERSION_0_2_5,
                            VERSION_0_2_6, VERSION_0_2_7, VERSION_0_2_8, VERSION_0_2_9, VERSION_0_2_10, VERSION_0_2_11,
                            VERSION_0_2_12,
                            VERSION_0_3_0, VERSION_0_3_1, VERSION_0_3_2, VERSION_0_3_3, VERSION_0_3_4, VERSION_0_3_5,
                            VERSION_0_3_6, VERSION_0_3_7, VERSION_0_3_8, VERSION_0_3_9, VERSION_0_3_10, VERSION_0_3_11,
                            VERSION_0_3_12, VERSION_0_3_13, VERSION_0_3_14, VERSION_0_3_15, VERSION_0_3_16,
                            VERSION_0_3_17, VERSION_0_3_18,
                            VERSION_0_4_0, VERSION_0_4_1, VERSION_0_4_2, VERSION_0_4_3, VERSION_0_4_4, VERSION_0_4_5,
                            ))
LATEST_VERSION = VERSION_0_2_12

UUID_RE = '^[a-z0-9-_]+$'
DOMAIN_NOT_FOUND_SECTION_NAME = 'domain_not_found'

DEFAULT_AWACS_INSTALLATION = 'awacs'


def fill_get_port_var_call(call_pb):
    """
    :type call_pb: modules_pb2.Call
    """
    call_pb.type = modules_pb2.Call.GET_PORT_VAR
    call_pb.get_port_var_params.var = 'port'


class L7MacroCompat(WrapperBase):
    __protobuf__ = modules_pb2.L7Macro.Compat

    MAXCONN_MIN = 1000
    MAXCONN_MAX = 10 ** 6
    MAXLEN_MIN = 64 * 1024
    MAXLEN_MAX = 512 * 1024
    MAXREQ_MIN = 64 * 1024
    MAXREQ_MAX = 512 * 1024

    ALLOWED_TCP_CONGESTION_CONTROL = ('bbr',)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        if self.pb.tcp_congestion_control and self.pb.tcp_congestion_control not in self.ALLOWED_TCP_CONGESTION_CONTROL:
            raise ValidationError('must be one of the "{}"'.format('", "'.join(self.ALLOWED_TCP_CONGESTION_CONTROL)),
                                  'tcp_congestion_control')
        with validate('enable_meta'):
            if self.pb.enable_meta:
                ctx.ensure_component_version(
                    model_pb2.ComponentMeta.PGINX_BINARY,
                    min=min_component_versions.META)
        if self.pb.disable_unistat and not self.pb.disable_sd:
            raise ValidationError(u'is not allowed without "disable_sd"', u'disable_unistat')
        if self.pb.HasField('maxconn'):
            with validate(u'maxconn'):
                validate_range(self.pb.maxconn.value, self.MAXCONN_MIN, self.MAXCONN_MAX)
        if self.pb.HasField('maxlen'):
            with validate(u'maxlen'):
                validate_range(self.pb.maxlen.value, self.MAXLEN_MIN, self.MAXLEN_MAX)
        if self.pb.HasField('maxreq'):
            with validate(u'maxreq'):
                validate_range(self.pb.maxreq.value, self.MAXREQ_MIN, self.MAXREQ_MAX)
        if self.pb.awacs_installation:
            with validate(u'awacs_installation'):
                if not appconfig.get_value('run.debug', False):
                    raise ValidationError('must not be set')
                if self.pb.awacs_installation == DEFAULT_AWACS_INSTALLATION:
                    raise ValidationError('must not be "{}"'.format(DEFAULT_AWACS_INSTALLATION))


class L7MacroAnnounceCheckReplySettingsCompat(WrapperBase):
    __protobuf__ = modules_pb2.L7Macro.AnnounceCheckReplySettings.Compat

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()


class L7MacroAnnounceCheckReplySettings(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7Macro.AnnounceCheckReplySettings

    compat = None  # type: L7MacroAnnounceCheckReplySettingsCompat | None

    REQUIRED = ['url_re']

    DEFAULT_URL_RE = u'/ping'
    REPORT_UUID = u'announce_check'

    def validate(self, version, ctx=DEFAULT_CTX, preceding_modules=()):
        """
        :type version: semantic_version.Version
        """
        self.auto_validate_required()

        if self.compat:
            with validate(u'compat'):
                self.compat.validate(ctx=ctx, preceding_modules=preceding_modules)

        with validate(u'url_re'):
            validate_pire_regexp(self.pb.url_re, lua_unescape_first=False)

        if self.pb.use_upstream_handler and self.is_graceful_shutdown_enabled(version):
            raise ValidationError(u'can not be set without "compat.disable_graceful_shutdown"',
                                  u'use_upstream_handler')

    def is_graceful_shutdown_enabled(self, version):
        """
        :type version: semantic_version.Version
        """
        return (version >= VERSION_0_2_9 and
                not self.pb.compat.disable_graceful_shutdown)

    def fill_regexp_section(self, regexp_section_pb, version, enable_monitoring=False):
        """
        :type regexp_section_pb: modules_pb2.RegexpSection
        :type version: semantic_version.Version
        :type enable_monitoring: bool
        """
        regexp_section_pb.matcher.match_fsm.url = dump_string(
            self.pb.url_re if self.pb.url_re else self.DEFAULT_URL_RE)

        holder_pb = regexp_section_pb.nested
        if enable_monitoring:
            report_pb = holder_pb.report
            report_pb.uuid = dump_string(self.REPORT_UUID)
            report_pb.ranges = dump_string(u'default')
            holder_pb = report_pb.nested

        if self.pb.use_upstream_handler:
            holder_pb.slb_ping_macro.use_shared_backends = True
        else:
            slb_ping_macro_pb = holder_pb.slb_ping_macro
            if self.is_graceful_shutdown_enabled(version):
                slb_ping_macro_pb.active_check_reply.zero_weight_at_shutdown.value = True
            else:
                slb_ping_macro_pb.errordoc = True


class L7MacroHealthCheckReplySettingsCompat(WrapperBase):
    __protobuf__ = modules_pb2.L7Macro.HealthCheckReplySettings.Compat

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()


class L7MacroHealthCheckReplySettings(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7Macro.HealthCheckReplySettings

    compat = None  # type: L7MacroHealthCheckReplySettingsCompat | None

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()

        if self.compat:
            with validate(u'compat'):
                self.compat.validate(ctx=ctx, preceding_modules=preceding_modules)


class L7MacroAntirobotSettingsCaptchaReplySettingsCompat(WrapperBase):
    __protobuf__ = modules_pb2.L7Macro.AntirobotSettings.CaptchaReplySettings.Compat

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()


class L7MacroAntirobotSettingsCaptchaReplySettings(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7Macro.AntirobotSettings.CaptchaReplySettings

    compat = None  # type: modules_pb2.L7Macro.AntirobotSettings.CaptchaReplySettings.Compat | None

    URI_RE = u'/x?(show|check)?captcha.*'
    REPORT_UUID = u'antirobot_captcha'

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()

    def fill_regexp_section(self, regexp_section_pb, version, trust_xffy, trust_icookie, trust_ja_x, service, req_group):
        """
        :type regexp_section_pb: modules_pb2.RegexpSection
        :type version: semantic_version.Version
        :type trust_xffy: bool
        :type trust_icookie: bool
        :type trust_ja_x: bool
        :type service: Optional[six.text_type]
        :type req_group: Optional[six.text_type]
        """
        regexp_section_pb.matcher.match_fsm.uri = dump_string(self.URI_RE)
        holder_pb = regexp_section_pb.nested
        captcha_macro_pb = holder_pb.captcha_macro
        if version >= VERSION_0_3_8:
            captcha_macro_pb.trust_x_yandex_ja_x = trust_ja_x
            captcha_macro_pb.trust_icookie = trust_icookie
            captcha_macro_pb.trust_x_forwarded_for_y = trust_xffy
            if service:
                captcha_macro_pb.service = service
        if req_group:
            captcha_macro_pb.req_group = req_group

        if version >= VERSION_0_4_5:
            captcha_macro_pb.version = u'0.0.5'
        elif version >= VERSION_0_4_1:
            captcha_macro_pb.version = u'0.0.4'
        elif version >= VERSION_0_3_18:
            captcha_macro_pb.version = u'0.0.5'
        elif version >= VERSION_0_3_8:
            if self.pb.compat and self.pb.compat.increased_cut_request_bytes:
                captcha_macro_pb.version = u'0.0.4'
            else:
                captcha_macro_pb.version = u'0.0.3'
        else:
            captcha_macro_pb.version = u'0.0.2'


class L7MacroAntirobotSettings(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7Macro.AntirobotSettings

    captcha_reply = None  # type: L7MacroAntirobotSettingsCaptchaReplySettings | None

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()

    def fill_holder_pb(self, holder_pb, version, trust_xffy, trust_icookie, trust_ja_x):
        antirobot_macro_pb = holder_pb.antirobot_macro
        antirobot_macro_version = u'0.0.2'  # use SD backends
        if version >= VERSION_0_2_4:
            antirobot_macro_version = u'0.0.3'  # use SD backends, add X-Forwarded-For-Y
        if version >= VERSION_0_3_0:
            antirobot_macro_version = u'0.0.4'  # allows disabling autosetting XFFY
            antirobot_macro_pb.trust_x_forwarded_for_y = trust_xffy
        if version >= VERSION_0_3_2:
            antirobot_macro_version = u'0.0.5'  # add icookie, allow trusting existing icookie
            antirobot_macro_pb.trust_icookie = trust_icookie
        if version >= VERSION_0_3_3:
            antirobot_macro_version = u'0.0.6'  # add ja 3/4
            antirobot_macro_pb.trust_x_yandex_ja_x = trust_ja_x
        if version >= VERSION_0_3_14:
            antirobot_macro_version = u'0.0.7'  # subnet_v4_mask=32, subnet_v6_mask=64 for hasher
        if version >= VERSION_0_3_15:
            antirobot_macro_version = u'0.0.8'  # take_ip_from='X-Forwarded-For-Y' for hasher, add X-Forwarded-For-Y before hasher module
        if version >= VERSION_0_3_17:
            antirobot_macro_version = u'0.0.9'  # select a random location for each attempt in myt, iva
        antirobot_macro_pb.version = antirobot_macro_version
        if self.pb.service:
            antirobot_macro_pb.service = self.pb.service
        if self.pb.req_group:
            antirobot_macro_pb.req_group = self.pb.req_group


class L7MacroWebauthSettings(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7Macro.WebauthSettings

    REQUIRED = ['mode', 'action']

    SIMPLE_AUTH_PATH = '/auth_request'
    EXTERNAL_AUTH_PATH = '/check_oauth_token'

    UNAUTHORIZED_SET_COOKIE = 'webauth_csrf_token={csrf_token}; Path=/'
    SIMPLE_UNAUTHORIZED_REDIRECT_OLD = 'https://passport.{yandex_domain}/auth?retpath={retpath}'
    SIMPLE_UNAUTHORIZED_REDIRECT_NEW = 'https://passport.yandex-team.ru/auth?retpath={retpath}'
    EXTERNAL_UNAUTHORIZED_REDIRECT = 'https://oauth.yandex-team.ru/authorize?response_type=code&client_id={app_id}&state={csrf_state}'

    SAVE_OAUTH_TOKEN_PATH = '/__webauth_save_oauth_token'
    SAVE_OAUTH_TOKEN_REWRITE_REGEXP = '/__webauth_save_oauth_token(.*)'
    SAVE_OAUTH_TOKEN_REWRITE_REWRITE = '/save_oauth_token%1'
    DENIED_RESPONSE_HEADER_NAME = 'Webauth-Denial-Reasons'

    WEBAUTH_URL = 'webauth.yandex-team.ru'
    CA_FILE = 'allCAs.pem'

    MODULE_NAME = 'webauth'

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_raise_if_blacklisted(ctx)
        self.auto_validate_required()
        if self.pb.pass_options_requests_through:
            with validate(u'pass_options_requests_through'):
                ctx.ensure_component_version(
                    model_pb2.ComponentMeta.PGINX_BINARY,
                    min=min_component_versions.WEBAUTH_PASS_OPTIONS_REQUESTS_THROUGH)
        if self.pb.action == self.pb.SAVE_OAUTH_TOKEN and self.pb.mode != self.pb.EXTERNAL:
            raise ValidationError(u'must not be "SAVE_OAUTH_TOKEN" if "mode" is "SIMPLE"', 'action')

    @staticmethod
    def get_namespace_webauth_idm_role(installation, namespace_id):
        return '/webauth-awacs/{}/{}/user/'.format(installation, namespace_id)

    def get_auth_path(self, installation, namespace_id):
        if self.pb.mode == self.pb.EXTERNAL:
            return self.EXTERNAL_AUTH_PATH
        if self.pb.action == self.pb.AUTHENTICATE_USING_IDM:
            return self.SIMPLE_AUTH_PATH + '?idm_role={}'.format(self.get_namespace_webauth_idm_role(installation,
                                                                                                     namespace_id))
        return self.SIMPLE_AUTH_PATH

    @classmethod
    def _fill_proxy_https_settings(cls, https_settings_pb):
        https_settings_pb.sni_host = cls.WEBAUTH_URL
        https_settings_pb.sni_on = True
        https_settings_pb.verify_depth = 3
        https_settings_pb.f_ca_file.type = modules_pb2.Call.GET_CA_CERT_PATH
        https_settings_pb.f_ca_file.get_ca_cert_path_params.name = cls.CA_FILE
        https_settings_pb.f_ca_file.get_ca_cert_path_params.default_ca_cert_dir = './'

    @classmethod
    def _fill_proxy(cls, proxy_pb):
        """
        :type proxy_pb: modules_pb2.ProxyModule
        """
        proxy_pb.host = cls.WEBAUTH_URL
        proxy_pb.port = 443
        proxy_pb.backend_timeout = '5s'
        cls._fill_proxy_https_settings(proxy_pb.https_settings)

    @classmethod
    def _fill_balancer2(cls, balancer2_pb, l7_macro_version):
        """
        :type proxy_pb: modules_pb2.Balancer2Module
        """
        balancer2_pb.attempts = 2
        balancer2_pb.attempts_rate_limiter.limit = 1.2
        balancer2_pb.connection_attempts = 2
        balancer2_pb.rr.SetInParent()
        if l7_macro_version >= VERSION_0_3_13:
            balancer2_pb.balancing_policy.retry_policy.balancing_policy.unique_policy.SetInParent()
        generated_proxy_backends_pb = balancer2_pb.generated_proxy_backends
        generated_proxy_backends_pb.instances.add(host=cls.WEBAUTH_URL, port=443, weight=1)
        generated_proxy_backends_pb.proxy_options.backend_timeout = '5s'
        cls._fill_proxy_https_settings(generated_proxy_backends_pb.proxy_options.https_settings)

    @classmethod
    def fill_save_oauth_token_section(cls, regexp_section_pb, l7_macro_version):
        regexp_section_pb.matcher.match_fsm.path = dump_string(cls.SAVE_OAUTH_TOKEN_PATH)
        regexp_section_pb.nested.headers.create.add(key='Host', value=dump_string(cls.WEBAUTH_URL))

        rewrite_pb = regexp_section_pb.nested.headers.nested.rewrite
        action_pb = rewrite_pb.actions.add()
        action_pb.regexp = dump_string(cls.SAVE_OAUTH_TOKEN_REWRITE_REGEXP)
        action_pb.split = 'path'
        action_pb.rewrite = dump_string(cls.SAVE_OAUTH_TOKEN_REWRITE_REWRITE)
        if l7_macro_version >= VERSION_0_3_9:
            cls._fill_balancer2(rewrite_pb.nested.balancer2, l7_macro_version)
        else:
            cls._fill_proxy(rewrite_pb.nested.proxy)

    def fill_holder_pb(self, holder_pb, namespace_id, awacs_installation, l7_macro_version):
        webauth_pb = holder_pb.webauth

        webauth_pb.allow_options_passthrough = self.pb.pass_options_requests_through

        webauth_pb.auth_path = self.get_auth_path(awacs_installation, namespace_id)
        if self.pb.action == self.pb.AUTHENTICATE_USING_IDM and self.pb.mode == self.pb.EXTERNAL:
            webauth_pb.role = self.get_namespace_webauth_idm_role(awacs_installation, namespace_id)

        webauth_pb.on_forbidden.errordocument.status = 403
        if l7_macro_version >= VERSION_0_2_8 or self.pb.mode == self.pb.EXTERNAL:
            webauth_pb.unauthorized_set_cookie = self.get_unauthorized_set_cookie()
            webauth_pb.unauthorized_redirect = self.get_unauthorized_redirect(self.pb.mode, l7_macro_version)
            webauth_pb.on_forbidden.errordocument.content = 'You are not authorized to access this resource.'
        else:
            webauth_pb.on_forbidden.errordocument.content = 'You are not authenticated or not authorized to access this resource.'

        headers_pb = webauth_pb.checker.headers
        headers_pb.create.add(key='Host', value=dump_string(self.WEBAUTH_URL))
        if l7_macro_version >= VERSION_0_3_5:
            headers_pb.append_func['X-Forwarded-For'] = 'realip'
        elif l7_macro_version >= VERSION_0_2_7:
            headers_pb.create_func['X-Forwarded-For'] = 'realip'
        if l7_macro_version >= VERSION_0_3_7:
            headers_pb.append_func['X-Forwarded-For-Y'] = 'realip'
        response_headers_pb = headers_pb.nested.log_headers
        response_headers_pb.response_name_re = self.DENIED_RESPONSE_HEADER_NAME

        if l7_macro_version >= VERSION_0_3_9:
            self._fill_balancer2(response_headers_pb.nested.balancer2, l7_macro_version)
            webauth_pb.on_error.errordocument.status = 504
            webauth_pb.on_error.errordocument.content = 'Failed to authenticate user.'
        else:
            self._fill_proxy(response_headers_pb.nested.proxy)

    @classmethod
    def get_unauthorized_redirect(cls, mode, l7_macro_version):
        if mode == modules_pb2.L7Macro.WebauthSettings.EXTERNAL:
            return dump_string(cls.EXTERNAL_UNAUTHORIZED_REDIRECT)
        else:
            if l7_macro_version >= VERSION_0_2_10:
                return dump_string(cls.SIMPLE_UNAUTHORIZED_REDIRECT_NEW)
            else:
                return dump_string(cls.SIMPLE_UNAUTHORIZED_REDIRECT_OLD)

    def get_unauthorized_set_cookie(self):
        return dump_string(self.UNAUTHORIZED_SET_COOKIE)


class L7MacroHttpSettingsRedirectToHttpsCompat(WrapperBase):
    __protobuf__ = modules_pb2.L7Macro.HttpSettings.RedirectToHttps.Compat

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()


class L7MacroHttpSettingsRedirectToHttps(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7Macro.HttpSettings.RedirectToHttps

    compat = None  # type: L7MacroHttpSettingsRedirectToHttpsCompat | None

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()

        if self.compat:
            with validate(u'compat'):
                self.compat.validate(ctx=ctx, preceding_modules=preceding_modules)


class L7MacroHttpSettingsCompat(WrapperBase):
    __protobuf__ = modules_pb2.L7Macro.HttpSettings.Compat

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        if self.pb.assign_shared_uuid and not re.match(UUID_RE, self.pb.assign_shared_uuid):
            raise ValidationError(u'must match {}'.format(UUID_RE), u'assign_shared_uuid')


class L7MacroHttpsSettingsCompat(WrapperBase):
    __protobuf__ = modules_pb2.L7Macro.HttpsSettings.Compat

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        if self.pb.assign_shared_uuid and not re.match(UUID_RE, self.pb.assign_shared_uuid):
            raise ValidationError(u'must match {}'.format(UUID_RE), u'assign_shared_uuid')


class L7MacroHttpsSettingsVerifyClientCert(WrapperBase):
    __protobuf__ = modules_pb2.L7Macro.HttpsSettings.VerifyClientCert

    CA_FILE = u'clientCAs.pem'

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()

    @classmethod
    def fill_ssl_sni_context_pb(cls, ssl_sni_context_pb):
        """
        :type ssl_sni_context_pb: modules_pb2.ExtendedHttpMacroSslSniContext
        """
        ssl_sni_context_pb.ca = cls.CA_FILE

        client_pb = ssl_sni_context_pb.client
        client_pb.verify_depth = 3
        client_pb.verify_peer = True
        client_pb.verify_once = True
        client_pb.fail_if_no_peer_cert = False


class L7MacroHttpsSettingsTls(WrapperBase):
    __protobuf__ = modules_pb2.L7Macro.HttpsSettings.TlsSettings

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()

    def has_non_default_preset(self):
        return self.pb.preset != modules_pb2.L7Macro.HttpsSettings.TlsSettings.DEFAULT

    def fill_ssl_sni_context_pb(self, ssl_sni_context_pb, enable_ecdsa_cipher):
        """
        :type ssl_sni_context_pb: modules_pb2.ExtendedHttpMacroSslSniContext
        :type enable_ecdsa_cipher: bool
        """
        ssl_sni_context_pb.ciphers = tls_settings.get_cipher_suite_for_preset(self.pb.preset, enable_ecdsa_cipher)
        ssl_protocols = tls_settings.get_ssl_protocols_for_preset(self.pb.preset)
        if ssl_protocols is not None:
            ssl_sni_context_pb.ssl_protocols.extend(ssl_protocols)


class L7MacroCoreSettings(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7Macro.CoreSettings

    compat = None  # type: L7MacroCoreSettingsCompat or None
    limits = None  # type: L7MacroCoreSettingsLimits or None

    REQUIRED = []

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()

        if self.compat:
            with validate(u'compat'):
                self.compat.validate(ctx=ctx, preceding_modules=preceding_modules)
        if self.limits:
            with validate(u'limits'):
                self.limits.validate(ctx=ctx, preceding_modules=preceding_modules)


class L7MacroCoreSettingsLimits(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7Macro.CoreSettings.Limits

    REQUIRED = []
    REQUIRED_ALLOFS = [
        ('req_line_max_len', 'req_line_plus_headers_max_len'),
    ]

    REQ_LINE_MAX_LEN_MIN = 16 * KILOBYTE
    REQ_LINE_MAX_LEN_MAX = 8 * MEGABYTE
    REQ_LINE_PLUS_HEADERS_MAX_LEN_MIN = 16 * KILOBYTE
    REQ_LINE_PLUS_HEADERS_MAX_LEN_MAX = 8 * MEGABYTE

    HEADERS_SIZE_MIN = 512

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()

        if self.pb.HasField('req_line_max_len'):
            with validate(u'req_line_max_len'):
                validate_range(self.pb.req_line_max_len.value,
                               self.REQ_LINE_MAX_LEN_MIN,
                               self.REQ_LINE_MAX_LEN_MAX)

            with validate(u'req_line_plus_headers_max_len'):
                validate_range(self.pb.req_line_plus_headers_max_len.value,
                               self.REQ_LINE_PLUS_HEADERS_MAX_LEN_MIN,
                               self.REQ_LINE_PLUS_HEADERS_MAX_LEN_MAX)

                if self.pb.req_line_plus_headers_max_len.value - self.pb.req_line_max_len.value < self.HEADERS_SIZE_MIN:
                    raise ValidationError(u'must be greater than req_line_max_len '
                                          u'by at least {} bytes'.format(self.HEADERS_SIZE_MIN))


class L7MacroCoreSettingsCompat(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7Macro.CoreSettings.Compat

    REQUIRED = []

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()


class L7MacroRpsLimiterLocal(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7Macro.RpsLimiterSettings.Local

    REQUIRED = ['max_requests', 'interval']

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        with validate(u'max_requests'):
            if self.pb.max_requests <= 0:
                raise ValidationError('must be greater than 0')
        if self.pb.HasField('max_requests_in_queue'):
            with validate(u'max_requests_in_queue'):
                validate_range(self.pb.max_requests_in_queue.value, 0, 1000)
        with validate(u'interval'):
            validate_timedelta_range(self.pb.interval, '1s', '60m')

    def fill_holder_pb(self, holder_pb):
        rate_limiter_pb = holder_pb.rate_limiter
        rate_limiter_pb.max_requests = self.pb.max_requests
        rate_limiter_pb.interval = self.pb.interval
        if self.pb.HasField('max_requests_in_queue'):
            rate_limiter_pb.max_requests_in_queue.value = self.pb.max_requests_in_queue.value


class L7MacroRpsLimiterExternal(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7Macro.RpsLimiterSettings.External

    REQUIRED = ['record_name']

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        rps_limiter_settings.validate_installation(self.pb.installation, ctx.rps_limiter_allowed_installations)

    def fill_holder_pb(self, holder_pb, rps_limiter_macro_version):
        rps_limiter_macro_pb = holder_pb.rps_limiter_macro

        if rps_limiter_macro_version:
            rps_limiter_macro_pb.version = rps_limiter_macro_version

        rps_limiter_macro_pb.record_name = self.pb.record_name
        rps_limiter_macro_pb.installation = self.pb.installation

    def get_full_backend_ids(self):
        inst = rps_limiter_settings.get_installation(self.pb.installation)
        return inst.get_full_backend_ids()


class L7MacroRpsLimiterSettings(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7Macro.RpsLimiterSettings

    REQUIRED_ONEOFS = [('local', 'external')]

    local = None  # type: L7MacroRpsLimiterLocal or None
    external = None  # type: L7MacroRpsLimiterExternal or None

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        if self.external:
            with validate('external'):
                self.external.validate(ctx=ctx, preceding_modules=preceding_modules)
        elif self.local:
            with validate('local'):
                self.local.validate(ctx=ctx, preceding_modules=preceding_modules)

    def fill_holder_pb(self, holder_pb, rps_limiter_macro_version):
        if self.external:
            self.external.fill_holder_pb(holder_pb, rps_limiter_macro_version)
        elif self.local:
            self.local.fill_holder_pb(holder_pb)


class L7MacroHttpSettings(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7Macro.HttpSettings

    compat = None  # type: L7MacroHttpSettingsCompat | None
    redirect_to_https = None  # type: L7MacroHttpSettingsRedirectToHttps | None

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()

        if self.compat:
            with validate(u'compat'):
                self.compat.validate(ctx=ctx, preceding_modules=preceding_modules)

        if self.pb.compat.use_instance_port_in_section_log_name and not self.pb.compat.bind_on_instance_port:
            raise ValidationError(u'use_instance_port_in_section_log_name can not be set without bind_on_instance_port',
                                  u'compat')

        if self.redirect_to_https:
            with validate(u'redirect_to_https'):
                if self.compat and (self.compat.pb.assign_shared_uuid or self.compat.pb.refer_shared_uuid):
                    raise ValidationError(u'can not be used together with '
                                          u'compat.assign_shared_uuid or compat.refer_shared_uuid')
                self.redirect_to_https.validate(ctx=ctx, preceding_modules=preceding_modules)


class L7MacroHttpsSettingsCert(WrapperBase):
    __protobuf__ = modules_pb2.L7Macro.HttpsSettings.Cert

    REQUIRED = ['id']

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()


class L7MacroHttpsSettings(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7Macro.HttpsSettings

    compat = None  # type: L7MacroHttpsSettingsCompat | None
    certs = ()  # type: list[L7MacroHttpsSettingsCert]
    verify_client_cert = None  # type: L7MacroHttpsSettingsVerifyClientCert | None
    tls_settings = None  # type: L7MacroHttpsSettingsTls | None

    REQUIRED = []

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()

        if self.compat:
            with validate(u'compat'):
                self.compat.validate(ctx=ctx, preceding_modules=preceding_modules)

        if self.verify_client_cert:
            with validate(u'verify_client_cert'):
                self.verify_client_cert.validate(ctx=ctx, preceding_modules=preceding_modules)

        if len(self.certs) > 1:
            raise ValidationError(u'must not contain more than one cert', u'certs')

        for i, cert in enumerate(self.certs):
            with validate(u'certs[{}]'.format(i)):
                cert.validate(ctx=ctx, preceding_modules=preceding_modules)

        if self.pb.compat.use_instance_port_in_section_log_name and not self.pb.compat.bind_on_instance_port:
            raise ValidationError(u'use_instance_port_in_section_log_name can not be set without bind_on_instance_port',
                                  u'compat')

        if self.tls_settings:
            with validate(u'tls_settings'):
                self.tls_settings.validate(ctx=ctx, preceding_modules=preceding_modules)
            if self.tls_settings.has_non_default_preset():
                if self.pb.compat.HasField('disable_rc4_sha_cipher'):
                    raise ValidationError(
                        u'"disable_rc4_sha_cipher" is deprecated when strong TLS settings preset is enabled', u'compat')
                if self.pb.compat.enable_sslv3:
                    raise ValidationError(
                        u'"enable_sslv3" cannot be used when strong TLS settings preset is enabled', u'compat')
                if self.pb.enable_tlsv1_3:
                    raise ValidationError(u'is deprecated when strong TLS settings preset is enabled',
                                          u'enable_tlsv1_3')


class L7MacroHttp2Settings(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7Macro.Http2Settings

    REQUIRED = []

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(),
                 enabled_in_http=False, enabled_in_https=False):
        assert enabled_in_http or enabled_in_https
        self.auto_validate_required()

        if self.pb.HasField('fraction'):
            with validate('fraction'):
                if enabled_in_http:
                    raise ValidationError('can not be used if "http.enable_http2" is set')
                validate_range(self.pb.fraction.value, 0, 1)

    @classmethod
    def fill_extended_http_macro_pb_in_http_section_with_defaults(cls, extended_http_macro_pb, version):
        """
        :type extended_http_macro_pb: modules_pb2.ExtendedHttpMacro
        :type version: semantic_version.Version
        """
        extended_http_macro_pb.http2_allow_without_ssl.value = True
        extended_http_macro_pb.http2_allow_sending_trailers.value = True

    def fill_extended_http_macro_pb_in_http_section(self, extended_http_macro_pb, version):
        """
        :type extended_http_macro_pb: modules_pb2.ExtendedHttpMacro
        :type version: semantic_version.Version
        """
        self.fill_extended_http_macro_pb_in_http_section_with_defaults(extended_http_macro_pb, version)

    @classmethod
    def fill_extended_http_macro_pb_in_https_section_with_defaults(cls, extended_http_macro_pb, version):
        if version >= VERSION_0_3_11:
            extended_http_macro_pb.http2_allow_without_ssl.value = False
            extended_http_macro_pb.http2_allow_sending_trailers.value = True

    def fill_extended_http_macro_pb_in_https_section(self, extended_http_macro_pb, version):
        """
        :type extended_http_macro_pb: modules_pb2.ExtendedHttpMacro
        :type version: semantic_version.Version
        """
        self.fill_extended_http_macro_pb_in_https_section_with_defaults(extended_http_macro_pb, version)
        if self.pb.HasField('fraction'):
            extended_http_macro_pb.http2_alpn_freq.value = self.pb.fraction.value


class L7MacroMonitoringSettings(WrapperBase):
    __protobuf__ = modules_pb2.L7Macro.MonitoringSettings

    REPORT_INPUT_SIZE_RANGES = '32,64,128,256,512,1024,4096,8192,16384,131072,524288,1048576,2097152'
    REPORT_OUTPUT_SIZE_RANGES = ('512,1024,4096,8192,16384,32768,65536,131072,'
                                 '262144,524288,1048576,2097152,4194304,8388608')

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()

    @classmethod
    def fill_extended_http_macro_pb_with_defaults(cls, extended_http_macro_pb, version):
        """
        :type extended_http_macro_pb: modules_pb2.ExtendedHttpMacro
        :type version: semantic_version.Version
        """
        if version >= VERSION_0_2_3:
            extended_http_macro_pb.report_input_size_ranges = cls.REPORT_INPUT_SIZE_RANGES
            extended_http_macro_pb.report_output_size_ranges = cls.REPORT_OUTPUT_SIZE_RANGES

    def fill_extended_http_macro_pb(self, extended_http_macro_pb, uuid, version):
        """
        :type extended_http_macro_pb: modules_pb2.ExtendedHttpMacro
        :type uuid: six.text_type
        :type version: semantic_version.Version
        """
        if self.pb.enable_total_signals:
            extended_http_macro_pb.report_uuid = uuid
        if self.pb.enable_molly_signals:
            extended_http_macro_pb.report_enable_molly_uuid = True
        if not self.pb.enable_total_signals and version >= VERSION_0_2_3:
            extended_http_macro_pb.report_input_size_ranges = self.REPORT_INPUT_SIZE_RANGES
            extended_http_macro_pb.report_output_size_ranges = self.REPORT_OUTPUT_SIZE_RANGES

    @staticmethod
    def _add_molly_matcher_map(report_pb):
        matcher_map_entry_pb = report_pb.matcher_map.add()
        matcher_map_entry_pb.key = u'molly'
        matcher_pb = matcher_map_entry_pb.value
        matcher_pb.match_fsm.cgi = u'.*everybodybecoolthisis=(crasher|molly).*'

    def fill_stats_storage_section(self, section_pb, version):
        """
        :type section_pb: modules_pb2.IpdispatchSection
        :type version: semantic_version.Version
        """
        section_pb.ips.add(value=u'127.0.0.4')
        port_pb = section_pb.ports.add()
        fill_get_port_var_call(port_pb.f_value)
        report_pb = section_pb.nested.report
        report_pb.uuid = u'service_total'
        report_pb.ranges = u'default'
        report_pb.just_storage = True
        if self.pb.enable_molly_signals:
            self._add_molly_matcher_map(report_pb)
        if version >= VERSION_0_2_3:
            report_pb.input_size_ranges = self.REPORT_INPUT_SIZE_RANGES
            report_pb.output_size_ranges = self.REPORT_OUTPUT_SIZE_RANGES
        report_pb.nested.http.nested.errordocument.status = 204


class L7MacroIncludeUpstreams(object):
    def __init__(self, l7_macro):
        """
        :type l7_macro: L7Macro
        """
        self._l7_macro = l7_macro

    def get_included_upstream_ids(self, current_namespace_id, upstream_ids):
        """
        :type current_namespace_id: six.text_type
        :type upstream_ids: iterable[(six.text_type, six.text_type)]
        :rtype: set[(six.text_type, six.text_type)]
        """
        excluded_upstream_ids = self._l7_macro.list_excluded_upstream_ids()
        included_upstream_ids = set()
        for (namespace_id, upstream_id) in upstream_ids:
            if is_upstream_internal(upstream_id):
                continue
            assert namespace_id == current_namespace_id  # safety check
            if upstream_id not in excluded_upstream_ids:
                included_upstream_ids.add((namespace_id, upstream_id))
        return included_upstream_ids


class L7MacroIncludeDomains(WrapperBase):
    __protobuf__ = modules_pb2.IncludeDomains

    def update_pb(self, pb=None):
        super(L7MacroIncludeDomains, self).update_pb(pb=pb)
        self.wrap_composite_fields()

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        if self.pb.type != modules_pb2.ALL:
            raise ValidationError(u'must be ALL', u'type')


Discriminator = collections.namedtuple('Discriminator', ('action', 'args'))


class L7MacroHeadersActionCreate(WrapperBase):
    __protobuf__ = modules_pb2.L7Macro.HeaderAction.Create

    REQUIRED = ['target']
    REQUIRED_ONEOFS = [('value', 'func')]

    def get_discriminator(self, i):
        return Discriminator(action='create', args=(bool(self.pb.func), self.pb.keep_existing))

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        with validate('target'):
            validate_header_name(self.pb.target)
        if self.pb.func:
            validate_header_func(self.pb.func, self.pb.target, hint='Maybe you meant "value" instead of "func"')


class L7MacroHeadersActionCopy(WrapperBase):
    __protobuf__ = modules_pb2.L7Macro.HeaderAction.Copy

    REQUIRED = ['target', 'source']

    def get_discriminator(self, i):
        return Discriminator(action='copy', args=(self.pb.keep_existing,))

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        with validate('target'):
            validate_header_name(self.pb.target)
        with validate('source'):
            validate_header_name(self.pb.source)


class L7MacroHeadersActionAppend(WrapperBase):
    __protobuf__ = modules_pb2.L7Macro.HeaderAction.Append

    REQUIRED = ['target']
    REQUIRED_ONEOFS = [('value', 'func')]

    @staticmethod
    def get_discriminator(i):
        # "append" actions with the same target do not override each other, as "create" and "copy" actions do.
        # Thus we always want them to appear in different groups.
        return Discriminator(action='append', args=(i,))

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        with validate('target'):
            validate_header_name(self.pb.target)
        if self.pb.func:
            validate_header_func(self.pb.func, self.pb.target, hint='Maybe you meant "value" instead of "func"')


class L7MacroHeadersActionDelete(WrapperBase):
    __protobuf__ = modules_pb2.L7Macro.HeaderAction.Delete

    REQUIRED = ['target_re']

    def get_discriminator(self, i):
        return Discriminator(action='delete', args=(self.pb.target_re,))

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        with validate('target_re'):
            validate_pire_regexp(self.pb.target_re, lua_unescape_first=False)


class L7MacroHeadersActionCopyFromRequest(WrapperBase):
    __protobuf__ = modules_pb2.L7Macro.HeaderAction.CopyFromRequest

    REQUIRED = ['source', 'target']

    def get_discriminator(self, i):
        return Discriminator(action='copy_from_request', args=())

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        with validate('target'):
            validate_header_name(self.pb.target)
        with validate('source'):
            validate_header_name(self.pb.source)


class L7MacroHeadersActionLog(WrapperBase):
    __protobuf__ = modules_pb2.L7Macro.HeaderAction.Log

    REQUIRED = ['target_re']

    def get_discriminator(self, i):
        return Discriminator(action='log', args=(self.pb.target_re, tuple(self.pb.cookie_fields)))

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        with validate('target_re'):
            validate_pire_regexp(self.pb.target_re, lua_unescape_first=False)


class L7MacroRewritePattern(WrapperBase):
    __protobuf__ = modules_pb2.L7Macro.RewritePattern

    DEFAULT_CASE_SENSITIVE = False

    REQUIRED = ['re']

    def augment_rewrite_action_pb(self, pb):
        """
        :type pb: modules_pb2.RewriteAction
        """
        pb.regexp = dump_string(self.pb.re)
        pb.literal = self.pb.literal.value
        setattr(pb, 'global', getattr(self.pb, 'global').value)
        if self.pb.HasField('case_sensitive'):
            case_sensitive = self.pb.case_sensitive.value
        else:
            case_sensitive = self.DEFAULT_CASE_SENSITIVE
        pb.case_insensitive.value = not case_sensitive

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        with validate(u're'):
            validate_re2_regexp(self.pb.re, lua_unescape_first=False)


class L7MacroHeadersActionRewrite(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7Macro.HeaderAction.Rewrite

    pattern = None  # type: L7MacroRewritePattern | None

    REQUIRED = ['target', 'pattern', 'replacement']

    @staticmethod
    def get_discriminator(i):
        return Discriminator(action=u'rewrite', args=(u'rewrite',))

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        with validate(u'target'):
            validate_header_name(self.pb.target)
        with validate(u'pattern'):
            self.pattern.validate(ctx=ctx, preceding_modules=preceding_modules)


class L7MacroHeadersActionUaas(WrapperBase):
    __protobuf__ = modules_pb2.L7Macro.HeaderAction.Uaas

    REQUIRED = ['service_name']

    @staticmethod
    def get_discriminator(i):
        return Discriminator(action=u'uaas', args=(i,))

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()


class L7MacroHeadersActionLaas(WrapperBase):
    __protobuf__ = modules_pb2.L7Macro.HeaderAction.Laas

    REQUIRED = []

    @staticmethod
    def get_discriminator(i):
        return Discriminator(action=u'laas', args=(i,))

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()


class L7MacroHeadersActionDecryptICookie(WrapperBase):
    __protobuf__ = modules_pb2.L7Macro.HeaderAction.ICookie

    REQUIRED = []

    @staticmethod
    def get_discriminator(i):
        return Discriminator(action=u'decrypt_icookie', args=(i,))

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()


class L7MacroHeaderAction(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7Macro.HeaderAction

    create = None  # type: L7MacroHeadersActionCreate | None
    copy = None  # type: L7MacroHeadersActionCopy | None
    append = None  # type: L7MacroHeadersActionAppend | None
    delete = None  # type: L7MacroHeadersActionDelete | None
    copy_from_request = None  # type: L7MacroHeadersActionCopyFromRequest | None
    log = None  # type: L7MacroHeadersActionLog | None
    rewrite = None  # type: L7MacroHeadersActionRewrite | None
    uaas = None  # type: L7MacroHeadersActionUaas | None
    laas = None  # type: L7MacroHeadersActionLaas | None
    decrypt_icookie = None  # type: L7MacroHeadersActionDecryptICookie

    REQUIRED_ONEOFS = [('create', 'append', 'copy', 'delete', 'copy_from_request', 'log', 'rewrite', 'uaas', 'laas',
                        'decrypt_icookie')]
    MODIFYING_KINDS = ['create', 'append', 'rewrite', 'copy', 'rewrite']

    def validate(self, mode, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        if mode == Mode.HEADERS:
            if self.kind == 'copy_from_request':
                raise ValidationError(u'is not supported for headers', self.kind)
        if mode == Mode.RESPONSE_HEADERS:
            if self.kind in ['rewrite', 'uaas', 'laas']:
                raise ValidationError(u'is not supported for response headers', self.kind)
        super(L7MacroHeaderAction, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def get_discriminator(self, i):
        """
        :type i: int
        :rtype: Discriminator
        """
        return self.action.get_discriminator(i)

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        with validate(self.kind):
            self.action.validate(ctx=ctx, preceding_modules=preceding_modules)

    @property
    def is_modifying(self):
        return self.kind in L7MacroHeaderAction.MODIFYING_KINDS

    @property
    def action(self):
        return getattr(self, self.kind)

    @property
    def kind(self):
        return self.pb.WhichOneof('kind')

    @property
    def target(self):
        if hasattr(self.action.pb, 'target'):
            return self.action.pb.target.lower()
        return None

    @property
    def func(self):
        if hasattr(self.action.pb, 'func'):
            return self.action.pb.func
        return None


def get_create_and_append_targets_from_actions(actions):
    return {(action.create or action.append).pb.target for action in actions}


def get_create_and_append_targets_from_module(headers_pb):
    return (
        set(entry_pb.key for entry_pb in headers_pb.create) |
        set(entry_pb.key for entry_pb in headers_pb.create_weak) |
        set(headers_pb.create_func) |
        set(headers_pb.create_func_weak) |
        set(entry_pb.key for entry_pb in headers_pb.append) |
        set(entry_pb.key for entry_pb in headers_pb.append_weak) |
        set(headers_pb.append_func) |
        set(headers_pb.append_func_weak)
    )


class Mode(enum.Enum):
    HEADERS = 0
    RESPONSE_HEADERS = 1


def to_header_module_pbs(actions, mode, use_sd_uaas=False, trust_icookie=False, geobase_macro_version='0.0.1'):
    """
    :type actions: Iterable[L7MacroHeaderAction]
    :param mode: Mode.HEADERS | Mode.RESPONSE_HEADERS
    :rtype: list[modules_pb2.HeadersModule | modules_pb2.ResponseHeadersModule | modules_pb2.LogHeadersModule]
    """
    header_module_pbs = []
    prev_actions_type = None
    prev_headers_pb = None

    enumerated_actions = enumerate(actions)
    for group_disc, actions_group in itertools.groupby(enumerated_actions, lambda i_act: i_act[1].get_discriminator(i_act[0])):
        actions_type = group_disc[0]
        actions_group = [a for _, a in actions_group]  # drop indices

        if ({prev_actions_type, actions_type}.issubset({'create', 'append'}) and
                not (get_create_and_append_targets_from_module(prev_headers_pb) &
                     get_create_and_append_targets_from_actions(actions_group))):
            module_pb = prev_headers_pb
        elif actions_type in ('create', 'append', 'copy', 'delete'):
            if mode == Mode.HEADERS:
                module_pb = modules_pb2.HeadersModule()
            elif mode == Mode.RESPONSE_HEADERS:
                module_pb = modules_pb2.ResponseHeadersModule()
            else:
                raise AssertionError()
            header_module_pbs.append(module_pb)
        elif actions_type == 'copy_from_request':
            assert mode == Mode.RESPONSE_HEADERS
            module_pb = modules_pb2.HeadersForwarderModule()
            header_module_pbs.append(module_pb)
        elif actions_type == 'log':
            module_pb = modules_pb2.LogHeadersModule()
            header_module_pbs.append(module_pb)

            # sanity check: if log actions are grouped together, we expect them to be exactly the same
            it = iter(actions_group)
            first_action = next(it)
            for action in it:
                assert action.pb == first_action.pb
        elif actions_type == 'rewrite':
            module_pb = modules_pb2.RewriteModule()
            header_module_pbs.append(module_pb)
        elif actions_type == 'uaas':
            module_pb = modules_pb2.ExpGetterMacro()
            if use_sd_uaas:
                module_pb._version = 3
            header_module_pbs.append(module_pb)
        elif actions_type == 'laas':
            module_pb = modules_pb2.GeobaseMacro()
            module_pb.version = geobase_macro_version
            header_module_pbs.append(module_pb)
        elif actions_type == 'decrypt_icookie':
            from awacs.wrappers import main
            module_pb = modules_pb2.IcookieModule()
            module_pb.use_default_keys = True
            module_pb.trust_parent.value = trust_icookie
            module_pb.domains.extend(main.ICOOKIE_DOMAINS)
            header_module_pbs.append(module_pb)
        else:
            raise AssertionError('got action_type {}'.format(actions_type))

        for action in actions_group:
            if actions_type == 'create':
                create_pb = action.create.pb

                key = create_pb.target
                attr = 'create'
                if create_pb.func:
                    attr += '_func'
                    value = create_pb.func
                else:
                    value = dump_string(create_pb.value)
                if create_pb.keep_existing:
                    attr += '_weak'
            elif actions_type == 'append':
                append_pb = action.append.pb

                key = append_pb.target
                attr = 'append'
                if append_pb.func:
                    attr += '_func'
                    value = append_pb.func
                else:
                    value = dump_string(append_pb.value)
                if append_pb.do_not_create_if_missing:
                    attr += '_weak'
            elif actions_type == 'copy':
                copy_pb = action.copy.pb

                key = copy_pb.source
                attr = 'copy'
                value = copy_pb.target
                if copy_pb.keep_existing:
                    attr += '_weak'
            elif actions_type == 'delete':
                delete_pb = action.delete.pb

                value = dump_string(delete_pb.target_re)
                setattr(module_pb, 'delete', value)
                continue
            elif actions_type == 'copy_from_request':
                copy_from_req_pb = action.copy_from_request.pb
                existing_item_pb = None
                for item_pb in module_pb.actions:
                    if item_pb.request_header == copy_from_req_pb.source and \
                            item_pb.response_header == copy_from_req_pb.target:
                        existing_item_pb = item_pb
                        break
                if existing_item_pb is None:
                    existing_item_pb = module_pb.actions.add()
                existing_item_pb.request_header = copy_from_req_pb.source
                existing_item_pb.response_header = copy_from_req_pb.target
                if copy_from_req_pb.keep_existing:
                    existing_item_pb.weak = True
                    existing_item_pb.erase_from_response = False
                else:
                    existing_item_pb.weak = False
                    existing_item_pb.erase_from_response = True
                continue
            elif actions_type == 'log':
                log_pb = action.log.pb

                value = dump_string(log_pb.target_re)
                if mode == Mode.HEADERS:
                    module_pb.name_re = value
                elif mode == Mode.RESPONSE_HEADERS:
                    module_pb.response_name_re = value
                else:
                    raise AssertionError
                module_pb.cookie_fields[:] = log_pb.cookie_fields
                continue
            elif actions_type == 'rewrite':
                pb = module_pb.actions.add()
                action.rewrite.pattern.augment_rewrite_action_pb(pb)
                pb.header_name = dump_string(action.rewrite.pb.target)
                pb.rewrite = dump_string(action.rewrite.pb.replacement)
                continue
            elif actions_type == 'uaas':
                module_pb.service_name = dump_string(action.uaas.pb.service_name)
                continue
            elif actions_type == 'laas':
                continue
            elif actions_type == 'decrypt_icookie':
                continue
            else:
                raise AssertionError()

            if attr in ('create', 'create_weak', 'append', 'append_weak', 'copy', 'copy_weak'):
                field = getattr(module_pb, attr)
                existing_item_pb = None
                for item_pb in field:
                    if item_pb.key == key:
                        existing_item_pb = item_pb
                        break
                if existing_item_pb is None:
                    field.add(key=key, value=value)
                else:
                    existing_item_pb.value = value
            elif attr in ('create_func', 'create_func_weak', 'append_func', 'append_func_weak'):
                field = getattr(module_pb, attr)
                field[key] = value
            else:
                raise AssertionError()

        prev_actions_type = actions_type
        prev_headers_pb = module_pb

    return header_module_pbs


class L7MacroRewriteAction(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7Macro.RewriteAction

    pattern = None  # type: L7MacroRewritePattern

    REQUIRED = ['target', 'pattern', 'replacement']

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        with validate('pattern'):
            self.pattern.validate(ctx=ctx, preceding_modules=preceding_modules)


def to_rewrite_module_pb(actions):
    """
    :type actions: Iterable[L7MacroRewriteAction]
    :rtype: modules_pb2.RewriteModule
    """
    rewrite_pb = modules_pb2.RewriteModule()
    for action in actions:
        rewrite_action_pb = rewrite_pb.actions.add()
        if action.pb.target == action.pb.URL:
            rewrite_action_pb.split = u'url'
        elif action.pb.target == action.pb.PATH:
            rewrite_action_pb.split = u'path'
        elif action.pb.target == action.pb.CGI:
            rewrite_action_pb.split = u'cgi'
        else:
            raise AssertionError()
        action.pattern.augment_rewrite_action_pb(rewrite_action_pb)
        rewrite_action_pb.rewrite = dump_string(action.pb.replacement)
    return rewrite_pb


class L7Macro(ModuleWrapperBase, MacroBase):
    __protobuf__ = modules_pb2.L7Macro

    compat = None  # type: L7MacroCompat | None
    core = None  # type: L7MacroCoreSettings | None
    http = None  # type: L7MacroHttpSettings | None
    https = None  # type: L7MacroHttpsSettings | None
    http2 = None  # type: L7MacroHttp2Settings | None
    announce_check_reply = None  # type: L7MacroAnnounceCheckReplySettings | None
    health_check_reply = None  # type: L7MacroHealthCheckReplySettings | None
    monitoring = None  # type: L7MacroMonitoringSettings | None
    headers = ()  # type: list[L7MacroHeaderAction]
    response_headers = ()  # type: list[L7MacroHeaderAction]
    # by romanovich@:
    # rewrite = ()  # type: list[L7MacroRewriteAction]

    include_domains = None  # type: L7MacroIncludeDomains or None
    antirobot = None  # type: L7MacroAntirobotSettings or None
    rps_limiter = None  # type: L7MacroRpsLimiterSettings or None
    webauth = None  # type: L7MacroWebauthSettings or None

    DEFAULT_HTTP_PORTS = [80]
    DEFAULT_HTTPS_PORTS = [443]
    DEFAULT_MAXCONN = 5000
    DEFAULT_MAXLEN = 64 * 1024
    DEFAULT_MAXREQ = 64 * 1024
    DEFAULT_HEALTH_CHECK_URL_RE = u'/awacs-balancer-health-check'
    DEFAULT_HEALTH_CHECK_KEY = u'awacs-balancer-health-check'
    DEFAULT_ANNOUNCE_CHECK_KEY = u'slbping'
    DEFAULT_REDIRECT_TO_HTTPS_KEY = u'http_to_https'
    ANTIROBOT_CAPTCHA_KEY = u'antirobot_captcha'
    TCP_LISTEN_QUEUE = 128
    STATE_DIRECTORY = '/dev/shm/balancer-state'
    LOG_DIR = '/place/db/www/logs/'
    CPU_LIMITER_DISABLE_FILE = './controls/cpu_limiter_disabled'
    CPU_LIMITER_ACTIVE_CHECK_SUBNET_FILE = './controls/active_check_subnets_list'
    WEBAUTH_SAVE_OAUTH_TOKEN_KEY = u'__webauth_save_oauth_token'

    REQUIRED = ['version']
    REQUIRED_ANYOFS = [('http', 'https')]
    HEADERS_MAX_COUNT = 20
    REWRITE_MAX_COUNT = 10

    def __init__(self, pb):
        from awacs.wrappers.main import InstanceMacro  # noqa # needed for include_domains expansion, do not remove
        super(L7Macro, self).__init__(pb)

    @property
    def include_upstreams(self):
        # a solution to meet the requirements of get_injected_full_upstream_ids
        return L7MacroIncludeUpstreams(self)

    def includes_domains(self):
        return bool(self.include_domains)

    def includes_backends(self):
        return False

    def would_include_backends(self):
        return True

    def get_version(self):
        return semantic_version.Version(self.pb.version)

    def get_http_ports(self):
        return self.pb.http.ports or self.DEFAULT_HTTP_PORTS

    def get_https_ports(self):
        return self.pb.https.ports or self.DEFAULT_HTTPS_PORTS

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        """
        :type ctx: ValidationCtx
        :param preceding_modules:
        """
        self.auto_validate_required()

        validate_version(self.pb.version, VALID_VERSIONS, field_name='version')
        if self.get_version() >= VERSION_0_4_0 or self.pb.compat.enable_meta:
            with validate('version'):
                ctx.ensure_component_version(
                    model_pb2.ComponentMeta.PGINX_BINARY,
                    min=min_component_versions.META)
        if self.get_version() >= VERSION_0_3_12 and ctx.balancer_env_type == model_pb2.BalancerSpec.L7_ENV_TESTING:
            with validate('version'):
                ctx.ensure_component_version(
                    model_pb2.ComponentMeta.PGINX_BINARY,
                    min=min_component_versions.ALLOW_EMPTY_ENDPOINT_SETS
                )

        if self.compat:
            with validate(u'compat'):
                self.compat.validate(ctx=ctx, preceding_modules=preceding_modules)

            if self.compat.pb.disable_sd and self._use_sd_uaas():
                with validate(u'version'):
                    raise ValidationError(u'"compat.disable_sd" must not be set')

            if self.core:
                # https://st.yandex-team.ru/AWACS-601
                # TODO: this validation is temporary
                if self.pb.compat.HasField('maxlen') and self.pb.core.limits.HasField('req_line_plus_headers_max_len'):
                    raise ValidationError(u'compat.maxlen and core.limits.req_line_plus_headers_max_len '
                                          u'must not be used together. '
                                          u'Please use core.limits.req_line_plus_headers_max_len')
                if self.pb.compat.HasField('maxreq') and self.pb.core.limits.HasField('req_line_max_len'):
                    raise ValidationError(u'compat.maxreq and core.limits.req_line_max_len must not be used together. '
                                          u'Please use core.limits.req_line_max_len')

        http_domains = DomainsSet()
        https_domains = DomainsSet()
        if self.include_domains:
            with validate(u'include_domains'):
                if ctx.config_type == ctx.CONFIG_TYPE_FULL:
                    # only check during async validation
                    if not ctx.domain_config_pbs:
                        raise ValidationError(u'no domains found')
                    ctx.validate_domains()
                    http_domains, https_domains = ctx.get_separate_http_and_https_domains()
                    if self.https and not self.https.verify_client_cert:
                        verify_client_cert_by_cert_id = {}
                        domain_ids_by_cert_id = collections.defaultdict(list)
                        for (ns_id, domain_id), domain_config_pb in https_domains.list_all_domains():
                            cert_id = domain_config_pb.cert.id
                            verify_client_cert = domain_config_pb.HasField('verify_client_cert') and domain_config_pb.verify_client_cert
                            domain_ids_by_cert_id[cert_id].append(domain_id)
                            if verify_client_cert_by_cert_id.get(cert_id, verify_client_cert) != verify_client_cert:
                                raise ValidationError(u'"verify_client_cert": can not be different in domains '
                                                      u'using same certificate: ("{}")'
                                                      .format(quote_join_sorted(domain_ids_by_cert_id[cert_id])))
                            verify_client_cert_by_cert_id[cert_id] = verify_client_cert
                    if self.http and self.http.redirect_to_https:
                        http_only_domains = set()
                        for (ns_id, domain_id), domain_config_pb in http_domains.list_all_domains():
                            if domain_config_pb.protocol == model_pb2.DomainSpec.Config.HTTP_ONLY:
                                http_only_domains.add(domain_id)
                        if http_only_domains:
                            raise ValidationError(
                                u'"http.redirect_to_https" is configured, but some domains are '
                                u'HTTP-only and will be inaccessible: "{}"'.format(
                                    quote_join_sorted(http_only_domains)))

        with validate('announce_check_reply'):
            if self.announce_check_reply:
                self.announce_check_reply.validate(version=self.get_version(),
                                                   ctx=ctx, preceding_modules=preceding_modules)
            else:
                if self.monitoring and self.monitoring.pb.enable_announce_check_signals:
                    raise ValidationError(u'must be set to use monitoring.enable_announce_check_signals')
                if self.include_domains:
                    raise ValidationError(u'must be set to use include_domains')
                if self.antirobot:
                    raise ValidationError(u'must be set to use antirobot')
                if self.rps_limiter:
                    raise ValidationError(u'must be set to use rps_limiter')

        if self.health_check_reply:
            with validate('health_check_reply'):
                self.health_check_reply.validate(ctx=ctx, preceding_modules=preceding_modules)
        else:
            if self.include_domains:
                raise ValidationError('must be set to use include_domains', 'health_check_reply')
            if self.antirobot:
                raise ValidationError('must be set to use antirobot', 'health_check_reply')
            if self.rps_limiter:
                raise ValidationError('must be set to use rps_limiter', 'health_check_reply')

        if self.antirobot:
            if self.pb.compat.disable_sd:
                raise ValidationError('cannot be used with antirobot', 'compat.disable_sd')
            if self.get_version() >= VERSION_0_3_4:
                for i, header_action in enumerate(self.headers):
                    with validate('headers[{}]'.format(i)):
                        if header_action.kind == 'decrypt_icookie':
                            raise ValidationError('cannot be used with antirobot', 'decrypt_icookie')
            with validate('antirobot'):
                self.antirobot.validate(ctx=ctx, preceding_modules=preceding_modules)

        if self.webauth:
            with validate(u'webauth'):
                if (self.get_version() >= VERSION_0_2_11 and self.http and not self.http.redirect_to_https and
                    self.webauth.pb.action != self.webauth.pb.SAVE_OAUTH_TOKEN):
                    raise ValidationError(u'"http.redirect_to_https" must be enabled')
                self.webauth.validate(ctx=ctx, preceding_modules=preceding_modules)

        if self.rps_limiter:
            if self.rps_limiter.external and self.pb.compat.disable_sd:
                raise ValidationError(u'cannot be used with external rps_limiter', u'compat.disable_sd')
            with validate(u'rps_limiter'):
                self.rps_limiter.validate(ctx=ctx, preceding_modules=preceding_modules)

        if self.core:
            with validate(u'core'):
                self.core.validate(ctx=ctx, preceding_modules=preceding_modules)
                if self.get_version() < VERSION_0_3_0 and self.core.pb.trust_x_forwarded_for_y:
                    raise ValidationError(u'trust_x_forwarded_for_y is not available in versions prior to 0.3.0')
                if self.core.pb.trust_icookie:
                    with validate(u'trust_icookie'):
                        if self.get_version() < VERSION_0_3_2:
                            raise ValidationError(u'is not available in versions prior to 0.3.2')

                        kinds = {action.kind for action in self.headers}
                        if 'decrypt_icookie' not in kinds and not self.antirobot:
                            raise ValidationError(u'cannot be set without decrypt_icookie action or configured antirobot')

                if self.core.pb.trust_x_yandex_ja_x:
                    if self.get_version() < VERSION_0_3_3:
                        raise ValidationError(u'is not available in versions prior to 0.3.3', u'trust_x_yandex_ja_x')
                    if not self.antirobot:
                        raise ValidationError(u'cannot be set without configuring antirobot', u'trust_x_yandex_ja_x')

        if self.http:
            with validate('http'):
                self.http.validate(ctx=ctx, preceding_modules=preceding_modules)

                if self.http.redirect_to_https:
                    with validate(u'redirect_to_https'):
                        if not self.announce_check_reply or not self.health_check_reply:
                            raise ValidationError(u'can only be used together with "announce_check_reply" and '
                                                  u'"health_check_reply"')
                        if not self.https:
                            raise ValidationError(u'can only be used together with "https"')

                if self.http.pb.compat.refer_shared_uuid:
                    referred_shared_uuid_exists = (
                        self.https and
                        self.https.pb.compat.assign_shared_uuid == self.http.pb.compat.refer_shared_uuid)
                    if not referred_shared_uuid_exists:
                        raise ValidationError('compat.refer_shared_uuid does not exist')
                if ctx.config_type == ctx.CONFIG_TYPE_FULL and self.include_domains:
                    if http_domains.empty():
                        raise ValidationError('no HTTP domains are found. Either remove "http: {}" section from '
                                              'balancer config, or change domain protocol to include HTTP.')
                    if https_domains.wildcard is not None and http_domains.wildcard is None:
                        raise ValidationError('Wildcard domain is HTTPS-only. Either remove "http: {}" section from '
                                              'balancer config, or change domain protocol to include HTTP.')
        elif not http_domains.empty():
            raise ValidationError('must be set because HTTP domains are present', 'http')

        if self.https:
            with validate('https'):
                if not self.include_domains and len(self.https.pb.certs) == 0:
                    raise ValidationError('one of "include_domains" or "https.certs" must be set')

                if self.include_domains:
                    if len(self.https.pb.certs) > 0:
                        raise ValidationError('cannot be used with "include_domains"', 'certs')
                    ctx.validate_domain_certs()

                self.https.validate(ctx=ctx, preceding_modules=preceding_modules)

                if self.https.pb.compat.refer_shared_uuid:
                    referred_shared_uuid_exists = (
                        self.http and
                        self.http.pb.compat.assign_shared_uuid == self.https.pb.compat.refer_shared_uuid)
                    if not referred_shared_uuid_exists:
                        raise ValidationError('compat.refer_shared_uuid does not exist')
                if ctx.config_type == ctx.CONFIG_TYPE_FULL and self.include_domains:
                    if https_domains.empty():
                        raise ValidationError('no HTTPS domains are found. Either remove "https: {}" section from '
                                              'balancer config, or change domain protocol to include HTTPS.')
                    if http_domains.wildcard is not None and https_domains.wildcard is None:
                        raise ValidationError('Wildcard domain is HTTP-only. Either remove "https: {}" section from '
                                              'balancer config, or change domain protocol to include HTTPS.')
        elif not https_domains.empty():
            raise ValidationError('must be set because HTTPS domains are present', 'https')

        if self.http and self.https:
            if (self.http.pb.compat.assign_shared_uuid and self.https.pb.compat.assign_shared_uuid and
                    self.http.pb.compat.assign_shared_uuid == self.https.pb.compat.assign_shared_uuid):
                raise ValidationError('compat.assign_shared_uuid can not be set to the same value for both '
                                      'http and https')
            if self.http.pb.compat.refer_shared_uuid and self.https.pb.compat.refer_shared_uuid:
                raise ValidationError('compat.refer_shared_uuid can not be set for both http and https')
            if self.http.pb.compat.bind_on_instance_port and self.https.pb.compat.bind_on_instance_port:
                raise ValidationError('compat.bind_on_instance_port can not be set for both http and https')
            if (self.http.pb.compat.use_instance_port_in_section_log_name and
                    self.https.pb.compat.use_instance_port_in_section_log_name):
                raise ValidationError('compat.use_instance_port_in_section_log_name '
                                      'can not be set for both http and https')
            if set(self.http.pb.ports) & set(self.https.pb.ports):
                raise ValidationError('http and https must not intersect by ports')

        if self.http2:
            with validate(u'http2'):
                http2_enabled_in_http = self.http.pb.enable_http2 if self.http else False
                http2_enabled_in_https = self.https.pb.enable_http2 if self.https else False
                if not http2_enabled_in_http and not http2_enabled_in_https:
                    raise ValidationError(u'can not be configured without "http.enable_http2" or "https.enable_http2"')
                self.http2.validate(ctx=ctx, preceding_modules=preceding_modules,
                                    enabled_in_http=http2_enabled_in_http, enabled_in_https=http2_enabled_in_https)

        if self.monitoring:
            with validate('monitoring'):
                self.monitoring.validate(ctx=ctx, preceding_modules=preceding_modules)

        # by romanovich@:
        # if len(self.rewrite) >= self.REWRITE_MAX_COUNT:
        #    raise ValidationError(u'exceeds allowed limit of {} actions'.format(self.REWRITE_MAX_COUNT), u'rewrite')

        # for i, action in enumerate(self.rewrite):
        #    with validate(u'rewrite[{}]'.format(i)):
        #        action.validate(ctx=ctx, preceding_modules=preceding_modules)

        if len(self.headers) > self.HEADERS_MAX_COUNT:
            raise ValidationError('exceeds allowed limit of {} actions'.format(self.HEADERS_MAX_COUNT),
                                  'headers')

        version = self.get_version()

        for i, action in enumerate(self.headers):
            with validate('headers[{}]'.format(i)):
                if version < VERSION_0_3_2 and action.kind == 'decrypt_icookie':
                    raise ValidationError('the `decrypt_icookie` header is not available in versions prior to 0.3.2')

                action.validate(mode=Mode.HEADERS, ctx=ctx, preceding_modules=preceding_modules)
                if version >= VERSION_0_3_0 and action.is_modifying and action.target == 'x-forwarded-for-y':
                    raise ValidationError(
                        "You can't modify the `X-Forwarded-For-Y` header manually "
                        "in l7_macro version 0.3.0 and above. The only available action is `log`")

        if len(self.response_headers) > self.HEADERS_MAX_COUNT:
            raise ValidationError('exceeds allowed limit of {} actions'.format(self.HEADERS_MAX_COUNT),
                                  'response_headers')

        for i, action in enumerate(self.response_headers):
            with validate('response_headers[{}]'.format(i)):
                action.validate(mode=Mode.RESPONSE_HEADERS, ctx=ctx, preceding_modules=preceding_modules)

    def get_included_cert_ids(self, namespace_id, ctx):
        """
        :type namespace_id: str
        :type ctx: ValidationCtx
        :rtype: set[(str, str)]
        """
        rv = set()
        if self.https:
            for cert_pb in self.https.pb.certs:
                rv.add((namespace_id, cert_pb.id))
                if cert_pb.secondary_id:
                    rv.add((namespace_id, cert_pb.secondary_id))
        return rv

    def includes_upstreams(self):
        return True

    def _fill_admin_section(self, section_pb):
        """
        :type section_pb: modules_pb2.IpdispatchSection
        """
        for ip in ['127.0.0.1', '::1']:
            section_pb.ips.add(value=ip)
        port_pb = section_pb.ports.add()
        fill_get_port_var_call(port_pb.f_value)
        section_pb.nested.http.nested.admin.SetInParent()

    @staticmethod
    def _fill_default_section_with_redirect_to_https(section_name, holder_pb, is_permanent):
        regexp_section_pb = holder_pb.regexp.sections.add(key=section_name).value
        regexp_section_pb.matcher.SetInParent()
        http_to_https_macro_pb = regexp_section_pb.nested.http_to_https_macro
        http_to_https_macro_pb.permanent = is_permanent

    @staticmethod
    def assign_header_module_to_holder(holder_pb, header_module_pb):
        """
        :type holder_pb: modules_pb2.Holder
        :type header_module_pb: modules_pb2.HeadersModule | modules_pb2.LogHeadersModule
        """
        if isinstance(header_module_pb, modules_pb2.HeadersModule):
            holder_pb.headers.CopyFrom(header_module_pb)
        elif isinstance(header_module_pb, modules_pb2.ResponseHeadersModule):
            holder_pb.response_headers.CopyFrom(header_module_pb)
        elif isinstance(header_module_pb, modules_pb2.HeadersForwarderModule):
            holder_pb.headers_forwarder.CopyFrom(header_module_pb)
        elif isinstance(header_module_pb, modules_pb2.LogHeadersModule):
            holder_pb.log_headers.CopyFrom(header_module_pb)
        elif isinstance(header_module_pb, modules_pb2.RewriteModule):
            holder_pb.rewrite.CopyFrom(header_module_pb)
        elif isinstance(header_module_pb, modules_pb2.ExpGetterMacro):
            holder_pb.exp_getter_macro.CopyFrom(header_module_pb)
        elif isinstance(header_module_pb, modules_pb2.GeobaseMacro):
            holder_pb.geobase_macro.CopyFrom(header_module_pb)
        elif isinstance(header_module_pb, modules_pb2.IcookieModule):
            holder_pb.icookie.CopyFrom(header_module_pb)
        else:
            raise ValueError(type(header_module_pb))

    def list_excluded_upstream_ids(self):
        """
        :rtype: set[six.text_type]
        """
        excluded_upstream_ids = set()
        if (self.announce_check_reply and
                self.announce_check_reply.compat and
                self.announce_check_reply.compat.pb.replaced_upstream_id):
            excluded_upstream_ids.add(self.announce_check_reply.compat.pb.replaced_upstream_id)
        if (self.health_check_reply and
                self.health_check_reply.compat and
                self.health_check_reply.compat.pb.replaced_upstream_id):
            excluded_upstream_ids.add(self.health_check_reply.compat.pb.replaced_upstream_id)
        if (self.http and
                self.http.redirect_to_https and
                self.http.redirect_to_https.compat and
                self.http.redirect_to_https.compat.pb.replaced_upstream_id):
            excluded_upstream_ids.add(self.http.redirect_to_https.compat.pb.replaced_upstream_id)
        return excluded_upstream_ids

    def _fill_check_reply_sections(self, regexp_section_pbs):
        if self.health_check_reply:
            key = dump_string(self.health_check_reply.compat and
                              self.health_check_reply.compat.pb.replaced_upstream_id or
                              self.DEFAULT_HEALTH_CHECK_KEY)
            regexp_section_pb = regexp_section_pbs.add(key=key).value
            regexp_section_pb.matcher.match_fsm.uri = dump_string(self.DEFAULT_HEALTH_CHECK_URL_RE)
            regexp_section_pb.nested.errordocument.status = 200

        if self.announce_check_reply:
            key = dump_string(self.announce_check_reply.compat and
                              self.announce_check_reply.compat.pb.replaced_upstream_id or
                              self.DEFAULT_ANNOUNCE_CHECK_KEY)
            regexp_section_pb = regexp_section_pbs.add(key=key).value
            version = self.get_version()
            self.announce_check_reply.fill_regexp_section(
                regexp_section_pb,
                version,
                enable_monitoring=self.monitoring and self.monitoring.pb.enable_announce_check_signals)

        if self.antirobot and self.antirobot.captcha_reply:
            key = dump_string(self.ANTIROBOT_CAPTCHA_KEY)
            regexp_section_pb = regexp_section_pbs.add(key=key).value
            trust_xffy = self.core.pb.trust_x_forwarded_for_y if self.core else False
            trust_icookie = self.core.pb.trust_icookie if self.core else False
            trust_ja_x = self.core.pb.trust_x_yandex_ja_x if self.core else False
            service = self.antirobot.pb.service or None
            req_group = self.antirobot.pb.req_group or None
            self.antirobot.captcha_reply.fill_regexp_section(
                regexp_section_pb, self.get_version(), trust_xffy, trust_icookie, trust_ja_x, service, req_group)

        if self.webauth and self.webauth.pb.mode == self.webauth.pb.EXTERNAL:
            regexp_section_pb = regexp_section_pbs.add(key=self.WEBAUTH_SAVE_OAUTH_TOKEN_KEY).value
            L7MacroWebauthSettings.fill_save_oauth_token_section(regexp_section_pb, version)

    def _fill_regexp_with_include_upstreams(self, regexp_pb):
        """
        :type regexp_pb: modules_pb2.RegexpModule
        """
        include_upstreams_pb = regexp_pb.include_upstreams
        include_upstreams_pb.order.label.name = 'order'
        excluded_upstream_ids = self.list_excluded_upstream_ids()
        if excluded_upstream_ids:
            getattr(include_upstreams_pb.filter, 'not').ids.extend(sorted(excluded_upstream_ids))
        else:
            include_upstreams_pb.filter.any = True

    def _fill_domain_holder_pb(self, holder_pb, full_domain_id, domain_config_pb, is_http_section):
        """
        :type holder_pb: modules_pb2.Holder
        :type full_domain_id: tuple[six.text_type, six.text_type]
        :type domain_config_pb: model_pb2.DomainSpec.Config
        :type is_http_section: bool
        """
        if self.get_version() >= VERSION_0_4_0 or self.pb.compat.enable_meta:
            holder_pb.meta.id = META_MODULE_ID
            holder_pb.meta.fields['domain'] = full_domain_id[1]
            holder_pb = holder_pb.meta.nested
        if is_http_section and domain_config_pb.HasField('redirect_to_https'):
            key = dump_string(self.DEFAULT_REDIRECT_TO_HTTPS_KEY + '_' + full_domain_id[1])
            self._fill_default_section_with_redirect_to_https(
                section_name=key,
                holder_pb=holder_pb,
                is_permanent=domain_config_pb.redirect_to_https.permanent)
        else:
            holder_pb.regexp.include_upstreams.CopyFrom(domain_config_pb.include_upstreams)

    def _fill_regexp_with_domains(self, regexp_pb, domains_set, allow_ports, is_http_section):
        """
        :type regexp_pb: modules_pb2.RegexpModule
        :type domains_set: DomainsSet
        :type allow_ports: list
        :type is_http_section: bool
        """
        default_section_pb = regexp_pb.sections.add(key='default').value  # type: modules_pb2.RegexpSection
        default_section_pb.matcher.SetInParent()
        regexp_host_pb = default_section_pb.nested.regexp_host  # type: modules_pb2.RegexpHostModule
        tld_regexp_pb = None
        for full_domain_id, domain_config_pb in domains_set.list_common_domains():
            entry_pb = regexp_host_pb.sections.add(key=full_domain_id[1])
            regexp_host_section_pb = entry_pb.value  # type: modules_pb2.RegexpHostSection
            regexp = hosts_to_regexp(itertools.chain(domain_config_pb.fqdns, domain_config_pb.shadow_fqdns))
            if allow_ports:
                if self.get_version() >= VERSION_0_2_5:
                    # https://st.yandex-team.ru/AWACS-602
                    regexp = u'({})(:{})?'.format(regexp, ports_to_regexp(allow_ports))
                else:
                    regexp = u'{}(:{})?'.format(regexp, ports_to_regexp(allow_ports))
            regexp_host_section_pb.pattern = dump_string(regexp)
            self._fill_domain_holder_pb(regexp_host_section_pb.nested, full_domain_id, domain_config_pb, is_http_section)
        if domains_set.yandex_tld is not None:
            full_domain_id, domain_config_pb = domains_set.yandex_tld
            entry_pb = regexp_host_pb.sections.add(key=full_domain_id[1])
            tld_regexp_pb = entry_pb.value.nested.regexp  # type: modules_pb2.RegexpModule
            regexp_section_pb = tld_regexp_pb.sections.add(key='from_term_layer').value
            header_matcher_pb = regexp_section_pb.matcher.match_fsm.header
            header_matcher_pb.name = 'X-Forwarded-For-Y'  # TODO: change to a more specific header after MINOTAUR-2486
            header_matcher_pb.value = '.*'
            self._fill_domain_holder_pb(regexp_section_pb.nested, full_domain_id, domain_config_pb, is_http_section)

        if domains_set.wildcard is not None:
            full_domain_id, domain_config_pb = domains_set.wildcard
            if domains_set.yandex_tld is not None:
                regexp_pb = tld_regexp_pb
            else:
                entry_pb = regexp_host_pb.sections.add(key=full_domain_id[1])
                regexp_pb = entry_pb.value.nested.regexp
            regexp_section_pb = regexp_pb.sections.add(key=dump_string('wildcard')).value
            regexp_section_pb.matcher.SetInParent()
            self._fill_domain_holder_pb(regexp_section_pb.nested, full_domain_id, domain_config_pb, is_http_section)
        elif self.get_version() >= VERSION_0_2_2:
            if domains_set.yandex_tld is not None:
                self._add_domain_not_found_section_tld(tld_regexp_pb)
            else:
                self._add_domain_not_found_section(regexp_host_pb)

    @staticmethod
    def _add_domain_not_found_section(regexp_host_pb):
        entry_pb = regexp_host_pb.sections.add(key=dump_string('domain_not_found')).value
        entry_pb.pattern = dump_string('.*')
        entry_pb.nested.errordocument.status = 404

    @staticmethod
    def balancer_env_type_to_log_record(balancer_env_type):
        prefix = 'L7_ENV_'
        env_type = model_pb2.BalancerSpec.EnvType.Name(balancer_env_type)
        assert env_type.startswith(prefix)
        return env_type[len(prefix):].lower()

    @staticmethod
    def _add_domain_not_found_section_tld(regexp_pb):
        entry_pb = regexp_pb.sections.add(key=dump_string('domain_not_found')).value
        entry_pb.matcher.match_fsm.host = dump_string('.*')
        entry_pb.nested.errordocument.status = 404

    def _use_sd_uaas(self):
        version = self.get_version()
        return version >= VERSION_0_3_1 or VERSION_0_2_12 <= version < VERSION_0_3_0

    def _fill_headers(self, holder_pb):
        version = self.get_version()

        if version >= VERSION_0_3_0:
            headers_pb = holder_pb.modules.add().headers
            if self.pb.core.trust_x_forwarded_for_y:
                headers_pb.create_func_weak['X-Forwarded-For-Y'] = 'realip'
            else:
                headers_pb.create_func['X-Forwarded-For-Y'] = 'realip'

        if self.headers:
            geobase_macro_version = '0.0.1'
            if version >= VERSION_0_3_10:
                geobase_macro_version = '0.0.3'
            for header_module_pb in to_header_module_pbs(
                    self.headers, mode=Mode.HEADERS,
                    use_sd_uaas=self._use_sd_uaas(),
                    trust_icookie=self.core.pb.trust_icookie if self.core else False,
                    geobase_macro_version=geobase_macro_version):
                self.assign_header_module_to_holder(holder_pb.modules.add(), header_module_pb)
        if self.response_headers:
            for header_module_pb in reversed(to_header_module_pbs(self.response_headers, mode=Mode.RESPONSE_HEADERS)):
                self.assign_header_module_to_holder(holder_pb.modules.add(), header_module_pb)

    def _fill_common_section_modules(self, holder_pb, namespace_id, refer_shared_uuid=None, domains_set=None, allow_ports=(),
                                     is_http_section=False, fill_check_repliers=False):
        """
        :type holder_pb: modules_pb2.Holder
        :type refer_shared_uuid: Optional[six.string_types]
        :type domains_set: DomainsSet
        :type allow_ports: list[int]
        :type is_http_section: bool
        """
        if refer_shared_uuid:
            holder_pb.shared.uuid = refer_shared_uuid
            return

        if self.core:
            if self.core.pb.merge_slashes:
                rewrite_pb = holder_pb.modules.add().rewrite
                rewrite_action_pb = rewrite_pb.actions.add(
                    regexp=dump_string(u'/+'),
                    rewrite=dump_string(u'/'),
                    split='path'
                )
                setattr(rewrite_action_pb, u'global', True)

        # by romanovich@:
        # if self.rewrite:
        #     holder_pb.modules.add().rewrite.CopyFrom(to_rewrite_module_pb(self.rewrite))

        version = self.get_version()
        if not is_http_section and version >= VERSION_0_3_6:
            holder_pb.modules.add().headers.create_weak.add(key='X-Yandex-HTTPS', value='yes')

        if self.antirobot:  # goes before everything else to reject bots
            trust_xffy = self.core.pb.trust_x_forwarded_for_y if self.core else False
            trust_icookie = self.core.pb.trust_icookie if self.core else False
            trust_ja_x = self.core.pb.trust_x_yandex_ja_x if self.core else False

            self.antirobot.fill_holder_pb(
                holder_pb.modules.add(), version, trust_xffy, trust_icookie, trust_ja_x)

        if self.webauth and self.webauth.pb.action != self.webauth.pb.SAVE_OAUTH_TOKEN:
            awacs_installation = self.pb.compat.awacs_installation or DEFAULT_AWACS_INSTALLATION
            self.webauth.fill_holder_pb(holder_pb.modules.add(), namespace_id, awacs_installation, version)

        if self.rps_limiter:
            rps_limiter_macro_version = None
            if VERSION_0_3_16 <= version < VERSION_0_4_0 or version >= VERSION_0_4_4:
                rps_limiter_macro_version = '0.0.4'
            elif VERSION_0_3_13 <= version < VERSION_0_4_0 or version >= VERSION_0_4_2:
                rps_limiter_macro_version = '0.0.3'
            self.rps_limiter.fill_holder_pb(holder_pb.modules.add(), rps_limiter_macro_version)

        self._fill_headers(holder_pb)

        regexp_pb = holder_pb.modules.add().regexp
        if self.include_domains:
            if fill_check_repliers:
                self._fill_check_reply_sections(regexp_pb.sections)
            self._fill_regexp_with_domains(regexp_pb, domains_set,
                                           allow_ports=allow_ports, is_http_section=is_http_section)
        elif self.include_upstreams:
            if fill_check_repliers:
                self._fill_check_reply_sections(regexp_pb.prepend_sections)
            self._fill_regexp_with_include_upstreams(regexp_pb)
        else:
            raise ValueError

    def _prefill_check_repliers(self, holder_pb):
        if self.get_version() < VERSION_0_2_1 and not self.antirobot and not self.rps_limiter and not self.webauth:
            return False, holder_pb
        regexp_sections_pb = holder_pb.regexp.sections
        self._fill_check_reply_sections(regexp_sections_pb)
        regexp_section_pb = regexp_sections_pb.add().value
        regexp_section_pb.matcher.SetInParent()
        holder_pb = regexp_section_pb.nested
        return True, holder_pb

    def _fill_http_section(self, section_pb, ctx, assign_shared_uuid=None, refer_shared_uuid=None, domains_set=None):
        """
        :type section_pb: modules_pb2.IpdispatchSection
        :type ctx: ValidationCtx
        :type assign_shared_uuid: Optional[six.string_types]
        :type refer_shared_uuid: Optional[six.string_types]
        :type domains_set: DomainsSet
        """
        assert bool(assign_shared_uuid) + bool(refer_shared_uuid) + bool(self.http.redirect_to_https) <= 1
        version = self.get_version()

        section_pb.ips.add(value='*')

        def add_section_ports():
            for port in self.get_http_ports():
                section_pb.ports.add(value=port)

        def maybe_add_instance_port():
            if self.pb.http.compat.bind_on_instance_port:
                port_pb = section_pb.ports.add()
                fill_get_port_var_call(port_pb.f_value)

        if self.pb.http.compat.use_instance_port_in_section_log_name:
            maybe_add_instance_port()
            add_section_ports()
        else:
            add_section_ports()
            maybe_add_instance_port()

        extended_http_macro_pb = section_pb.nested.extended_http_macro
        extended_http_macro_pb.allow_webdav = self.pb.core.allow_webdav

        # https://st.yandex-team.ru/AWACS-601
        # TODO: reading from compat.{maxlen,maxreq} is temporary
        extended_http_macro_pb.maxlen = (self.pb.compat.maxlen.value if self.pb.compat.HasField('maxlen')
                                         else self.DEFAULT_MAXLEN)
        extended_http_macro_pb.maxreq = (self.pb.compat.maxreq.value if self.pb.compat.HasField('maxreq')
                                         else self.DEFAULT_MAXREQ)
        # TODO: in future we will only read from core.limits.{req_line_max_len,req_line_plus_headers_max_len}
        if self.pb.core.limits.HasField('req_line_max_len'):
            extended_http_macro_pb.maxreq = self.pb.core.limits.req_line_max_len.value
        if self.pb.core.limits.HasField('req_line_plus_headers_max_len'):
            extended_http_macro_pb.maxlen = self.pb.core.limits.req_line_plus_headers_max_len.value

        if self.pb.http.compat.use_instance_port_in_section_log_name:
            fill_get_port_var_call(extended_http_macro_pb.f_port)
        if self.pb.core.compat.HasField('keepalive_drop_probability'):
            extended_http_macro_pb.keepalive_drop_probability.CopyFrom(self.pb.core.compat.keepalive_drop_probability)

        if self.monitoring:
            self.monitoring.fill_extended_http_macro_pb(extended_http_macro_pb, uuid=u'http', version=version)
        else:
            L7MacroMonitoringSettings.fill_extended_http_macro_pb_with_defaults(extended_http_macro_pb, version=version)
        if version >= VERSION_0_1_1:
            extended_http_macro_pb.yandex_cookie_policy = extended_http_macro_pb.YCP_STABLE

        if assign_shared_uuid:
            shared_pb = extended_http_macro_pb.nested.shared
            shared_pb.uuid = assign_shared_uuid
            holder_pb = shared_pb.nested
        else:
            holder_pb = extended_http_macro_pb.nested

        if version >= VERSION_0_4_0 or self.pb.compat.enable_meta:
            holder_pb.meta.id = META_MODULE_ID
            holder_pb.meta.fields['namespace'] = ctx.namespace_id
            holder_pb.meta.fields['env_type'] = self.balancer_env_type_to_log_record(ctx.balancer_env_type)
            holder_pb = holder_pb.meta.nested

        if self.pb.http.enable_http2:
            extended_http_macro_pb.enable_http2 = True
            if self.http2:
                self.http2.fill_extended_http_macro_pb_in_http_section(extended_http_macro_pb, version)
            else:
                L7MacroHttp2Settings.fill_extended_http_macro_pb_in_http_section_with_defaults(extended_http_macro_pb, version)

        if self.http.redirect_to_https:
            self._fill_check_reply_sections(holder_pb.regexp.sections)
            key = dump_string(self.http.redirect_to_https.compat and
                              self.http.redirect_to_https.compat.pb.replaced_upstream_id or
                              self.DEFAULT_REDIRECT_TO_HTTPS_KEY)
            self._fill_default_section_with_redirect_to_https(section_name=key,
                                                              holder_pb=holder_pb,
                                                              is_permanent=self.http.redirect_to_https.pb.permanent)
        else:
            check_repliers_are_prefilled, holder_pb = self._prefill_check_repliers(holder_pb)
            allow_ports = ()
            if self.include_domains and version >= VERSION_0_0_5:
                allow_ports = self.get_http_ports()
            self._fill_common_section_modules(holder_pb,
                                              namespace_id=ctx.namespace_id,
                                              refer_shared_uuid=refer_shared_uuid,
                                              domains_set=domains_set,
                                              allow_ports=allow_ports,
                                              is_http_section=True,
                                              fill_check_repliers=not check_repliers_are_prefilled)

    def _fill_get_log_path(self, call_pb, name):
        """
        :type call_pb: awacs.proto.modules_pb2.Call
        :type name: six.text_type
        """
        call_pb.type = modules_pb2.Call.GET_LOG_PATH
        params_pb = call_pb.get_log_path_params
        params_pb.name = name
        fill_get_port_var_call(params_pb.f_port)
        params_pb.default_log_dir = self.LOG_DIR

    def _fill_ssl_sni_context_pb(self, ssl_sni_context_pb, ctx, cert_id, secondary_cert_id, servername_regexp,
                                 verify_client_cert):
        ssl_sni_context_pb.c_cert.id = cert_id
        ssl_sni_context_pb.servername_regexp = servername_regexp
        if verify_client_cert:
            L7MacroHttpsSettingsVerifyClientCert.fill_ssl_sni_context_pb(ssl_sni_context_pb)
        if secondary_cert_id:
            ssl_sni_context_pb.c_secondary_cert.id = secondary_cert_id
        if self.pb.https.compat.HasField('disable_rc4_sha_cipher'):  # TODO: legacy field, superseded by tls_settings
            ssl_sni_context_pb.disable_rc4_sha_cipher.CopyFrom(self.pb.https.compat.disable_rc4_sha_cipher)
        if self.https.tls_settings:
            enable_ecdsa_cipher = ctx.is_ecc_cert(cert_id) or ctx.is_ecc_cert(secondary_cert_id)
            self.https.tls_settings.fill_ssl_sni_context_pb(ssl_sni_context_pb, enable_ecdsa_cipher)

    def _fill_https_section(self, section_pb, ctx, assign_shared_uuid=None, refer_shared_uuid=None,
                            domains_set=None, ecc_certs_are_used=False):
        """
        :type section_pb: modules_pb2.IpdispatchSection
        :type assign_shared_uuid: Optional[six.text_type]
        :type refer_shared_uuid: Optional[six.text_type]
        :type domains_set: DomainsSet
        :type ctx: ValidationCtx
        """
        assert not assign_shared_uuid or not refer_shared_uuid
        version = self.get_version()

        section_pb.ips.add(value=u'*')

        def add_section_ports():
            for port in self.get_https_ports():
                section_pb.ports.add(value=port)

        def maybe_add_instance_port():
            if self.pb.https.compat.bind_on_instance_port:
                port_pb = section_pb.ports.add()
                fill_get_port_var_call(port_pb.f_value)

        if self.pb.https.compat.use_instance_port_in_section_log_name:
            maybe_add_instance_port()
            add_section_ports()
        else:
            add_section_ports()
            maybe_add_instance_port()

        extended_http_macro_pb = section_pb.nested.extended_http_macro
        extended_http_macro_pb.allow_webdav = self.pb.core.allow_webdav

        # https://st.yandex-team.ru/AWACS-601
        # TODO: reading from compat.{maxlen,maxreq} is temporary
        extended_http_macro_pb.maxlen = (self.pb.compat.maxlen.value if self.pb.compat.HasField('maxlen')
                                         else self.DEFAULT_MAXLEN)
        extended_http_macro_pb.maxreq = (self.pb.compat.maxreq.value if self.pb.compat.HasField('maxreq')
                                         else self.DEFAULT_MAXREQ)
        # TODO: in future we will only read from core.limits.{req_line_max_len,req_line_plus_headers_max_len}
        if self.pb.core.limits.HasField('req_line_max_len'):
            extended_http_macro_pb.maxreq = self.pb.core.limits.req_line_max_len.value
        if self.pb.core.limits.HasField('req_line_plus_headers_max_len'):
            extended_http_macro_pb.maxlen = self.pb.core.limits.req_line_plus_headers_max_len.value

        if self.pb.https.compat.use_instance_port_in_section_log_name:
            fill_get_port_var_call(extended_http_macro_pb.f_port)
        if self.pb.core.compat.HasField('keepalive_drop_probability'):
            extended_http_macro_pb.keepalive_drop_probability.CopyFrom(self.pb.core.compat.keepalive_drop_probability)

        if self.monitoring:
            self.monitoring.fill_extended_http_macro_pb(extended_http_macro_pb, uuid=u'https', version=version)
        else:
            L7MacroMonitoringSettings.fill_extended_http_macro_pb_with_defaults(extended_http_macro_pb, version=version)
        if version >= VERSION_0_1_1:
            extended_http_macro_pb.yandex_cookie_policy = extended_http_macro_pb.YCP_STABLE

        extended_http_macro_pb.enable_ssl = True

        if self.pb.https.enable_tlsv1_3:
            extended_http_macro_pb.disable_tlsv1_3.value = False
        if not self.pb.https.compat.enable_sslv3:
            extended_http_macro_pb.disable_sslv3 = True

        if self.https.tls_settings and self.https.tls_settings.has_non_default_preset():
            # these options conflict with "ssl_protocols"
            extended_http_macro_pb.ClearField('disable_sslv3')
            extended_http_macro_pb.ClearField('disable_tlsv1_3')

        if self.pb.https.enable_http2:
            extended_http_macro_pb.enable_http2 = True
            if self.http2:
                self.http2.fill_extended_http_macro_pb_in_https_section(extended_http_macro_pb, version)
            else:
                L7MacroHttp2Settings.fill_extended_http_macro_pb_in_https_section_with_defaults(extended_http_macro_pb, version)

        if version >= VERSION_0_0_2:
            extended_http_macro_pb.ssl_sni_ja3_enabled = True

        if self.include_domains:
            cert_fqdns = OrderedDict()
            secondary_cert_ids = {}
            verify_client_cert_by_cert_id = {}
            for domain_id, domain_config_pb in domains_set.list_common_domains():
                cert_id = dump_string(domain_config_pb.cert.id)
                if cert_id not in cert_fqdns:
                    cert_fqdns[cert_id] = []
                cert_fqdns[cert_id].extend(itertools.chain(domain_config_pb.fqdns, domain_config_pb.shadow_fqdns))
                if domain_config_pb.secondary_cert.id:
                    secondary_cert_ids[cert_id] = dump_string(domain_config_pb.secondary_cert.id)
                verify_client_cert_by_cert_id[cert_id] = domain_config_pb.HasField('verify_client_cert')

            last_ssl_sni_context_pb = None
            for cert_id, fqdns in six.iteritems(cert_fqdns):
                ssl_sni_context_pb = extended_http_macro_pb.ssl_sni_contexts.add(key=cert_id).value
                servername_regexp = dump_string(hosts_to_regexp(fqdns))
                secondary_cert_id = secondary_cert_ids.get(cert_id)
                self._fill_ssl_sni_context_pb(ssl_sni_context_pb, ctx, cert_id, secondary_cert_id, servername_regexp,
                                              self.https.verify_client_cert or verify_client_cert_by_cert_id[cert_id])
                last_ssl_sni_context_pb = ssl_sni_context_pb

            if domains_set.wildcard is not None:
                domain_id, domain_config_pb = domains_set.wildcard
                default_ssl_sni_context_pb = extended_http_macro_pb.ssl_sni_contexts.add(key=u'default').value
                cert_id = dump_string(domain_config_pb.cert.id)
                secondary_cert_id = dump_string(domain_config_pb.secondary_cert.id)
                self._fill_ssl_sni_context_pb(default_ssl_sni_context_pb, ctx, cert_id, secondary_cert_id, 'default',
                                              self.https.verify_client_cert or domain_config_pb.HasField('verify_client_cert'))
            elif last_ssl_sni_context_pb is not None:
                # BALANCER-2829: pginx requires us to specify default ssl sni context,
                # let's specify a random (last) one for now:
                default_ssl_sni_context_pb = extended_http_macro_pb.ssl_sni_contexts.add(key=u'default').value
                default_ssl_sni_context_pb.CopyFrom(last_ssl_sni_context_pb)
                default_ssl_sni_context_pb.servername_regexp = u'default'
        else:
            assert len(self.pb.https.certs) == 1
            cert_id = self.pb.https.certs[0].id
            ssl_sni_context_pb = extended_http_macro_pb.ssl_sni_contexts.add(key=cert_id).value
            ssl_sni_context_pb.c_cert.id = cert_id
            ssl_sni_context_pb.servername_regexp = u'default'
            if self.https.verify_client_cert:
                self.https.verify_client_cert.fill_ssl_sni_context_pb(ssl_sni_context_pb)
            secondary_cert_id = self.pb.https.certs[0].secondary_id
            if secondary_cert_id:
                ssl_sni_context_pb.c_secondary_cert.id = secondary_cert_id
            if self.pb.https.compat.HasField('disable_rc4_sha_cipher'):  # TODO: legacy field
                ssl_sni_context_pb.disable_rc4_sha_cipher.CopyFrom(self.pb.https.compat.disable_rc4_sha_cipher)
            if self.https.tls_settings:
                self.https.tls_settings.fill_ssl_sni_context_pb(ssl_sni_context_pb,
                                                                enable_ecdsa_cipher=ctx.ecc_certs_are_used)

        if assign_shared_uuid:
            shared_pb = extended_http_macro_pb.nested.shared
            shared_pb.uuid = assign_shared_uuid
            holder_pb = shared_pb.nested
        else:
            holder_pb = extended_http_macro_pb.nested

        if version >= VERSION_0_4_0 or self.pb.compat.enable_meta:
            holder_pb.meta.id = META_MODULE_ID
            holder_pb.meta.fields['namespace'] = ctx.namespace_id
            holder_pb.meta.fields['env_type'] = self.balancer_env_type_to_log_record(ctx.balancer_env_type)
            holder_pb = holder_pb.meta.nested

        check_repliers_are_prefilled, holder_pb = self._prefill_check_repliers(holder_pb)
        allow_ports = ()
        if self.include_domains and version >= VERSION_0_0_5:
            allow_ports = self.get_https_ports()
        self._fill_common_section_modules(holder_pb,
                                          namespace_id=ctx.namespace_id,
                                          refer_shared_uuid=refer_shared_uuid,
                                          domains_set=domains_set,
                                          allow_ports=allow_ports,
                                          fill_check_repliers=not check_repliers_are_prefilled)

    def expand(self, ctx=DEFAULT_CTX, preceding_modules=()):
        """
        :param ctx:
        :param preceding_modules:
        """
        version = self.get_version()

        holder_pb = modules_pb2.Holder()
        instance_macro_pb = holder_pb.instance_macro
        instance_macro_pb.f_workers.type = modules_pb2.Call.GET_WORKERS
        instance_macro_pb.f_workers.get_workers_params.SetInParent()
        instance_macro_pb.maxconn = (self.pb.compat.maxconn.value if self.pb.compat.HasField('maxconn')
                                     else self.DEFAULT_MAXCONN)
        if self.pb.compat.tcp_congestion_control:
            instance_macro_pb.tcp_congestion_control = self.pb.compat.tcp_congestion_control
        if self.announce_check_reply and self.announce_check_reply.is_graceful_shutdown_enabled(version=version):
            instance_macro_pb.shutdown_close_using_bpf.value = True
        if version >= VERSION_0_1_1:
            instance_macro_pb.config_check.SetInParent()
        if version >= VERSION_0_2_0 or self.pb.compat.enable_persistent_sd_cache:
            instance_macro_pb.version = '0.0.2'

        if not self.pb.compat.disable_tcp_listen_queue_limit:
            instance_macro_pb.tcp_listen_queue = self.TCP_LISTEN_QUEUE

        if not self.pb.compat.disable_unistat:
            instance_macro_pb.unistat.SetInParent()
            if version >= VERSION_0_2_6 and not self.pb.compat.enable_unistat_legacy_signals:
                instance_macro_pb.unistat.hide_legacy_signals = True

        if not self.pb.compat.disable_sd:
            instance_macro_pb.sd.SetInParent()
            if version >= VERSION_0_3_12 and ctx.balancer_env_type == model_pb2.BalancerSpec.L7_ENV_TESTING:
                instance_macro_pb.sd.allow_empty_endpoint_sets = True
            if VERSION_0_3_14 <= version < VERSION_0_4_0 or version >= VERSION_0_4_3:
                instance_macro_pb.sd.connect_timeout = '1s'
                instance_macro_pb.sd.request_timeout = '3s'

        admin_section_pb = instance_macro_pb.sections.add(key='admin').value
        self._fill_admin_section(admin_section_pb)

        if self.include_domains:
            http_domains, https_domains = ctx.get_separate_http_and_https_domains()
        else:
            http_domains = DomainsSet()
            https_domains = DomainsSet()

        if self.monitoring and self.monitoring.pb.enable_total_signals:
            stats_storage_section_pb = instance_macro_pb.sections.add(key='stats_storage').value
            self.monitoring.fill_stats_storage_section(stats_storage_section_pb, version=version)

        if self.http and self.https and self.https.pb.compat.place_first:
            https_section_pb = instance_macro_pb.sections.add(key='https_section').value
            self._fill_https_section(https_section_pb,
                                     assign_shared_uuid=self.https.pb.compat.assign_shared_uuid,
                                     refer_shared_uuid=self.https.pb.compat.refer_shared_uuid,
                                     domains_set=https_domains,
                                     ctx=ctx,
                                     )

        if self.http:
            http_section_pb = instance_macro_pb.sections.add(key='http_section').value
            self._fill_http_section(http_section_pb,
                                    assign_shared_uuid=self.http.pb.compat.assign_shared_uuid,
                                    refer_shared_uuid=self.http.pb.compat.refer_shared_uuid,
                                    domains_set=http_domains,
                                    ctx=ctx)

        if self.https and not self.https.pb.compat.place_first:
            https_section_pb = instance_macro_pb.sections.add(key='https_section').value
            self._fill_https_section(https_section_pb,
                                     assign_shared_uuid=self.https.pb.compat.assign_shared_uuid,
                                     refer_shared_uuid=self.https.pb.compat.refer_shared_uuid,
                                     domains_set=https_domains,
                                     ctx=ctx,
                                     )

        version = self.get_version()
        if version >= VERSION_0_0_3:
            instance_macro_pb.state_directory = self.STATE_DIRECTORY
            instance_macro_pb.pinger_required = True

        if version >= VERSION_0_0_4:
            self._fill_get_log_path(instance_macro_pb.f_pinger_log, 'pinger_log')
            self._fill_get_log_path(instance_macro_pb.f_dynamic_balancing_log, 'dynamic_balancing_log')

        if version >= VERSION_0_1_0:
            instance_macro_pb.cpu_limiter.disable_file = self.CPU_LIMITER_DISABLE_FILE
            instance_macro_pb.cpu_limiter.active_check_subnet_file = self.CPU_LIMITER_ACTIVE_CHECK_SUBNET_FILE

        return [holder_pb]

    @staticmethod
    def get_uaas_full_backend_ids(use_sd_uaas):
        from awacs.wrappers.main import ExpGetterMacro
        if use_sd_uaas:
            backends = ExpGetterMacro.UAAS_FULL_BACKEND_IDS_VERSION_3
        else:
            backends = ExpGetterMacro.UAAS_FULL_BACKEND_IDS_VERSION_2
        return set(six.itervalues(backends))

    @staticmethod
    def get_antirobot_full_backend_ids():
        from awacs.wrappers.main import AntirobotMacro
        return set(six.itervalues(AntirobotMacro.ANTIROBOT_FULL_SD_BACKEND_IDS))

    def get_rps_limiter_full_backend_ids(self):
        return set(six.itervalues(self.rps_limiter.external.get_full_backend_ids()))

    def get_would_be_included_full_backend_ids(self, current_namespace_id):
        rv = set()
        if any(action.uaas for action in self.headers):
            rv |= self.get_uaas_full_backend_ids(use_sd_uaas=self._use_sd_uaas())
        if self.antirobot:
            rv |= self.get_antirobot_full_backend_ids()
        if self.rps_limiter and self.rps_limiter.external:
            rv |= self.get_rps_limiter_full_backend_ids()
        return rv
