# coding: utf-8
import itertools
import re

import semantic_version
import six
from six.moves import http_client as httplib

from infra.awacs.proto import modules_pb2, model_pb2
from awacs.lib.strutils import to_full_ids
from awacs.model.util import is_upstream_internal

from . import min_component_versions
from .base import ConfigWrapperBase, ModuleWrapperBase, MacroBase, DEFAULT_CTX, add_module, find_module
from .main import AttemptsRateLimiter, Report, InstanceMacro, Main, Compressor
from .errors import ValidationError
from .luautil import dump_string
from .tls_settings import CipherSuite
from .util import (validate, validate_pire_regexp, validate_range, validate_timedeltas,
                   validate_status_codes, validate_timedelta, validate_request_line, validate_version,
                   validate_header_name, validate_item_uniqueness, META_MODULE_ID)
from . import l7macro


BalancerSettingsCompat = modules_pb2.L7UpstreamMacro.BalancerSettings.Compat

GO_TO_ON_ERROR = modules_pb2.L7UpstreamMacro.BalancerSettings.RetryHttpResponses.GO_TO_ON_ERROR
PROXY_RESPONSE_AS_IS = modules_pb2.L7UpstreamMacro.BalancerSettings.RetryHttpResponses.PROXY_RESPONSE_AS_IS

VERSION_0_0_1 = semantic_version.Version(u'0.0.1')
VERSION_0_0_2 = semantic_version.Version(u'0.0.2')
VERSION_0_1_0 = semantic_version.Version(u'0.1.0')
VERSION_0_1_1 = semantic_version.Version(u'0.1.1')
VERSION_0_1_2 = semantic_version.Version(u'0.1.2')
VERSION_0_2_0 = semantic_version.Version(u'0.2.0')
VERSION_0_2_1 = semantic_version.Version(u'0.2.1')
VERSION_0_2_2 = semantic_version.Version(u'0.2.2')
VERSION_0_2_3 = semantic_version.Version(u'0.2.3')
VERSION_0_3_0 = semantic_version.Version(u'0.3.0')
VERSION_0_3_1 = semantic_version.Version(u'0.3.1')
VALID_VERSIONS = frozenset((VERSION_0_0_1, VERSION_0_0_2, VERSION_0_1_0, VERSION_0_1_1,
                            VERSION_0_1_2, VERSION_0_2_0, VERSION_0_2_1, VERSION_0_2_2, VERSION_0_2_3,
                            VERSION_0_3_0, VERSION_0_3_1))
INITIAL_VERSION = VERSION_0_0_1
LATEST_VERSION = VERSION_0_0_2

UUID_RE = r'^[a-z0-9_-]+$'
YEAR_IN_SECONDS = 3600 * 24 * 365


class L7UpstreamMacroCompat(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7UpstreamMacro.Compat

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        with validate('enable_meta'):
            if self.pb.enable_meta:
                ctx.ensure_component_version(
                    model_pb2.ComponentMeta.PGINX_BINARY,
                    min=min_component_versions.META)


class L7UpstreamMacroMonitoringSettings(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7UpstreamMacro.MonitoringSettings

    REQUIRED = ['uuid']

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()

        if not re.match(UUID_RE, self.pb.uuid):
            raise ValidationError(u'must match {}'.format(UUID_RE), u'uuid')

        if self.pb.ranges:
            with validate(u'ranges'):
                validate_timedeltas(self.pb.ranges)

        if self.pb.response_codes:
            with validate(u'response_codes'):
                validate_status_codes(self.pb.response_codes, allow_families=False)

    def fill_report_pb(self, report_pb):
        report_pb.uuid = self.pb.uuid
        report_pb.ranges = self.pb.ranges or Report.DEFAULT_RANGES_ALIAS
        report_pb.outgoing_codes.extend(self.pb.response_codes)

    @classmethod
    def fill_default_report_pb(cls, report_pb, uuid):
        report_pb.uuid = uuid
        report_pb.ranges = Report.DEFAULT_RANGES_ALIAS


class L7UpstreamMacroCompression(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7UpstreamMacro.Compression

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()

        with validate('codecs'):
            validate_item_uniqueness(self.pb.codecs)

            used_codecs = frozenset(self.pb.codecs)
            unknown_codecs = used_codecs - Compressor.COMPRESSION_CODECS_NAMES
            if unknown_codecs:
                raise ValidationError(
                    u'unknown compression codec names: [{}]'.format(
                        ', '.join(sorted(unknown_codecs))))

    def fill_holder_pb(self, holder_pb):
        """
        :type holder_pb: modules_pb2.Holder
        """
        holder_pb.compressor.enable_compression.value = True
        holder_pb.compressor.compression_codecs.extend(self.pb.codecs)


class L7UpstreamMacroBalancerSettingsCompatWatermarkPolicy(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7UpstreamMacro.BalancerSettings.Compat.WatermarkPolicy

    REQUIRED = ['lo', 'hi']

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        if not (0 <= self.pb.lo <= 1):
            raise ValidationError(u'must be between 0 and 1', u'lo')
        if not (0 <= self.pb.hi <= 1):
            raise ValidationError(u'must be between 0 and 1', u'hi')
        if self.pb.lo > self.pb.hi:
            raise ValidationError(u'lo > hi ({} > {}), which is meaningless'.format(self.pb.lo, self.pb.hi))


class L7UpstreamMacroBalancerCompat(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7UpstreamMacro.BalancerSettings.Compat

    watermark_policy = None  # type: L7UpstreamMacroBalancerSettingsCompatWatermarkPolicy | None

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()

        if self.watermark_policy:
            with validate(u'watermark_policy'):
                self.watermark_policy.validate(ctx=ctx, preceding_modules=preceding_modules)


class L7UpstreamMacroBalancerRetryHttpResponses(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7UpstreamMacro.BalancerSettings.RetryHttpResponses

    REQUIRED = ['codes']

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()

        with validate(u'codes'):
            validate_status_codes(self.pb.codes)

        if self.pb.exceptions:
            with validate(u'exceptions'):
                validate_status_codes(self.pb.exceptions)

    def includes_503(self):
        return ((u'5xx' in self.pb.codes or u'503' in self.pb.codes) and
                not (u'5xx' in self.pb.exceptions or u'503' in self.pb.exceptions))


class L7UpstreamMacroHealthCheckCompat(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7UpstreamMacro.HealthCheckSettings.Compat

    def validate(self, compat_method, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        if self.pb.not_steady and compat_method != BalancerSettingsCompat.ACTIVE:
            raise ValidationError(u'can only be set if "compat.method" set to ACTIVE', u'not_steady')


class L7UpstreamMacroHealthCheck(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7UpstreamMacro.HealthCheckSettings

    compat = None  # type: L7UpstreamMacroHealthCheckCompat | None

    REQUIRED = ['delay', 'request']

    def validate(self, compat_method, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()

        if self.compat:
            with validate(u'compat'):
                self.compat.validate(compat_method, ctx=ctx, preceding_modules=preceding_modules)

        with validate(u'delay'):
            validate_timedelta(self.pb.delay)

        with validate(u'request'):
            validate_request_line(self.pb.request)


class L7UpstreamMacroUseHttpsToEndpointsSettings(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7UpstreamMacro.UseHttpsToEndpointsSettings

    REQUIRED = []

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()


class L7UpstreamMacroBalancerSettings(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7UpstreamMacro.BalancerSettings

    REQUIRED = ['backend_timeout']
    REQUIRED_ONEOFS = [('attempts', 'attempt_all_endpoints'),
                       ('do_not_retry_http_responses', 'retry_http_responses'), ]
    OPTIONAL_TIMEOUTS = [
        'connect_timeout',
        'backend_read_timeout',
        'backend_write_timeout',
        'client_read_timeout',
        'client_write_timeout',
        'client_read_timeout',
    ]

    compat = None  # type: L7UpstreamMacroBalancerCompat | None
    health_check = None  # type: L7UpstreamMacroHealthCheck | None
    retry_http_responses = None  # type: L7UpstreamMacroBalancerRetryHttpResponses | None
    use_https_to_endpoints = None  # type: L7UpstreamMacroUseHttpsToEndpointsSettings | None

    DEFAULT_CONNECT_TIMEOUT = u'100ms'
    DEFAULT_QUORUM = 0.35

    MAX_MAX_PESSIMIZED_SHARE = .5
    DYNAMIC_MIN_PESSIMIZATION_COEFF = 0.1
    DYNAMIC_WEIGHT_INCREASE_STEP = 0.1
    DYNAMIC_HISTORY_INTERVAL = u'10s'
    INFINITE_SWITCHED_BACKEND_TIMEOUT = u'{}s'.format(YEAR_IN_SECONDS)

    CA_FILE = u'allCAs.pem'

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()

        if self.compat:
            with validate(u'compat'):
                self.compat.validate(ctx=ctx, preceding_modules=preceding_modules)

        compat_method = self.compat.pb.method if self.compat else BalancerSettingsCompat.METHOD_NONE
        has_watermark_policy = bool(self.compat and self.compat.watermark_policy)

        if ctx.config_type == ctx.CONFIG_TYPE_FULL and compat_method == BalancerSettingsCompat.METHOD_NONE:
            top_level_module = find_module(preceding_modules, (l7macro.L7Macro, InstanceMacro, Main))
            ok = True
            if top_level_module:
                if isinstance(top_level_module, l7macro.L7Macro):
                    ok = semantic_version.Version(top_level_module.pb.version) >= l7macro.VERSION_0_0_3
                elif isinstance(top_level_module, (InstanceMacro, Main)):
                    ok = bool(top_level_module.pb.state_directory)
                else:
                    raise AssertionError()
            if not ok:
                raise ValidationError(
                    u'can only be used without "compat.method" '
                    u'if preceded by instance_macro or main module with state_directory, '
                    u'or l7_macro of version 0.0.3+')

        with validate(u'backend_timeout'):
            validate_timedelta(self.pb.backend_timeout)

        for field in self.OPTIONAL_TIMEOUTS:
            value = getattr(self.pb, field)
            if value:
                with validate(field):
                    validate_timedelta(value)

        if self.pb.max_reattempts_share and self.pb.do_not_limit_reattempts:
            raise ValidationError(u'"max_reattempts_share" and "do_not_limit_reattempts" can not be used together')

        if not has_watermark_policy and (not self.pb.max_reattempts_share and not self.pb.do_not_limit_reattempts):
            raise ValidationError(u'either "max_reattempts_share" or "do_not_limit_reattempts" must be set')

        if has_watermark_policy and (self.pb.max_reattempts_share or self.pb.do_not_limit_reattempts):
            raise ValidationError(u'neither "max_reattempts_share" nor "do_not_limit_reattempts" '
                                  u'must be set if "compat.watermark_policy" is used')

        with validate(u'max_pessimized_endpoints_share'):
            if not compat_method and not self.pb.max_pessimized_endpoints_share:
                raise ValidationError(u'is required')
            if compat_method and self.pb.max_pessimized_endpoints_share:
                raise ValidationError(u'can not be used with "compat.method" set to {}'.format(
                    BalancerSettingsCompat.Method.Name(compat_method)))
            if self.pb.max_pessimized_endpoints_share:
                validate_range(self.pb.max_pessimized_endpoints_share,
                               min_=0,
                               max_=L7UpstreamMacroBalancerSettings.MAX_MAX_PESSIMIZED_SHARE,
                               exclusive_min=True,
                               exclusive_max=False)

        if self.pb.max_reattempts_share:
            with validate(u'max_reattempts_share'):
                validate_range(self.pb.max_reattempts_share,
                               min_=0,
                               max_=AttemptsRateLimiter.MAX_LIMIT,
                               exclusive_min=True,
                               exclusive_max=False)

        with validate(u'health_check'):
            if self.health_check:
                if compat_method not in (BalancerSettingsCompat.METHOD_NONE, BalancerSettingsCompat.ACTIVE):
                    raise ValidationError(u'can not be used with "compat.method" set to {}'.format(
                        BalancerSettingsCompat.Method.Name(self.compat.pb.method)))
                self.health_check.validate(compat_method=compat_method, ctx=ctx, preceding_modules=preceding_modules)
            else:
                if compat_method == BalancerSettingsCompat.ACTIVE:
                    raise ValidationError(u'must be set if "compat.method" set to ACTIVE')

        with validate(u'use_https_to_endpoints'):
            if self.use_https_to_endpoints:
                self.use_https_to_endpoints.validate(ctx=ctx, preceding_modules=preceding_modules)

        if ((self.pb.fast_attempts or self.pb.fast_attempt_all_endpoints) and
                self.pb.fast_attempts_type == self.pb.CONNECT_FAILURE_AND_503):
            if not self.retry_http_responses:
                raise ValidationError(u'must be set if "fast_attempts_type" is CONNECT_FAILURE_AND_503',
                                      u'retry_http_responses')
            if not self.retry_http_responses.includes_503():
                raise ValidationError(u'must include 503 if "fast_attempts_type" is CONNECT_FAILURE_AND_503',
                                      u'retry_http_responses')

        if self.retry_http_responses:
            with validate(u'retry_http_responses'):
                self.retry_http_responses.validate(ctx=ctx, preceding_modules=preceding_modules)

    def _uses_dynamic(self):
        return not self.compat or self.compat.pb.method == self.compat.pb.METHOD_NONE

    def _uses_health_checks(self):
        return self.health_check

    def augment_outer_balancer2(self, balancer2_pb, version):
        """
        :type balancer2_pb: modules_pb2.Balancer2Module
        :type version: semantic_version.Version
        """
        if self.retry_http_responses and self.retry_http_responses.pb.on_last_failed_retry == PROXY_RESPONSE_AS_IS:
            balancer2_pb.status_code_blacklist.extend(self.retry_http_responses.pb.codes)
            balancer2_pb.status_code_blacklist_exceptions.extend(self.retry_http_responses.pb.exceptions)
            balancer2_pb.return_last_5xx = True

        if version >= VERSION_0_2_1:
            balancer2_pb.use_on_error_for_non_idempotent.value = True

    def fill_balancer2(self, balancer2_pb, name, version):
        """
        :type balancer2_pb: modules_pb2.Balancer2Module
        :type name: six.text_type
        :type version: semantic_version.Version
        """
        compat_method = self.compat.pb.method if self.compat else BalancerSettingsCompat.METHOD_NONE
        if compat_method != BalancerSettingsCompat.METHOD_NONE:
            if compat_method == BalancerSettingsCompat.RR:
                balancer2_pb.rr.SetInParent()
            elif compat_method == BalancerSettingsCompat.WEIGHTED2:
                balancer2_pb.weighted2.SetInParent()
            elif compat_method == BalancerSettingsCompat.ACTIVE:
                balancer2_pb.active.SetInParent()
            else:
                raise AssertionError()
        else:
            d_pb = balancer2_pb.dynamic
            d_pb.min_pessimization_coeff = self.DYNAMIC_MIN_PESSIMIZATION_COEFF
            d_pb.weight_increase_step = self.DYNAMIC_WEIGHT_INCREASE_STEP
            d_pb.history_interval = self.DYNAMIC_HISTORY_INTERVAL

        if self.pb.attempts:
            balancer2_pb.attempts = self.pb.attempts
        elif self.pb.attempt_all_endpoints:
            call_pb = balancer2_pb.f_attempts
            call_pb.type = modules_pb2.Call.COUNT_BACKENDS
            call_pb.count_backends_params.compat_enable_sd_support.value = True
        else:
            raise AssertionError()

        if self.pb.fast_attempts or self.pb.fast_attempt_all_endpoints:
            if self.pb.fast_attempts_type == self.pb.CONNECT_FAILURE:
                if self.pb.fast_attempts:
                    balancer2_pb.connection_attempts = self.pb.fast_attempts
                elif self.pb.fast_attempt_all_endpoints:
                    call_pb = balancer2_pb.f_connection_attempts
                    call_pb.type = modules_pb2.Call.COUNT_BACKENDS
                    call_pb.count_backends_params.compat_enable_sd_support.value = True
                else:
                    raise AssertionError()
            elif self.pb.fast_attempts_type == self.pb.CONNECT_FAILURE_AND_503:
                balancer2_pb.fast_503.value = True
                if self.pb.fast_attempts:
                    balancer2_pb.fast_attempts = self.pb.fast_attempts
                elif self.pb.fast_attempt_all_endpoints:
                    call_pb = balancer2_pb.f_fast_attempts
                    call_pb.type = modules_pb2.Call.COUNT_BACKENDS
                    call_pb.count_backends_params.compat_enable_sd_support.value = True
                else:
                    raise AssertionError()
            else:
                raise AssertionError()

        if self.health_check:
            if compat_method == BalancerSettingsCompat.ACTIVE:
                balancer2_pb.active.delay = self.health_check.pb.delay
                balancer2_pb.active.request = self.health_check.pb.request
                if self.health_check.compat and self.health_check.compat.pb.not_steady:
                    balancer2_pb.active.steady.value = False
            else:
                balancer2_pb.dynamic.active.delay = self.health_check.pb.delay
                balancer2_pb.dynamic.active.request = self.health_check.pb.request

        if self.compat and self.compat.watermark_policy:
            wm_pb = balancer2_pb.balancing_policy.watermark_policy
            wm_pb.hi = self.compat.watermark_policy.pb.hi
            wm_pb.lo = self.compat.watermark_policy.pb.lo
            wm_pb.balancing_policy.unique_policy.SetInParent()
        elif self.pb.do_not_limit_reattempts:
            pass
        elif self.pb.max_reattempts_share:
            balancer2_pb.attempts_rate_limiter.limit = self.pb.max_reattempts_share
            balancer2_pb.attempts_rate_limiter.coeff = AttemptsRateLimiter.DEFAULT_COEFF
        else:
            raise AssertionError()

        uses_dynamic = self._uses_dynamic()
        if self.pb.max_pessimized_endpoints_share:
            assert uses_dynamic
            balancer2_pb.dynamic.max_pessimized_share = self.pb.max_pessimized_endpoints_share
        else:
            assert not uses_dynamic

        po_pb = balancer2_pb.generated_proxy_backends.proxy_options

        if self.pb.protocol == self.pb.HTTP1:
            pass
        elif self.pb.protocol == self.pb.HTTP2:
            po_pb.http2_backend = True

        po_pb.fail_on_5xx.value = False
        if self.pb.do_not_retry_http_responses:
            pass
        elif self.retry_http_responses:
            if (self.retry_http_responses.pb.codes == [u'5xx'] and
                    not self.retry_http_responses.pb.exceptions and
                    self.retry_http_responses.pb.on_last_failed_retry == GO_TO_ON_ERROR):
                po_pb.fail_on_5xx.value = True
            else:
                if self.retry_http_responses.pb.on_last_failed_retry == GO_TO_ON_ERROR:
                    po_pb.status_code_blacklist.extend(self.retry_http_responses.pb.codes)
                    po_pb.status_code_blacklist_exceptions.extend(self.retry_http_responses.pb.exceptions)
                elif self.retry_http_responses.pb.on_last_failed_retry == PROXY_RESPONSE_AS_IS:
                    balancer2_pb.status_code_blacklist.extend(self.retry_http_responses.pb.codes)
                    balancer2_pb.status_code_blacklist_exceptions.extend(self.retry_http_responses.pb.exceptions)
                    balancer2_pb.return_last_5xx = True
                else:
                    raise AssertionError()
        else:
            raise AssertionError()

        if self.pb.HasField('retry_non_idempotent'):
            balancer2_pb.retry_non_idempotent.CopyFrom(self.pb.retry_non_idempotent)

        if version >= VERSION_0_2_0:
            balancer2_pb.use_on_error_for_non_idempotent.value = True

        po_pb.connect_timeout = self.pb.connect_timeout or self.DEFAULT_CONNECT_TIMEOUT
        po_pb.backend_timeout = self.pb.backend_timeout

        po_pb.backend_read_timeout = self.pb.backend_read_timeout
        po_pb.backend_write_timeout = self.pb.backend_write_timeout
        po_pb.client_read_timeout = self.pb.client_read_timeout
        po_pb.client_write_timeout = self.pb.client_write_timeout
        po_pb.allow_connection_upgrade = self.pb.allow_connection_upgrade
        po_pb.keepalive_count = self.pb.keepalive_count
        po_pb.buffering = self.pb.buffering
        po_pb.watch_client_close = self.pb.watch_client_close

        if self.compat and self.compat.pb.use_infinite_switched_backend_timeout:
            # see https://st.yandex-team.ru/SWAT-7181 for details
            po_pb.switched_backend_timeout = self.INFINITE_SWITCHED_BACKEND_TIMEOUT

        if self.use_https_to_endpoints:
            po_pb.https_settings.f_ca_file.type = modules_pb2.Call.GET_CA_CERT_PATH
            po_pb.https_settings.f_ca_file.get_ca_cert_path_params.name = self.CA_FILE
            po_pb.https_settings.f_ca_file.get_ca_cert_path_params.default_ca_cert_dir = u'./'
            if not self.use_https_to_endpoints.pb.disable_sni:
                # https://st.yandex-team.ru/AWACS-889
                po_pb.https_settings.sni_on = True
            po_pb.https_settings.verify_depth = 3
            po_pb.https_settings.ciphers = CipherSuite.DEFAULT

        if version >= VERSION_0_1_0 and version < VERSION_0_2_0:
            # https://st.yandex-team.ru/SWAT-6521
            check_backends_pb = balancer2_pb.check_backends
            check_backends_pb.name = name
            check_backends_pb.quorum.value = self.DEFAULT_QUORUM


class L7UpstreamMacroOnError(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7UpstreamMacro.OnError

    static = None  # type: L7UpstreamMacroStaticResponse | None

    REQUIRED_ONEOFS = [('rst', 'static',)]

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()

        if self.static:
            with validate(u'static'):
                self.static.validate(ctx=ctx, preceding_modules=preceding_modules)

    def fill_holder_pb(self, on_error_pb):
        """
        :type balancer2_pb: modules_pb2.Holder
        """
        if self.pb.rst:
            pass
        elif self.static:
            self.static.fill_holder_pb(on_error_pb)
        else:
            raise AssertionError()


class L7UpstreamMacroStaticResponse(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7UpstreamMacro.StaticResponse

    REQUIRED = ['status']

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()

    def fill_holder_pb(self, holder_pb):
        """
        :type holder_pb: modules_pb2.Holder
        """
        holder_pb.errordocument.status = self.pb.status
        holder_pb.errordocument.content = dump_string(self.pb.content)


class L7UpstreamMacroFlatSchemeCompat(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7UpstreamMacro.FlatScheme.Compat

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()


class L7UpstreamMacroFlatScheme(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7UpstreamMacro.FlatScheme

    compat = None  # type: L7UpstreamMacroFlatSchemeCompat | None
    balancer = None  # type: L7UpstreamMacroBalancerSettings | None
    on_error = None  # type: L7UpstreamMacroOnError | None
    on_fast_error = None  # type: L7UpstreamMacroOnError | None

    REQUIRED = ['balancer', 'backend_ids', 'on_error']

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), top_level_can_handle_announce_checks=False):
        self.auto_validate_required()

        if self.compat:
            with validate(u'compat'):
                self.compat.validate(ctx=ctx, preceding_modules=preceding_modules)

        if top_level_can_handle_announce_checks and self.pb.can_handle_announce_checks:
            raise ValidationError(u'is already set at the top level', u'can_handle_announce_checks')

        with validate(u'balancer'):
            self.balancer.validate(ctx=ctx, preceding_modules=preceding_modules)

        with validate(u'on_error'):
            self.on_error.validate(ctx=ctx, preceding_modules=preceding_modules)

        if self.on_fast_error:
            with validate(u'on_fast_error'):
                ctx.ensure_component_version(
                    model_pb2.ComponentMeta.PGINX_BINARY,
                    min=min_component_versions.ON_FAST_ERROR)
                self.on_fast_error.validate(ctx=ctx, preceding_modules=preceding_modules)

    def expand(self, upstream_id, version):
        """
        :type upstream_id: six.text_type
        :type version: semantic_version.Version
        """
        rv = []
        if self.pb.can_handle_announce_checks:
            h_pb = modules_pb2.Holder()
            h_pb.shared.uuid = u'backends'
            rv.append(h_pb)

        h_pb = modules_pb2.Holder()
        self.balancer.fill_balancer2(h_pb.balancer2,
                                     name=upstream_id,
                                     version=version)

        ib_pb = h_pb.balancer2.generated_proxy_backends.include_backends
        ib_pb.type = modules_pb2.IncludeBackends.BY_ID
        ib_pb.ids.extend(self.pb.backend_ids)
        self.on_error.fill_holder_pb(h_pb.balancer2.on_error)
        if self.on_fast_error:
            self.on_fast_error.fill_holder_pb(h_pb.balancer2.on_fast_error)
        rv.append(h_pb)

        return rv


class L7UpstreamMacroDcBalancerSettingsCompat(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7UpstreamMacro.DcBalancerSettings.Compat

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()


class L7UpstreamMacroDcBalancerSettings(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7UpstreamMacro.DcBalancerSettings

    REQUIRED = ['method']
    REQUIRED_ONEOFS = [('attempts', 'attempt_all_dcs')]

    compat = None  # type: L7UpstreamMacroDcBalancerSettingsCompat | None

    DEFAULT_GEO = u'random'
    WEIGHTS_FILE = u'./controls/traffic_control.weights'

    def fill_balancer2(self, balancer2_pb, dcs_count):
        """
        :type balancer2_pb: modules_pb2.Balancer2Module
        :type dcs_count: int
        """
        balancer2_pb.rr.SetInParent()
        if not (self.compat and self.compat.pb.disable_dynamic_weights):
            balancer2_pb.rr.weights_file = self.WEIGHTS_FILE
        if self.pb.attempts:
            balancer2_pb.attempts = self.pb.attempts
        elif self.pb.attempt_all_dcs:
            balancer2_pb.attempts = dcs_count
        else:
            raise AssertionError()
        if self.pb.method == self.pb.BY_DC_WEIGHT:
            balancer2_pb.balancing_policy.unique_policy.SetInParent()
        elif self.pb.method == self.pb.LOCAL_THEN_BY_DC_WEIGHT:
            by_name_policy_pb = balancer2_pb.balancing_policy.by_name_policy
            f_name_pb = by_name_policy_pb.f_name
            f_name_pb.type = f_name_pb.GET_GEO
            f_name_pb.get_geo_params.name = self.pb.weights_section_id + u'_'
            f_name_pb.get_geo_params.default_geo = self.DEFAULT_GEO
            by_name_policy_pb.balancing_policy.unique_policy.SetInParent()
        else:
            raise AssertionError()

    def validate(self, dcs_count, ctx=DEFAULT_CTX, preceding_modules=()):
        """
        :type dcs_count: int
        """
        self.auto_validate_required()

        if self.compat:
            with validate(u'compat'):
                self.compat.validate(ctx=ctx, preceding_modules=preceding_modules)

        if self.pb.attempts > dcs_count:
            raise ValidationError(u'exceeds the number of configured dcs ({})'.format(dcs_count), u'attempts')

        if not self.pb.weights_section_id and not (self.compat and self.compat.pb.disable_dynamic_weights):
            raise ValidationError(u'is required', u'weights_section_id')


class L7UpstreamMacroDevnullCompat(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7UpstreamMacro.Devnull.Compat


class L7UpstreamMacroDevnull(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7UpstreamMacro.Devnull

    compat = None  # type: L7UpstreamMacroDevnullCompat | None
    monitoring = None  # type: L7UpstreamMacroMonitoringSettings | None
    static = None  # type: L7UpstreamMacroStaticResponse | None

    REQUIRED_ONEOFS = [('static',)]

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), top_level_disable_monitoring=False):
        self.auto_validate_required()

        if self.compat:
            with validate(u'compat'):
                self.compat.validate(ctx=ctx, preceding_modules=preceding_modules)

        if self.monitoring:
            with validate(u'monitoring'):
                if top_level_disable_monitoring or (self.compat and self.compat.pb.disable_monitoring):
                    raise ValidationError(u'can not be used with "compat.disable_monitoring"')
                self.monitoring.validate(ctx=ctx, preceding_modules=preceding_modules)

        if self.static:
            with validate(u'static'):
                self.static.validate(ctx=ctx, preceding_modules=preceding_modules)

    @classmethod
    def _get_default_report_uuid(cls, upstream_id):
        """
        :type upstream_id: six.text_type
        """
        return u'requests_' + upstream_id + u'_to_devnull'

    @classmethod
    def fill_holder_pb_with_default_config(cls, holder_pb, upstream_id, top_level_disable_monitoring=False):
        """
        :type holder_pb: modules_pb2.Holder
        :type upstream_id: six.text_type
        :type top_level_disable_monitoring: bool
        """
        if top_level_disable_monitoring:
            pass
        else:
            holder_pb.report.uuid = cls._get_default_report_uuid(upstream_id)
            holder_pb.report.ranges = Report.DEFAULT_RANGES_ALIAS
            holder_pb = holder_pb.report.nested
        holder_pb.errordocument.status = httplib.NO_CONTENT

    def fill_holder_pb(self, holder_pb, upstream_id, top_level_disable_monitoring=False):
        """
        :type holder_pb: modules_pb2.Holder
        :type upstream_id: six.text_type
        :type top_level_disable_monitoring: bool
        """
        if top_level_disable_monitoring or (self.compat and self.compat.pb.disable_monitoring):
            pass
        else:
            report_pb = holder_pb.report
            if self.monitoring:
                self.monitoring.fill_report_pb(report_pb)
            else:
                L7UpstreamMacroMonitoringSettings.fill_default_report_pb(
                    report_pb, uuid=self._get_default_report_uuid(upstream_id))
            holder_pb = holder_pb.report.nested

        if self.static:
            self.static.fill_holder_pb(holder_pb)
        else:
            raise AssertionError()


class L7UpstreamMacroDcCompat(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7UpstreamMacro.Dc.Compat

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()


class L7UpstreamMacroDc(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7UpstreamMacro.Dc

    REQUIRED = ['name', 'backend_ids']

    ALLOWED_DC_NAMES = (u'sas', u'man', u'vla', u'iva', u'myt')

    compat = None  # type: L7UpstreamMacroDcCompat | None
    monitoring = None  # type: L7UpstreamMacroMonitoringSettings | None

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), top_level_disable_monitoring=False):
        self.auto_validate_required()

        if self.compat:
            with validate(u'compat'):
                self.compat.validate(ctx=ctx, preceding_modules=preceding_modules)

        if self.pb.name not in self.ALLOWED_DC_NAMES:
            raise ValidationError(u'is not allowed', u'name')

        if self.pb.weight and self.pb.weight not in (-1, 1):
            raise ValidationError(u'must be either 1 or -1', u'weight')

        if self.monitoring:
            with validate(u'monitoring'):
                if top_level_disable_monitoring or (self.compat and self.compat.pb.disable_monitoring):
                    raise ValidationError(u'can not be used with "compat.disable_monitoring"')

                self.monitoring.validate(ctx=ctx, preceding_modules=preceding_modules)


class L7UpstreamMacroByDcSchemeCompat(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7UpstreamMacro.ByDcScheme.Compat

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()


class L7UpstreamMacroByDcScheme(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7UpstreamMacro.ByDcScheme

    REQUIRED = ['dc_balancer', 'balancer', 'dcs', 'on_error']

    compat = None  # type: L7UpstreamMacroByDcSchemeCompat | None
    dc_balancer = None  # type: L7UpstreamMacroDcBalancerSettings | None
    balancer = None  # type: L7UpstreamMacroBalancerSettings | None
    dcs = ()  # type: collections.Iterable[L7UpstreamMacroDc]
    on_error = None  # type: L7UpstreamMacroOnError | None
    on_fast_error = None  # type: L7UpstreamMacroOnError | None
    devnull = None  # type: L7UpstreamMacroDevnull | None

    DEFAULT_DC_WEIGHT = 1

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(),
                 top_level_can_handle_announce_checks=False,
                 top_level_disable_monitoring=False):
        self.auto_validate_required()

        if self.compat:
            with validate(u'compat'):
                self.compat.validate(ctx=ctx, preceding_modules=preceding_modules)

        if top_level_can_handle_announce_checks and self.pb.can_handle_announce_checks:
            raise ValidationError(u'is already set at the top level', u'can_handle_announce_checks')

        with validate(u'dc_balancer'):
            self.dc_balancer.validate(dcs_count=len(self.dcs), ctx=ctx, preceding_modules=preceding_modules)

        with validate(u'balancer'):
            self.balancer.validate(ctx=ctx, preceding_modules=preceding_modules)

        seen_dc_names = set()
        for i, dc in enumerate(self.dcs):
            with validate(u'dcs[{}]'.format(i)):
                dc.validate(ctx=ctx, preceding_modules=preceding_modules,
                            top_level_disable_monitoring=top_level_disable_monitoring)
                if dc.pb.name in seen_dc_names:
                    raise ValidationError(u'duplicate name "{}"'.format(dc.pb.name), u'name')
                seen_dc_names.add(dc.pb.name)

        with validate(u'on_error'):
            self.on_error.validate(ctx=ctx, preceding_modules=preceding_modules)

        if self.on_fast_error:
            with validate(u'on_fast_error'):
                ctx.ensure_component_version(
                    model_pb2.ComponentMeta.PGINX_BINARY,
                    min=min_component_versions.ON_FAST_ERROR)
                self.on_fast_error.validate(ctx=ctx, preceding_modules=preceding_modules)

        if self.devnull:
            with validate(u'devnull'):
                if self._is_devnull_disabled():
                    raise ValidationError(u'can not be used with "compat.disable_devnull"')
                self.devnull.validate(ctx=ctx, preceding_modules=preceding_modules,
                                      top_level_disable_monitoring=top_level_disable_monitoring)

    def _is_devnull_disabled(self):
        return self.compat and self.compat.pb.disable_devnull

    def expand(self, upstream_id, version, weight_section_spec_pb=None, top_level_disable_monitoring=False):
        """
        :type upstream_id: six.text_type
        :type version: semantic_version.Version
        :type weight_section_spec_pbs: model_pb2.WeightSectionSpec | None
        :type top_level_disable_monitoring: bool
        """
        rv = []

        if self.pb.can_handle_announce_checks:
            h_pb = modules_pb2.Holder()
            h_pb.shared.uuid = u'backends'
            rv.append(h_pb)

        h_pb = modules_pb2.Holder()
        rv.append(h_pb)
        balancer2_pb = h_pb.balancer2
        self.dc_balancer.fill_balancer2(balancer2_pb, dcs_count=len(self.dcs))
        self.balancer.augment_outer_balancer2(balancer2_pb, version)

        for dc in self.dcs:
            weight = dc.pb.weight or self.DEFAULT_DC_WEIGHT
            if weight_section_spec_pb:
                for location in weight_section_spec_pb.locations:
                    if dc.pb.name == location.name.lower():
                        weight = location.default_weight
                        break

            dc_backend_pb = balancer2_pb.backends.add(
                name=self.dc_balancer.pb.weights_section_id + u'_' + dc.pb.name,
                weight=weight)
            dc_module_pbs = dc_backend_pb.nested.modules

            if top_level_disable_monitoring or (dc.compat and dc.compat.pb.disable_monitoring):
                pass
            else:
                report_pb = dc_module_pbs.add().report
                if dc.monitoring:
                    dc.monitoring.fill_report_pb(report_pb)
                else:
                    L7UpstreamMacroMonitoringSettings.fill_default_report_pb(
                        report_pb, uuid=u'requests_' + upstream_id + u'_to_' + dc.pb.name)

            dc_balancer2_pb = dc_module_pbs.add().balancer2
            self.balancer.fill_balancer2(dc_balancer2_pb,
                                         name=upstream_id + u'#' + dc.pb.name,
                                         version=version)

            ib_pb = dc_balancer2_pb.generated_proxy_backends.include_backends
            ib_pb.type = modules_pb2.IncludeBackends.BY_ID
            ib_pb.ids.extend(dc.pb.backend_ids)

        disable_devnull = self._is_devnull_disabled()
        if not disable_devnull:
            dc_backend_pb = balancer2_pb.backends.add(
                name=self.dc_balancer.pb.weights_section_id + u'_devnull',
                weight=-1)
            if self.devnull:
                self.devnull.fill_holder_pb(
                    dc_backend_pb.nested, upstream_id=upstream_id,
                    top_level_disable_monitoring=top_level_disable_monitoring)
            else:
                L7UpstreamMacroDevnull.fill_holder_pb_with_default_config(
                    dc_backend_pb.nested, upstream_id=upstream_id,
                    top_level_disable_monitoring=top_level_disable_monitoring)

        self.on_error.fill_holder_pb(balancer2_pb.on_error)
        if self.on_fast_error:
            self.on_fast_error.fill_holder_pb(balancer2_pb.on_fast_error)

        return rv


class L7UpstreamMacroSplitRouteCompat(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7UpstreamMacro.SplitRoute.Compat

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()


class L7UpstreamMacroSplitRoute(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7UpstreamMacro.SplitRoute

    REQUIRED = ['name', 'upstream_id']

    compat = None  # type: L7UpstreamMacroSplitRouteCompat | None

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()

        if self.compat:
            with validate(u'compat'):
                self.compat.validate(ctx=ctx, preceding_modules=preceding_modules)


class L7UpstreamMacroTrafficSplitCompat(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7UpstreamMacro.TrafficSplit.Compat

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()


class L7UpstreamMacroTrafficSplit(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7UpstreamMacro.TrafficSplit

    WEIGHTS_FILE = L7UpstreamMacroDcBalancerSettings.WEIGHTS_FILE
    REQUIRED = ['attempts', 'routes']

    compat = None  # type: L7UpstreamMacroTrafficSplitCompat | None
    routes = ()  # type: collections.Iterable[L7UpstreamMacroSplitRoute]

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()

        if self.compat:
            with validate(u'compat'):
                self.compat.validate(ctx=ctx, preceding_modules=preceding_modules)

        if self.pb.attempts > len(self.routes):
            raise ValidationError(u'exceeds the number of configured routes ({})'.format(len(self.routes)), u'attempts')

        if not self.pb.weights_section_id and not (self.compat and self.compat.pb.disable_dynamic_weights):
            raise ValidationError(u'is required', u'weights_section_id')

        seen_route_names = set()
        for i, route in enumerate(self.routes):
            with validate(u'routes[{}]'.format(i)):
                route.validate(ctx=ctx, preceding_modules=preceding_modules)
                if route.pb.name in seen_route_names:
                    raise ValidationError(u'duplicate name "{}"'.format(route.pb.name), u'name')
                seen_route_names.add(route.pb.name)

                if not is_upstream_internal(route.pb.upstream_id):
                    raise ValidationError(u'"{}" is not an internal upstream'.format(route.pb.upstream_id), u'upstream_id')

                if ctx.config_type != ctx.CONFIG_TYPE_FULL:
                    continue
                upstream_spec_pb = ctx.upstream_spec_pbs.get((ctx.namespace_id, route.pb.upstream_id))
                if upstream_spec_pb is None:
                    raise ValidationError(u'upstream "{}" not found'.format(route.pb.upstream_id), u'upstream_id')
                if not upstream_spec_pb.yandex_balancer.config.HasField('l7_upstream_macro'):
                    raise ValidationError(u'upstream "{}" must contain l7_upstream_macro'.format(route.pb.upstream_id),
                                          u'upstream_id')

    def fill_balancer2(self, balancer2_pb):
        """
        :type balancer2_pb: modules_pb2.Balancer2Module
        """
        balancer2_pb.rr.SetInParent()
        if not (self.compat and self.compat.pb.disable_dynamic_weights):
            balancer2_pb.rr.weights_file = self.WEIGHTS_FILE
        balancer2_pb.attempts = self.pb.attempts
        balancer2_pb.balancing_policy.unique_policy.SetInParent()

    def expand(self, namespace_id, upstream_spec_pbs, weight_section_spec_pb=None):
        """
        :type upstream_id: six.text_type
        :type version: semantic_version.Version
        :type weight_section_spec_pbs: model_pb2.WeightSectionSpec | None
        """
        rv = []
        h_pb = modules_pb2.Holder()
        rv.append(h_pb)
        balancer2_pb = h_pb.balancer2
        self.fill_balancer2(balancer2_pb)

        for route in self.routes:
            weight = 1
            if weight_section_spec_pb:
                for location in weight_section_spec_pb.locations:
                    if route.pb.name == location.name.lower():
                        weight = location.default_weight
                        break

            route_backend_pb = balancer2_pb.backends.add(
                name=self.pb.weights_section_id + u'_' + route.pb.name,
                weight=weight)
            upstream_spec_pb = upstream_spec_pbs.get((namespace_id, route.pb.upstream_id))
            route_backend_pb.nested.l7_upstream_macro.CopyFrom(upstream_spec_pb.yandex_balancer.config.l7_upstream_macro)
        return rv


class L7UpstreamMacroMatcherHeader(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7UpstreamMacro.Matcher.Header

    REQUIRED = ['name', 're']

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        with validate(u'name'):
            validate_header_name(self.pb.name)
        with validate(u're'):
            validate_pire_regexp(self.pb.re, lua_unescape_first=False)


class L7UpstreamMacroMatcher(ConfigWrapperBase):
    __protobuf__ = modules_pb2.L7UpstreamMacro.Matcher

    header = None  # type: L7UpstreamMacroMatcherHeader | None
    and_ = []  # type: list[L7UpstreamMacroMatcher]
    or_ = []  # type: list[L7UpstreamMacroMatcher]
    not_ = None  # type: L7UpstreamMacroMatcher | None

    REQUIRED_ONEOFS = [
        ('any', 'url_re', 'path_re', 'uri_re', 'host_re', 'cgi_re', 'header', 'method', 'and_', 'or_', 'not_')
    ]

    # As defined by https://tools.ietf.org/html/rfc7231
    # plus PATCH
    # minus TRACE (because it requires allow_trace flag in http module)
    ALLOWED_METHODS = (u'GET', u'HEAD', u'POST', u'PUT', u'PATCH', u'DELETE', u'CONNECT', u'OPTIONS')

    def validate(self, upstream_id, ctx=DEFAULT_CTX, preceding_modules=()):
        """
        :type upstream_id: six.text_type
        """
        self.auto_validate_required()

        if upstream_id == u'default' and not self.pb.any:
            with validate(u'any'):
                raise ValidationError(u'must be set if upstream is "{}"'.format(upstream_id))

        if self.pb.url_re:
            with validate(u'url_re'):
                validate_pire_regexp(self.pb.url_re, lua_unescape_first=False)
        if self.pb.path_re:
            with validate(u'path_re'):
                validate_pire_regexp(self.pb.path_re, lua_unescape_first=False)
        if self.pb.uri_re:
            with validate(u'uri_re'):
                validate_pire_regexp(self.pb.uri_re, lua_unescape_first=False)
        if self.pb.host_re:
            with validate(u'host_re'):
                validate_pire_regexp(self.pb.host_re, lua_unescape_first=False)
        if self.pb.cgi_re:
            with validate(u'cgi_re'):
                validate_pire_regexp(self.pb.cgi_re, lua_unescape_first=False)
        if self.header:
            with validate(u'header'):
                self.header.validate(ctx=ctx, preceding_modules=preceding_modules)
        if self.pb.method:
            if self.pb.method not in self.ALLOWED_METHODS:
                raise ValidationError(u'must be one of the following: {}'.format(
                    u', '.join(self.ALLOWED_METHODS), u'method'))

        for i, matcher in enumerate(self.and_):
            with validate(u'and_[{}]'.format(i)):
                matcher.validate(upstream_id=upstream_id, ctx=ctx, preceding_modules=preceding_modules)
                if matcher.contains_any():
                    raise ValidationError(u'nested matcher must not contain "any: true"')
        for i, matcher in enumerate(self.or_):
            with validate(u'or_[{}]'.format(i)):
                matcher.validate(upstream_id=upstream_id, ctx=ctx, preceding_modules=preceding_modules)
                if matcher.contains_any():
                    raise ValidationError(u'nested matcher must not contain "any: true"')
        if self.not_:
            with validate(u'not_'):
                self.not_.validate(upstream_id=upstream_id, ctx=ctx, preceding_modules=preceding_modules)
                if self.not_.contains_any():
                    raise ValidationError(u'nested matcher must not contain "any: true"')

    def fill_matcher(self, matcher_pb):
        """
        :type matcher_pb: modules_pb2.Matcher
        """
        if self.pb.any:
            matcher_pb.SetInParent()
        elif self.pb.url_re:
            matcher_pb.match_fsm.url = dump_string(self.pb.url_re)
        elif self.pb.path_re:
            matcher_pb.match_fsm.path = dump_string(self.pb.path_re)
        elif self.pb.uri_re:
            matcher_pb.match_fsm.uri = dump_string(self.pb.uri_re)
        elif self.pb.host_re:
            matcher_pb.match_fsm.host = dump_string(self.pb.host_re)
        elif self.pb.cgi_re:
            matcher_pb.match_fsm.cgi = dump_string(self.pb.cgi_re)
        elif self.header:
            matcher_pb.match_fsm.header.name = dump_string(self.header.pb.name)
            matcher_pb.match_fsm.header.value = dump_string(self.header.pb.re)
        elif self.pb.method:
            matcher_pb.match_method.methods.append(self.pb.method.lower())
        elif self.and_:
            for matcher in self.and_:
                matcher.fill_matcher(matcher_pb.match_and.add())
        elif self.or_:
            for matcher in self.or_:
                matcher.fill_matcher(matcher_pb.match_or.add())
        elif self.not_:
            self.not_.fill_matcher(matcher_pb.match_not)
        else:
            raise AssertionError()

    def contains_any(self):
        return (self.pb.any or
                (self.not_ and self.not_.contains_any()) or
                any(matcher.contains_any() for matcher in self.and_) or
                any(matcher.contains_any() for matcher in self.or_))


class L7UpstreamMacro(ModuleWrapperBase, MacroBase):
    __protobuf__ = modules_pb2.L7UpstreamMacro

    compat = None  # type: L7UpstreamMacroCompat | None
    matcher = None  # type: L7UpstreamMacroMatcher | None
    monitoring = None  # type: L7UpstreamMacroMonitoringSettings | None
    rewrite = ()  # type: collections.Iterable[l7macro.L7MacroRewriteAction]
    rps_limiter = None  # type: l7macro.L7MacroRpsLimiterSettings or None
    headers = ()  # type: collections.Iterable[l7macro.L7MacroHeaderAction]
    response_headers = ()  # type: collections.Iterable[l7macro.L7MacroHeaderAction]
    flat_scheme = None  # type: L7UpstreamMacroFlatScheme | None
    by_dc_scheme = None  # type: L7UpstreamMacroByDcScheme | None
    traffic_split = None  # type: L7UpstreamMacroTrafficSplit | None
    static_response = None  # type: L7UpstreamMacroStaticResponse | None
    compression = None # type: L7UpstreamMacroCompression | None

    HEADERS_MAX_COUNT = 10
    REWRITE_MAX_COUNT = 10

    REQUIRED = ['version', 'id']
    REQUIRED_ONEOFS = [('flat_scheme', 'by_dc_scheme', 'traffic_split', 'static_response')]

    def get_version(self):
        if self.pb.version:
            return semantic_version.Version(self.pb.version)
        else:
            return INITIAL_VERSION

    def _validate_rewrite(self, ctx=DEFAULT_CTX, preceding_modules=()):
        if len(self.rewrite) >= self.REWRITE_MAX_COUNT:
            raise ValidationError(u'exceeds allowed limit of {} actions'.format(self.REWRITE_MAX_COUNT), u'rewrite')
        for i, action in enumerate(self.rewrite):
            with validate(u'rewrite[{}]'.format(i)):
                action.validate(ctx=ctx, preceding_modules=preceding_modules)

    def _validate_headers(self, ctx=DEFAULT_CTX, preceding_modules=()):
        if len(self.headers) >= self.HEADERS_MAX_COUNT:
            raise ValidationError(u'exceeds allowed limit of {} actions'.format(self.HEADERS_MAX_COUNT), u'headers')
        for i, action in enumerate(self.headers):
            with validate(u'headers[{}]'.format(i)):
                action.validate(mode=l7macro.Mode.HEADERS, ctx=ctx, preceding_modules=preceding_modules)

    def _validate_response_headers(self, ctx=DEFAULT_CTX, preceding_modules=()):
        if len(self.response_headers) >= self.HEADERS_MAX_COUNT:
            raise ValidationError(u'exceeds allowed limit of {} actions'.format(self.HEADERS_MAX_COUNT),
                                  u'response_headers')
        for i, action in enumerate(self.response_headers):
            with validate(u'response_headers[{}]'.format(i)):
                action.validate(mode=l7macro.Mode.RESPONSE_HEADERS, ctx=ctx, preceding_modules=preceding_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        validate_version(self.pb.version, VALID_VERSIONS, field_name='version')

        if self.get_version() >= VERSION_0_3_0 or self.pb.compat.enable_meta:
            with validate('version'):
                ctx.ensure_component_version(
                    model_pb2.ComponentMeta.PGINX_BINARY,
                    min=min_component_versions.META)
        new_preceding_modules = add_module(preceding_modules, self)

        if not re.match(UUID_RE, self.pb.id):
            raise ValidationError(u'must match {}'.format(UUID_RE), u'id')

        if self.compat:
            with validate(u'compat'):
                self.compat.validate(ctx=ctx, preceding_modules=new_preceding_modules)

        if ctx.upstream_type == model_pb2.YandexBalancerUpstreamSpec.INTERNAL:
            if self.pb.HasField('matcher'):
                raise ValidationError(u'"matcher": must not be set')
        else:
            if not self.pb.HasField('matcher'):
                raise ValidationError(u'"matcher": must be set')

        if self.matcher:
            with validate(u'matcher'):
                self.matcher.validate(upstream_id=self.pb.id, ctx=ctx, preceding_modules=new_preceding_modules)

        disable_monitoring = bool(self.compat and self.compat.pb.disable_monitoring)

        if self.monitoring:
            with validate(u'monitoring'):
                if disable_monitoring:
                    raise ValidationError(u'can not be used with "compat.disable_monitoring"')
                self.monitoring.validate(ctx=ctx, preceding_modules=new_preceding_modules)

        if self.rps_limiter:
            with validate(u'rps_limiter'):
                self.rps_limiter.validate(ctx=ctx, preceding_modules=new_preceding_modules)
        self._validate_rewrite(ctx=ctx, preceding_modules=new_preceding_modules)
        self._validate_headers(ctx=ctx, preceding_modules=new_preceding_modules)
        self._validate_response_headers(ctx=ctx, preceding_modules=new_preceding_modules)

        if self.flat_scheme:
            with validate(u'flat_scheme'):
                self.flat_scheme.validate(ctx=ctx, preceding_modules=new_preceding_modules,
                                          top_level_can_handle_announce_checks=self.pb.can_handle_announce_checks)

        if self.by_dc_scheme:
            with validate(u'by_dc_scheme'):
                self.by_dc_scheme.validate(ctx=ctx, preceding_modules=new_preceding_modules,
                                           top_level_disable_monitoring=disable_monitoring,
                                           top_level_can_handle_announce_checks=self.pb.can_handle_announce_checks)

        if self.traffic_split:
            if is_upstream_internal(self.pb.id):
                raise ValidationError(u'traffic_split can not be used in internal upstream')
            with validate(u'traffic_split'):
                self.traffic_split.validate(ctx=ctx, preceding_modules=new_preceding_modules)

        if self.static_response:
            with validate(u'static_response'):
                self.static_response.validate(ctx=ctx, preceding_modules=new_preceding_modules)

        if self.compression:
            with validate(u'compression'):
                self.compression.validate(ctx=ctx, preceding_modules=new_preceding_modules)

        for i, action in enumerate(self.headers):
            with validate('headers[{}]'.format(i)):
                if action.kind == 'decrypt_icookie':
                    raise ValidationError(
                        'the `decrypt_icookie` action is not available in l7_upstream_macro')

    def includes_backends(self):
        return False

    def would_include_backends(self):
        return True

    def get_would_be_included_full_backend_ids(self, current_namespace_id):
        flat_backend_ids = set()
        if self.flat_scheme:
            flat_backend_ids = set(self.flat_scheme.pb.backend_ids)
        elif self.by_dc_scheme:
            for dc in self.by_dc_scheme.dcs:
                flat_backend_ids.update(dc.pb.backend_ids)
        rv = set(to_full_ids(current_namespace_id, flat_backend_ids))
        if any(action.uaas for action in self.headers):
            rv.update(l7macro.L7Macro.get_uaas_full_backend_ids(use_sd_uaas=self._use_sd_uaas()))
        if self.rps_limiter and self.rps_limiter.external:
            rv.update(six.itervalues(self.rps_limiter.external.get_full_backend_ids()))
        return rv

    def would_include_weight_sections(self):
        return True

    def get_would_be_included_full_weight_section_ids(self, namespace_id):
        """
        :type namespace_id: str
        :rtype: set[(str, str)]
        """
        rv = set()
        if self.by_dc_scheme and self.by_dc_scheme.dc_balancer.pb.weights_section_id:
            rv.add((namespace_id, self.by_dc_scheme.dc_balancer.pb.weights_section_id))
        if self.traffic_split and self.traffic_split.pb.weights_section_id:
            rv.add((namespace_id, self.traffic_split.pb.weights_section_id))
        return rv

    def would_include_internal_upstreams(self):
        return True

    def get_would_be_included_full_internal_upstream_ids(self, namespace_id):
        """
        :type namespace_id: str
        :rtype: set[(str, str)]
        """
        rv = set()
        if self.traffic_split:
            for route in self.traffic_split.routes:
                rv.add((namespace_id, route.pb.upstream_id))
        return rv

    def to_regexp_section_pb(self):
        """
        :rtype: modules_pb2.RegexpSection
        """
        pb = modules_pb2.RegexpSection()
        self.matcher.fill_matcher(pb.matcher)
        pb.nested.l7_upstream_macro.CopyFrom(self.pb)
        return pb

    def _use_sd_uaas(self):
        version = self.get_version()
        return version >= VERSION_0_1_2 or VERSION_0_0_2 <= version < VERSION_0_1_0

    def expand(self, ctx=DEFAULT_CTX, preceding_modules=()):
        # we can safely ignore our .matcher here,
        # it was handled by to_regexp_section_pb
        version = self.get_version()

        shared_h_pb = None
        if self.pb.can_handle_announce_checks:
            shared_h_pb = modules_pb2.Holder()
            shared_h_pb.shared.uuid = u'backends'

        threshold_h_pb = None
        if self.compat:
            if self.compat.pb.threshold_profile == self.compat.pb.THRESHOLD_PROFILE_NONE:
                pass
            elif self.compat.pb.threshold_profile == self.compat.pb.THRESHOLD_PROFILE_CORE_MAPS:
                threshold_h_pb = modules_pb2.Holder()
                threshold_h_pb.threshold.lo_bytes = 734003
                threshold_h_pb.threshold.hi_bytes = 838860
                threshold_h_pb.threshold.pass_timeout = u'10s'
                threshold_h_pb.threshold.recv_timeout = u'1s'
            else:
                raise AssertionError()

        report_h_pb = None
        disable_monitoring = bool(self.compat and self.compat.pb.disable_monitoring)
        if disable_monitoring:
            pass
        else:
            report_h_pb = modules_pb2.Holder()
            if self.monitoring:
                self.monitoring.fill_report_pb(report_h_pb.report)
            else:
                L7UpstreamMacroMonitoringSettings.fill_default_report_pb(report_h_pb.report, uuid=self.pb.id)

        rewrite_h_pb = None
        if self.rewrite:
            rewrite_h_pb = modules_pb2.Holder()
            rewrite_h_pb.rewrite.CopyFrom(l7macro.to_rewrite_module_pb(self.rewrite))

        rps_limiter_h_pb = None
        if self.rps_limiter:
            rps_limiter_macro_version = None
            if VERSION_0_2_3 <= version < VERSION_0_3_0 or version >= VERSION_0_3_1:
                rps_limiter_macro_version = '0.0.3'

            rps_limiter_h_pb = modules_pb2.Holder()
            self.rps_limiter.fill_holder_pb(rps_limiter_h_pb, rps_limiter_macro_version)

        header_pbs = []
        if self.headers:
            geobase_macro_version = '0.0.1'
            if version >= VERSION_0_2_2:
                geobase_macro_version = '0.0.3'
            for header_module_pb in l7macro.to_header_module_pbs(self.headers, mode=l7macro.Mode.HEADERS,
                                                                 use_sd_uaas=self._use_sd_uaas(),
                                                                 geobase_macro_version=geobase_macro_version):
                h_pb = modules_pb2.Holder()
                l7macro.L7Macro.assign_header_module_to_holder(h_pb, header_module_pb)
                header_pbs.append(h_pb)

        response_header_pbs = []
        if self.response_headers:
            for header_module_pb in reversed(l7macro.to_header_module_pbs(self.response_headers,
                                                                          mode=l7macro.Mode.RESPONSE_HEADERS)):
                h_pb = modules_pb2.Holder()
                l7macro.L7Macro.assign_header_module_to_holder(h_pb, header_module_pb)
                response_header_pbs.append(h_pb)

        balancer_pbs = []
        if self.flat_scheme:
            balancer_pbs.extend(self.flat_scheme.expand(upstream_id=self.pb.id, version=version))
        elif self.by_dc_scheme:
            weight_section_spec_pb = ctx.weight_section_spec_pbs.get((ctx.namespace_id, self.by_dc_scheme.dc_balancer.pb.weights_section_id))
            balancer_pbs.extend(self.by_dc_scheme.expand(upstream_id=self.pb.id,
                                                         version=version,
                                                         top_level_disable_monitoring=disable_monitoring,
                                                         weight_section_spec_pb=weight_section_spec_pb))
        elif self.traffic_split:
            weight_section_spec_pb = ctx.weight_section_spec_pbs.get((ctx.namespace_id, self.traffic_split.pb.weights_section_id))
            balancer_pbs.extend(self.traffic_split.expand(ctx.namespace_id, ctx.upstream_spec_pbs,
                                                          weight_section_spec_pb=weight_section_spec_pb))
        elif self.static_response:
            h_pb = modules_pb2.Holder()
            self.static_response.fill_holder_pb(h_pb)
            balancer_pbs.append(h_pb)
        else:
            raise AssertionError()

        compression_h_pb = None
        if self.compression is not None:
            compression_h_pb = modules_pb2.Holder()
            self.compression.fill_holder_pb(compression_h_pb)

        if rps_limiter_h_pb is not None or compression_h_pb is not None or version >= VERSION_0_1_1:
            # newer and more correct module order
            module_pbs = (report_h_pb, compression_h_pb, threshold_h_pb, rps_limiter_h_pb, shared_h_pb, rewrite_h_pb)
        else:
            # old module order
            module_pbs = (shared_h_pb, threshold_h_pb, report_h_pb, rewrite_h_pb)

        rv = []
        if version >= VERSION_0_3_0 or self.pb.compat.enable_meta:
            h_pb = modules_pb2.Holder()
            h_pb.meta.id = META_MODULE_ID
            h_pb.meta.fields['upstream'] = self.pb.id
            rv.append(h_pb)
        for h_pb in itertools.chain(module_pbs, header_pbs, response_header_pbs, balancer_pbs):
            if h_pb is not None:
                rv.append(h_pb)

        return rv
