# coding: utf-8
import six
import ast
import tokenize
import collections

import funcparserlib.parser as p
from six import unichr, StringIO


class Reader(object):
    EOS = object()

    def __init__(self, string):
        self._string = string
        self._i = 0
        self._parts = []

    def look_around(self):
        return self._string[self._i - 2:self._i + 2]

    @property
    def current(self):
        if self._i < len(self._string):
            return self._string[self._i]
        else:
            return self.EOS

    def next(self):
        self._i += 1
        return self.current

    def save(self, part):
        self._parts.append(part)

    def build(self):
        return ''.join(self._parts)


hexdigits = '0123456789abcdefABCDEF'


def ishexdigit(d):
    return d in hexdigits


def hexdigittovalue(d):
    if d.isdigit():
        return ord(d) - ord('0')
    else:
        return ord(d.lower()) - ord('a') + 10


def read_hex(reader):
    d1 = reader.next()
    if not ishexdigit(d1):
        raise ValueError('hexadecimal digit expected')
    d2 = reader.next()
    if not ishexdigit(d2):
        raise ValueError('hexadecimal digit expected')
    return chr((hexdigittovalue(d1) << 4) + hexdigittovalue(d2))


def read_unicode(reader):
    cur = reader.next()
    if cur != '{':
        raise ValueError('missing "{"')
    cur = reader.next()
    if not ishexdigit(cur):
        raise ValueError('hexadecimal digit expected')
    rv = hexdigittovalue(cur)
    cur = reader.next()
    while ishexdigit(cur):
        rv = (rv << 4) + hexdigittovalue(cur)
        if rv > 0x10FFFF:
            raise ValueError('UTF-8 value too large')
        cur = reader.next()
    if cur != '}':
        raise ValueError('missing "}"')
    return unichr(rv)


def read_string(string):
    # a function that mimics Lua's parser read_string behaviour
    # https://github.com/lua/lua/blob/8edbf57fb8e3c20aac9560b7e4cce7583d14ebf6/llex.c#L366
    reader = Reader(string)
    while reader.current != reader.EOS:
        cur = reader.current
        if cur == '\\':
            cur = reader.next()
            if cur == 'a':
                reader.save('\a')
            elif cur == 'b':
                reader.save('\b')
            elif cur == 'f':
                reader.save('\f')
            # elif cur == 'n': reader.save('\n')
            # elif cur == 'r': reader.save('\r')
            elif cur == 't':
                reader.save('\t')
            elif cur == 'v':
                reader.save('\v')
            elif cur == 'x':
                s = read_hex(reader)
                reader.save(s)
            elif cur == 'u':
                s = read_unicode(reader)
                reader.save(s)
            elif cur in ('\\', '\'', '"'):
                reader.save(cur)
            elif cur == reader.EOS:
                raise ValueError('unfinished string')
            elif cur == 'z':
                raise ValueError('Lua\'s \z is not supported')
            elif cur in ('n', 'r'):
                raise ValueError('multiline strings are not supported')
            else:
                raise ValueError('invalid escape sequence: {}'.format(reader.look_around()))
        elif cur in ('\n', '\r'):
            raise ValueError('multiline strings are not supported')
        else:
            reader.save(cur)
        reader.next()
    return reader.build()


RawCall = collections.namedtuple('Call', ['func', 'args'])

negate = lambda x: -x
get_val = lambda t: t[1]
p_op = p.some(lambda t: t[0] == tokenize.OP)
p_minus = p.some(lambda t: t[0] == tokenize.OP and t[1] == '-')

p_id = p.some(lambda t: t[0] == tokenize.NAME) >> get_val
p_true = p.some(lambda t: t[0] == tokenize.NAME and t[1] == 'true') >> (lambda t: True)
p_false = p.some(lambda t: t[0] == tokenize.NAME and t[1] == 'false') >> (lambda t: False)
p_number = p.some(lambda t: t[0] == tokenize.NUMBER) >> get_val >> ast.literal_eval
p_negative_number = p.skip(p_minus) + p_number >> negate
p_string = p.some(lambda t: t[0] == tokenize.STRING) >> get_val >> ast.literal_eval
p_newline = p.maybe(p.some(lambda t: t[0] == tokenize.NEWLINE))
p_endmarker = p.some(lambda t: t[0] == tokenize.ENDMARKER)


@p.with_forward_decls
def arg():
    return p_number | p_negative_number | p_string | p_true | p_false | p_call


def create_call(name):
    return RawCall(name, [])


def add_args_to_call(z):
    func, args = z
    if args:
        func.args.append(args[0])
        func.args.extend(args[1])
    return func


p_call = (
    (p_id >> create_call) +  # parse function name and create RawCall object
    p.skip(p_op) +  # skip left bracker
    p.maybe(
        arg +  # parse first argument
        p.many(p.skip(p_op) + arg)  # parse the rest of the arguments
    ) +
    p.skip(p_op) >>  # skip right bracket
    add_args_to_call  # add arguments to the Call object
)

p_root = p_call + p.skip(p_newline + p_endmarker + p.finished)


def parse_call(call_expr):
    """
    :param six.text_type call_expr: e.g., get_global("log_dir")
    :rtype: RawCall
    """
    # useful for debugging:
    # tokenize.tokenize(StringIO(s).readline)
    tokens = []
    try:
        for token in tokenize.generate_tokens(StringIO(call_expr).readline):
            if token[0] == tokenize.ERRORTOKEN:
                raise ValueError('unexpected token {!r} at position {}'.format(token[1], token[2][1]))
            tokens.append(token)
    except tokenize.TokenError as e:
        raise ValueError('{} at position {}'.format(e.args[0], e.args[1][0]))
    try:
        return p_root.parse(tokens)
    except p.NoParseError as e:
        token = tokens[e.state.pos]
        raise ValueError('unexpected token {!r} at position {}'.format(token[1], token[2][1]))


def dump_call(raw_call):
    """
    :type raw_call: RawCall
    :rtype: str
    """
    str_args = []
    for arg in raw_call.args:
        if isinstance(arg, RawCall):
            arg_str = dump_call(arg)
        elif isinstance(arg, bool):
            arg_str = 'true' if arg else 'false'
        elif isinstance(arg, int):
            arg_str = str(arg)
        elif isinstance(arg, float):
            arg_str = '{0:.3f}'.format(arg)
        elif isinstance(arg, six.string_types):
            arg_str = '"{}"'.format(arg)
        else:
            raise RuntimeError('unexpected arg type: {}'.format(type(arg)))
        str_args.append(arg_str)
    return '{}({})'.format(raw_call.func, ', '.join(str_args))
