# https://a.yandex-team.ru/arc/trunk/arcadia/contrib/libs/lua/lua-5.2.0/src/llex.c?rev=3803864#L329


HEXDIGITS = '0123456789abcdefABCDEF'


def ishexdigit(d):
    return d in HEXDIGITS


def hexdigit_to_value(d):
    if d.isdigit():
        return ord(d) - ord('0')
    else:
        return ord(d.lower()) - ord('a') + 10


def read_hex(reader):
    """
    :type reader: Reader
    :rtype: str
    """
    d1 = reader.next()
    if not ishexdigit(d1):
        raise ValueError('hexadecimal digit expected')
    d2 = reader.next()
    if not ishexdigit(d2):
        raise ValueError('hexadecimal digit expected')
    return chr((hexdigit_to_value(d1) << 4) + hexdigit_to_value(d2))


def read_dec(reader):
    """
    :type reader: Reader
    :rtype: str
    """
    r = 0
    i = 0
    while i < 3 and reader.current != EOS and reader.current.isdigit():
        cur = reader.current
        r = 10 * r + (ord(cur) - ord('0'))
        reader.next()
        i += 1
    if r > 255:
        raise ValueError('decimal escape too large')
    return chr(r)


EOS = object()


class Reader(object):
    __slots__ = ('buf', 'i', 'parts')

    def __init__(self, buf):
        self.buf = buf
        self.i = 0
        self.parts = []

    def look_around(self):
        return self.buf[self.i - 4:self.i + 4]

    @property
    def current(self):
        if self.i < len(self.buf):
            return self.buf[self.i]
        else:
            return EOS

    def next(self):
        self.i += 1
        return self.current

    def save(self, part):
        self.parts.append(part)

    def build(self):
        return ''.join(self.parts)


def read_string(buf):
    # https://a.yandex-team.ru/arc/trunk/arcadia/contrib/libs/lua/lua-5.2.0/src/llex.c?rev=3803864#L329
    reader = Reader(buf)
    while reader.current != EOS:
        cur = reader.current
        if cur in ('\n', '\r'):
            raise ValueError('unfinished string')
        elif cur == '\\':
            cur = reader.next()
            if cur == 'a':
                reader.save('\a')
            elif cur == 'b':
                reader.save('\b')
            elif cur == 'f':
                reader.save('\f')
            elif cur == 'n':
                reader.save('\n')
            elif cur == 'r':
                reader.save('\r')
            elif cur == 't':
                reader.save('\t')
            elif cur == 'v':
                reader.save('\v')
            elif cur == 'x':
                reader.save(read_hex(reader))
            elif cur in ('\n', '\r'):
                reader.save('\n')
            elif cur in ('\\', '\'', '"'):
                reader.save(cur)
            elif cur == EOS:
                raise ValueError('unfinished string')
            elif cur == 'z':
                raise ValueError('Lua\'s \z is not supported')
            else:
                if cur.isdigit():
                    reader.save(read_dec(reader))
                else:
                    raise ValueError('invalid escape sequence')
        else:
            reader.save(cur)
        reader.next()
    return reader.build()


def dump_string(string):
    """
    :type string: six.text_type
    :rtype: six.text_type
    """
    rv = u''
    for c in string:
        if c == '\a':
            rv += r'\a'
        elif c == '\b':
            rv += r'\b'
        elif c == '\f':
            rv += r'\f'
        elif c == '\n':
            rv += r'\n'
        elif c == '\r':
            rv += r'\r'
        elif c == '\t':
            rv += r'\t'
        elif c == '\v':
            rv += r'\v'
        elif 0x0 <= ord(c) <= 0x1f or ord(c) == 0x7f:
            rv += r'\x' + '{:02x}'.format(ord(c))
        elif c == '"':
            rv += r'\"'
        elif c == '\\':
            rv += r'\\'
        else:
            rv += c
    return rv
