# coding: utf-8
import operator
import re
import itertools
import collections

import six
from six.moves import http_client as httplib
import semantic_version

from infra.awacs.proto import modules_pb2 as proto, model_pb2
from awacs.lib import OrderedDict
from awacs.lib.strutils import flatten_full_id, to_full_ids, join_full_uids2, quote_join_sorted
from awacs.model.util import is_upstream_internal
from awacs.wrappers import defs, base, errors
from awacs.wrappers.funcs import validate_args

from . import min_component_versions
from .base import (
    find_module, add_module, find_last_module,
    ConfigWrapperBase, ModuleWrapperBase, ChainableModuleWrapperBase,
    Holder, WrapperBase, MacroBase, ANY_MODULE, DEFAULT_CTX)
from .config import Config, Call as ConfigCall
from .errors import ValidationError, CertDoesNotExist
from .tls_settings import CipherSuite
from .util import (
    validate, format_ip, format_ip_port, format_addr, validate_timedeltas, validate_timedelta,
    validate_ip, validate_port, validate_pire_regexp, validate_key_uniqueness, validate_long_timedelta,
    validate_status_codes, validate_header_name, is_addr_local, validate_status_code_or_family,
    LOCAL_V4_ADDR, LOCAL_V6_ADDR, is_addr_external, contains_ip, contains_addr, intersect_addrs,
    validate_request_line, validate_timedelta_range, Value, is_close, validate_cookie_name,
    validate_comma_separated_ints, timedelta_to_ms, validate_range, validate_func_name_one_of,
    fill_get_public_cert_path, fill_get_private_cert_path, validate_item_uniqueness,
    validate_re2_regexp, validate_version, validate_header_func, cluster_name_from_alias,
    validate_status_code_3xx)
from . import rps_limiter_settings
from . import l7macro
from functools import reduce


DEFAULT_LOG_DIR = '/place/db/www/logs'
DEFAULT_PUBLIC_CERT_DIR = '/dev/shm/balancer'
DEFAULT_PRIVATE_CERT_DIR = '/dev/shm/balancer/priv'
DEFAULT_CA_CERT_DIR = './'

ICOOKIE_DOMAINS = [
    '.yandex.ru',
    '.yandex.ua',
    '.yandex.uz',
    '.yandex.by',
    '.yandex.kz',
    '.yandex.com',
    '.yandex.com.tr',
    '.yandex.com.ge',
    '.yandex.fr',
    '.yandex.az',
    '.yandex.com.am',
    '.yandex.co.il',
    '.yandex.kg',
    '.yandex.lt',
    '.yandex.lv',
    '.yandex.md',
    '.yandex.tj',
    '.yandex.tm',
    '.yandex.ee',
    '.yandex.eu',
    '.yandex.fi',
    '.yandex.pl',
    '.ya.ru',
    '.kinopoisk.ru',
]


def require_sd(preceding_modules, field_name=None):
    top_level_module = find_module(preceding_modules, (l7macro.L7Macro, InstanceMacro, Main))
    if isinstance(top_level_module, l7macro.L7Macro):
        sd_disabled = top_level_module.compat and top_level_module.compat.pb.disable_sd
    else:
        sd_disabled = not top_level_module.sd
    if sd_disabled:
        raise ValidationError('can only be used if preceded by l7_macro, instance_macro or main module with '
                              'enabled SD', field_name)


def require_state_directory(preceding_modules, field_name=None):
    top_level_module = find_module(preceding_modules, (l7macro.L7Macro, InstanceMacro, Main))
    ok = True
    if top_level_module:
        if isinstance(top_level_module, l7macro.L7Macro):
            ok = semantic_version.Version(top_level_module.pb.version) >= l7macro.VERSION_0_0_3
        else:
            ok = bool(top_level_module.pb.state_directory)
    if not ok:
        raise ValidationError(
            u'can only be used if preceded by instance_macro or main module with state_directory, '
            u'or l7_macro of version 0.0.3+', field_name)


def require_hashing_module(preceding_modules, field_name=None):
    if (ANY_MODULE not in preceding_modules and
            not find_module(preceding_modules, Hasher) and
            not find_module(preceding_modules, HeadersHasher) and
            not find_module(preceding_modules, CookieHasher) and
            not find_module(preceding_modules, CgiHasher)):
        raise ValidationError(
            'must be preceded by "hasher" or "headers_hasher" or "cookie_hasher" or "cgi_hasher" module',
            field_name
        )


class IncludeUpstreamsFilterSpec(WrapperBase):
    __protobuf__ = proto.IncludeUpstreams.FilterSpec

    and_ = []  # type: list[IncludeUpstreamsFilterSpec]
    or_ = []  # type: list[IncludeUpstreamsFilterSpec]
    not_ = None  # type: IncludeUpstreamsFilterSpec | None

    PB_FIELD_TO_CLS_ATTR_MAPPING = {
        'and': 'and_',
        'or': 'or_',
        'not': 'not_',
    }

    FIELDS = ('any', 'id', 'ids', 'id_prefix', 'id_prefix_in', 'id_suffix', 'id_suffix_in')
    OP_FIELDS = ('and', 'or', 'not')
    ALL_FIELDS = FIELDS + OP_FIELDS

    REQUIRED_ONEOFS = [ALL_FIELDS]

    def update_pb(self, pb=None):
        super(IncludeUpstreamsFilterSpec, self).update_pb(pb=pb)
        self.wrap_composite_fields()

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()

        for i, and_ in enumerate(self.and_):
            with validate('and[{}]'.format(i)):
                and_.validate(ctx=ctx, preceding_modules=preceding_modules)

        for i, or_ in enumerate(self.or_):
            with validate('or[{}]'.format(i)):
                or_.validate(ctx=ctx, preceding_modules=preceding_modules)

        if self.not_:
            with validate('not'):
                self.not_.validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_predicate(self):
        """
        :rtype: callable
        """
        if self.pb.any:
            return lambda upstream_id: True
        elif self.pb.id:
            return lambda upstream_id: upstream_id == self.pb.id
        elif self.pb.ids:
            return lambda upstream_id: upstream_id in self.pb.ids
        elif self.pb.id_prefix:
            return lambda upstream_id: upstream_id.startswith(self.pb.id_prefix)
        elif self.pb.id_prefix_in:
            return lambda upstream_id: upstream_id.startswith(tuple(self.pb.id_prefix_in))
        elif self.pb.id_suffix:
            return lambda upstream_id: upstream_id.endswith(self.pb.id_suffix)
        elif self.pb.id_suffix_in:
            return lambda upstream_id: upstream_id.endswith(tuple(self.pb.id_suffix_in))
        elif self.and_:
            predicates = [a.to_predicate() for a in self.and_]
            return lambda upstream_id: reduce(operator.and_, [p(upstream_id) for p in predicates])
        elif self.or_:
            predicates = [o.to_predicate() for o in self.or_]
            return lambda upstream_id: reduce(operator.or_, [p(upstream_id) for p in predicates])
        elif self.not_:
            p = self.not_.to_predicate()
            return lambda upstream_id: not p(upstream_id)
        else:
            raise RuntimeError('failed to build predicate')


class IncludeUpstreamsOrderSpecLabel(WrapperBase):
    __protobuf__ = proto.IncludeUpstreams.OrderSpec.Label

    REQUIRED = ['name']

    def get_default_value(self):
        """
        :rtype: str | None
        """
        if self.pb.HasField('default_value'):
            return self.pb.default_value.value
        else:
            return None

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()


class IncludeUpstreamsOrderSpec(WrapperBase):
    __protobuf__ = proto.IncludeUpstreams.OrderSpec

    label = None  # type: IncludeUpstreamsOrderSpecLabel | None

    REQUIRED = ['label']

    def update_pb(self, pb=None):
        super(IncludeUpstreamsOrderSpec, self).update_pb(pb=pb)
        self.wrap_composite_fields()

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        with validate('label'):
            self.label.validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_key_func(self):
        """
        :rtype: callable
        """
        if self.label:
            label_name = self.label.pb.name
            label_default = self.label.get_default_value()
            if label_default is None:
                def key(item):
                    full_upstream_id, labels = item
                    if label_name not in labels:
                        raise ValidationError('can not sort upstreams: upstream "{}" does not have '
                                              'required label "{}"'.format(full_upstream_id[1], label_name))
                    return labels[label_name]

                return key
            else:
                return lambda full_upstream_id_labels: full_upstream_id_labels[1].get(label_name, label_default)
        else:
            raise RuntimeError('failed to build key func')


class IncludeUpstreams(WrapperBase):
    __protobuf__ = proto.IncludeUpstreams

    filter = None  # type: IncludeUpstreamsFilterSpec | None
    order = None  # type: IncludeUpstreamsOrderSpec | None

    def update_pb(self, pb=None):
        super(IncludeUpstreams, self).update_pb(pb=pb)
        self.wrap_composite_fields()

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        if self.pb.type == proto.NONE:
            if self.pb.ids:
                raise ValidationError('must not be set', 'ids')
            with validate('filter'):
                self.require_value(self.filter)
                self.filter.validate(ctx=ctx, preceding_modules=preceding_modules)
            if self.order:
                with validate('order'):
                    self.order.validate(ctx=ctx, preceding_modules=preceding_modules)

        elif self.pb.type == proto.ALL:
            if self.filter or self.order or self.pb.ids:
                raise ValidationError('neither of "filter", "order", "ids" must be set')

        elif self.pb.type == proto.BY_ID:
            if self.filter or self.order:
                raise ValidationError('neither of "filter", "order" must be set')
            self.require_value(self.pb.ids, 'ids')
            for i, upstream_id in enumerate(self.pb.ids):
                with validate('ids[{}]'.format(i)):
                    self.require_value(upstream_id)
                    if upstream_id.count('/') > 0:
                        raise ValidationError('must not contain slashes')

    def filter_upstreams(self, current_namespace_id, upstreams, labels=None):
        """
        :type current_namespace_id: str
        :type upstreams: dict[(str, str), awacs.wrappers.base.Holder]
        :type labels: dict[(str, str), dict[str, str]]
        :rtype: dict[(str, str), Holder]
        """
        if self.pb.type == proto.NONE:
            predicate = self.filter.to_predicate()

            included_full_upstream_ids = []
            for full_upstream_id in sorted(upstreams.keys()):
                namespace_id, upstream_id = full_upstream_id
                assert namespace_id == current_namespace_id  # safety check until and if we flatten upstreams
                if predicate(upstream_id):
                    included_full_upstream_ids.append(full_upstream_id)

            if self.order:
                labels = labels or {}
                key = self.order.to_key_func()
                annotated_filtered_upstreams = [(full_upstream_id, labels.get(full_upstream_id, {}))
                                                for full_upstream_id in included_full_upstream_ids]
                annotated_filtered_upstreams = sorted(annotated_filtered_upstreams, key=key)
                included_full_upstream_ids = [full_upstream_id for full_upstream_id, _ in annotated_filtered_upstreams]

            filtered_upstreams = OrderedDict()
            for full_upstream_id in included_full_upstream_ids:
                if is_upstream_internal(full_upstream_id[1]):
                    continue
                filtered_upstreams[full_upstream_id] = upstreams[full_upstream_id]
            return filtered_upstreams
        elif self.pb.type == proto.ALL:
            return OrderedDict(sorted(upstream for upstream in upstreams.items() if not is_upstream_internal(upstream[0][1])))
        elif self.pb.type == proto.BY_ID:
            missing_full_upstream_ids = set()
            filtered_upstreams = OrderedDict()
            for full_upstream_id in to_full_ids(current_namespace_id, self.pb.ids):
                assert full_upstream_id[0] == current_namespace_id
                if full_upstream_id in upstreams and not is_upstream_internal(full_upstream_id[1]):
                    filtered_upstreams[full_upstream_id] = upstreams[full_upstream_id]
                else:
                    missing_full_upstream_ids.add(full_upstream_id)
            if missing_full_upstream_ids:
                list_str = join_full_uids2(current_namespace_id, missing_full_upstream_ids)
                raise errors.UpstreamDoesNotExist('Some of the included upstreams are missing: "{}"'.format(list_str))
            else:
                return filtered_upstreams
        else:
            raise RuntimeError('unexpected type {}'.format(self.pb.type))

    def get_included_upstream_ids(self, current_namespace_id, upstream_ids):
        """
        :type current_namespace_id: str
        :type upstream_ids: iterable[(str, str)]
        :rtype: set[(str, str)]
        """
        if self.pb.type == proto.NONE:
            rv = set()
            predicate = self.filter.to_predicate()
            for full_upstream_id in upstream_ids:
                namespace_id, upstream_id = full_upstream_id
                assert namespace_id == current_namespace_id  # safety check until and if we flatten upstreams
                if predicate(upstream_id):
                    rv.add(full_upstream_id)
        elif self.pb.type == proto.ALL:
            rv = set(upstream_ids)
        elif self.pb.type == proto.BY_ID:
            rv = set(to_full_ids(current_namespace_id, self.pb.ids))
        else:
            raise RuntimeError('unexpected type {}'.format(self.pb.type))
        return set(upstream_id for upstream_id in rv if not is_upstream_internal(upstream_id[1]))


class IncludeBackends(WrapperBase):
    __protobuf__ = proto.IncludeBackends

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        if self.pb.type != proto.IncludeBackends.BY_ID:
            raise ValidationError('must be equal to "BY_ID"', 'type')
        self.require_value(self.pb.ids, 'ids')
        for i, backend_id in enumerate(self.pb.ids):
            with validate('ids[{}]'.format(i)):
                self.require_value(backend_id)
                if backend_id.count('/') > 1:
                    raise ValidationError('must contain at most one slash')

    def get_dynamic_backends_name(self):
        return '#'.join(sorted(self.pb.ids))

    def filter(self, namespace_id, backend_specs, endpoint_set_specs):
        """
        :type namespace_id: str
        :type backend_specs: dict[(str, str), awacs.proto.model_pb2.BackendSpec]
        :type endpoint_set_specs: dict[(str, str), awacs.proto.model_pb2.EndpointSetSpec]
        :rtype: (dict[(str, str), awacs.proto.model_pb2.BackendSpec],
                 dict[(str, str), awacs.proto.model_pb2.EndpointSetSpec])
        :raises: errors.EndpointSetsDoNotExist, errors.BackendsDoNotExist, errors.ValidationError
        """
        if self.pb.type == proto.IncludeBackends.BY_ID:
            backends = OrderedDict()
            missing_full_backend_ids = set()
            for full_backend_id in to_full_ids(namespace_id, self.pb.ids):
                if full_backend_id in backend_specs:
                    backends[full_backend_id] = backend_specs[full_backend_id]
                else:
                    missing_full_backend_ids.add(full_backend_id)
            if missing_full_backend_ids:
                message = 'Some of the included backends are missing: "{}"'.format(
                    join_full_uids2(namespace_id, missing_full_backend_ids))
                raise errors.BackendsDoNotExist(message, full_ids=missing_full_backend_ids)

            full_backend_ids_by_types = collections.defaultdict(set)
            for full_backend_id, backend_spec_pb in six.iteritems(backends):
                full_backend_ids_by_types[backend_spec_pb.selector.type].add(full_backend_id)

            all_backends_are_sd = False
            if model_pb2.BackendSelector.YP_ENDPOINT_SETS_SD in full_backend_ids_by_types:
                if len(full_backend_ids_by_types) > 1:
                    yp_endpoint_set_sd_full_backend_ids = full_backend_ids_by_types.pop(
                        model_pb2.BackendSelector.YP_ENDPOINT_SETS_SD)
                    other_backend_ids = set.union(*six.itervalues(full_backend_ids_by_types))
                    message = (
                        'YP_ENDPOINT_SETS_SD-backends ("{}") can not be used '
                        'together with backends of different types ("{}")').format(
                        join_full_uids2(namespace_id, yp_endpoint_set_sd_full_backend_ids),
                        join_full_uids2(namespace_id, other_backend_ids)
                    )
                    raise errors.ValidationError(message)
                else:
                    assert len(full_backend_ids_by_types) == 1
                    all_backends_are_sd = True

            endpoint_sets = OrderedDict()
            if not all_backends_are_sd:
                missing_full_endpoint_set_ids = set()
                for full_endpoint_set_id in to_full_ids(namespace_id, self.pb.ids):
                    if full_endpoint_set_id in endpoint_set_specs:
                        endpoint_sets[full_endpoint_set_id] = endpoint_set_specs[full_endpoint_set_id]
                    else:
                        missing_full_endpoint_set_ids.add(full_endpoint_set_id)
                if missing_full_endpoint_set_ids:
                    message = 'Some included backends are missing or not resolved yet: "{}"'.format(
                        join_full_uids2(namespace_id, missing_full_endpoint_set_ids))
                    raise errors.EndpointSetsDoNotExist(message, full_ids=missing_full_endpoint_set_ids)

            return backends, endpoint_sets
        else:
            raise RuntimeError('unexpected type {}'.format(self.pb.type))

    def get_included_full_backend_ids(self, current_namespace_id):
        """
        :type current_namespace_id: str
        :rtype: set[(str, str)]
        """
        if self.pb.type == proto.IncludeBackends.BY_ID:
            return set(to_full_ids(current_namespace_id, self.pb.ids))
        else:
            raise RuntimeError('unexpected type {}'.format(self.pb.type))


def validate_ip_call(call):
    validate_func_name_one_of(call, (defs.get_ip_by_iproute.name,))
    call.validate()


def validate_log_call(call):
    validate_func_name_one_of(call, (defs.get_str_var.name, defs.get_log_path.name))
    call.validate()


def validate_name_call(call):
    validate_func_name_one_of(call, (defs.get_str_var.name, defs.get_geo.name,
                                     defs.prefix_with_dc.name, defs.suffix_with_dc.name))
    call.validate()


def validate_dns_ttl_call(call):
    validate_func_name_one_of(call, (defs.get_random_timedelta.name,))
    call.validate()


def validate_port_call(call):
    validate_func_name_one_of(call, (defs.get_int_var.name, defs.get_port_var.name,))
    call.validate()


def validate_hysteresis_call(call):
    """
    :type call: Call
    """
    validate_func_name_one_of(call, (defs.get_total_weight_percent.name,))
    call.validate()


def validate_quorum_call(call):
    """
    :type call: Call
    """
    validate_func_name_one_of(call, (defs.get_total_weight_percent.name,))
    call.validate()


def validate_workers_call(call):
    """
    :type call: Call
    """
    validate_func_name_one_of(call, (defs.get_workers.name,))
    call.validate()


def validate_header_value_call(call):
    """
    :type call: Call
    """
    validate_func_name_one_of(call, (defs.get_str_env_var.name,))
    call.validate()


class Ip(ConfigWrapperBase):
    __protobuf__ = proto.IpdispatchSection.Ip

    REQUIRED = ['value']
    ALLOWED_CALLS = {
        'value': [defs.get_ip_by_iproute.name],
    }

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        value = self.get('value')
        if value.is_func():
            validate_ip_call(value.value)
        else:
            if value.value not in (LOCAL_V4_ADDR, LOCAL_V6_ADDR):
                validate_ip(value.value)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        value = self.get('value')
        if value.is_func():
            return value.to_config(ctx=ctx)
        else:
            if value.value == LOCAL_V4_ADDR:
                return ConfigCall('get_ip_by_iproute', ['v4'])
            elif value.value == LOCAL_V6_ADDR:
                return ConfigCall('get_ip_by_iproute', ['v6'])
            return value.to_config(ctx=ctx)


class Port(ConfigWrapperBase):
    __protobuf__ = proto.IpdispatchSection.Port

    REQUIRED = ['value']
    ALLOWED_CALLS = {
        'value': [defs.get_int_var.name, defs.get_port_var.name],
    }

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        value = self.get('value')
        if value.is_func():
            validate_port_call(value.value)
        else:
            validate_port(value.value)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        return self.get('value').to_config(ctx=ctx)


class Addr(ConfigWrapperBase):
    __protobuf__ = proto.Addr

    REQUIRED = ['ip', 'port']
    ALLOWED_CALLS = {
        'ip': [defs.get_ip_by_iproute.name],
        'port': [defs.get_int_var.name, defs.get_port_var.name],
    }

    @validate('ip')
    def _validate_ip(self):
        ip = self.get('ip')
        if ip.is_func():
            validate_ip_call(ip.value)
        else:
            if ip.value not in (LOCAL_V4_ADDR, LOCAL_V6_ADDR):
                validate_ip(ip.value)

    @validate('port')
    def _validate_port(self):
        port = self.get('port')
        if port.is_func():
            validate_port_call(port.value)
        else:
            validate_port(self.pb.port)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        self._validate_ip()
        self._validate_port()
        super(Addr, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        ip = self.get('ip')
        if ip.is_func():
            ip_config = ip.to_config(ctx=ctx)
        else:
            if ip.value == LOCAL_V4_ADDR:
                ip_config = ConfigCall('get_ip_by_iproute', ['v4'])
            elif ip.value == LOCAL_V6_ADDR:
                ip_config = ConfigCall('get_ip_by_iproute', ['v6'])
            else:
                ip_config = ip.to_config(ctx=ctx)
        port_config = self.get('port').to_config(ctx=ctx)
        table = {
            'ip': ip_config,
            'port': port_config,
        }
        if is_addr_external(ip):
            table['disabled'] = ConfigCall('get_int_var', ['disable_external', 0])
        return Config(table)


DEFAULT_EVENTS = OrderedDict([('stats', 'report')])


class ConfigCheck(ConfigWrapperBase):
    __protobuf__ = proto.MainModule.ConfigCheck

    DEFAULT_QUORUMS_FILE = u'./controls/backend_check_quorums'

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        super(ConfigCheck, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict([
            ('quorums_file', self.pb.quorums_file or self.DEFAULT_QUORUMS_FILE),
        ])
        return Config(table)


class Unistat(ConfigWrapperBase):
    __protobuf__ = proto.MainModule.Unistat

    addrs = []  # type: list[Addr]

    DEFAULT_UNISTAT_PORT_OFFSET = 2

    def list_addrs(self):
        ip_ports = set()
        for addr in self.addrs:
            ip_port = addr.get('ip'), addr.get('port')
            ip_ports.add(ip_port)
        return ip_ports

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()

        ip_ports = set()
        for i, addr in enumerate(self.addrs):
            with validate('addrs[{}]'.format(i)):
                addr.validate(ctx=ctx, preceding_modules=preceding_modules)
            ip = addr.get('ip')
            port = addr.get('port')
            ip_port = ip, port
            tmp_addr = contains_addr(ip_ports, ip_port)
            if not tmp_addr:
                ip_ports.add(ip_port)
            else:
                raise ValidationError('contain duplicate address: {}'.format(format_addr(tmp_addr)), 'addrs')

        super(Unistat, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        if self.addrs:
            addrs = Config(array=[addr.to_config(ctx=ctx) for addr in self.addrs])
        else:
            addr_pb = proto.Addr()
            addr_pb.ip = '*'
            addr_pb.f_port.type = proto.Call.GET_PORT_VAR
            addr_pb.f_port.get_port_var_params.var = 'port'
            addr_pb.f_port.get_port_var_params.offset.value = self.DEFAULT_UNISTAT_PORT_OFFSET
            addrs = Config(array=[Addr(addr_pb).to_config(ctx=ctx)])
        table = OrderedDict([
            ('addrs', addrs),
        ])
        if self.pb.hide_legacy_signals:
            table['hide_legacy_signals'] = self.pb.hide_legacy_signals
        return Config(table)


class LimitsMixin(object):
    # type: proto.MainModule.CpuLimiter.ConnReject | proto.MainModule.CpuLimiter.Http2Drop | proto.MainModule.CpuLimiter.KeepaliveClose

    pb = None

    def validate_lo_and_hi(self):
        with validate('lo'):
            validate_range(self.pb.lo.value, 0, 1)
        with validate('hi'):
            validate_range(self.pb.hi.value, 0, 1)
        if self.pb.lo.value > self.pb.hi.value:
            raise ValidationError('must be less or equal to "hi"', 'lo')


class ConnReject(WrapperBase, LimitsMixin):
    __protobuf__ = proto.MainModule.CpuLimiter.ConnReject

    REQUIRED = ['lo', 'hi']

    def update_pb(self, pb=None):
        super(ConnReject, self).update_pb(pb=pb)
        self.wrap_composite_fields()

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        self.validate_lo_and_hi()
        if self.pb.HasField('conn_hold_count'):
            if self.pb.conn_hold_count.value < 0:
                raise ValidationError('must be non-negative', 'conn_hold_count')
        if self.pb.conn_hold_duration:
            with validate('conn_hold_duration'):
                validate_timedelta(self.pb.conn_hold_duration)


class Http2Drop(WrapperBase, LimitsMixin):
    __protobuf__ = proto.MainModule.CpuLimiter.Http2Drop

    REQUIRED = ['lo', 'hi']

    def update_pb(self, pb=None):
        super(Http2Drop, self).update_pb(pb=pb)
        self.wrap_composite_fields()

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        self.validate_lo_and_hi()


class KeepaliveClose(WrapperBase, LimitsMixin):
    __protobuf__ = proto.MainModule.CpuLimiter.KeepaliveClose

    REQUIRED = ['lo', 'hi']

    def update_pb(self, pb=None):
        super(KeepaliveClose, self).update_pb(pb=pb)
        self.wrap_composite_fields()

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        self.validate_lo_and_hi()


class CpuLimiter(ConfigWrapperBase):
    __protobuf__ = proto.MainModule.CpuLimiter

    conn_reject = None  # type: ConnReject
    http2_drop = None  # type: Http2Drop
    keepalive_close = None  # type: KeepaliveClose

    ALLOWED_KNOBS = {
        'disable_file': model_pb2.KnobSpec.BOOLEAN,
        'disable_http2_file': model_pb2.KnobSpec.BOOLEAN,
    }

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=()):
        if self.conn_reject:
            with validate('conn_reject'):
                self.conn_reject.validate(ctx=ctx, preceding_modules=preceding_modules)
        if self.http2_drop:
            with validate('http2_drop'):
                self.http2_drop.validate(ctx=ctx, preceding_modules=preceding_modules)
        if self.keepalive_close:
            with validate('keepalive_close'):
                self.keepalive_close.validate(ctx=ctx, preceding_modules=preceding_modules)
        if not self.keepalive_close and self.conn_reject:
            with validate('conn_reject'):
                if self.conn_reject.pb.HasField('conn_hold_count'):
                    raise ValidationError('must not be set if "keepalive_close" is not enabled', 'conn_hold_count')
                if self.conn_reject.pb.conn_hold_duration:
                    raise ValidationError('must not be set if "keepalive_close" is not enabled', 'conn_hold_duration')

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        self.validate_knob('disable_file', ctx=ctx)
        self.validate_knob('disable_http2_file', ctx=ctx)
        if self.pb.HasField('cpu_usage_coeff'):
            with validate('cpu_usage_coeff'):
                validate_range(self.pb.cpu_usage_coeff.value, 0, 1)
        super(CpuLimiter, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict([
            ('active_check_subnet_default', True)
        ])
        if self.pb.HasField('cpu_usage_coeff'):
            table['cpu_usage_coeff'] = self.pb.cpu_usage_coeff.value
        if self.conn_reject:
            table['enable_conn_reject'] = True
            table['conn_reject_lo'] = self.conn_reject.pb.lo.value
            table['conn_reject_hi'] = self.conn_reject.pb.hi.value
            if self.conn_reject.pb.HasField('conn_hold_count'):
                table['conn_hold_count'] = self.conn_reject.pb.conn_hold_count.value
            if self.conn_reject.pb.conn_hold_duration:
                table['conn_hold_duration'] = self.conn_reject.pb.conn_hold_duration
        if self.http2_drop:
            table['enable_http2_drop'] = True
            table['http2_drop_lo'] = self.http2_drop.pb.lo.value
            table['http2_drop_hi'] = self.http2_drop.pb.hi.value
        if self.keepalive_close:
            table['enable_keepalive_close'] = True
            table['keepalive_close_lo'] = self.keepalive_close.pb.lo.value
            table['keepalive_close_hi'] = self.keepalive_close.pb.hi.value
        disable_file = self.get('disable_file')
        if disable_file.value:
            table['disable_file'] = disable_file.to_config(ctx)
        http2_disable_file = self.get('disable_http2_file')
        if http2_disable_file.value:
            table['disable_http2_file'] = http2_disable_file.to_config(ctx)
        if self.pb.active_check_subnet_file:
            table['active_check_subnet_file'] = self.pb.active_check_subnet_file
        return Config(table)


class ServiceDiscovery(ConfigWrapperBase):
    __protobuf__ = proto.MainModule.ServiceDiscovery

    DEFAULT_HOST = 'sd.yandex.net'
    DEFAULT_PORT = 8080
    DEFAULT_CLIENT_NAME = 'unknown-awacs-l7-balancer'  # should never be used actually
    DEFAULT_CONNECT_TIMEOUT = '50ms'
    DEFAULT_REQUEST_TIMEOUT = '1s'
    DEFAULT_CACHE_DIR = './sd_cache'
    DEFAULT_UPDATE_FREQUENCY = '60s'

    DEFAULTS = {
        'host': DEFAULT_HOST,
        'port': DEFAULT_HOST,
        'connect_timeout': DEFAULT_CONNECT_TIMEOUT,
        'request_timeout': DEFAULT_REQUEST_TIMEOUT,
        'cache_dir': DEFAULT_CACHE_DIR,
        'update_frequency': DEFAULT_UPDATE_FREQUENCY,
    }

    @validate('cache_dir')
    def _validate_cache_dir(self):
        cache_dir = self.get('cache_dir')
        if not cache_dir.value:
            return
        if cache_dir.is_func():
            validate_func_name_one_of(cache_dir.value, (defs.get_str_var.name,))
            cache_dir.value.validate()

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        self._validate_cache_dir()
        if self.pb.port:
            validate_port(self.pb.port, 'port')
        if self.pb.connect_timeout:
            with validate('connect_timeout'):
                validate_timedelta(self.pb.connect_timeout)
        if self.pb.request_timeout:
            with validate('request_timeout'):
                validate_timedelta(self.pb.request_timeout)
        if self.pb.allow_empty_endpoint_sets:
            ctx.ensure_component_version(model_pb2.ComponentMeta.PGINX_BINARY,
                                         min=min_component_versions.ALLOW_EMPTY_ENDPOINT_SETS)
        super(ServiceDiscovery, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        client_name = self.pb.client_name or ctx.sd_client_name or self.DEFAULT_CLIENT_NAME
        cache_dir = self.get('cache_dir')
        if cache_dir.value:
            cache_dir = cache_dir.to_config(ctx=ctx)
        else:
            cache_dir = self.DEFAULT_CACHE_DIR
        table = OrderedDict([
            ('client_name', client_name),
            ('host', self.pb.host or self.DEFAULT_HOST),
            ('port', self.pb.port or self.DEFAULT_PORT),
            ('connect_timeout', self.pb.connect_timeout or self.DEFAULT_CONNECT_TIMEOUT),
            ('request_timeout', self.pb.request_timeout or self.DEFAULT_REQUEST_TIMEOUT),
            ('cache_dir', cache_dir),
        ])
        if self.pb.log:
            table['log'] = self.pb.log
        if self.pb.allow_empty_endpoint_sets:
            table['allow_empty_endpoint_sets'] = self.pb.allow_empty_endpoint_sets
        return Config(table)


class Main(ChainableModuleWrapperBase):
    __protobuf__ = proto.MainModule

    addrs = []  # type: list[Addr]
    admin_addrs = []  # type: list[Addr]
    unistat = None  # type: Unistat | None
    cpu_limiter = None  # type: CpuLimiter | None
    sd = None  # type: ServiceDiscovery
    config_check = None  # type: ConfigCheck | None

    SHAREABLE = False

    DEFAULT_BUFFER = 65536
    DEFAULT_ENABLE_REUSE_PORT = True
    DEFAULT_MAXCONN = 5000
    DEFAULT_TCP_RST_ON_ERROR = True
    DEFAULT_PRIVATE_ADDRESS = '127.0.0.10'
    DEFAULT_RESET_DNS_CACHE_FILE = './controls/reset_dns_cache_file'

    ALLOWED_KNOBS = {
        'reset_dns_cache_file': model_pb2.KnobSpec.BOOLEAN,
    }
    DEFAULT_KNOB_IDS = {
        'reset_dns_cache_file': 'reset_dns_cache',
    }

    ALLOWED_CALLS = {
        'log': (defs.get_str_var.name, defs.get_log_path.name),
        'dns_ttl': (defs.get_random_timedelta.name,),
        'dynamic_balancing_log': (defs.get_str_var.name, defs.get_log_path.name),
        'workers': (defs.get_workers.name,),
        'pinger_log': (defs.get_str_var.name, defs.get_log_path.name),
    }

    DEFAULTS = {
        'maxconn': DEFAULT_MAXCONN,
        'buffer': DEFAULT_BUFFER,
        'private_address': DEFAULT_PRIVATE_ADDRESS,
        'enable_reuse_port': DEFAULT_ENABLE_REUSE_PORT,
        'reset_dns_cache_file': DEFAULT_RESET_DNS_CACHE_FILE,
    }

    @classmethod
    def _validate_addrs(cls, ctx, addrs, field_name, preceding_modules=()):
        ip_ports = set()
        for i, addr in enumerate(addrs):
            with validate('{}[{}]'.format(field_name, i)):
                addr.validate(ctx=ctx, preceding_modules=preceding_modules)
            ip = addr.get('ip')
            port = addr.get('port')
            ip_port = ip, port
            tmp_addr = contains_addr(ip_ports, ip_port)
            if not tmp_addr:
                ip_ports.add(ip_port)
            else:
                raise ValidationError('contain duplicate address: {}'.format(format_addr(tmp_addr)), field_name)
        return ip_ports

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        super(Main, self).validate_composite_fields(ctx=ctx,
                                                    preceding_modules=preceding_modules,
                                                    chained_modules=chained_modules)

        new_preceding_modules = add_module(preceding_modules, self)
        addrs = self._validate_addrs(
            ctx, self.addrs, 'addrs', preceding_modules=new_preceding_modules)
        admin_addrs = self._validate_addrs(
            ctx, self.admin_addrs, 'admin_addrs', preceding_modules=new_preceding_modules)

        common_addrs = intersect_addrs(addrs, admin_addrs)
        if common_addrs:
            common_addrs_str = ', '.join(format_addr(addr) for addr in common_addrs)
            raise ValidationError('addrs and admin_addrs intersect: {}'.format(common_addrs_str))

        if self.unistat:
            unistat_addrs = self.unistat.list_addrs()

            common_unistat_addrs = intersect_addrs(addrs, unistat_addrs)
            if common_unistat_addrs:
                common_unistat_addrs_str = ', '.join(format_addr(addr) for addr in common_unistat_addrs)
                raise ValidationError(
                    'addrs and unistat addrs intersect: {}'.format(common_unistat_addrs_str))

            common_unistat_admin_addrs = intersect_addrs(admin_addrs, unistat_addrs)
            if common_unistat_admin_addrs:
                common_unistat_admin_addrs_str = ', '.join(format_addr(addr) for addr in common_unistat_admin_addrs)
                raise ValidationError(
                    'admin_addrs and unistat addrs intersect: {}'.format(common_unistat_admin_addrs_str))

        nested_module = self.nested or (chained_modules and chained_modules[0])
        if nested_module and nested_module.module_name != 'ipdispatch':
            raise ValidationError('main module must be followed by ipdispatch')

        ipdispatch = nested_module.module  # type: Ipdispatch
        if not ipdispatch.includes_upstreams():
            ipdispatch_addrs = set(ipdispatch.list_addrs())
            if ipdispatch_addrs != addrs:
                unused_addrs = ', '.join(format_ip_port(ip.value, port.value)
                                         for ip, port in addrs - ipdispatch_addrs)
                extra_addrs = ', '.join(format_ip_port(ip.value, port.value)
                                        for ip, port in ipdispatch_addrs - addrs)
                msg_parts = []
                if unused_addrs:
                    msg_parts.append('addrs not used by non-admin '
                                     'ipdispatch sections: {}'.format(unused_addrs))
                if extra_addrs:
                    msg_parts.append('addrs used by non-admin ipdispatch '
                                     'sections not listed in main module: {}'.format(extra_addrs))
                raise ValidationError('; '.join(msg_parts))

            ipdispatch_admin_addrs = set(ipdispatch.list_admin_addrs())
            if ipdispatch_admin_addrs != admin_addrs:
                unused_addrs = ', '.join(format_ip_port(ip.value, port.value)
                                         for ip, port in admin_addrs - ipdispatch_admin_addrs)
                extra_addrs = ', '.join(format_ip_port(ip.value, port.value)
                                        for ip, port in ipdispatch_admin_addrs - admin_addrs)
                msg_parts = []
                if unused_addrs:
                    msg_parts.append('admin_addrs not used by admin '
                                     'ipdispatch sections: {}'.format(unused_addrs))
                if extra_addrs:
                    msg_parts.append('admin_addrs used by admin ipdispatch '
                                     'sections not listed in main module: {}'.format(extra_addrs))
                raise ValidationError('; '.join(msg_parts))

        if self.unistat:
            with validate('unistat'):
                self.unistat.validate(ctx=ctx, preceding_modules=preceding_modules)

        if self.cpu_limiter:
            with validate('cpu_limiter'):
                self.cpu_limiter.validate(ctx=ctx, preceding_modules=preceding_modules)

        if self.sd:
            with validate('sd'):
                if not self.unistat:
                    raise ValidationError('can not be enabled without unistat')
                self.sd.validate(ctx=ctx, preceding_modules=preceding_modules)

        if self.config_check:
            with validate('config_check'):
                self.config_check.validate(ctx=ctx, preceding_modules=preceding_modules)

    @validate('maxconn')
    def _validate_maxconn(self):
        if self.pb.maxconn < 0:
            raise ValidationError('must be non-negative')

    @validate('buffer')
    def _validate_buffer(self):
        if self.pb.buffer < 0:
            raise ValidationError('must be non-negative')

    @validate('workers')
    def _validate_workers(self):
        workers = self.get('workers')
        if not workers.value:
            return

        if workers.is_func():
            validate_workers_call(workers.value)
        else:
            if self.pb.workers < 0:
                raise ValidationError('must be positive or zero')

    @validate('log')
    def _validate_log(self):
        log = self.get('log')
        if log.is_func():
            validate_log_call(log.value)

    @validate('dynamic_balancing_log')
    def _validate_dynamic_balancing_log(self):
        dynamic_balancing_log = self.get('dynamic_balancing_log')
        if not dynamic_balancing_log.value:
            return

        if dynamic_balancing_log.is_func():
            validate_log_call(dynamic_balancing_log.value)

    @validate('pinger_log')
    def _validate_pinger_log(self):
        pinger_log = self.get('pinger_log')
        if not pinger_log.value:
            return

        if pinger_log.is_func():
            validate_log_call(pinger_log.value)

    @validate('private_address')
    def _validate_private_address(self):
        if self.pb.private_address:
            if not is_addr_local(self.pb.private_address):
                raise ValidationError('must be local v4 address')

    @validate('tcp_fastopen')
    def _validate_tcp_fastopen(self):
        if self.pb.tcp_fastopen < 0:
            raise ValidationError('must be non-negative')

    @validate('dns_ttl')
    def _validate_dns_ttl(self):
        dns_ttl = self.get('dns_ttl')
        if not dns_ttl.value:
            return

        if dns_ttl.is_func():
            validate_dns_ttl_call(dns_ttl.value)
        else:
            validate_timedelta(self.pb.dns_ttl)

    @validate('state_directory')
    def _validate_state_directory(self):
        if self.pb.pinger_required and not self.pb.state_directory:
            raise ValidationError('must be set if pinger_required is set')

    @validate('tcp_listen_queue')
    def _validate_tcp_listen_queue(self):
        if self.pb.tcp_listen_queue < 0:
            raise ValidationError('must be non-negative')

    def _validate_reset_dns_cache_file(self, ctx):
        self.validate_knob('reset_dns_cache_file', ctx=ctx)

    @validate('worker_start_delay')
    def _validate_worker_start_delay(self):
        if self.pb.worker_start_delay:
            validate_timedelta(self.pb.worker_start_delay)

    @validate('worker_start_duration')
    def _validate_worker_start_duration(self):
        if self.pb.worker_start_duration:
            validate_timedelta(self.pb.worker_start_duration)

    @validate('coro_stack_size')
    def _validate_coro_stack_size(self):
        if self.pb.coro_stack_size:
            min_16_kb = 16384
            max_64_mb = 64 * 2 ** 20
            validate_range(self.pb.coro_stack_size, min_16_kb, max_64_mb)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        if preceding_modules:
            raise ValidationError('must go first in a config tree')
        self._validate_maxconn()
        self._validate_buffer()
        self._validate_workers()
        self._validate_log()
        self._validate_dynamic_balancing_log()
        self._validate_pinger_log()
        self._validate_state_directory()
        self._validate_private_address()
        self._validate_tcp_fastopen()
        self._validate_dns_ttl()
        self._validate_tcp_listen_queue()
        self._validate_reset_dns_cache_file(ctx)
        self._validate_worker_start_delay()
        self._validate_worker_start_duration()
        self._validate_coro_stack_size()
        super(Main, self).validate(ctx=ctx, preceding_modules=preceding_modules,
                                   chained_modules=chained_modules)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        events = DEFAULT_EVENTS
        if self.pb.events:
            events = OrderedDict(sorted(self.pb.events.items()))
        addrs = Config(array=[addr.to_config(ctx=ctx) for addr in self.addrs])
        admin_addrs = Config(array=[addr.to_config(ctx=ctx) for addr in self.admin_addrs])

        default_tcp_rst_on_error = self.DEFAULT_TCP_RST_ON_ERROR
        if self.pb.HasField('default_tcp_rst_on_error'):
            default_tcp_rst_on_error = self.pb.default_tcp_rst_on_error.value

        default_enable_reuse_port = self.DEFAULT_ENABLE_REUSE_PORT
        if self.pb.disable_reuse_port:
            default_enable_reuse_port = False

        workers = self.get('workers')
        if workers.value:
            workers = workers.to_config(ctx=ctx)
        else:
            workers = self.pb.workers

        table = {
            'addrs': addrs,
            'admin_addrs': admin_addrs,
            'maxconn': self.pb.maxconn or self.DEFAULT_MAXCONN,
            'workers': workers,
            'buffer': self.pb.buffer or self.DEFAULT_BUFFER,
            'private_address': self.pb.private_address or self.DEFAULT_PRIVATE_ADDRESS,
            'enable_reuse_port': default_enable_reuse_port,
            'events': Config(events),
            'default_tcp_rst_on_error': default_tcp_rst_on_error,
            'tcp_fastopen': self.pb.tcp_fastopen,
            'reset_dns_cache_file': self.get('reset_dns_cache_file', self.DEFAULT_RESET_DNS_CACHE_FILE).to_config(ctx),
        }
        if self.pb.thread_mode:
            table['thread_mode'] = self.pb.thread_mode
        if self.pb.tcp_listen_queue:
            table['tcp_listen_queue'] = self.pb.tcp_listen_queue
        if self.pb.tcp_congestion_control:
            table['tcp_congestion_control'] = self.pb.tcp_congestion_control
        log = self.get('log')
        if log.value:
            table['log'] = log.to_config(ctx=ctx)
        dynamic_balancing_log = self.get('dynamic_balancing_log')
        if dynamic_balancing_log.value:
            table['dynamic_balancing_log'] = dynamic_balancing_log.to_config(ctx=ctx)
        pinger_log = self.get('pinger_log')
        if pinger_log.value:
            table['pinger_log'] = pinger_log.to_config(ctx=ctx)
        if self.pb.pinger_required:
            table['pinger_required'] = self.pb.pinger_required
        if self.pb.state_directory:
            table['state_directory'] = self.pb.state_directory
        if self.pb.config_tag:
            table['config_tag'] = self.pb.config_tag
        dns_ttl = self.get('dns_ttl')
        if dns_ttl.value:
            table['dns_ttl'] = dns_ttl.to_config(ctx=ctx)
        if self.pb.worker_start_delay:
            table['worker_start_delay'] = self.pb.worker_start_delay
        if self.pb.worker_start_duration:
            table['worker_start_duration'] = self.pb.worker_start_duration
        if self.pb.coro_stack_size:
            table['coro_stack_size'] = self.pb.coro_stack_size
        if self.pb.HasField('coro_stack_guard'):
            table['coro_stack_guard'] = self.pb.coro_stack_guard.value
        if self.pb.HasField('unistat'):
            table['unistat'] = self.unistat.to_config(ctx=ctx)
        if self.pb.HasField('cpu_limiter'):
            table['cpu_limiter'] = self.cpu_limiter.to_config(ctx=ctx)
        if self.pb.HasField('sd'):
            table['sd'] = self.sd.to_config(ctx=ctx)
        if self.pb.HasField('config_check'):
            table['config_check'] = self.config_check.to_config(ctx=ctx)
        if self.pb.storage_gc_required:
            table['storage_gc_required'] = self.pb.storage_gc_required
        if self.pb.HasField('shutdown_accept_connections'):
            table['shutdown_accept_connections'] = self.pb.shutdown_accept_connections.value
        if self.pb.HasField('shutdown_close_using_bpf'):
            table['shutdown_close_using_bpf'] = self.pb.shutdown_close_using_bpf.value
        return Config(table)


class Accesslog(ChainableModuleWrapperBase):
    __protobuf__ = proto.AccesslogModule

    REQUIRED = ['log']

    # info attrs:
    ALLOWED_CALLS = {
        'log': (defs.get_str_var.name, defs.get_log_path.name),
    }

    @validate('log')
    def _validate_log(self):
        log = self.get('log')
        if log.is_func():
            validate_log_call(log.value)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()
        self._validate_log()
        super(Accesslog, self).validate(ctx=ctx, preceding_modules=preceding_modules, chained_modules=chained_modules)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = {
            'log': self.get('log').to_config(ctx=ctx),
        }
        if self.pb.additional_ip_header:
            table['additional_ip_header'] = self.pb.additional_ip_header
        if self.pb.additional_port_header:
            table['additional_port_header'] = self.pb.additional_port_header
        return Config(table)


class Meta(ChainableModuleWrapperBase):
    __protobuf__ = proto.MetaModule

    REQUIRED = ['id', 'fields']

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()
        super(Meta, self).validate(ctx=ctx, preceding_modules=preceding_modules, chained_modules=chained_modules)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict([
            ('id', self.pb.id),
            ('fields', Config(OrderedDict(sorted(self.pb.fields.items())))),
        ])
        return Config(table)


class Errorlog(ChainableModuleWrapperBase):
    __protobuf__ = proto.ErrorlogModule

    REQUIRED = ['log']

    LOG_LEVELS = ('CRITICAL', 'ERROR', 'INFO', 'DEBUG')
    DEFAULT_LOG_LEVEL = 'ERROR'

    # only for docs for now:
    ALLOWED_CALLS = {
        'log': [defs.get_str_var.name, defs.get_log_path.name],
    }
    DEFAULTS = {
        'log_level': DEFAULT_LOG_LEVEL,
    }

    @validate('log')
    def _validate_log(self):
        log = self.get('log')
        if log.is_func():
            validate_log_call(log.value)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()
        self._validate_log()
        if self.pb.log_level and self.pb.log_level not in self.LOG_LEVELS:
            raise ValidationError('must be one of the {}'.format(', '.join(self.LOG_LEVELS)), 'log_level')
        super(Errorlog, self).validate(ctx=ctx, preceding_modules=preceding_modules, chained_modules=chained_modules)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = {
            'log': self.get('log').to_config(ctx=ctx),
            'log_level': self.pb.log_level or self.DEFAULT_LOG_LEVEL,
        }
        return Config(table)


class Admin(ModuleWrapperBase):
    __protobuf__ = proto.AdminModule

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        if not find_module(preceding_modules, Http):
            raise ValidationError('must preceded by an "http" module')

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        top_level_module = find_module(preceding_modules, (Main, InstanceMacro))
        assert top_level_module
        table = OrderedDict()
        if top_level_module.unistat or self.pb.disable_xml_stats:
            table['disable_xml_stats'] = True
        return Config(table)


class IpdispatchSection(ChainableModuleWrapperBase):
    __protobuf__ = proto.IpdispatchSection

    ips = []  # type: list[Ip]
    local_ips = []  # type: list[Ip]
    ports = []  # type: list[Port]
    local_ports = []  # type: list[Port]

    REQUIRED = ['ips', 'ports']

    SHAREABLE = False

    @classmethod
    def _validate_ips(cls, ips, field_name):
        checked_ips = set()
        for i, ip in enumerate(ips):
            with validate('{}[{}]'.format(field_name, i)):
                ip.validate()
            ip_value = ip.get('value')
            tmp_ip = contains_ip(checked_ips, ip_value)
            if not tmp_ip:
                checked_ips.add(ip_value)
            else:
                raise ValidationError('contains duplicate ip: {}'.format(format_ip(tmp_ip.value)), field_name)

    @classmethod
    def _validate_ports(cls, ports, field_name):
        if ports:
            for i, port in enumerate(ports):
                with validate('{}[{}]'.format(field_name, i)):
                    port.validate()
            ports_counter = collections.Counter([p.get('value') for p in ports])
            port, count = ports_counter.most_common(1)[0]
            if count > 1:
                raise ValidationError('contains duplicate port: {}'.format(port.value), field_name)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()
        if preceding_modules and not isinstance(preceding_modules[-1], (Ipdispatch, InstanceMacro)):
            raise ValidationError('must either be a child of "ipdispatch" or "instance_macro" module '
                                  'or go first in a config tree')

        self._validate_ips(self.ips, 'ips')
        self._validate_ips(self.local_ips, 'local_ips')
        self._validate_ports(self.ports, 'ports')
        self._validate_ports(self.local_ports, 'local_ports')
        if (self.local_ips and not self.local_ports) or (not self.local_ips and self.local_ports):
            raise ValidationError('local_ips and local_ports must be used together')

        if self.is_admin() and self.is_external():
            raise ValidationError('section contains an admin module and binds on external ip addresses')

        super(IpdispatchSection, self).validate(ctx=ctx, preceding_modules=preceding_modules,
                                                chained_modules=chained_modules)

    def list_addrs(self):
        addrs = itertools.product(
            [ip.get('value') for ip in self.ips],
            [port.get('value') for port in self.ports]
        )
        local_addrs = itertools.product(
            [ip.get('value') for ip in self.local_ips],
            [port.get('value') for port in self.local_ports]
        )
        return itertools.chain(addrs, local_addrs)

    def is_admin(self):
        for m in self.nested.walk_chain():
            if isinstance(m, Admin):
                return True
        return False

    def is_external(self):
        for ip_wrapper in self.ips + self.local_ips:
            ip = ip_wrapper.get('value')
            if is_addr_external(ip):
                return True
        return False

    def to_config(self, ctx=DEFAULT_CTX, use_local_addrs=False, preceding_modules=(), *args, **kwargs):
        new_preceding_modules = add_module(preceding_modules, self)
        ips = self.ips if not use_local_addrs else self.local_ips
        ports = self.ports if not use_local_addrs else self.local_ports
        ip_configs = [ip.to_config(ctx=ctx, preceding_modules=new_preceding_modules) for ip in ips]
        port_configs = [port.to_config(ctx=ctx, preceding_modules=new_preceding_modules) for port in ports]
        config = Config(OrderedDict([
            ('ips', Config(array=ip_configs)),
            ('ports', Config(array=port_configs)),
        ]))
        self._add_nested_module_to_config(config, preceding_modules=preceding_modules)
        config.shareable = False
        return config


class Ipdispatch(ModuleWrapperBase):
    __protobuf__ = proto.IpdispatchModule

    __slots__ = ('sections',)

    # sections = OrderedDict()  # type: dict[basestring, IpdispatchSection]
    section_items = []  # type: list[(basestring, IpdispatchSection)]
    include_upstreams = None  # type: IncludeUpstreams | None

    REQUIRED_ONEOFS = [('sections', 'include_upstreams')]

    PB_FIELD_TO_CLS_ATTR_MAPPING = {'sections': 'section_items'}

    def wrap_composite_fields(self):
        super(Ipdispatch, self).wrap_composite_fields()
        self.sections = OrderedDict(self.section_items)

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=()):
        new_preceding_modules = add_module(preceding_modules, self)

        if self.include_upstreams:
            with validate('include_upstreams'):
                if ctx.config_type == ctx.CONFIG_TYPE_UPSTREAM:
                    raise ValidationError('is not allowed in upstream config')
                self.include_upstreams.validate(ctx=ctx, preceding_modules=new_preceding_modules)

        with validate('sections'):
            validate_key_uniqueness(self.section_items)

        for key, section in six.iteritems(self.sections):
            with validate('sections[{}]'.format(key)):
                section.validate(ctx=ctx, preceding_modules=new_preceding_modules)

        for key_1, key_2 in itertools.combinations(list(self.sections), 2):
            section_1 = self.sections[key_1]
            section_2 = self.sections[key_2]
            common_addrs = intersect_addrs(section_1.list_addrs(), section_2.list_addrs())
            if common_addrs:
                common_addrs_str = ', '.join(format_ip_port(ip.value, port.value)
                                             for ip, port in sorted(common_addrs))
                raise ValidationError('{} and {} sections intersect by '
                                      'addresses: {}'.format(key_1, key_2, common_addrs_str))

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        super(Ipdispatch, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def list_addrs(self):
        addrs = []
        for section in six.itervalues(self.sections):
            if not section.is_admin():
                addrs.extend(section.list_addrs())
        return addrs

    def list_admin_addrs(self):
        addrs = []
        for section in six.itervalues(self.sections):
            if section.is_admin():
                addrs.extend(section.list_addrs())
        return addrs

    @staticmethod
    def _port_value_to_slug(value, ctx):
        if value.is_func():
            # replace all the symbols like ()|'", with _
            return re.sub(r'[()"\' ,]+', '_', value.value.to_config(ctx=ctx).to_lua()).strip('_')
        else:
            return value.value

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        preceding_modules = add_module(preceding_modules, self)
        table = OrderedDict()
        for key, section in six.iteritems(self.sections):
            if section.local_ips:
                local_suffix = self._port_value_to_slug(section.local_ports[0].get('value'), ctx)
                local_section_key = '{}_{}'.format(key, local_suffix)
                remote_suffix = self._port_value_to_slug(section.ports[0].get('value'), ctx)
                remote_section_key = '{}_{}'.format(key, remote_suffix)
                if local_section_key == remote_section_key:
                    local_section_key += '_local'
                    remote_section_key += '_remote'
                table[remote_section_key] = section.to_config(ctx=ctx, preceding_modules=preceding_modules)
                table[local_section_key] = section.to_config(ctx=ctx, use_local_addrs=True,
                                                             preceding_modules=preceding_modules)
            else:
                table[key] = section.to_config(ctx=ctx, preceding_modules=preceding_modules)
        return Config(table, shareable=False)

    def get_branches(self):
        return six.itervalues(self.sections)

    def get_named_branches(self):
        return self.sections

    def includes_upstreams(self):
        return self.include_upstreams


class Cutter(ChainableModuleWrapperBase):
    __protobuf__ = proto.CutterModule

    DEFAULT_BYTES = 512
    DEFAULT_TIMEOUT = '0.1s'

    # info attrs:
    DEFAULTS = {
        'bytes': DEFAULT_BYTES,
        'timeout': DEFAULT_TIMEOUT,
    }

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        if self.pb.bytes < 0:
            raise ValidationError('bytes should be a non-negative integer')

        if self.pb.timeout:
            with validate('timeout'):
                validate_timedelta(self.pb.timeout)

        super(Cutter, self).validate(ctx=ctx, preceding_modules=preceding_modules, chained_modules=chained_modules)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict([
            ('bytes', self.pb.bytes or self.DEFAULT_BYTES),
            ('timeout', self.pb.timeout or self.DEFAULT_TIMEOUT),
        ])
        return Config(table)


class H100(ChainableModuleWrapperBase):
    __protobuf__ = proto.H100Module

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        super(H100, self).validate(ctx=ctx, preceding_modules=preceding_modules, chained_modules=chained_modules)


class Hasher(ChainableModuleWrapperBase):
    __protobuf__ = proto.HasherModule

    REQUIRED = ['mode']

    MODES = ('barnavig', 'subnet', 'request', 'text', 'random')

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()
        if self.pb.mode not in self.MODES:
            raise ValidationError('must be one of the {}'.format(', '.join(self.MODES)), 'mode')

        if self.pb.take_ip_from and self.pb.mode != 'subnet':
            raise ValidationError('must not be set if mode is "{}"'.format(self.pb.mode), 'take_ip_from')

        if self.pb.subnet_v4_mask < 0 or self.pb.subnet_v4_mask > 32:
            raise ValidationError('incorrect value on subnet mask for ipv4')

        if self.pb.subnet_v6_mask < 0 or self.pb.subnet_v6_mask > 128:
            raise ValidationError('incorrect value on subnet mask for ipv6')

        super(Hasher, self).validate(ctx=ctx, preceding_modules=preceding_modules, chained_modules=chained_modules)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict([
            ('mode', self.pb.mode),
        ])
        if self.pb.take_ip_from:
            table['take_ip_from'] = self.pb.take_ip_from
        if self.pb.subnet_v4_mask:
            table['subnet_v4_mask'] = self.pb.subnet_v4_mask
        if self.pb.subnet_v6_mask:
            table['subnet_v6_mask'] = self.pb.subnet_v6_mask
        return Config(table)


class CertMixin(object):
    def include_certs(self, namespace_id, cert_spec_pbs, ctx):
        """
        :type namespace_id: six.text_type
        :type cert_spec_pbs: dict[(six.text_type, six.text_type), model_pb2.CertificateSpec]
        :type ctx: ValidationCtx
        :rtype: set[(six.text_type, six.text_type)]
        """
        rv = set()
        for field_name, cert in six.iteritems(self._certs):
            full_cert_id = (namespace_id, cert.id)
            flat_cert_id = flatten_full_id(namespace_id, full_cert_id)
            cert_spec_pb = cert_spec_pbs.get(full_cert_id)
            self.validate_cert(field_name, ctx=ctx)
            if cert_spec_pb is None:
                raise CertDoesNotExist('cert "{}" is missing'.format(flat_cert_id))

            public_cert_call_pb = getattr(self.pb, 'f_cert')
            private_cert_call_pb = getattr(self.pb, 'f_priv')

            fill_get_public_cert_path(call_pb=public_cert_call_pb,
                                      name='allCAs-{}.pem'.format(flat_cert_id),
                                      default_public_cert_dir=DEFAULT_PUBLIC_CERT_DIR)
            fill_get_private_cert_path(call_pb=private_cert_call_pb,
                                       name='{}.pem'.format(flat_cert_id),
                                       default_private_cert_dir=DEFAULT_PRIVATE_CERT_DIR)
            self.pb.ClearField('c_' + field_name)
            rv.add(full_cert_id)
        self.wrap_calls_and_knobs_and_certs()
        return rv

    def _validate_certs(self, ctx):
        self.validate_cert('cert', ctx=ctx)
        self.validate_cert('priv', ctx=ctx)

        cert_is_a_reference = self.get('cert').is_cert()
        priv_is_set = bool(self.get('priv').value)
        with validate('priv'):
            if cert_is_a_reference and priv_is_set:
                raise ValidationError('using "priv" opt while !c-value is used in "cert" opt is prohibited')
            if not cert_is_a_reference and not priv_is_set:
                raise ValidationError('is required')


class HttpMixin(object):
    def _validate_no_keepalive_file(self, ctx):
        self.validate_knob('no_keepalive_file', ctx)

    def _validate_keepalive_params(self, ctx):
        # type self.pb: Http or ExtendedHttpMacro
        keepalive_set_and_enabled = False
        keepalive_set_and_disabled = False
        if self.pb.HasField('keepalive'):
            keepalive_set_and_enabled = self.pb.keepalive.value
            keepalive_set_and_disabled = not self.pb.keepalive.value

        if self.pb.keepalive_timeout:
            with validate('keepalive_timeout'):
                validate_timedelta(self.pb.keepalive_timeout)
                keepalive_timeout_ms = timedelta_to_ms(self.pb.keepalive_timeout)
                if keepalive_set_and_enabled and keepalive_timeout_ms == 0:
                    raise ValidationError('can not be 0 if "keepalive" set to true')
                if keepalive_set_and_disabled and keepalive_timeout_ms != 0:
                    raise ValidationError('must be 0 if "keepalive" set to false')

        if self.pb.HasField('keepalive_requests'):
            keepalive_requests = self.pb.keepalive_requests.value
            with validate('keepalive_requests'):
                if keepalive_requests < 0:
                    raise ValidationError('must be non-negative')
                if keepalive_set_and_enabled and keepalive_requests == 0:
                    raise ValidationError('can not be 0 if "keepalive" set to true')
                if keepalive_set_and_disabled and keepalive_requests != 0:
                    raise ValidationError('must be 0 if "keepalive" set to false')

        if self.pb.HasField('keepalive_drop_probability'):
            keepalive_drop_probability = self.pb.keepalive_drop_probability.value
            with validate('keepalive_drop_probability'):
                if not 0 <= keepalive_drop_probability <= 1:
                    raise ValidationError('must be between or equal to 0 or 1')
                if keepalive_set_and_enabled and keepalive_drop_probability == 1:
                    raise ValidationError('can not be 1 if "keepalive" set to true')
                if keepalive_set_and_disabled and keepalive_drop_probability != 1:
                    raise ValidationError('must be 1 if "keepalive" set to false')


class Http(ChainableModuleWrapperBase, HttpMixin):
    __protobuf__ = proto.HttpModule

    DEFAULT_MAXLEN = 65536
    DEFAULT_MAXREQ = 65536
    DEFAULT_KEEPALIVE = True
    DEFAULT_NO_KEEPALIVE_FILE = './controls/keepalive_disabled'

    ALLOWED_KNOBS = {
        'no_keepalive_file': model_pb2.KnobSpec.BOOLEAN,
        'disable_client_hints_restore_file': model_pb2.KnobSpec.BOOLEAN,
    }
    DEFAULT_KNOB_IDS = {
        'no_keepalive_file': 'balancer_disable_keepalive',
    }

    DEFAULTS = {
        'maxlen': DEFAULT_MAXLEN,
        'maxreq': DEFAULT_MAXREQ,
        'keepalive': DEFAULT_KEEPALIVE,
        'no_keepalive_file': DEFAULT_NO_KEEPALIVE_FILE,
    }

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        if self.pb.maxlen < 0:
            raise ValidationError('must be positive', 'maxlen')
        if self.pb.maxreq < 0:
            raise ValidationError('must be positive', 'maxreq')
        if self.pb.HasField('maxheaders'):
            if self.pb.maxheaders.value <= 0:
                raise ValidationError('must be positive', 'maxheaders')

        self._validate_no_keepalive_file(ctx)
        self._validate_keepalive_params(ctx)
        if self.pb.client_hints_ua_header:
            with validate('client_hints_ua_header'):
                validate_header_name(self.pb.client_hints_ua_header)
        if self.pb.client_hints_ua_proto_header:
            with validate('client_hints_ua_proto_header'):
                validate_header_name(self.pb.client_hints_ua_proto_header)

        super(Http, self).validate(ctx=ctx, preceding_modules=preceding_modules, chained_modules=chained_modules)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        events = DEFAULT_EVENTS
        if self.pb.events:
            events = OrderedDict(sorted(self.pb.events.items()))
        keepalive = self.DEFAULT_KEEPALIVE
        if self.pb.HasField('keepalive'):
            keepalive = self.pb.keepalive.value
        table = OrderedDict([
            ('maxlen', self.pb.maxlen or self.DEFAULT_MAXLEN),
            ('maxreq', self.pb.maxreq or self.DEFAULT_MAXREQ),
            ('keepalive', keepalive),
            ('no_keepalive_file', self.get('no_keepalive_file', self.DEFAULT_NO_KEEPALIVE_FILE).to_config(ctx)),
        ])
        if self.pb.keepalive_timeout:
            table['keepalive_timeout'] = self.pb.keepalive_timeout
        if self.pb.HasField('keepalive_requests'):
            table['keepalive_requests'] = self.pb.keepalive_requests.value
        if self.pb.HasField('keepalive_drop_probability'):
            table['keepalive_drop_probability'] = self.pb.keepalive_drop_probability.value
        table['events'] = Config(events)
        if self.pb.HasField('maxheaders'):
            table['maxheaders'] = self.pb.maxheaders.value
        if self.pb.allow_trace:
            table['allow_trace'] = self.pb.allow_trace
        if self.pb.allow_webdav:
            table['allow_webdav'] = self.pb.allow_webdav
        if self.pb.allow_client_hints_restore:
            table['allow_client_hints_restore'] = self.pb.allow_client_hints_restore
        if self.pb.client_hints_ua_header:
            table['client_hints_ua_header'] = self.pb.client_hints_ua_header
        if self.pb.client_hints_ua_proto_header:
            table['client_hints_ua_proto_header'] = self.pb.client_hints_ua_proto_header
        disable_client_hints_restore_file = self.get('disable_client_hints_restore_file')
        if disable_client_hints_restore_file.value:
            table['disable_client_hints_restore_file'] = disable_client_hints_restore_file.to_config(ctx)
        return Config(table)


class Http2(ChainableModuleWrapperBase):
    __protobuf__ = proto.Http2Module

    ALLOWED_CALLS = {
        'debug_log_name': (defs.get_str_var.name, defs.get_log_path.name),
    }

    DEFAULT_GOAWAY_DEBUG_DATA_ENABLED = False
    DEFAULT_DEBUG_LOG_ENABLED = False

    DEFAULTS = {
        'goaway_debug_data_enabled': DEFAULT_GOAWAY_DEBUG_DATA_ENABLED,
        'debug_log_enabled': DEFAULT_DEBUG_LOG_ENABLED,
    }

    @validate('debug_log_name')
    def _validate_debug_log_name(self):
        debug_log_name = self.get('debug_log_name')
        self.require_value(debug_log_name.value)
        if debug_log_name.is_func():
            validate_log_call(debug_log_name.value)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        ssl_sni = find_last_module(preceding_modules, SslSni)
        if not ssl_sni and not self.pb.allow_http2_without_ssl.value:
            raise ValidationError('must be preceded by ssl_sni module')
        if self.pb.debug_log_enabled:
            self._validate_debug_log_name()
        super(Http2, self).validate(ctx=ctx, preceding_modules=preceding_modules, chained_modules=chained_modules)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict([
            ('goaway_debug_data_enabled', self.pb.goaway_debug_data_enabled),
            ('debug_log_enabled', self.pb.debug_log_enabled),
        ])
        if self.pb.debug_log_enabled:
            table['debug_log_name'] = self.get('debug_log_name').to_config(ctx=ctx)
        if self.pb.HasField('allow_http2_without_ssl'):
            table['allow_http2_without_ssl'] = self.pb.allow_http2_without_ssl.value
        if self.pb.HasField('allow_sending_trailers'):
            table['allow_sending_trailers'] = self.pb.allow_sending_trailers.value

        events = DEFAULT_EVENTS
        if self.pb.events:
            events = OrderedDict(sorted(self.pb.events.items()))
        table['events'] = Config(events)

        return Config(table)


class Report(ChainableModuleWrapperBase):
    __protobuf__ = proto.ReportModule

    __slots__ = ('matcher_map',)

    # matcher_map = OrderedDict()  # type: dict[basestring, Matcher]
    matcher_map_items = []  # type: list[(basestring, Matcher)]

    PB_FIELD_TO_CLS_ATTR_MAPPING = {'matcher_map': 'matcher_map_items'}

    DEFAULT_RANGES_ALIAS = 'default'
    DEFAULT_RANGES = (
        '1ms,4ms,7ms,11ms,17ms,26ms,39ms,58ms,87ms,131ms,197ms,296ms,444ms,666ms,'
        '1000ms,1500ms,2250ms,3375ms,5062ms,7593ms,11390ms,17085ms,30000ms,60000ms,150000ms'
    )
    DEFAULT_JUST_STORAGE = False
    DEFAULT_DISABLE_ROBOTNESS = True
    DEFAULT_DISABLE_SSLNESS = True

    DEFAULTS = {
        'just_storage': DEFAULT_JUST_STORAGE,
        'ranges': DEFAULT_RANGES_ALIAS,
    }

    UUID_RE = r'^[,a-z0-9_-]+$'

    def wrap_composite_fields(self):
        super(Report, self).wrap_composite_fields()
        self.matcher_map = OrderedDict(self.matcher_map_items)

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        with validate('matcher_map'):
            validate_key_uniqueness(self.matcher_map_items)
        if self.pb.outgoing_codes:
            with validate('outgoing_codes'):
                validate_status_codes(self.pb.outgoing_codes, allow_families=False)
        new_preceding_modules = add_module(preceding_modules, self)
        for key, value in six.iteritems(self.matcher_map):
            with validate('matcher_map[{}]'.format(key)):
                value.validate(ctx=ctx, preceding_modules=new_preceding_modules)

        super(Report, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        if not self.pb.uuid and not self.pb.refers:
            raise ValidationError('either "uuid" or "refers" must be specified')
        if self.pb.uuid and not re.match(self.UUID_RE, self.pb.uuid):
            raise ValidationError('must match {}'.format(self.UUID_RE), 'uuid')
        if not self.pb.ranges and not self.pb.refers:
            raise ValidationError('either "ranges" or "refers" must be specified')
        if self.pb.just_storage and self.pb.refers:
            raise ValidationError('"refers" and "just_storage" can not be used together')
        if self.pb.ranges:
            if self.pb.ranges != Report.DEFAULT_RANGES_ALIAS:
                with validate('ranges'):
                    validate_timedeltas(self.pb.ranges)
        if self.pb.backend_time_ranges:
            with validate('backend_time_ranges'):
                validate_timedeltas(self.pb.backend_time_ranges)
        if self.pb.input_size_ranges:
            with validate('input_size_ranges'):
                validate_comma_separated_ints(self.pb.input_size_ranges)
        if self.pb.output_size_ranges:
            with validate('output_size_ranges'):
                validate_comma_separated_ints(self.pb.output_size_ranges)
        if self.pb.disable_signals:
            with validate('disable_signals'):
                validate_item_uniqueness(self.pb.disable_signals)
        super(Report, self).validate(ctx=ctx, preceding_modules=preceding_modules, chained_modules=chained_modules)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        events = DEFAULT_EVENTS
        if self.pb.events:
            events = OrderedDict(sorted(self.pb.events.items()))
        global_vars = {}
        table = OrderedDict()
        if self.pb.uuid:
            table['uuid'] = self.pb.uuid
        if self.pb.refers:
            table['refers'] = self.pb.refers
        if self.pb.ranges:
            if self.pb.ranges == self.DEFAULT_RANGES_ALIAS:
                table['ranges'] = ConfigCall('get_str_var', ('default_ranges',))
                global_vars['default_ranges'] = self.DEFAULT_RANGES
            else:
                table['ranges'] = self.pb.ranges
        if self.pb.backend_time_ranges:
            table['backend_time_ranges'] = self.pb.backend_time_ranges
        if self.pb.input_size_ranges:
            table['input_size_ranges'] = self.pb.input_size_ranges
        if self.pb.output_size_ranges:
            table['output_size_ranges'] = self.pb.output_size_ranges
        if self.pb.matcher_map:
            matcher_map_table = OrderedDict()
            for key, value in six.iteritems(self.matcher_map):
                matcher_map_table[key] = value.to_config(ctx=ctx)
            table['matcher_map'] = Config(matcher_map_table)
        if self.pb.outgoing_codes:
            table['outgoing_codes'] = ','.join(self.pb.outgoing_codes)
        if self.pb.labels:
            table['labels'] = Config(OrderedDict(sorted(self.pb.labels.items())))
        if self.pb.disable_signals:
            table['disable_signals'] = ','.join(self.pb.disable_signals)
        table.update([
            ('just_storage', self.pb.just_storage or self.DEFAULT_JUST_STORAGE),
            ('disable_robotness', self.DEFAULT_DISABLE_ROBOTNESS),
            ('disable_sslness', self.DEFAULT_DISABLE_SSLNESS),
            ('events', Config(events)),
        ])
        return Config(table, global_vars=global_vars, report=True)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=(), *args, **kwargs):
        config = self._to_params_config(ctx=ctx, preceding_modules=preceding_modules)
        self._add_nested_module_to_config(config, ctx=ctx, preceding_modules=preceding_modules)

        # We mark report module as non-shareable if it uses matcher_map,
        # please see SWAT-5801 for details.
        config.shareable = True
        if self.matcher_map and ctx.config_type == ctx.CONFIG_TYPE_FULL:
            top_level_module = find_module(preceding_modules, (l7macro.L7Macro, InstanceMacro, Main))
            if isinstance(top_level_module, l7macro.L7Macro):
                matcher_map_fix_enabled = top_level_module.get_version() >= l7macro.VERSION_0_1_2
            elif isinstance(top_level_module, (InstanceMacro, Main)):
                matcher_map_fix_enabled = top_level_module.pb.enable_matcher_map_fix
            else:
                raise AssertionError(u'top level module is not found in {!r}'.format(preceding_modules))
            if matcher_map_fix_enabled:
                config.shareable = False

        return config


class Threshold(ChainableModuleWrapperBase):
    __protobuf__ = proto.ThresholdModule

    on_pass_timeout_failure = None  # type: Holder | None

    # all real-life configs specify both hi_bytes and lo_bytes, as well as timeouts,
    # so the new logic is to make all fields required, hope this will fit
    REQUIRED = ['hi_bytes', 'lo_bytes', 'pass_timeout', 'recv_timeout']

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        new_preceding_modules = add_module(preceding_modules, self)
        if self.on_pass_timeout_failure:
            with validate('on_pass_timeout_failure'):
                self.on_pass_timeout_failure.validate(ctx=ctx, preceding_modules=new_preceding_modules)
        super(Threshold, self).validate_composite_fields(ctx=ctx,
                                                         preceding_modules=preceding_modules,
                                                         chained_modules=chained_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()
        if self.pb.hi_bytes <= 0:
            raise ValidationError('must be positive', 'hi_bytes')
        if self.pb.lo_bytes <= 0:
            raise ValidationError('must be positive', 'lo_bytes')
        if self.pb.lo_bytes > self.pb.hi_bytes:
            raise ValidationError('lo_bytes > hi_bytes ({} > {}), '
                                  'which is meaningless'.format(self.pb.lo_bytes, self.pb.hi_bytes))
        super(Threshold, self).validate(ctx=ctx, preceding_modules=preceding_modules, chained_modules=chained_modules)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict([
            ('lo_bytes', self.pb.lo_bytes),
            ('hi_bytes', self.pb.hi_bytes),
            ('recv_timeout', self.pb.recv_timeout),
            ('pass_timeout', self.pb.pass_timeout),
        ])
        if self.on_pass_timeout_failure:
            table['on_pass_timeout_failure'] = self.on_pass_timeout_failure.to_config(
                ctx=ctx, preceding_modules=add_module(preceding_modules, self))

        return Config(table)

    def get_branches(self):
        if self.on_pass_timeout_failure:
            yield self.on_pass_timeout_failure

    def get_named_branches(self):
        rv = {}
        if self.on_pass_timeout_failure:
            rv['on_pass_timeout_failure'] = self.on_pass_timeout_failure
        return rv


class Rr(ConfigWrapperBase):
    __protobuf__ = proto.Rr

    ALLOWED_KNOBS = {
        'weights_file': model_pb2.KnobSpec.YB_BACKEND_WEIGHTS,
    }

    def _validate_weights_file(self, ctx):
        self.validate_knob('weights_file', ctx=ctx)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()
        self._validate_weights_file(ctx)
        if self.pb.randomize_initial_state and self.pb.count_of_randomized_requests_on_weights_application:
            raise ValidationError('"randomize_initial_state" and "count_of_randomized_requests_on_weights_application" '
                                  'can not be used together')
        super(Rr, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict()
        weights_file = self.get('weights_file')
        if weights_file.value:
            table['weights_file'] = weights_file.to_config(ctx)
        if self.pb.count_of_randomized_requests_on_weights_application:
            table['count_of_randomized_requests_on_weights_application'] = \
                self.pb.count_of_randomized_requests_on_weights_application
        if self.pb.randomize_initial_state:
            table['randomize_initial_state'] = self.pb.randomize_initial_state
        return Config(table)


class Weighted2(ConfigWrapperBase):
    __protobuf__ = proto.Weighted2

    ALLOWED_KNOBS = {
        'weights_file': model_pb2.KnobSpec.YB_BACKEND_WEIGHTS,
    }

    DEFAULT_MIN_WEIGHT = 0.05
    DEFAULT_MAX_WEIGHT = 5.0
    DEFAULT_PLUS_DIFF_PER_SEC = 0.05
    DEFAULT_MINUS_DIFF_PER_SEC = 0.1
    DEFAULT_HISTORY_TIME = '100s'
    DEFAULT_FEEDBACK_TIME = '300s'
    DEFAULT_SLOW_REPLY_TIME = '1s'

    DEFAULTS = {
        'max_weight': DEFAULT_MAX_WEIGHT,
        'min_weight': DEFAULT_MIN_WEIGHT,
        'history_time': DEFAULT_HISTORY_TIME,
        'feedback_time': DEFAULT_FEEDBACK_TIME,
        'plus_diff_per_sec': DEFAULT_PLUS_DIFF_PER_SEC,
        'minus_diff_per_sec': DEFAULT_MINUS_DIFF_PER_SEC,
        'slow_reply_time': DEFAULT_SLOW_REPLY_TIME,
    }

    def _validate_weights_file(self, ctx):
        self.validate_knob('weights_file', ctx=ctx)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        if self.pb.history_time:
            with validate('history_time'):
                validate_timedelta(self.pb.history_time)

        if self.pb.feedback_time:
            with validate('feedback_time'):
                validate_timedelta(self.pb.feedback_time)

        if self.pb.slow_reply_time:
            with validate('slow_reply_time'):
                validate_timedelta(self.pb.slow_reply_time)

        self._validate_weights_file(ctx)

        super(Weighted2, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        correction_params_config = Config(OrderedDict([
            ('max_weight', self.pb.max_weight or self.DEFAULT_MAX_WEIGHT),
            ('min_weight', self.pb.min_weight or self.DEFAULT_MIN_WEIGHT),
            ('history_time', self.pb.history_time or self.DEFAULT_HISTORY_TIME),
            ('feedback_time', self.pb.feedback_time or self.DEFAULT_FEEDBACK_TIME),
            ('plus_diff_per_sec', self.pb.plus_diff_per_sec or self.DEFAULT_PLUS_DIFF_PER_SEC),
            ('minus_diff_per_sec', self.pb.minus_diff_per_sec or self.DEFAULT_MINUS_DIFF_PER_SEC),
        ]))
        table = OrderedDict()
        weights_file = self.get('weights_file')
        if weights_file.value:
            table['weights_file'] = weights_file.to_config(ctx)
        table.update([
            ('slow_reply_time', self.pb.slow_reply_time or self.DEFAULT_SLOW_REPLY_TIME),
            ('correction_params', correction_params_config),
        ])
        return Config(table)


class Hashing(ConfigWrapperBase):
    __protobuf__ = proto.Hashing

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        require_hashing_module(preceding_modules)
        if bool(self.pb.delay) ^ bool(self.pb.request):
            raise ValidationError('both options "delay" and "request" should be set '
                                  'or not set simultaneously for active + hashing')

        if self.pb.request:
            with validate('request'):
                validate_request_line(self.pb.request)

        if self.pb.delay:
            with validate('delay'):
                validate_timedelta(self.pb.delay)

        super(Hashing, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict()
        if self.pb.delay:
            table['delay'] = self.pb.delay
        if self.pb.request:
            table['request'] = self.pb.request
        if self.pb.HasField('steady'):
            table['steady'] = self.pb.steady.value
        return Config(table)


class Hysteresis(ConfigWrapperBase):
    __protobuf__ = proto.Hysteresis

    REQUIRED = ['value']
    ALLOWED_CALLS = {
        'value': [defs.get_total_weight_percent.name],
    }

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        balancer2 = find_module(preceding_modules, Balancer2)  # type: Balancer2
        assert balancer2
        balancer2_uses_endpoint_sets = (balancer2.generated_proxy_backends and
                                        balancer2.generated_proxy_backends.endpoint_sets)
        self.auto_validate_required()
        value = self.get('value')
        if value.is_func():
            func = value.value
            validate_hysteresis_call(func)
            if func.func_name == defs.get_total_weight_percent.name and balancer2_uses_endpoint_sets:
                raise ValidationError('!f {}() can not be used with endpoint sets'.format(func.func_name))
            if not (0 <= func.func_params.pb.value <= 100):
                raise ValidationError('hysteresis percent must be between 0% and 100%')
        else:
            if not balancer2_uses_endpoint_sets:
                raise ValidationError('absolute value is not allowed, please use get_total_weight_percent')

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        return self.get('value').to_config(ctx=ctx)

    def get_value(self):
        value = self.get('value')
        if value.is_func():
            func = value.value
            validate_hysteresis_call(func)  # just in case
            return func.func_params.pb.value, Active.TYPE_PERCENT
        else:
            return value.value, Active.TYPE_ABSOLUTE


class Quorum(ConfigWrapperBase):
    __protobuf__ = proto.Quorum

    REQUIRED = ['value']
    ALLOWED_CALLS = {
        'value': [defs.get_total_weight_percent.name],
    }

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        balancer2 = find_module(preceding_modules, Balancer2)  # type: Balancer2
        assert balancer2
        balancer2_uses_endpoint_sets = (balancer2.generated_proxy_backends and
                                        balancer2.generated_proxy_backends.endpoint_sets)
        self.auto_validate_required()
        value = self.get('value')
        if value.is_func():
            func = value.value
            validate_quorum_call(func)
            if func.func_name == defs.get_total_weight_percent.name and balancer2_uses_endpoint_sets:
                raise ValidationError('!f {}() can not be used with endpoint sets'.format(func.func_name))
            if not (1 <= func.func_params.pb.value <= 100):
                raise ValidationError('quorum percent must be between 1% and 100%')
        else:
            if not balancer2_uses_endpoint_sets:
                raise ValidationError('absolute value is not allowed, please use get_total_weight_percent')

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        return self.get('value').to_config(ctx=ctx)

    def get_value(self):
        value = self.get('value')
        if value.is_func():
            func = value.value
            validate_quorum_call(func)  # just in case
            return func.func_params.pb.value, Active.TYPE_PERCENT
        else:
            return value.value, Active.TYPE_ABSOLUTE


class Active(ConfigWrapperBase):
    __protobuf__ = proto.Active

    quorum = None  # type: Quorum | None
    hysteresis = None  # type: Hysteresis | None

    REQUIRED = ['request', 'delay']

    TYPE_ABSOLUTE = 0
    TYPE_PERCENT = 1

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=()):
        if self.quorum:
            with validate('quorum'):
                self.quorum.validate(ctx=ctx, preceding_modules=preceding_modules)

        if self.hysteresis:
            with validate('hysteresis'):
                self.hysteresis.validate(ctx=ctx, preceding_modules=preceding_modules)
                if self.quorum:
                    quorum_value, quorum_type = self.quorum.get_value()
                    hysteresis_value, hysteresis_type = self.hysteresis.get_value()
                    if quorum_type != hysteresis_type:
                        raise ValidationError('must have the same units as "quorum"')
                    if quorum_value < hysteresis_value:
                        raise ValidationError('must be less or equal to "quorum"')
                else:
                    raise ValidationError('can not be used without "quorum"')
        super(Active, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        with validate('request'):
            validate_request_line(self.pb.request)
        with validate('delay'):
            validate_timedelta(self.pb.delay)
        super(Active, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def allows_different_backend_weights(self):
        rv = True
        if self.quorum:
            quorum_value, quorum_type = self.quorum.get_value()
            if quorum_type == self.TYPE_PERCENT:
                func = self.quorum.get('value').value
                rv &= (func.func_params.pb.HasField('allow_different_backend_weights') and
                       func.func_params.pb.allow_different_backend_weights.value)
        if self.hysteresis:
            hysteresis_value, hysteresis_type = self.hysteresis.get_value()
            if hysteresis_type == self.TYPE_PERCENT:
                func = self.hysteresis.get('value').value
                rv &= (func.func_params.pb.HasField('allow_different_backend_weights') and
                       func.func_params.pb.allow_different_backend_weights.value)
        return rv

    def to_config(self, total_weight, ctx=DEFAULT_CTX, preceding_modules=()):
        """
        :type total_weight: float
        """
        table = OrderedDict()
        if self.pb.delay:
            table['delay'] = self.pb.delay
        if self.pb.request:
            table['request'] = self.pb.request
        if self.pb.HasField('steady'):
            table['steady'] = self.pb.steady.value

        if self.quorum:
            quorum_value = self.quorum.get('value')
            if quorum_value.is_func():
                quorum_func = quorum_value.value
                if quorum_func.func_name == defs.get_total_weight_percent.name:
                    quorum_value = quorum_func.func_params.pb.value
                    table['quorum'] = total_weight * quorum_value / 100.
                else:
                    raise RuntimeError('unexpected function {}'.format(quorum_func.func_name))
            else:
                table['quorum'] = quorum_value.value

            if self.hysteresis:
                hysteresis_value = self.hysteresis.get('value')
                if hysteresis_value.is_func():
                    hysteresis_func = hysteresis_value.value
                    if hysteresis_func.func_name == defs.get_total_weight_percent.name:
                        hysteresis_value = hysteresis_func.func_params.pb.value
                        table['hysteresis'] = total_weight * hysteresis_value / 100.
                    else:
                        raise RuntimeError('unexpected function {}'.format(hysteresis_value.func_name))
                else:
                    table['hysteresis'] = hysteresis_value.value

        return Config(table)


class DynamicActive(ConfigWrapperBase):
    __protobuf__ = proto.Balancer2Module.Dynamic.Active

    REQUIRED = ['weight_normalization_coeff', 'request', 'delay']

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        with validate('request'):
            validate_request_line(self.pb.request)
        with validate('delay'):
            validate_timedelta(self.pb.delay)
        super(DynamicActive, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict([
            ('request', self.pb.request),
            ('delay', self.pb.delay),
            ('weight_normalization_coeff', self.pb.weight_normalization_coeff),
            ('use_backend_weight', self.pb.use_backend_weight),
        ])
        return Config(table)


class DynamicHashing(ConfigWrapperBase):
    __protobuf__ = proto.Balancer2Module.Dynamic.Hashing

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()


class Dynamic(ConfigWrapperBase):
    __protobuf__ = proto.Balancer2Module.Dynamic

    active = None  # type: DynamicActive | None
    hashing = None  # type: DynamicHashing | None

    DEFAULT_MIN_PESSIMIZATION_COEFF = .1
    DEFAULT_WEIGHT_INCREASE_STEP = .1
    DEFAULT_HISTORY_INTERVAL = '10s'

    REQUIRED = ['max_pessimized_share']

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=()):
        if self.active:
            with validate('active'):
                self.active.validate(ctx=ctx, preceding_modules=preceding_modules)
        if ctx.config_type == ctx.CONFIG_TYPE_FULL:
            require_state_directory(preceding_modules)
        super(Dynamic, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        if self.hashing:
            with validate('hashing'):
                require_hashing_module(preceding_modules)
        with validate('history_interval'):
            if self.pb.history_interval:
                validate_timedelta(self.pb.history_interval)
        super(Dynamic, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict([
            ('max_pessimized_share', self.pb.max_pessimized_share),
            ('min_pessimization_coeff', self.pb.min_pessimization_coeff or self.DEFAULT_MIN_PESSIMIZATION_COEFF),
            ('weight_increase_step', self.pb.weight_increase_step or self.DEFAULT_WEIGHT_INCREASE_STEP),
            ('history_interval', self.pb.history_interval or self.DEFAULT_HISTORY_INTERVAL),
            ('backends_name', self.pb.backends_name),
        ])
        if self.active:
            table['active'] = self.active.to_config(ctx=ctx)
        return Config(table)


class RendezvousHashing(ConfigWrapperBase):
    __protobuf__ = proto.RendezvousHashing

    ALLOWED_KNOBS = {
        'weights_file': model_pb2.KnobSpec.YB_BACKEND_WEIGHTS,
    }

    def _validate_weights_file(self, ctx):
        self.validate_knob('weights_file', ctx=ctx)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        require_hashing_module(preceding_modules)
        if self.pb.reload_duration:
            with validate('reload_duration'):
                validate_timedelta(self.pb.reload_duration)

        if bool(self.pb.delay) ^ bool(self.pb.request):
            raise ValidationError('both options "delay" and "request" should be set '
                                  'or not set simultaneously for active + hashing')

        if self.pb.request:
            with validate('request'):
                validate_request_line(self.pb.request)

        if self.pb.delay:
            with validate('delay'):
                validate_timedelta(self.pb.delay)

        self._validate_weights_file(ctx)

        super(RendezvousHashing, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict()
        weights_file = self.get('weights_file')
        if weights_file.value:
            table['weights_file'] = weights_file.to_config(ctx)
        if self.pb.reload_duration:
            table['reload_duration'] = self.pb.reload_duration
        if self.pb.delay:
            table['delay'] = self.pb.delay
        if self.pb.request:
            table['request'] = self.pb.request
        if self.pb.HasField('steady'):
            table['steady'] = self.pb.steady.value
        return Config(table)


class LoadFactorCriterion(ConfigWrapperBase):
    __protobuf__ = proto.LoadFactorCriterion

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        return Config({})


class FailRateCriterion(ConfigWrapperBase):
    __protobuf__ = proto.FailRateCriterion

    DEFAULT_HISTORY_TIME = '20s'

    DEFAULTS = {
        'history_time': DEFAULT_HISTORY_TIME,
    }

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        if self.pb.history_time:
            with validate('history_time'):
                validate_timedelta(self.pb.history_time)
        super(FailRateCriterion, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict([
            ('history_time', self.pb.history_time or self.DEFAULT_HISTORY_TIME),
        ])
        return Config(table)


class BackendWeightCriterion(ConfigWrapperBase):
    __protobuf__ = proto.BackendWeightCriterion

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        return Config({})


class RequestDurationCriterion(ConfigWrapperBase):
    __protobuf__ = proto.RequestDurationCriterion

    DEFAULT_SLOW_REPLY_TIME = '10s'
    DEFAULT_HISTORY_TIME = '20s'

    DEFAULTS = {
        'history_time': DEFAULT_HISTORY_TIME,
        'slow_reply_time': DEFAULT_SLOW_REPLY_TIME,
    }

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        if self.pb.history_time:
            with validate('history_time'):
                validate_timedelta(self.pb.history_time)
        if self.pb.slow_reply_time:
            with validate('slow_reply_time'):
                validate_timedelta(self.pb.slow_reply_time)
        super(RequestDurationCriterion, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict([
            ('history_time', self.pb.history_time or self.DEFAULT_HISTORY_TIME),
            ('slow_reply_time', self.pb.slow_reply_time or self.DEFAULT_SLOW_REPLY_TIME),
        ])
        return Config(table)


class CombinedCriterionWeightedCriterion(ConfigWrapperBase):
    __protobuf__ = proto.CombinedCriterion.WeightedCriterion

    criterion = None  # type: Pwr2 | None

    REQUIRED = ['weight', 'criterion']

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=()):
        with validate('criterion'):
            self.criterion.validate(ctx=ctx, preceding_modules=preceding_modules)
        super(CombinedCriterionWeightedCriterion, self).validate_composite_fields(ctx=ctx,
                                                                                  preceding_modules=preceding_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        with validate('weight'):
            if self.pb.weight <= 0:
                raise ValidationError('criterion weight must be positive: {} <= 0'.format(self.pb.weight))
        super(CombinedCriterionWeightedCriterion, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict()
        table['weight'] = self.pb.weight
        config = Config(table)
        config.extend(self.criterion.to_config(ctx=ctx, preceding_modules=preceding_modules))
        return config


class CombinedCriterion(ConfigWrapperBase):
    __protobuf__ = proto.CombinedCriterion

    criteria = []  # type: list[CombinedCriterionWeightedCriterion]

    REQUIRED = ['criteria']

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=()):
        for i, criterion in enumerate(self.criteria):
            with validate('criterion[{}]'.format(i)):
                criterion.validate(ctx=ctx, preceding_modules=preceding_modules)
        super(CombinedCriterion, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        super(CombinedCriterion, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        return Config(array=[c.to_config(ctx=ctx, preceding_modules=preceding_modules) for c in self.criteria])


class Pwr2(ConfigWrapperBase):
    __protobuf__ = proto.Pwr2

    load_factor_criterion = None  # type: LoadFactorCriterion | None
    fail_rate_criterion = None  # type: FailRateCriterion | None
    backend_weight_criterion = None  # type: BackendWeightCriterion | None
    request_duration_criterion = None  # type: RequestDurationCriterion | None
    combined_criterion = None  # type: CombinedCriterion | None

    REQUIRED_PB_ONEOFS = ['kind']

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=()):
        kind = self.pb.WhichOneof('kind')
        with validate(kind):
            getattr(self, kind).validate(ctx=ctx, preceding_modules=preceding_modules)
        super(Pwr2, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        super(Pwr2, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        kind = self.pb.WhichOneof('kind')
        table = OrderedDict()
        table[kind] = getattr(self, kind).to_config(ctx=ctx, preceding_modules=preceding_modules)
        return Config(table)


class Leastconn(ConfigWrapperBase):
    __protobuf__ = proto.Leastconn

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        return Config({})


class Balancer2Backend(ChainableModuleWrapperBase):
    __protobuf__ = proto.Balancer2Backend

    REQUIRED = ['weight']
    ALLOWED_CALLS = {
        'name': [defs.get_str_var.name, defs.get_geo.name,
                 defs.prefix_with_dc.name, defs.suffix_with_dc.name]
    }

    # since it's not a "real" module, it can't be shared
    SHAREABLE = False

    @validate('name')
    def _validate_name(self):
        name = self.get('name')
        if name.is_func():
            validate_name_call(name.value)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        if not self.pb.weight:
            raise ValidationError('is required and must not be 0; '
                                  'please use -1 instead of 0 to assign zero weight to backend', 'weight')
        self.auto_validate_required()
        if not preceding_modules or not isinstance(preceding_modules[-1], Balancer2):
            raise ValidationError('must be a child of "balancer2" module')
        self._validate_name()
        super(Balancer2Backend, self).validate(ctx=ctx, preceding_modules=preceding_modules,
                                               chained_modules=chained_modules)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict()
        table['weight'] = float(self.pb.weight)
        return Config(table)


class ProxyHttpsSettings(ConfigWrapperBase):
    __protobuf__ = proto.ProxyHttpsSettings

    REQUIRED = ['ca_file', 'verify_depth']
    DEFAULTS = {
        'ciphers': CipherSuite.DEFAULT,
        'sni_on': False,
        'verify_depth': False,
    }

    @validate('ca_file')
    def _validate_ca_file(self):
        ca_file = self.get('ca_file')
        if ca_file.is_func():
            validate_func_name_one_of(ca_file.value, (defs.get_str_var.name, defs.get_ca_cert_path.name))
            ca_file.value.validate()

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        self._validate_ca_file()
        with validate('verify_depth'):
            if self.pb.verify_depth <= 0:
                raise ValidationError('must be positive')
        super(ProxyHttpsSettings, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict([
            ('ciphers', self.pb.ciphers or CipherSuite.DEFAULT),
            ('ca_file', self.get('ca_file').to_config(ctx=ctx)),
            ('sni_on', self.pb.sni_on),
            ('verify_depth', self.pb.verify_depth)
        ])
        if self.pb.sni_host:
            table['sni_host'] = self.pb.sni_host
        return Config(table)


class ProxyOptionsMixin(object):
    """
    :type pb: proto.GeneratedProxyBackends.ProxyOptions | proto.ProxyModule
    """
    HTTP_BACKEND = True

    DEFAULT_FAIL_ON_5XX = True
    DEFAULT_RESOLVE_TIMEOUT = '10ms'
    DEFAULT_CONNECT_TIMEOUT = '100ms'
    DEFAULT_BACKEND_TIMEOUT = '10s'
    DEFAULT_NEED_RESOLVE = True

    DEFAULTS = {
        'fail_on_5xx': DEFAULT_FAIL_ON_5XX,
        'need_resolve': DEFAULT_NEED_RESOLVE,
        'resolve_timeout': DEFAULT_RESOLVE_TIMEOUT,
        'connect_timeout': DEFAULT_CONNECT_TIMEOUT,
        'backend_timeout': DEFAULT_BACKEND_TIMEOUT,
    }

    def validate_proxy_options(self, ctx=DEFAULT_CTX, preceding_modules=()):
        # scalar fields:
        if self.pb.resolve_timeout:
            with validate('resolve_timeout'):
                validate_timedelta(self.pb.resolve_timeout)
        if self.pb.connect_timeout:
            with validate('connect_timeout'):
                validate_timedelta(self.pb.connect_timeout)
        if self.pb.backend_timeout:
            with validate('backend_timeout'):
                validate_timedelta(self.pb.backend_timeout)
        if self.pb.keepalive_count < 0:
            raise ValidationError('must be non-negative', 'keepalive_count')
        if self.pb.status_code_blacklist:
            with validate('status_code_blacklist'):
                validate_status_codes(self.pb.status_code_blacklist)
        if self.pb.status_code_blacklist_exceptions:
            with validate('status_code_blacklist_exceptions'):
                validate_status_codes(self.pb.status_code_blacklist_exceptions)
        if self.pb.keepalive_timeout:
            with validate('keepalive_timeout'):
                if not self.pb.keepalive_count:
                    raise ValidationError('can only be used with keepalive_count > 0')
                validate_timedelta(self.pb.keepalive_timeout)
        if self.pb.switched_backend_timeout:
            with validate('switched_backend_timeout'):
                validate_timedelta(self.pb.switched_backend_timeout)
        if self.pb.backend_read_timeout:
            with validate('backend_read_timeout'):
                validate_timedelta(self.pb.backend_read_timeout)
        if self.pb.client_read_timeout:
            with validate('client_read_timeout'):
                validate_timedelta(self.pb.client_read_timeout)
        if self.pb.backend_write_timeout:
            with validate('backend_write_timeout'):
                validate_timedelta(self.pb.backend_write_timeout)
        if self.pb.client_write_timeout:
            with validate('client_write_timeout'):
                validate_timedelta(self.pb.client_write_timeout)
        # composite fields:
        if self.https_settings:
            with validate('https_settings'):
                self.https_settings.validate(ctx=ctx, preceding_modules=preceding_modules)

    def get_proxy_options_config(self, ctx=DEFAULT_CTX, preceding_modules=(),
                                 force_need_resolve_to_false=False):
        fail_on_5xx = self.pb.fail_on_5xx.value if self.pb.HasField('fail_on_5xx') else self.DEFAULT_FAIL_ON_5XX
        need_resolve = self.pb.need_resolve.value if self.pb.HasField('need_resolve') else self.DEFAULT_NEED_RESOLVE
        if force_need_resolve_to_false:
            need_resolve = False
        table = OrderedDict([
            ('resolve_timeout', self.pb.resolve_timeout or self.DEFAULT_RESOLVE_TIMEOUT),
            ('connect_timeout', self.pb.connect_timeout or self.DEFAULT_CONNECT_TIMEOUT),
            ('backend_timeout', self.pb.backend_timeout or self.DEFAULT_BACKEND_TIMEOUT),
            ('fail_on_5xx', fail_on_5xx),
            ('http_backend', self.HTTP_BACKEND),
            ('buffering', self.pb.buffering),
            ('keepalive_count', self.pb.keepalive_count),
            ('need_resolve', need_resolve),
        ])
        if self.pb.keepalive_timeout:
            table['keepalive_timeout'] = self.pb.keepalive_timeout
        if self.pb.status_code_blacklist:
            table['status_code_blacklist'] = Config(array=self.pb.status_code_blacklist)
        if self.pb.status_code_blacklist_exceptions:
            table['status_code_blacklist_exceptions'] = Config(array=self.pb.status_code_blacklist_exceptions)
        if self.https_settings:
            table['https_settings'] = self.https_settings.to_config(ctx=ctx)
        if self.pb.switched_backend_timeout:
            table['switched_backend_timeout'] = self.pb.switched_backend_timeout
        if self.pb.backend_read_timeout:
            table['backend_read_timeout'] = self.pb.backend_read_timeout
        if self.pb.client_read_timeout:
            table['client_read_timeout'] = self.pb.client_read_timeout
        if self.pb.allow_connection_upgrade:
            table['allow_connection_upgrade'] = self.pb.allow_connection_upgrade
        if self.pb.backend_write_timeout:
            table['backend_write_timeout'] = self.pb.backend_write_timeout
        if self.pb.client_write_timeout:
            table['client_write_timeout'] = self.pb.client_write_timeout
        if self.pb.allow_connection_upgrade_without_connection_header:
            table[
                'allow_connection_upgrade_without_connection_header'] = self.pb.allow_connection_upgrade_without_connection_header
        if self.pb.watch_client_close:
            table['watch_client_close'] = self.pb.watch_client_close
        if self.pb.http2_backend:
            table['http2_backend'] = self.pb.http2_backend

        return Config(table)


class GeneratedProxyBackendsProxyOptions(ConfigWrapperBase, ProxyOptionsMixin):
    __protobuf__ = proto.GeneratedProxyBackends.ProxyOptions

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.validate_proxy_options(ctx=ctx, preceding_modules=preceding_modules)  # method from ProxyOptionsMixin
        super(GeneratedProxyBackendsProxyOptions, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=(), force_need_resolve_to_false=False):
        return self.get_proxy_options_config(ctx=ctx, preceding_modules=preceding_modules,
                                             force_need_resolve_to_false=force_need_resolve_to_false)  # method from ProxyOptionsMixin


class GeneratedProxyBackendsInstance(ConfigWrapperBase):
    __protobuf__ = proto.GeneratedProxyBackends.Instance

    REQUIRED = ['host', 'port', 'weight']

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        if not self.pb.weight:
            raise ValidationError('is required and must not be 0; '
                                  'please use -1 instead of 0 to assign zero weight to backend', 'weight')
        self.auto_validate_required()
        validate_port(self.pb.port, 'port')
        if self.pb.cached_ip:
            validate_ip(self.pb.cached_ip, 'cached_ip')
            # optimization: do not call parent's validate as we know it does nothing useful

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        array = [self.pb.host, self.pb.port, float(self.pb.weight)]
        if self.pb.cached_ip:
            array.append(self.pb.cached_ip)
        return Config(array=array, compact=True)


class GeneratedProxyBackendsEndpointSet(ConfigWrapperBase):
    __protobuf__ = proto.GeneratedProxyBackends.EndpointSet

    REQUIRED = ['cluster', 'id']

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        # optimization: do not call parent's validate as we know it does nothing useful
        pass

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        return Config(OrderedDict([
            ('cluster_name', cluster_name_from_alias(self.pb.cluster)),
            ('endpoint_set_id', self.pb.id),
        ]))


class GeneratedProxyBackendsNannySnapshot(ConfigWrapperBase):
    __protobuf__ = proto.GeneratedProxyBackends.NannySnapshot

    REQUIRED = ['service_id', 'snapshot_id']

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        if self.pb.HasField('port'):
            validate_port(self.pb.port.value, 'port')
        super(GeneratedProxyBackendsNannySnapshot, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        return Config()


class GeneratedProxyBackendsGencfgGroup(ConfigWrapperBase):
    __protobuf__ = proto.GeneratedProxyBackends.GencfgGroup

    REQUIRED = ['name', 'version']

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        if self.pb.version != 'trunk' and not self.pb.version.startswith('tags/'):
            raise ValidationError('must be either equal to "trunk" or start with "tags/" prefix', 'version')
        if self.pb.HasField('port'):
            validate_port(self.pb.port.value, 'port')
        super(GeneratedProxyBackendsGencfgGroup, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        return Config()


class GeneratedProxySdOptions(ConfigWrapperBase):
    __protobuf__ = proto.GeneratedProxyBackends.SdOptions

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        if self.pb.termination_delay:
            with validate('termination_delay'):
                validate_timedelta(self.pb.termination_delay)
        if self.pb.termination_deadline:
            with validate('termination_deadline'):
                validate_timedelta(self.pb.termination_deadline)
        super(GeneratedProxySdOptions, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict()
        if self.pb.termination_delay:
            table['termination_delay'] = self.pb.termination_delay
        if self.pb.termination_delay:
            table['termination_deadline'] = self.pb.termination_deadline
        return Config(table)


class AttemptsRateLimiter(ConfigWrapperBase):
    __protobuf__ = proto.AttemptsRateLimiter

    DEFAULT_COEFF = 0.99
    MAX_LIMIT = 3

    REQUIRED = ['limit']
    DEFAULTS = {
        'coeff': DEFAULT_COEFF,
    }

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        balancer2 = find_last_module(preceding_modules, Balancer2)  # type: Balancer2
        assert balancer2
        attempts = balancer2.get('attempts')
        with validate('limit'):
            validate_range(self.pb.limit, 0, self.MAX_LIMIT, exclusive_min=True, exclusive_max=False)
        if self.pb.coeff and self.pb.max_budget:
            raise ValidationError('at most one of "coeff", "max_budget" must be specified')
        if self.pb.coeff:
            with validate('coeff'):
                validate_range(self.pb.coeff, 0, 1, exclusive_min=True, exclusive_max=True)
        if self.pb.max_budget:
            with validate('max_budget'):
                if self.pb.max_budget <= 0:
                    raise ValidationError('must be greater than 0')

        super(AttemptsRateLimiter, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict([
            ('limit', self.pb.limit),
        ])
        if self.pb.max_budget:
            table['max_budget'] = self.pb.max_budget
        else:
            table['coeff'] = self.pb.coeff or self.DEFAULT_COEFF
        table['switch_default'] = True
        return Config(table)


class GeneratedProxyBackends(ModuleWrapperBase):
    __protobuf__ = proto.GeneratedProxyBackends

    proxy_options = None  # type: GeneratedProxyBackendsProxyOptions | None
    sd_options = None  # type: GeneratedProxySdOptions | None
    include_backends = None  # type: IncludeBackends | None
    instances = []  # type: list[GeneratedProxyBackendsInstance]
    nanny_snapshots = []  # type: list[GeneratedProxyBackendsNannySnapshot]
    gencfg_groups = []  # type: list[GeneratedProxyBackendsGencfgGroup]
    endpoint_sets = []  # type: list[GeneratedProxyBackendsEndpointSet]

    INSTANCES_LIMIT = 2000
    GENCFG_GROUPS_LIMIT = 20
    NANNY_SNAPSHOTS_LIMIT = 20
    ENDPOINT_SETS_LIMIT = 10

    REQUIRED = ['proxy_options']
    REQUIRED_ONEOFS = [('instances', 'nanny_snapshots', 'gencfg_groups', 'endpoint_sets', 'include_backends')]

    def includes_backends(self):
        return bool(self.include_backends)

    def get_would_be_included_full_backend_ids(self, current_namespace_id):
        if self.include_backends:
            return self.include_backends.get_included_full_backend_ids(current_namespace_id)
        else:
            return set()

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        super(GeneratedProxyBackends, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=()):
        preceding_modules = add_module(preceding_modules, self)
        with validate('proxy_options'):
            self.proxy_options.validate(ctx=ctx, preceding_modules=preceding_modules)
        if self.sd_options:
            with validate('sd_options'):
                self.sd_options.validate(ctx=ctx, preceding_modules=preceding_modules)

        if len(self.instances) > self.INSTANCES_LIMIT:
            raise ValidationError(
                'number of instances ({}) exceeds allowed limit of {}'.format(len(self.instances),
                                                                              self.INSTANCES_LIMIT),
                'instances')

        if len(self.gencfg_groups) > self.GENCFG_GROUPS_LIMIT:
            raise ValidationError(
                'number of gencfg groups ({}) exceeds allowed limit of {}'.format(len(self.gencfg_groups),
                                                                                  self.GENCFG_GROUPS_LIMIT),
                'gencfg_groups')

        if len(self.nanny_snapshots) > self.NANNY_SNAPSHOTS_LIMIT:
            raise ValidationError(
                'number of Nanny snapshots ({}) exceeds allowed limit of {}'.format(len(self.nanny_snapshots),
                                                                                    self.NANNY_SNAPSHOTS_LIMIT),
                'nanny_snapshots')

        if len(self.endpoint_sets) > self.ENDPOINT_SETS_LIMIT:
            raise ValidationError(
                'number of endpoint sets ({}) exceeds allowed limit of {}'.format(len(self.endpoint_sets),
                                                                                  self.ENDPOINT_SETS_LIMIT),
                'endpoint_sets')

        if self.include_backends:
            with validate('include_backends'):
                self.include_backends.validate(ctx=ctx, preceding_modules=preceding_modules)

        if self.endpoint_sets and ctx.config_type == ctx.CONFIG_TYPE_FULL:
            require_sd(preceding_modules, field_name='endpoint_sets')

        instances_by_spec = set()
        for i, instance in enumerate(self.instances):
            # do not use
            # with validate('instances[{}]'.format(i)):
            # use faster try-except
            try:
                instance.validate(ctx=ctx, preceding_modules=preceding_modules)
            except ValidationError as e:
                field_name = 'instances[{}]'.format(i)
                e.path.appendleft(field_name)
            spec = (instance.pb.host, instance.pb.port)
            if spec in instances_by_spec and not self.pb.ignore_duplicates:
                field_name = 'instances[{}]'.format(i)
                raise ValidationError('duplicate host and port: {}:{}'.format(*spec), field_name)
            instances_by_spec.add(spec)

        snapshots_by_spec = set()
        for i, snapshot in enumerate(self.nanny_snapshots):
            with validate('nanny_snapshots[{}]'.format(i)):
                snapshot.validate(ctx=ctx, preceding_modules=preceding_modules)
                spec = (snapshot.pb.service_id, snapshot.pb.snapshot_id)
                if spec in snapshots_by_spec:
                    raise ValidationError('duplicate service_id and snapshot_id: {}:{}'.format(*spec))
                snapshots_by_spec.add(spec)

        gencfg_groups_by_spec = set()
        for i, group in enumerate(self.gencfg_groups):
            with validate('gencfg_groups[{}]'.format(i)):
                group.validate(ctx=ctx, preceding_modules=preceding_modules)
                spec = (group.pb.name, group.pb.version)
                if spec in gencfg_groups_by_spec:
                    raise ValidationError('duplicate name and version: {}:{}'.format(*spec))
                gencfg_groups_by_spec.add(spec)

        endpoint_sets_by_spec = set()
        for i, yp_service in enumerate(self.endpoint_sets):
            with validate('endpoint_sets[{}]'.format(i)):
                yp_service.validate(ctx=ctx, preceding_modules=preceding_modules)
                spec = (cluster_name_from_alias(yp_service.pb.cluster), yp_service.pb.id)
                if spec in endpoint_sets_by_spec:
                    raise ValidationError('duplicate cluster and id: {}:{}'.format(*spec))
                endpoint_sets_by_spec.add(spec)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        if self.endpoint_sets:
            endpoint_set_configs = [
                endpoint_set.to_config(ctx=ctx, preceding_modules=preceding_modules)
                for endpoint_set in self.endpoint_sets
            ]
            # we use force_need_resolve_to_false here due to https://st.yandex-team.ru/BALANCER-2679
            proxy_options_config = self.proxy_options.to_config(ctx=ctx, preceding_modules=preceding_modules,
                                                                force_need_resolve_to_false=True)

            table = OrderedDict([
                ('endpoint_sets', Config(array=endpoint_set_configs)),
                ('proxy_options', proxy_options_config),
            ])
            if self.sd_options:
                table['sd'] = self.sd_options.to_config(ctx=ctx, preceding_modules=preceding_modules)
            return Config(table)
        else:
            instances = self.instances
            if self.pb.ignore_duplicates:
                seen = set()
                filtered_instances = []
                for i in instances:
                    if (i.pb.host, i.pb.port) not in seen:
                        seen.add((i.pb.host, i.pb.port))
                        filtered_instances.append(i)
                instances = filtered_instances
            instances = sorted(instances, key=lambda i: (i.pb.host, i.pb.port))
            args = [
                Config(array=[i.to_config(ctx=ctx) for i in instances]),
                self.proxy_options.to_config(ctx=ctx, preceding_modules=preceding_modules),
            ]
            return Config(array=ConfigCall('gen_proxy_backends', args))

    def get_total_weight(self, allow_different_backend_weights=False):
        """
        :type allow_different_backend_weights: bool
        :rtype: float
        :raises: ValidationError
        """
        it = iter(self.instances)
        instance = next(it)
        rv = instance.pb.weight
        for next_instance in it:
            if not allow_different_backend_weights and not is_close(instance.pb.weight, next_instance.pb.weight):
                raise ValidationError('weights of instances {}:{} and {}:{} differ. '
                                      'if this difference is expected and you are sure that specified '
                                      'quorum and hysteresis percentages are still valid, '
                                      'please set "allow_different_backend_weights" to true'.format(
                                          instance.pb.host, instance.pb.port, next_instance.pb.host, next_instance.pb.port))
            instance = next_instance
            rv += instance.pb.weight
        return rv


class SimplePolicy(ConfigWrapperBase):
    __protobuf__ = proto.SimplePolicy

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        return Config({})


class UniquePolicy(ConfigWrapperBase):
    __protobuf__ = proto.UniquePolicy

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        return Config({})


class DependendPolicyMixin(object):
    balancing_policy = None  # type: BalancingPolicy

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=()):
        with validate('balancing_policy'):
            self.balancing_policy.validate(ctx=ctx, preceding_modules=preceding_modules)
        super(DependendPolicyMixin, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules)


class TimeoutPolicy(DependendPolicyMixin, ConfigWrapperBase):
    __protobuf__ = proto.TimeoutPolicy

    REQUIRED = ['timeout', 'balancing_policy']

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        with validate('timeout'):
            validate_timedelta(self.pb.timeout)
        super(TimeoutPolicy, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        config = Config(OrderedDict([
            ('timeout', self.pb.timeout),
        ]))
        config.extend(self.balancing_policy.to_config(ctx=ctx, preceding_modules=preceding_modules))
        return config


class ActivePolicy(DependendPolicyMixin, ConfigWrapperBase):
    __protobuf__ = proto.ActivePolicy

    REQUIRED = ['balancing_policy']

    ALLOWED_CALLS = {
        'skip_attempts': [defs.count_backends.name],
    }

    @validate('skip_attempts')
    def _validate_skip_attempts(self, preceding_modules=()):
        skip_attempts = self.get('skip_attempts')

        if not skip_attempts.value:
            return

        balancer2 = find_last_module(preceding_modules, Balancer2)
        if not balancer2:
            raise ValidationError('must be preceded by balancer2 module')

        if not balancer2.hashing:
            raise ValidationError('can only be used with "hashing" balancing type')

        if skip_attempts.is_func():
            validate_func_name_one_of(skip_attempts.value, [defs.count_backends.name])
        else:
            if skip_attempts.value < 0:
                raise ValidationError('must be non-negative')

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        self._validate_skip_attempts(preceding_modules=preceding_modules)
        super(ActivePolicy, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict()

        skip_attempts = self.get('skip_attempts')
        if skip_attempts.is_func():
            if skip_attempts.value.func_name == defs.count_backends.name:
                balancer2 = find_last_module(preceding_modules, Balancer2)

                if balancer2.generated_proxy_backends and balancer2.generated_proxy_backends.endpoint_sets:
                    # https://st.yandex-team.ru/AWACS-511
                    # skip_attempts is infinite by default (https://wiki.yandex-team.ru/balancer/cookbook/#activepolicy),
                    # and active_policy can only be followed by unique_policy, so the resulting number of attempts
                    # would be equal to number of endpoints
                    pass
                else:
                    table['skip_attempts'] = balancer2.count_backends()
        elif skip_attempts.value:
            table['skip_attempts'] = skip_attempts.value

        config = Config(table)
        config.extend(self.balancing_policy.to_config(ctx=ctx, preceding_modules=preceding_modules))
        return config


class ByNamePolicy(DependendPolicyMixin, ConfigWrapperBase):
    __protobuf__ = proto.ByNamePolicy

    REQUIRED = ['name', 'balancing_policy']

    ALLOWED_CALLS = {
        'name': [defs.get_str_var.name, defs.get_geo.name,
                 defs.prefix_with_dc.name, defs.suffix_with_dc.name]
    }

    @validate('name')
    def _validate_name(self):
        name = self.get('name')
        if name.is_func():
            validate_name_call(name.value)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        self._validate_name()
        super(ByNamePolicy, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict([
            ('name', self.get('name').to_config(ctx=ctx)),
        ])
        if self.pb.allow_zero_weights:
            table['allow_zero_weights'] = self.pb.allow_zero_weights
        if self.pb.strict:
            table['strict'] = self.pb.strict
        config = Config(table)
        config.extend(self.balancing_policy.to_config(ctx=ctx, preceding_modules=preceding_modules))
        return config


class ByNameFromHeaderPolicyHint(ConfigWrapperBase):
    __protobuf__ = proto.ByNameFromHeaderPolicy.Hint

    REQUIRED = ['hint', 'backend']

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        super(ByNameFromHeaderPolicyHint, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        return Config(OrderedDict([
            ('hint', self.pb.hint),
            ('backend', self.pb.backend),
        ]))


class ByNameFromHeaderPolicy(DependendPolicyMixin, ConfigWrapperBase):
    __protobuf__ = proto.ByNameFromHeaderPolicy

    hints = []  # type: list[ByNameFromHeaderPolicyHint]

    REQUIRED = ['hints', 'balancing_policy']

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=()):
        hint_names = set()
        for i, hint in enumerate(self.hints):
            with validate('hints[{}]'.format(i)):
                hint_name = hint.pb.hint
                if hint_name:
                    if hint_name in hint_names:
                        raise ValidationError('duplicate hint: "{}"'.format(hint_name), 'hint')
                    hint_names.add(hint_name)
                hint.validate(ctx=ctx, preceding_modules=preceding_modules)

        if self.pb.header_name:
            with validate('header_name'):
                validate_header_name(self.pb.header_name)

        super(ByNameFromHeaderPolicy, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        super(ByNameFromHeaderPolicy, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict()
        if self.pb.header_name:
            table['header_name'] = self.pb.header_name
        if self.pb.allow_zero_weights:
            table['allow_zero_weights'] = self.pb.allow_zero_weights
        if self.pb.strict:
            table['strict'] = self.pb.strict
        hints = [hint.to_config(ctx=ctx, preceding_modules=preceding_modules)
                 for hint in self.hints]
        table['hints'] = Config(array=hints)
        config = Config(table)
        config.extend(self.balancing_policy.to_config(ctx=ctx, preceding_modules=preceding_modules))
        return config


class ByHashPolicy(DependendPolicyMixin, ConfigWrapperBase):
    __protobuf__ = proto.ByHashPolicy

    REQUIRED = ['balancing_policy']

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        super(ByHashPolicy, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        return self.balancing_policy.to_config(ctx=ctx, preceding_modules=preceding_modules)


class RetryPolicy(DependendPolicyMixin, ConfigWrapperBase):
    __protobuf__ = proto.RetryPolicy

    REQUIRED = ['balancing_policy']

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        super(RetryPolicy, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        return self.balancing_policy.to_config(ctx=ctx, preceding_modules=preceding_modules)


class WatermarkPolicy(DependendPolicyMixin, ConfigWrapperBase):
    __protobuf__ = proto.WatermarkPolicy

    REQUIRED = ['balancing_policy']

    DEFAULT_PARAMS_FILE = './controls/watermark_policy.params_file'

    DEFAULTS = {
        'params_file': DEFAULT_PARAMS_FILE,
    }
    ALLOWED_KNOBS = {
        'params_file': model_pb2.KnobSpec.YB_WATERMARK_POLICY_PARAMS,
    }
    DEFAULT_KNOB_IDS = {
        'params_file': 'common_watermark_policy_params_file',
    }

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        if self.pb.lo and not (0 <= self.pb.lo <= 1):
            raise ValidationError('must be between 0 and 1', 'lo')
        if self.pb.hi and not (0 <= self.pb.hi <= 1):
            raise ValidationError('must be between 0 and 1', 'hi')
        if self.pb.lo > self.pb.hi:
            raise ValidationError('lo > hi ({} > {}), which is meaningless'.format(self.pb.lo, self.pb.hi))
        if self.pb.coeff and not (0 <= self.pb.coeff <= 1):
            raise ValidationError('must be between 0 and 1', 'coeff')
        super(WatermarkPolicy, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict([
            ('lo', self.pb.lo),
            ('hi', self.pb.hi),
            ('params_file', self.get('params_file', self.DEFAULT_PARAMS_FILE).to_config(ctx)),
        ])
        if self.pb.HasField('shared'):
            table['shared'] = self.pb.shared.value
        if self.pb.coeff:
            table['coeff'] = self.pb.coeff

        config = Config(table)

        config.extend(self.balancing_policy.to_config(ctx=ctx, preceding_modules=preceding_modules))
        return config


class BalancingPolicy(ConfigWrapperBase):
    __protobuf__ = proto.BalancingPolicy

    simple_policy = None  # type: SimplePolicy | None
    unique_policy = None  # type: UniquePolicy | None
    timeout_policy = None  # type: TimeoutPolicy | None
    active_policy = None  # type: ActivePolicy | None
    by_name_policy = None  # type: ByNamePolicy | None
    by_hash_policy = None  # type: ByHashPolicy | None
    retry_policy = None  # type: RetryPolicy | None
    watermark_policy = None  # type: WatermarkPolicy | None

    REQUIRED_PB_ONEOFS = ['kind']

    def list_policy_kinds(self):
        kind = self.pb.WhichOneof('kind')
        nested_policy = getattr(self, kind)

        rv = [kind]
        if nested_policy and hasattr(nested_policy, 'balancing_policy'):
            rv += nested_policy.balancing_policy.list_policy_kinds()
        return rv

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=()):
        kind = self.pb.WhichOneof('kind')
        with validate(kind):
            getattr(self, kind).validate(ctx=ctx, preceding_modules=preceding_modules)
        super(BalancingPolicy, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        super(BalancingPolicy, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        kind = self.pb.WhichOneof('kind')
        table = OrderedDict()
        table[kind] = getattr(self, kind).to_config(ctx=ctx, preceding_modules=preceding_modules)
        return Config(table)


class Balancer2DelaySettings(ConfigWrapperBase):
    __protobuf__ = proto.Balancer2Module.DelaySettings

    REQUIRED = ['first_delay', 'delay_multiplier', 'delay_on_fast', 'max_random_delay']

    MAX_FIRST_DELAY = '10s'
    MAX_MAX_RANDOM_DELAY = '10s'

    @validate('first_delay')
    def _validate_first_delay(self):
        validate_timedelta(self.pb.first_delay)
        validate_timedelta_range(self.pb.first_delay, '0ms', self.MAX_FIRST_DELAY)

    @validate('delay_multiplier')
    def _validate_delay_multiplier(self):
        validate_range(self.pb.delay_multiplier, 1, 2)

    @validate('max_random_delay')
    def _validate_max_random_delay(self):
        validate_timedelta(self.pb.max_random_delay)
        validate_timedelta_range(self.pb.max_random_delay, '0ms', self.MAX_MAX_RANDOM_DELAY)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        self._validate_first_delay()
        self._validate_delay_multiplier()
        self._validate_max_random_delay()


class Balancer2CheckBackends(ConfigWrapperBase):
    __protobuf__ = proto.Balancer2Module.CheckBackends

    REQUIRED = ['quorum', 'name']

    @validate('quorum')
    def _validate_quorum(self):
        validate_range(self.pb.quorum.value, 0, 1)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        self._validate_quorum()

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        return Config(OrderedDict([
            ('quorum', self.pb.quorum.value),
            ('name', self.pb.name),
        ]))


class Balancer2(ModuleWrapperBase):
    __protobuf__ = proto.Balancer2Module

    __slots__ = ('on_status_code',)

    rr = None  # type: Rr | None
    weighted2 = None  # type: Weighted2 | None
    hashing = None  # type: Hashing | None
    active = None  # type: Active | None
    rendezvous_hashing = None  # type: RendezvousHashing | None
    pwr2 = None  # type: Pwr2 | None
    leastconn = None  # type: Leastconn | None
    dynamic = None  # type: Dynamic | None
    balancing_policy = None  # type: BalancingPolicy | None
    on_error = None  # type: Holder | None
    on_fast_error = None  # type: Holder | None
    backends = []  # type: list[Balancer2Backend]
    generated_proxy_backends = None  # type: GeneratedProxyBackends | None
    attempts_rate_limiter = None  # type: AttemptsRateLimiter | None
    # on_status_code = OrderedDict()  # type: dict[basestring, Holder]
    on_status_code_items = []  # type: list[(basestring, Holder)]
    delay_settings = None  # type: Balancer2DelaySettings | None
    check_backends = None  # type: Balancer2CheckBackends | None

    REQUIRED = ['attempts']
    REQUIRED_ONEOFS = [
        ('rr', 'weighted2', 'hashing', 'active', 'rendezvous_hashing', 'pwr2', 'leastconn', 'dynamic'),
        ('backends', 'generated_proxy_backends'),
    ]
    DEFAULT_RETRY_NON_IDEMPOTENT = True

    # info attrs:
    DEFAULTS = {
        'retry_non_idempotent': DEFAULT_RETRY_NON_IDEMPOTENT,
    }
    ALLOWED_CALLS = {
        'attempts': [defs.count_backends.name, defs.count_backends_sd.name],
        'connection_attempts': [defs.count_backends.name, defs.count_backends_sd.name],
        'fast_attempts': [defs.count_backends.name, defs.count_backends_sd.name],
    }
    ALLOWED_KNOBS = {
        'attempts_file': model_pb2.KnobSpec.INTEGER,
    }

    PB_FIELD_TO_CLS_ATTR_MAPPING = {'on_status_code': 'on_status_code_items'}

    def wrap_composite_fields(self):
        super(Balancer2, self).wrap_composite_fields()
        self.on_status_code = OrderedDict(self.on_status_code_items)

    def has_endpoint_sets(self):
        return bool(self.generated_proxy_backends and self.generated_proxy_backends.endpoint_sets)

    def has_instances(self):
        return bool(self.generated_proxy_backends and self.generated_proxy_backends.instances)

    @validate('attempts')
    def _validate_attempts(self):
        attempts = self.get('attempts')
        if attempts.is_func():
            validate_func_name_one_of(attempts.value, [defs.count_backends.name, defs.count_backends_sd.name])

            if (attempts.value.func_name == defs.count_backends.name and
                    (attempts.value.func_params.pb.HasField('compat_enable_sd_support') and
                     not attempts.value.func_params.pb.compat_enable_sd_support.value) and
                    self.has_endpoint_sets()):
                raise ValidationError('!f count_backends(false) can not be used with endpoint sets, '
                                      'please use !f count_backends() instead')

            if attempts.value.func_name == defs.count_backends_sd.name and self.has_instances():
                raise ValidationError('!f count_backends_sd() can only be used with endpoint sets, '
                                      'please use !f count_backends() instead')
        else:
            if attempts.value < 0:
                raise ValidationError('must be positive')

    @validate('rewind_limit')
    def _validate_rewind_limit(self):
        if self.pb.rewind_limit < 0:
            raise ValidationError('must be positive')

    @validate('connection_attempts')
    def _validate_connection_attempts(self):
        connection_attempts = self.get('connection_attempts')
        if connection_attempts.is_func():
            validate_func_name_one_of(connection_attempts.value,
                                      [defs.count_backends.name, defs.count_backends_sd.name])

            if (connection_attempts.value.func_name == defs.count_backends.name and
                    (connection_attempts.value.func_params.pb.HasField('compat_enable_sd_support') and
                     not connection_attempts.value.func_params.pb.compat_enable_sd_support.value) and
                    self.has_endpoint_sets()):
                raise ValidationError('!f count_backends(false) can not be used with endpoint sets, '
                                      'please use !f count_backends() instead')

            if connection_attempts.value.func_name == defs.count_backends_sd.name and self.has_instances():
                raise ValidationError('!f count_backends_sd() can only be used with endpoint sets, '
                                      'please use !f count_backends() instead')
        else:
            if connection_attempts.value < 0:
                raise ValidationError('must be positive')

    @validate('fast_attempts')
    def _validate_fast_attempts(self):
        fast_attempts = self.get('fast_attempts')
        if fast_attempts.is_func():
            validate_func_name_one_of(fast_attempts.value, [defs.count_backends.name, defs.count_backends_sd.name])

            if (fast_attempts.value.func_name == defs.count_backends.name and
                    (fast_attempts.value.func_params.pb.HasField('compat_enable_sd_support') and
                     not fast_attempts.value.func_params.pb.compat_enable_sd_support.value) and
                    self.has_endpoint_sets()):
                raise ValidationError('!f count_backends(false) can not be used with endpoint sets, '
                                      'please use !f count_backends() instead')

            if fast_attempts.value.func_name == defs.count_backends_sd.name and self.has_instances():
                raise ValidationError('!f count_backends_sd() can only be used with endpoint sets, '
                                      'please use !f count_backends() instead')
        else:
            if fast_attempts.value < 0:
                raise ValidationError('must be positive')

    @validate('hedged_delay')
    def _validate_hedged_delay(self):
        if self.pb.hedged_delay:
            if not self.attempts_rate_limiter:
                raise ValidationError('"attempts_rate_limiter" should be used if "hedged_delay" specified')
            if not self.attempts_rate_limiter.pb.max_budget:
                raise ValidationError('"attempts_rate_limiter.max_budget" should be used if "hedged_delay" specified')
            validate_timedelta(self.pb.hedged_delay)

    def _validate_attempts_file(self, ctx):
        self.validate_knob('attempts_file', ctx=ctx)

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=()):
        preceding_modules = add_module(preceding_modules, self)

        if self.rr:
            with validate('rr'):
                self.rr.validate(ctx=ctx, preceding_modules=preceding_modules)

        if self.weighted2:
            with validate('weighted2'):
                self.weighted2.validate(ctx=ctx, preceding_modules=preceding_modules)

        if self.hashing:
            with validate('hashing'):
                self.hashing.validate(ctx=ctx, preceding_modules=preceding_modules)

        if self.active:
            # call get_total_weight just to validate
            self.get_total_weight(allow_different_backend_weights=self.active.allows_different_backend_weights())
            with validate('active'):
                self.active.validate(ctx=ctx, preceding_modules=preceding_modules)

        if self.rendezvous_hashing:
            with validate('rendezvous_hashing'):
                self.rendezvous_hashing.validate(ctx=ctx, preceding_modules=preceding_modules)

        if self.pwr2:
            with validate('pwr2'):
                self.pwr2.validate(ctx=ctx, preceding_modules=preceding_modules)

        if self.leastconn:
            with validate('leastconn'):
                self.leastconn.validate(ctx=ctx, preceding_modules=preceding_modules)

        if self.balancing_policy:
            with validate('balancing_policy'):
                self.balancing_policy.validate(ctx=ctx, preceding_modules=preceding_modules)

                # https://st.yandex-team.ru/SWAT-7537
                policy_kinds = self.balancing_policy.list_policy_kinds()
                if 'active_policy' in policy_kinds:
                    i = policy_kinds.index('active_policy')
                    if policy_kinds[i + 1:] != ['unique_policy']:
                        raise ValidationError('active_policy can only be followed by unique_policy')

        if self.attempts_rate_limiter:
            with validate('attempts_rate_limiter'):
                if self.pb.disable_attempts_rate_limiter.value:
                    raise ValidationError('cannot be used together with disable_attempts_rate_limiter')

                self.attempts_rate_limiter.validate(ctx=ctx, preceding_modules=preceding_modules)

        backend_names = set()
        for i, backend in enumerate(self.backends):
            with validate('backends[{}]'.format(i)):
                name = backend.pb.name
                if name:
                    if name in backend_names:
                        raise ValidationError('duplicate backend name: "{}"'.format(name), 'name')
                    backend_names.add(name)
                backend.validate(ctx=ctx, preceding_modules=preceding_modules)

        if self.generated_proxy_backends:
            with validate('generated_proxy_backends'):
                self.generated_proxy_backends.validate(ctx=ctx, preceding_modules=preceding_modules)

            connection_attempts = self.get('connection_attempts')
            fast_attempts = self.get('fast_attempts')
            if fast_attempts.is_func() or fast_attempts.value != 0:
                if connection_attempts.is_func() or connection_attempts.value != 0:
                    raise ValidationError('options "fast_attempts" and "connection_attempts" are mutually exclusive')
            else:
                with validate('fast_503'):
                    if self.pb.HasField('fast_503'):
                        raise ValidationError('can only be used with fast_attempts option')

        if self.dynamic:
            with validate('dynamic'):
                if not self.generated_proxy_backends:
                    raise ValidationError('can only be used with generated_proxy_backends')
                with validate('backends_name'):
                    if not self.dynamic.pb.backends_name and (not self.generated_proxy_backends or
                                                              not self.generated_proxy_backends.include_backends):
                        raise ValidationError(
                            'must be set if balancer2.generated_proxy_backends.include_backends is not used')
                self.dynamic.validate(ctx=ctx, preceding_modules=preceding_modules)

        if self.on_status_code:
            with validate('on_status_code'):
                validate_key_uniqueness(self.on_status_code_items)
            for status_code, holder in six.iteritems(self.on_status_code):
                with validate('on_status_code[{}]'.format(status_code)):
                    holder.validate(ctx=ctx, preceding_modules=preceding_modules)

        if self.on_error:
            with validate('on_error'):
                self.on_error.validate(ctx=ctx, preceding_modules=preceding_modules)

        if self.on_fast_error:
            with validate('on_fast_error'):
                ctx.ensure_component_version(
                    model_pb2.ComponentMeta.PGINX_BINARY,
                    min=min_component_versions.ON_FAST_ERROR)
                self.on_fast_error.validate(ctx=ctx, preceding_modules=preceding_modules)

        if self.pb.status_code_blacklist:
            with validate('status_code_blacklist'):
                validate_status_codes(self.pb.status_code_blacklist)

        if self.pb.status_code_blacklist_exceptions:
            with validate('status_code_blacklist_exceptions'):
                validate_status_codes(self.pb.status_code_blacklist_exceptions)

        if self.delay_settings:
            with validate('delay_settings'):
                self.delay_settings.validate(ctx=ctx, preceding_modules=preceding_modules)

        super(Balancer2, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        self._validate_attempts()
        self._validate_connection_attempts()
        self._validate_fast_attempts()
        self._validate_attempts_file(ctx)
        self._validate_rewind_limit()
        self._validate_hedged_delay()
        if self.pb.return_last_5xx:
            found_5xx_in_blacklist = False
            for status_code in self.pb.status_code_blacklist:
                if status_code.startswith('5'):
                    found_5xx_in_blacklist = True
                    break
            if not found_5xx_in_blacklist:
                raise ValidationError('at least one 5xx must be specified in status_code_blacklist', 'return_last_5xx')
        if self.check_backends and not self.generated_proxy_backends:
            raise ValidationError('can only be used with "generated_proxy_backends"', 'check_backends')
        super(Balancer2, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def count_backends(self):
        if self.backends:
            return len(self.backends)
        elif self.generated_proxy_backends:
            return len(self.generated_proxy_backends.instances)

    def get_total_weight(self, allow_different_backend_weights=False):
        """
        :type allow_different_backend_weights: bool
        :rtype: float
        :raises: ValidationError
        """
        if self.backends:
            it = iter(self.backends)
            backend = next(it)
            rv = backend.pb.weight
            for i, next_backend in enumerate(it, start=1):
                if not allow_different_backend_weights and not is_close(backend.pb.weight, next_backend.pb.weight):
                    raise ValidationError('weights of backends[{}] and backends[{}] differ. '
                                          'if this difference is expected and you are sure that specified '
                                          'quorum and hysteresis percentages are still valid, '
                                          'please set "allow_different_backend_weights" to true'.format(i - 1, i))
                backend = next_backend
                rv += backend.pb.weight
            return rv
        elif self.generated_proxy_backends:
            if self.generated_proxy_backends.instances:
                return self.generated_proxy_backends.get_total_weight(
                    allow_different_backend_weights=allow_different_backend_weights)
            else:
                return 0
        else:
            raise AssertionError('neither self.backends not self.generated_proxy_backends is present')

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        preceding_modules = add_module(preceding_modules, self)

        if self.balancing_policy:
            config = self.balancing_policy.to_config(ctx=ctx, preceding_modules=preceding_modules)
        else:
            config = Config(OrderedDict())
            config.table['unique_policy'] = Config()

        attempts = self.get('attempts')
        if attempts.is_func():
            if attempts.value.func_name == defs.count_backends.name:
                if attempts.value.func_params.pb.HasField('compat_enable_sd_support'):
                    use_count_backends = attempts.value.func_params.pb.compat_enable_sd_support.value
                else:
                    use_count_backends = True
                if self.generated_proxy_backends and self.generated_proxy_backends.endpoint_sets:
                    config.table['attempts'] = ('count_backends'
                                                if use_count_backends else self.count_backends())
                else:
                    config.table['attempts'] = self.count_backends()
            elif attempts.value.func_name == defs.count_backends_sd.name:
                config.table['attempts'] = 'count_backends'
        elif attempts.value:
            config.table['attempts'] = attempts.value
        else:
            raise AssertionError('attempts are not set. How come?')

        if self.pb.rewind_limit:
            config.table['rewind_limit'] = self.pb.rewind_limit

        attempts_file = self.get('attempts_file')
        if attempts_file.value:
            config.table['attempts_file'] = attempts_file.to_config(ctx)

        connection_attempts = self.get('connection_attempts')
        if connection_attempts.is_func():
            if connection_attempts.value.func_name == defs.count_backends.name:
                if connection_attempts.value.func_params.pb.HasField('compat_enable_sd_support'):
                    use_count_backends = connection_attempts.value.func_params.pb.compat_enable_sd_support.value
                else:
                    use_count_backends = True
                if self.generated_proxy_backends and self.generated_proxy_backends.endpoint_sets:
                    config.table['connection_attempts'] = ('count_backends'
                                                           if use_count_backends else self.count_backends())
                else:
                    config.table['connection_attempts'] = self.count_backends()
            elif connection_attempts.value.func_name == defs.count_backends_sd.name:
                config.table['connection_attempts'] = 'count_backends'
        elif connection_attempts.value:
            config.table['connection_attempts'] = connection_attempts.value

        fast_attempts = self.get('fast_attempts')
        if fast_attempts.is_func():
            if fast_attempts.value.func_name == defs.count_backends.name:
                if fast_attempts.value.func_params.pb.HasField('compat_enable_sd_support'):
                    use_count_backends = fast_attempts.value.func_params.pb.compat_enable_sd_support.value
                else:
                    use_count_backends = True

                if self.generated_proxy_backends and self.generated_proxy_backends.endpoint_sets:
                    config.table['fast_attempts'] = ('count_backends'
                                                     if use_count_backends else self.count_backends())
                else:
                    config.table['fast_attempts'] = self.count_backends()
            elif fast_attempts.value.func_name == defs.count_backends_sd.name:
                config.table['fast_attempts'] = 'count_backends'
        elif fast_attempts.value:
            config.table['fast_attempts'] = fast_attempts.value

        if self.pb.HasField('fast_503'):
            config.table['fast_503'] = self.pb.fast_503.value
        if self.pb.HasField('retry_non_idempotent'):
            config.table['retry_non_idempotent'] = self.pb.retry_non_idempotent.value
        if self.pb.HasField('use_on_error_for_non_idempotent'):
            config.table['use_on_error_for_non_idempotent'] = self.pb.use_on_error_for_non_idempotent.value
        if self.pb.return_last_5xx:
            config.table['return_last_5xx'] = self.pb.return_last_5xx
        if self.pb.status_code_blacklist:
            config.table['status_code_blacklist'] = Config(array=self.pb.status_code_blacklist)
        if self.pb.status_code_blacklist_exceptions:
            config.table['status_code_blacklist_exceptions'] = Config(array=self.pb.status_code_blacklist_exceptions)
        if self.pb.hedged_delay:
            config.table['hedged_delay'] = self.pb.hedged_delay
        if self.pb.HasField('delay_settings'):
            config.table['first_delay'] = self.pb.delay_settings.first_delay
            config.table['delay_multiplier'] = self.pb.delay_settings.delay_multiplier
            config.table['delay_on_fast'] = self.pb.delay_settings.delay_on_fast.value
            config.table['max_random_delay'] = self.pb.delay_settings.max_random_delay
        if self.check_backends:
            config.table['check_backends'] = self.check_backends.to_config(ctx=ctx)

        if self.rr:
            balancing_options_key = 'rr'
            balancing_options = self.rr.to_config(
                ctx=ctx, preceding_modules=preceding_modules)
        elif self.weighted2:
            balancing_options_key = 'weighted2'
            balancing_options = self.weighted2.to_config(
                ctx=ctx, preceding_modules=preceding_modules)
        elif self.hashing:
            balancing_options_key = 'hashing'
            balancing_options = self.hashing.to_config(
                ctx=ctx, preceding_modules=preceding_modules)
        elif self.active:
            total_weight = self.get_total_weight(
                allow_different_backend_weights=self.active.allows_different_backend_weights())
            balancing_options_key = 'active'
            balancing_options = self.active.to_config(
                total_weight=total_weight,
                ctx=ctx,
                preceding_modules=preceding_modules)
        elif self.rendezvous_hashing:
            balancing_options_key = 'rendezvous_hashing'
            balancing_options = self.rendezvous_hashing.to_config(
                ctx=ctx, preceding_modules=preceding_modules)
        elif self.pwr2:
            balancing_options_key = 'pwr2'
            balancing_options = self.pwr2.to_config(
                ctx=ctx, preceding_modules=preceding_modules)
        elif self.leastconn:
            balancing_options_key = 'leastconn'
            balancing_options = self.leastconn.to_config(
                ctx=ctx, preceding_modules=preceding_modules)
        elif self.dynamic:
            balancing_options_key = 'dynamic_hashing' if self.dynamic.hashing else 'dynamic'
            balancing_options = self.dynamic.to_config(
                ctx=ctx, preceding_modules=preceding_modules)
        else:
            raise AssertionError()

        if self.backends:
            backends_table = OrderedDict([(backend.get('name').to_config(ctx=ctx),
                                           backend.to_config(ctx=ctx, preceding_modules=preceding_modules))
                                          for backend in self.backends if backend.get('name').value])
            backends_array = [backend.to_config(ctx=ctx, preceding_modules=preceding_modules)
                              for backend in self.backends if not backend.get('name').value]
            balancing_options.extend(Config(table=backends_table, array=backends_array))
            config.table[balancing_options_key] = balancing_options
        elif self.generated_proxy_backends:
            gpb_config = self.generated_proxy_backends.to_config(ctx=ctx, preceding_modules=preceding_modules)
            if self.generated_proxy_backends.endpoint_sets:
                gpb_config.table[balancing_options_key] = balancing_options
                config.table['sd'] = gpb_config
            else:
                balancing_options.extend(gpb_config)
                config.table[balancing_options_key] = balancing_options
        else:
            raise AssertionError('neither self.backends nor self.generated_proxy_backends is present')

        if self.attempts_rate_limiter:
            config.table['attempts_rate_limiter'] = self.attempts_rate_limiter.to_config(
                ctx=ctx, preceding_modules=preceding_modules)

        if self.on_error:
            config.table['on_error'] = self.on_error.to_config(
                ctx=ctx, preceding_modules=preceding_modules)

        if self.on_fast_error:
            config.table['on_fast_error'] = self.on_fast_error.to_config(
                ctx=ctx, preceding_modules=preceding_modules)

        if self.on_status_code:
            on_status_code_config = Config(OrderedDict())
            for status_code, holder in sorted(six.iteritems(self.on_status_code)):
                on_status_code_config.table[status_code] = holder.to_config(
                    ctx=ctx, preceding_modules=preceding_modules)
            config.table['on_status_code'] = on_status_code_config

        config.shareable = True
        return config

    def get_branches(self):
        for backend in self.backends:
            yield backend
        if self.generated_proxy_backends:
            yield self.generated_proxy_backends
        if self.on_error:
            yield self.on_error
        if self.on_fast_error:
            yield self.on_fast_error
        for on_status_code in six.itervalues(self.on_status_code):
            yield on_status_code

    def get_named_branches(self):
        rv = {}
        for i, backend in enumerate(self.backends):
            name = 'backends[{}]'.format(i)
            rv[name] = backend
        if self.generated_proxy_backends:
            rv['generated_proxy_backends'] = self.generated_proxy_backends
        if self.on_error:
            rv['on_error'] = self.on_error
        if self.on_fast_error:
            rv['on_fast_error'] = self.on_fast_error
        for name, on_status_code in six.iteritems(self.on_status_code):
            name = 'on_status_code[{}]'.format(name)
            rv[name] = on_status_code
        return rv


class Header(ConfigWrapperBase):
    __protobuf__ = proto.Header

    REQUIRED = ['name', 'value']

    ALLOWED_CALLS = {
        'value': [defs.get_str_env_var.name],
    }

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        with validate('name'):
            validate_header_name(self.pb.name)
        with validate('value'):
            value = self.get('value')
            if value.is_func():
                validate_header_value_call(value.value)
        super(Header, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        return Config(OrderedDict([
            ('name', self.pb.name),
            ('value', self.get('value').to_config(ctx=ctx)),
        ]))


class MatchFsm(ConfigWrapperBase):
    __protobuf__ = proto.MatchFsm

    header = None  # type: Header | None

    DEFAULT_CASE_INSENSITIVE = True

    DEFAULT = {
        'case_insensitive': DEFAULT_CASE_INSENSITIVE,
    }

    def list_set_fields(self):
        matchers = OrderedDict([
            ('host', self.pb.host),
            ('path', self.pb.path),
            ('cgi', self.pb.cgi),
            ('uri', self.pb.uri),
            ('url', self.pb.url),
            ('match', self.pb.match),
            ('cookie', self.pb.cookie),
            ('header', self.header),
            ('upgrade', self.pb.upgrade),
        ])
        set_fields = list()
        for name, value in six.iteritems(matchers):
            if value:
                set_fields.append(name)
        return set_fields

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=()):
        if self.header:
            with validate('header'):
                self.header.validate(ctx=ctx, preceding_modules=preceding_modules)

        super(MatchFsm, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        for f in ('host', 'path', 'cgi', 'uri', 'url', 'match', 'cookie', 'upgrade'):
            value = getattr(self.pb, f)
            if value:
                with validate(f):
                    validate_pire_regexp(value)
        s = self.list_set_fields()
        if len(s) > 1:
            raise ValidationError('at most one of the "{}" '
                                  'must be specified'.format('", "'.join(s)))

        self.validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        case_insensitive = self.DEFAULT_CASE_INSENSITIVE
        if self.pb.HasField('case_insensitive'):
            case_insensitive = self.pb.case_insensitive.value
        table = OrderedDict()
        if self.pb.host:
            table['host'] = self.pb.host
        if self.pb.path:
            table['path'] = self.pb.path
        if self.pb.cgi:
            table['cgi'] = self.pb.cgi
        if self.pb.uri:
            table['URI'] = self.pb.uri
        if self.pb.url:
            table['url'] = self.pb.url
        if self.pb.match:
            table['match'] = self.pb.match
        if self.pb.cookie:
            table['cookie'] = self.pb.cookie
        if self.header:
            table['header'] = self.header.to_config(ctx=ctx, preceding_modules=preceding_modules)
        if self.pb.upgrade:
            table['upgrade'] = self.pb.upgrade
        table.update([
            ('case_insensitive', case_insensitive),
            ('surround', self.pb.surround),
        ])
        return Config(table)


class MatchSourceIp(ConfigWrapperBase):
    __protobuf__ = proto.MatchSourceIp

    REQUIRED = ['source_mask']

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        # TODO validate source_mask
        super(MatchSourceIp, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        return Config({
            'source_mask': self.pb.source_mask,
        })


class MatchMethod(ConfigWrapperBase):
    __protobuf__ = proto.MatchMethod

    REQUIRED = ['methods']
    SUPPORTED_METHODS = frozenset(['head', 'get', 'options', 'post', 'put',
                                   'patch', 'delete', 'connect', 'trace'])

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        seen_methods = set()
        for i, method in enumerate(self.pb.methods):
            field_name = 'methods[{}]'.format(i)
            if not method.islower():
                raise ValidationError('must be all lowercase', field_name=field_name)
            if method not in self.SUPPORTED_METHODS:
                raise ValidationError('is not supported', field_name=field_name)
            if method in seen_methods:
                raise ValidationError('is duplicate'.format(method), field_name=field_name)
            if method == 'trace':
                http = find_module(preceding_modules, Http)
                if not http or not http.pb.allow_trace:
                    raise ValidationError('using "trace" method requires preceding "http" module with '
                                          'enabled "allow_trace" option', field_name=field_name)
            seen_methods.add(method)
        super(MatchMethod, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        return Config({
            'methods': Config(array=list(self.pb.methods), compact=True),
        })


class Matcher(ConfigWrapperBase):
    __protobuf__ = proto.Matcher

    match_fsm = None  # type: MatchFsm | None
    match_source_ip = None  # type: MatchSourceIp | None
    match_method = None  # type: MatchMethod | None
    match_not = None  # type: Matcher | None
    match_and = []  # type: list[Matcher]
    match_or = []  # type: list[Matcher]

    def _count_set_fields(self):
        return len([f for f in (self.match_fsm,
                                self.match_source_ip,
                                self.match_method,
                                self.match_not,
                                self.match_and,
                                self.match_or) if f])

    def is_empty(self):
        return self._count_set_fields() == 0

    def validate(self, ctx=DEFAULT_CTX, is_nested=False, preceding_modules=()):
        self.auto_validate_required()
        self.validate_composite_fields(ctx=ctx, is_nested=is_nested, preceding_modules=preceding_modules)

    def validate_composite_fields(self, ctx=DEFAULT_CTX, is_nested=False, preceding_modules=()):
        c = self._count_set_fields()
        if is_nested and c == 0:
            raise ValidationError('either "match_fsm", "match_source_ip", "match_method", '
                                  '"match_not", "match_and" or "match_or" must be specified')
        if c > 1:
            raise ValidationError('at most one of the "match_fsm", "match_source_ip", "match_method", '
                                  '"match_not", "match_and" or "match_or" must be specified')

        if self.match_fsm:
            with validate('match_fsm'):
                self.match_fsm.validate(ctx=ctx, preceding_modules=preceding_modules)
        if self.match_source_ip:
            with validate('match_source_ip'):
                self.match_source_ip.validate(ctx=ctx, preceding_modules=preceding_modules)
        if self.match_method:
            with validate('match_method'):
                self.match_method.validate(ctx=ctx, preceding_modules=preceding_modules)
        if self.match_not:
            with validate('match_not'):
                self.match_not.validate(is_nested=True, ctx=ctx, preceding_modules=preceding_modules)
        for i, matcher in enumerate(self.match_or):
            with validate('match_or[{}]'.format(i)):
                matcher.validate(is_nested=True, ctx=ctx, preceding_modules=preceding_modules)
        for i, matcher in enumerate(self.match_and):
            with validate('match_and[{}]'.format(i)):
                matcher.validate(is_nested=True, ctx=ctx, preceding_modules=preceding_modules)

        super(Matcher, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = {}
        if self.match_fsm:
            table['match_fsm'] = self.match_fsm.to_config(ctx=ctx, preceding_modules=preceding_modules)
        if self.match_source_ip:
            table['match_source_ip'] = self.match_source_ip.to_config(ctx=ctx, preceding_modules=preceding_modules)
        if self.match_method:
            table['match_method'] = self.match_method.to_config(ctx=ctx, preceding_modules=preceding_modules)
        if self.match_not:
            table['match_not'] = self.match_not.to_config(ctx=ctx, preceding_modules=preceding_modules)
        if self.match_or:
            table['match_or'] = Config(array=[m.to_config(ctx=ctx, preceding_modules=preceding_modules)
                                              for m in self.match_or])
        if self.match_and:
            table['match_and'] = Config(array=[m.to_config(ctx=ctx, preceding_modules=preceding_modules)
                                               for m in self.match_and])
        return Config(table)


class Srcrwr(ChainableModuleWrapperBase):
    __protobuf__ = proto.SrcrwrModule

    REQUIRED = ['match_host', 'match_source_mask']
    REQUIRED_ONEOFS = [('id', 'id_deprecated')]

    MODULE_NAME = 'srcrwr'

    def compose_matcher(self):
        host_matcher_pb = proto.Matcher(match_fsm=proto.MatchFsm(host=self.pb.match_host))
        source_mask_matcher_pb = proto.Matcher(
            match_source_ip=proto.MatchSourceIp(source_mask=self.pb.match_source_mask))
        matcher_pb = proto.Matcher(match_and=[host_matcher_pb, source_mask_matcher_pb])
        return Matcher(matcher_pb)

    @validate('match_host')
    def _validate_match_host(self):
        validate_pire_regexp(self.pb.match_host)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_raise_if_blacklisted(ctx)
        self.auto_validate_required()
        self.require_nested(chained_modules)

        first_chained_module = self._get_first_chained_module(chained_modules)
        assert first_chained_module is not None

        if not isinstance(first_chained_module, Balancer2):
            raise ValidationError('must be followed by balancer2 module')
        self._validate_match_host()
        super(Srcrwr, self).validate(ctx=ctx, preceding_modules=preceding_modules, chained_modules=chained_modules)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        config = Config(OrderedDict())
        config.table['id'] = self.pb.id or self.pb.id_deprecated

        config.extend(self.compose_matcher().to_config(ctx=ctx, preceding_modules=add_module(preceding_modules, self)))
        return config


class SrcrwrExt(ChainableModuleWrapperBase):
    __protobuf__ = proto.SrcrwrExtModule

    REQUIRED = ['remove_prefix', 'domains']

    MODULE_NAME = 'srcrwr_ext'

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_raise_if_blacklisted(ctx)
        self.auto_validate_required()

        self.require_nested(chained_modules)
        last_chained_module = self._get_last_chained_module(chained_modules)
        assert last_chained_module is not None

        if not (isinstance(last_chained_module, Proxy) or
                (isinstance(last_chained_module, Balancer2) and last_chained_module.generated_proxy_backends)):
            raise ValidationError(u'must contain proxy or balancer2 with generated_proxy_backends '
                                  u'as terminal module')
        super(SrcrwrExt, self).validate(ctx=ctx, preceding_modules=preceding_modules, chained_modules=chained_modules)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict()
        table['remove_prefix'] = self.pb.remove_prefix
        table['domains'] = self.pb.domains
        return Config(table)


class Cache2(ChainableModuleWrapperBase):
    __protobuf__ = proto.Cache2Module

    REQUIRED = ['cache_ttl', 'shard_number']

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()
        with validate('cache_ttl'):
            validate_timedelta(self.pb.cache_ttl)
        with validate('shard_number'):
            validate_range(self.pb.shard_number, min_=1, max_=1000)
        super(Cache2, self).validate(ctx=ctx, preceding_modules=preceding_modules, chained_modules=chained_modules)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict()
        table['cache_ttl'] = self.pb.cache_ttl
        table['shard_number'] = self.pb.shard_number
        if self.pb.HasField('ignore_cgi'):
            table['ignore_cgi'] = self.pb.ignore_cgi.value
        return Config(table)


class CookiePolicy(ChainableModuleWrapperBase):
    __protobuf__ = proto.CookiePolicyModule

    REQUIRED = ['uuid']

    UUID_RE = r'^[a-zA-Z0-9_-]{1,32}$'
    ALLOWED_DEFAULT_YANDEX_POLICIES = (u'off', u'unstable', u'stable')
    ALLOWED_KNOBS = {
        'file_switch': model_pb2.KnobSpec.BOOLEAN,
    }

    def _validate_file_switch(self, ctx):
        self.validate_knob('file_switch', ctx=ctx)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()
        if not re.match(self.UUID_RE, self.pb.uuid):
            raise ValidationError(u'must be {}'.format(self.UUID_RE), u'uuid')
        self._validate_file_switch(ctx)
        if (self.pb.default_yandex_policies and
                self.pb.default_yandex_policies not in self.ALLOWED_DEFAULT_YANDEX_POLICIES):
            raise ValidationError(u'must be one of the {}'.format(
                ', '.join(self.ALLOWED_DEFAULT_YANDEX_POLICIES)), u'default_yandex_policies')
        super(CookiePolicy, self).validate(ctx=ctx, preceding_modules=preceding_modules,
                                           chained_modules=chained_modules)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict()
        table['uuid'] = self.pb.uuid
        file_switch = self.get('file_switch')
        if file_switch.value:
            table['file_switch'] = file_switch.to_config(ctx)
        # if self.pb.parser_mode:
        #    table['parser_mode'] = self.pb.parser_mode
        if self.pb.default_yandex_policies:
            table['default_yandex_policies'] = self.pb.default_yandex_policies
        return Config(table)


class RegexpSection(ChainableModuleWrapperBase):
    __protobuf__ = proto.RegexpSection

    matcher = None  # type: Matcher | None

    REQUIRED = ['matcher']

    # since it's not a "real" module, it can't be shared
    SHAREABLE = False

    DEFAULT_KEY = 'default'

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=(), key=None):
        self.auto_validate_required()
        if not preceding_modules or not (
                isinstance(preceding_modules[-1], Regexp) or preceding_modules[-1] == ANY_MODULE):
            raise ValidationError('must be a child of "regexp" module')
        self.require_nested(chained_modules)
        self.validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules,
                                       chained_modules=chained_modules,
                                       key=key)

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=(), key=None):
        with validate('matcher'):
            self.matcher.validate(ctx=ctx, preceding_modules=add_module(preceding_modules, self))
        if key == self.DEFAULT_KEY and not self.matcher.is_empty():
            raise ValidationError('"default" section must have an empty matcher', 'sections[default]')
        super(RegexpSection, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules,
                                                             chained_modules=chained_modules)

    def _to_params_config(self, priority, ctx=DEFAULT_CTX, preceding_modules=()):
        # we want priority and matcher go first, thus make table an ordered dict
        config = Config(OrderedDict())
        config.table['priority'] = priority
        if self.matcher:
            config.extend(self.matcher.to_config(ctx=ctx, preceding_modules=add_module(preceding_modules, self)))
        return config


class Regexp(ModuleWrapperBase):
    __protobuf__ = proto.RegexpModule

    __slots__ = ('sections', 'prepend_sections')

    # sections = OrderedDict()  # type: dict[basestring, RegexpSection]
    section_items = []  # type: list[(basestring, RegexpSection)]
    # prepend_sections = OrderedDict()  # type: dict[basestring, RegexpSection]
    prepend_sections_items = []  # type: list[(basestring, RegexpSection)]
    include_upstreams = None  # type: IncludeUpstreams | None

    REQUIRED_ONEOFS = [('sections', 'include_upstreams')]

    PB_FIELD_TO_CLS_ATTR_MAPPING = {'sections': 'section_items', 'prepend_sections': 'prepend_sections_items'}

    def wrap_composite_fields(self):
        super(Regexp, self).wrap_composite_fields()
        self.sections = OrderedDict(self.section_items)
        self.prepend_sections = OrderedDict(self.prepend_sections_items)

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=()):
        if self.include_upstreams:
            if ctx.config_type == ctx.CONFIG_TYPE_UPSTREAM:
                raise ValidationError('is not allowed in upstream config', 'include_upstreams')
            with validate('include_upstreams'):
                self.include_upstreams.validate(ctx=ctx, preceding_modules=preceding_modules)
        else:
            if self.sections and self.prepend_sections:
                raise ValidationError('sections and prepend_sections must not be used together')
            with validate('sections'):
                if not self.sections:
                    raise ValidationError('must contain at least one section')
                validate_key_uniqueness(self.section_items)

                sections_w_empty_matcher = set()
                # at this point nested sections are not validated yet, so
                # we have to check if `section.matcher` is not None
                for key, section in six.iteritems(self.sections):
                    if section.matcher and section.matcher.is_empty():
                        sections_w_empty_matcher.add(key)
                if len(sections_w_empty_matcher) > 1:
                    raise ValidationError(
                        'too many sections with an empty matcher: "{}". '
                        'at most one section can have an empty matcher and '
                        'hence become the default one.'.format(quote_join_sorted(sections_w_empty_matcher)
                                                               )
                    )

            if len(sections_w_empty_matcher) == 1:
                last_key, last_section = self.section_items[-1]
                if not last_section.matcher or not last_section.matcher.is_empty():
                    raise ValidationError('section with an empty matcher must go last',
                                          'sections[{}]'.format(sections_w_empty_matcher.pop()))

        new_preceding_modules = add_module(preceding_modules, self)
        for key, section in six.iteritems(self.sections):
            with validate('sections[{}]'.format(key)):
                section.validate(ctx=ctx, preceding_modules=new_preceding_modules, key=key)
        super(Regexp, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        super(Regexp, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        preceding_modules = add_module(preceding_modules, self)
        table = OrderedDict()
        n = len(self.sections)
        outlets = []
        for i, (key, section) in enumerate(six.iteritems(self.sections)):
            if section.matcher.is_empty():
                key = 'default'
            section_config = section.to_config(priority=n - i, ctx=ctx, preceding_modules=preceding_modules)
            table[key] = section_config
            outlets.extend(section_config.get_outlets())
        return Config(table, shareable=True, outlets=outlets)

    def get_branches(self):
        return six.itervalues(self.sections)

    def get_named_branches(self):
        return self.sections

    def includes_upstreams(self):
        return self.include_upstreams


class RegexpPathSection(ChainableModuleWrapperBase):
    __protobuf__ = proto.RegexpPathSection

    # since it's not a "real" module, it can't be shared
    SHAREABLE = False

    DEFAULT_KEY = 'default'
    DEFAULT_CASE_INSENSITIVE = True

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=(), key=None):
        if not preceding_modules or not \
                (isinstance(preceding_modules[-1], RegexpPath) or preceding_modules[-1] == ANY_MODULE):
            raise ValidationError('must be a child of "regexp_path" module')
        self.require_nested(chained_modules)
        self.validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules,
                                       chained_modules=chained_modules,
                                       key=key)

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=(), key=None):
        if key == self.DEFAULT_KEY and self.pb.pattern:
            raise ValidationError('"default" section must have an empty pattern', 'sections[default]')
        super(RegexpPathSection, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules,
                                                                 chained_modules=chained_modules)

    def _to_params_config(self, priority, ctx=DEFAULT_CTX, preceding_modules=()):
        # we want priority and pattern go first, thus make table an ordered dict
        table = OrderedDict()
        config = Config(table)
        if self.pb.pattern:
            table['priority'] = priority
            table['pattern'] = self.pb.pattern

            case_insensitive = self.DEFAULT_CASE_INSENSITIVE
            if self.pb.HasField('case_insensitive'):
                case_insensitive = self.pb.case_insensitive.value
            table['case_insensitive'] = case_insensitive

        return config


class RegexpPath(ModuleWrapperBase):
    __protobuf__ = proto.RegexpPathModule

    __slots__ = ('sections',)

    # sections = OrderedDict()  # type: dict[basestring, RegexpPathSection]
    section_items = []  # type: list[(basestring, RegexpPathSection)]
    include_upstreams = None  # type: IncludeUpstreams | None

    REQUIRED_ONEOFS = [('sections', 'include_upstreams')]

    PB_FIELD_TO_CLS_ATTR_MAPPING = {'sections': 'section_items'}

    def wrap_composite_fields(self):
        super(RegexpPath, self).wrap_composite_fields()
        self.sections = OrderedDict(self.section_items)

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=()):
        if self.include_upstreams:
            if ctx.config_type == ctx.CONFIG_TYPE_UPSTREAM:
                raise ValidationError('is not allowed in upstream config')
        else:
            with validate('sections'):
                if not self.sections:
                    raise ValidationError('must contain at least one section')
                validate_key_uniqueness(self.section_items)

                sections_w_empty_pattern = set()
                # at this point nested sections are not validated yet, so
                # we have to check if `section.pb.pattern` is empty
                for key, section in six.iteritems(self.sections):
                    if not section.pb.pattern:
                        sections_w_empty_pattern.add(key)
                if len(sections_w_empty_pattern) > 1:
                    raise ValidationError(
                        'too many sections with an empty pattern: "{}". '
                        'at most one section can have an empty pattern and '
                        'hence become the default one.'.format(quote_join_sorted(sections_w_empty_pattern)
                                                               )
                    )

            if len(sections_w_empty_pattern) == 1:
                last_key, last_section = self.section_items[-1]
                if last_section.pb.pattern:
                    raise ValidationError('section with an empty pattern must go last',
                                          'sections[{}]'.format(sections_w_empty_pattern.pop()))

        new_preceding_modules = add_module(preceding_modules, self)
        for key, section in six.iteritems(self.sections):
            with validate('sections[{}]'.format(key)):
                section.validate(ctx=ctx, preceding_modules=new_preceding_modules, key=key)
        super(RegexpPath, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        super(RegexpPath, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        preceding_modules = add_module(preceding_modules, self)
        table = OrderedDict()
        n = len(self.sections)
        outlets = []
        for i, (key, section) in enumerate(six.iteritems(self.sections)):
            if not section.pb.pattern:
                key = 'default'
            section_config = section.to_config(priority=n - i, ctx=ctx, preceding_modules=preceding_modules)
            table[key] = section_config
            outlets.extend(section_config.get_outlets())
        return Config(table, shareable=True, outlets=outlets)

    def get_branches(self):
        return six.itervalues(self.sections)

    def get_named_branches(self):
        return self.sections

    def includes_upstreams(self):
        return self.include_upstreams


class PrefixPathRouterSection(ChainableModuleWrapperBase):
    __protobuf__ = proto.PrefixPathRouterSection

    # since it's not a "real" module, it can't be shared
    SHAREABLE = False

    DEFAULT_KEY = 'default'

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=(), key=None):
        if not preceding_modules or not \
                (isinstance(preceding_modules[-1], PrefixPathRouter) or preceding_modules[-1] == ANY_MODULE):
            raise ValidationError('must be a child of "prefix_path_router" module')
        self.require_nested(chained_modules)
        self.validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules,
                                       chained_modules=chained_modules,
                                       key=key)

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=(), key=None):
        if key == self.DEFAULT_KEY and self.pb.route:
            raise ValidationError('"default" section must have an empty route', 'sections[default]')
        super(PrefixPathRouterSection, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules,
                                                                       chained_modules=chained_modules)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        # we want route go first, thus make table an ordered dict
        table = OrderedDict()
        if self.pb.route:
            table['route'] = self.pb.route
        return Config(table)


class PrefixPathRouter(ModuleWrapperBase):
    __protobuf__ = proto.PrefixPathRouterModule

    __slots__ = ('sections',)

    # sections = OrderedDict()  # type: dict[basestring, PrefixPathRouterSection]
    section_items = []  # type: list[(basestring, PrefixPathRouterSection)]
    include_upstreams = None  # type: IncludeUpstreams | None

    REQUIRED_ONEOFS = [('sections', 'include_upstreams')]

    PB_FIELD_TO_CLS_ATTR_MAPPING = {'sections': 'section_items'}

    DEFAULT_CASE_INSENSITIVE = True

    def wrap_composite_fields(self):
        super(PrefixPathRouter, self).wrap_composite_fields()
        self.sections = OrderedDict(self.section_items)

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=()):
        if self.include_upstreams:
            if ctx.config_type == ctx.CONFIG_TYPE_UPSTREAM:
                raise ValidationError('is not allowed in upstream config')
        else:
            with validate('sections'):
                if not self.sections:
                    raise ValidationError('must contain at least one section')
                validate_key_uniqueness(self.section_items)

                sections_w_empty_route = set()
                # at this point nested sections are not validated yet, so
                # we have to check if `section.pb.route` is empty
                for key, section in six.iteritems(self.sections):
                    if not section.pb.route:
                        sections_w_empty_route.add(key)
                if len(sections_w_empty_route) > 1:
                    raise ValidationError(
                        'too many sections with an empty route: "{}". '
                        'at most one section can have an empty route and '
                        'hence become the default one.'.format(quote_join_sorted(sections_w_empty_route)
                                                               )
                    )

            if len(sections_w_empty_route) == 1:
                last_key, last_section = self.section_items[-1]
                if last_section.pb.route:
                    raise ValidationError('section with an empty route must go last',
                                          'sections[{}]'.format(sections_w_empty_route.pop()))

        new_preceding_modules = add_module(preceding_modules, self)
        for key, section in six.iteritems(self.sections):
            with validate('sections[{}]'.format(key)):
                section.validate(ctx=ctx, preceding_modules=new_preceding_modules, key=key)
        super(PrefixPathRouter, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        super(PrefixPathRouter, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        preceding_modules = add_module(preceding_modules, self)
        table = OrderedDict()
        case_insensitive = self.DEFAULT_CASE_INSENSITIVE
        if self.pb.HasField('case_insensitive'):
            case_insensitive = self.pb.case_insensitive.value
        table['case_insensitive'] = case_insensitive

        outlets = []
        for i, (key, section) in enumerate(six.iteritems(self.sections)):
            if not section.pb.route:
                key = 'default'
            section_config = section.to_config(ctx=ctx, preceding_modules=preceding_modules)
            table[key] = section_config
            outlets.extend(section_config.get_outlets())
        return Config(table, shareable=True, outlets=outlets)

    def get_branches(self):
        return six.itervalues(self.sections)

    def get_named_branches(self):
        return self.sections

    def includes_upstreams(self):
        return self.include_upstreams


class RegexpHostSection(ChainableModuleWrapperBase):
    __protobuf__ = proto.RegexpHostSection

    # since it's not a "real" module, it can't be shared
    SHAREABLE = False

    DEFAULT_KEY = 'default'
    DEFAULT_CASE_INSENSITIVE = True

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=(), key=None):
        if not preceding_modules or not \
                (isinstance(preceding_modules[-1], RegexpHost) or preceding_modules[-1] == ANY_MODULE):
            raise ValidationError('must be a child of "regexp_host" module')
        self.require_nested(chained_modules)
        self.validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules,
                                       chained_modules=chained_modules,
                                       key=key)

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=(), key=None):
        if key == self.DEFAULT_KEY and self.pb.pattern:
            raise ValidationError('"default" section must have an empty pattern', 'sections[default]')
        super(RegexpHostSection, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules,
                                                                 chained_modules=chained_modules)

    def _to_params_config(self, priority, ctx=DEFAULT_CTX, preceding_modules=()):
        # we want priority and pattern go first, thus make table an ordered dict
        table = OrderedDict()
        config = Config(table)
        if self.pb.pattern:
            table['priority'] = priority
            table['pattern'] = self.pb.pattern

            case_insensitive = self.DEFAULT_CASE_INSENSITIVE
            if self.pb.HasField('case_insensitive'):
                case_insensitive = self.pb.case_insensitive.value
            table['case_insensitive'] = case_insensitive

        return config


class RegexpHost(ModuleWrapperBase):
    __protobuf__ = proto.RegexpHostModule

    __slots__ = ('sections',)

    # sections = OrderedDict()  # type: dict[six.string_types, RegexpHostSection]
    section_items = []  # type: list[(six.string_types, RegexpHostSection)]
    include_upstreams = None  # type: IncludeUpstreams or None

    REQUIRED_ONEOFS = [('sections', 'include_upstreams')]

    PB_FIELD_TO_CLS_ATTR_MAPPING = {'sections': 'section_items'}

    def wrap_composite_fields(self):
        super(RegexpHost, self).wrap_composite_fields()
        self.sections = OrderedDict(self.section_items)

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=()):
        if self.include_upstreams:
            if ctx.config_type == ctx.CONFIG_TYPE_UPSTREAM:
                raise ValidationError('is not allowed in upstream config')
        else:
            with validate('sections'):
                if not self.sections:
                    raise ValidationError('must contain at least one section')
                validate_key_uniqueness(self.section_items)

                sections_w_empty_pattern = set()
                # at this point nested sections are not validated yet, so
                # we have to check if `section.pb.pattern` is empty
                for key, section in six.iteritems(self.sections):
                    if not section.pb.pattern:
                        sections_w_empty_pattern.add(key)
                if len(sections_w_empty_pattern) > 1:
                    raise ValidationError(
                        'too many sections with an empty pattern: "{}". '
                        'at most one section can have an empty pattern and '
                        'hence become the default one.'.format(quote_join_sorted(sections_w_empty_pattern))
                    )

            if len(sections_w_empty_pattern) == 1:
                last_key, last_section = self.section_items[-1]
                if last_section.pb.pattern:
                    raise ValidationError('section with an empty pattern must go last',
                                          'sections[{}]'.format(sections_w_empty_pattern.pop()))

        new_preceding_modules = add_module(preceding_modules, self)
        for key, section in six.iteritems(self.sections):
            with validate('sections[{}]'.format(key)):
                section.validate(ctx=ctx, preceding_modules=new_preceding_modules, key=key)
        super(RegexpHost, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        super(RegexpHost, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        preceding_modules = add_module(preceding_modules, self)
        table = OrderedDict()
        n = len(self.sections)
        outlets = []
        for i, (key, section) in enumerate(six.iteritems(self.sections)):
            if not section.pb.pattern:
                key = 'default'
            section_config = section.to_config(priority=n - i, ctx=ctx, preceding_modules=preceding_modules)
            table[key] = section_config
            outlets.extend(section_config.get_outlets())
        return Config(table, shareable=True, outlets=outlets)

    def get_branches(self):
        return six.itervalues(self.sections)

    def get_named_branches(self):
        return self.sections

    def includes_upstreams(self):
        return self.include_upstreams


class HeaderMapEntry(WrapperBase):
    __protobuf__ = proto.HeaderMapEntry

    REQUIRED = ['key', 'value']
    ALLOWED_CALLS = {
        'value': [defs.get_str_env_var.name],
    }

    def validate(self):
        self.auto_validate_required()
        with validate('key'):
            validate_header_name(self.pb.key)
        with validate('value'):
            value = self.get('value')
            if value.is_func():
                validate_header_value_call(value.value)

    def to_config_item(self, ctx=DEFAULT_CTX):
        return self.pb.key, self.get('value').to_config(ctx=ctx)


class HeadersBase(ChainableModuleWrapperBase):
    create = []  # type: list[HeaderMapEntry]
    create_weak = []  # type: list[HeaderMapEntry]

    REQUIRED_ANYOFS = [('delete', 'create_func', 'create_func_weak', 'create', 'create_weak',
                        'append_func', 'append_func_weak', 'append', 'append_weak', 'copy', 'copy_weak')]

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()
        for param in ('create', 'create_weak', 'append', 'append_weak', 'copy', 'copy_weak'):
            for i, header_entry in enumerate(getattr(self, param)):
                with validate('{}[{}]'.format(param, i)):
                    header_entry.validate()
        for param in ('create_func', 'create_func_weak', 'append_func', 'append_func_weak'):
            with validate(param):
                for header_name in six.iterkeys(getattr(self.pb, param)):
                    validate_header_name(header_name)
        if self.pb.delete:
            with validate('delete'):
                validate_pire_regexp(self.pb.delete)
        with validate('create_func'):
            for header, func in six.iteritems(self.pb.create_func):
                validate_header_func(func, header, hint='Try create instead of create_func')
        with validate('create_func_weak'):
            for header, func in six.iteritems(self.pb.create_func_weak):
                validate_header_func(func, header, hint='Try create_weak instead of create_func_weak')
        with validate('append_func'):
            for header, func in six.iteritems(self.pb.append_func):
                validate_header_func(func, header, hint='Try append instead of append_func')
        with validate('append_func_weak'):
            for header, func in six.iteritems(self.pb.append_func_weak):
                validate_header_func(func, header, hint='Try append_weak instead of append_func_weak')
        super(HeadersBase, self).validate(ctx=ctx, preceding_modules=preceding_modules,
                                          chained_modules=chained_modules)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict()
        if self.pb.delete:
            table['delete'] = self.pb.delete
        if self.pb.create_func:
            table['create_func'] = Config(OrderedDict(sorted(self.pb.create_func.items())))
        if self.pb.create_func_weak:
            table['create_func_weak'] = Config(OrderedDict(sorted(self.pb.create_func_weak.items())))

        if self.pb.create:
            create_table = OrderedDict(sorted([entry.to_config_item(ctx=ctx) for entry in self.create]))
            table['create'] = Config(create_table)
        if self.pb.create_weak:
            create_weak_table = OrderedDict(sorted([entry.to_config_item(ctx=ctx) for entry in self.create_weak]))
            table['create_weak'] = Config(create_weak_table)

        if self.pb.append_func:
            table['append_func'] = Config(OrderedDict(sorted(self.pb.append_func.items())))
        if self.pb.append_func_weak:
            table['append_func_weak'] = Config(OrderedDict(sorted(self.pb.append_func_weak.items())))

        if self.pb.append:
            append_table = OrderedDict(sorted([entry.to_config_item(ctx=ctx) for entry in self.append]))
            table['append'] = Config(append_table)
        if self.pb.append_weak:
            append_weak_table = OrderedDict(sorted([entry.to_config_item(ctx=ctx) for entry in self.append_weak]))
            table['append_weak'] = Config(append_weak_table)

        if self.pb.copy:
            copy_table = OrderedDict(sorted([entry.to_config_item(ctx=ctx) for entry in self.copy]))
            table['copy'] = Config(copy_table)
        if self.pb.copy_weak:
            copy_weak_table = OrderedDict(sorted([entry.to_config_item(ctx=ctx) for entry in self.copy_weak]))
            table['copy_weak'] = Config(copy_weak_table)

        return Config(table)


class Headers(HeadersBase):
    __protobuf__ = proto.HeadersModule


class ResponseHeaders(HeadersBase):
    __protobuf__ = proto.ResponseHeadersModule


class CookieMapEntry(WrapperBase):
    __protobuf__ = proto.CookieMapEntry

    REQUIRED = ['key', 'value']

    def validate(self):
        self.auto_validate_required()
        with validate('key'):
            validate_cookie_name(self.pb.key)

    def to_config_item(self, ctx=DEFAULT_CTX):
        return self.pb.key, self.pb.value


class CookiesModule(ChainableModuleWrapperBase):
    __protobuf__ = proto.CookiesModule
    create = []  # type: list[CookieMapEntry]
    create_weak = []  # type: list[CookieMapEntry]

    REQUIRED_ANYOFS = [('delete', 'create', 'create_weak')]

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()
        for param in ('create', 'create_weak'):
            for i, cookie_entry in enumerate(getattr(self, param)):
                with validate('{}[{}]'.format(param, i)):
                    cookie_entry.validate()
        if self.pb.delete:
            with validate('delete'):
                validate_pire_regexp(self.pb.delete)
        super(CookiesModule, self).validate(
            preceding_modules=preceding_modules,
            chained_modules=chained_modules
        )

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict()
        if self.pb.delete:
            table['delete'] = self.pb.delete
        if self.create:
            create_table = OrderedDict(sorted([entry.to_config_item(ctx=ctx) for entry in self.create]))
            table['create'] = Config(create_table)
        if self.create_weak:
            create_weak_table = OrderedDict(sorted([entry.to_config_item(ctx=ctx) for entry in self.create_weak]))
            table['create_weak'] = Config(create_weak_table)
        return Config(table)


class Proxy(ModuleWrapperBase, ProxyOptionsMixin):
    __protobuf__ = proto.ProxyModule

    REQUIRED = ['host', 'port']

    def validate_instance_params(self):
        with validate('port'):
            validate_port(self.pb.port)

        if self.pb.cached_ip:
            with validate('cached_ip'):
                validate_ip(self.pb.cached_ip)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        self.validate_instance_params()
        self.validate_proxy_options(ctx=ctx)  # method from ProxyOptionsMixin
        super(Proxy, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict([
            ('host', self.pb.host),
            ('port', self.pb.port),
        ])
        if self.pb.cached_ip:
            table['cached_ip'] = self.pb.cached_ip
        config = Config(table)
        config.extend(
            self.get_proxy_options_config(ctx=ctx,
                                          preceding_modules=preceding_modules))  # method from ProxyOptionsMixin
        return config


class ErrorDocument(ModuleWrapperBase):
    __protobuf__ = proto.ErrorDocumentModule

    REQUIRED = ['status']

    # info attrs:
    DEFAULTS = {
        'force_conn_close': False,
    }

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        if not (0 < self.pb.status < 600):
            raise ValidationError('is not a valid HTTP status', 'status')
        c = 0
        if self.pb.file:
            c += 1
        if self.pb.content:
            c += 1
        if self.pb.base64:
            c += 1
        if c > 1:
            raise ValidationError('"file", "content" and "base64" are mutually exclusive')
        super(ErrorDocument, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = {
            'status': self.pb.status,
            'force_conn_close': self.pb.force_conn_close,
        }
        if self.pb.file:
            table['file'] = self.pb.file
        if self.pb.content:
            table['content'] = self.pb.content
        if self.pb.base64:
            table['base64'] = self.pb.base64
        if self.pb.remain_headers:
            table['remain_headers'] = self.pb.remain_headers
        return Config(table)


class RewriteAction(ConfigWrapperBase):
    __protobuf__ = proto.RewriteAction

    VALID_SPLIT_VALUES = (
        'url',
        'path',
        'cgi',
    )

    REQUIRED = ['rewrite']

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        if self.pb.split and self.pb.split not in self.VALID_SPLIT_VALUES:
            raise ValidationError('must be empty or equal to one of the '
                                  'following: {}'.format(', '.join(self.VALID_SPLIT_VALUES)), 'split')
        with validate('regexp'):
            validate_re2_regexp(self.pb.regexp)
        # TODO validate more thoroughly
        super(RewriteAction, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = {
            'literal': self.pb.literal,
            'global': getattr(self.pb, 'global'),
        }
        case_insensitive = self.pb.deprecated_case_insensitive
        if self.pb.HasField('case_insensitive'):
            case_insensitive = self.pb.case_insensitive.value
        table['case_insensitive'] = case_insensitive

        if self.pb.split:
            table['split'] = self.pb.split
        if self.pb.regexp:
            table['regexp'] = self.pb.regexp
        if self.pb.header_name:
            table['header_name'] = self.pb.header_name
        table['rewrite'] = self.pb.rewrite
        return Config(table)


class Rewrite(ChainableModuleWrapperBase):
    __protobuf__ = proto.RewriteModule

    actions = []  # type: list[RewriteAction]

    REQUIRED = ['actions']

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        new_preceding_modules = add_module(preceding_modules, self)
        for i, action in enumerate(self.actions):
            with validate('actions[{}]'.format(i)):
                action.validate(ctx=ctx, preceding_modules=new_preceding_modules)
        super(Rewrite, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules,
                                                       chained_modules=chained_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()
        super(Rewrite, self).validate(ctx=ctx, preceding_modules=preceding_modules, chained_modules=chained_modules)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        action_configs = Config(array=[action.to_config(ctx=ctx, preceding_modules=preceding_modules)
                                       for action in self.actions])
        table = OrderedDict([('actions', action_configs)])
        return Config(table)


class RedirectsUrlParts(ConfigWrapperBase):
    __protobuf__ = proto.RedirectsModule.UrlParts
    DEFAULT_PATH = True

    DEFAULTS = {
        'path': DEFAULT_PATH,
    }

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        super(RedirectsUrlParts, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict()
        if self.pb.HasField('host'):
            table['host'] = self.pb.host.value
        table['path'] = self.pb.path.value if self.pb.HasField('path') else self.DEFAULT_PATH
        if self.pb.HasField('query'):
            table['query'] = self.pb.query.value
        if self.pb.HasField('fragment'):
            table['fragment'] = self.pb.fragment.value
        return Config(table)


class RedirectsRewrite(ConfigWrapperBase):
    __protobuf__ = proto.RedirectsModule.Rewrite

    url = None  # type: RedirectsUrlParts | None

    REQUIRED = ['regexp']

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=()):
        field = self.pb.WhichOneof('what')
        if field:
            with validate(field):
                getattr(self, field).validate(ctx=ctx, preceding_modules=preceding_modules)
        super(RedirectsRewrite, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        with validate('regexp'):
            validate_re2_regexp(self.pb.regexp)
        super(RedirectsRewrite, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict()
        table['regexp'] = self.pb.regexp
        table['rewrite'] = self.pb.rewrite
        global_ = getattr(self.pb, 'global')
        if global_:
            table['global'] = global_
        field = self.pb.WhichOneof('what')
        if field:
            table[field] = getattr(self, field).to_config(ctx=ctx, preceding_modules=preceding_modules)
        return Config(table)


class RedirectsRedirect(ConfigWrapperBase):
    __protobuf__ = proto.RedirectsModule.Redirect

    dst_rewrites = []  # type: list[RedirectsRewrite]

    REQUIRED = ['dst', 'code']

    DEFAULT_LEGACY_RSTRIP = True

    DEFAULTS = {
        'legacy_rstrip': DEFAULT_LEGACY_RSTRIP
    }

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=()):
        for i, dst_rewrite in enumerate(self.dst_rewrites):
            with validate('dst_rewrites[{}]'.format(i)):
                dst_rewrite.validate(ctx=ctx, preceding_modules=preceding_modules)
        super(RedirectsRedirect, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        with validate('code'):
            # str() is used because of checker operates with string
            validate_status_code_3xx(str(self.pb.code))
        super(RedirectsRedirect, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict()
        table['dst'] = self.pb.dst
        table['legacy_rstrip'] = \
            self.pb.legacy_rstrip.value if self.pb.HasField('legacy_rstrip') else self.DEFAULT_LEGACY_RSTRIP
        table['code'] = self.pb.code
        table['dst_rewrites'] = Config(array=[dst_rewrite.to_config(ctx=ctx, preceding_modules=preceding_modules)
                                              for dst_rewrite in self.dst_rewrites])
        return Config(table)


class RedirectsForward(ConfigWrapperBase):
    __protobuf__ = proto.RedirectsModule.Forward

    dst_rewrites = []  # type: list[RedirectsRewrite]
    nested = None  # type: Holder

    REQUIRED = ['dst', 'nested']

    DEFAULT_LEGACY_RSTRIP = True

    DEFAULTS = {
        'legacy_rstrip': DEFAULT_LEGACY_RSTRIP
    }

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=()):
        with validate('nested'):
            self.nested.validate(ctx=ctx, preceding_modules=preceding_modules)
        for i, dst_rewrite in enumerate(self.dst_rewrites):
            with validate('dst_rewrites[{}]'.format(i)):
                dst_rewrite.validate(ctx=ctx, preceding_modules=preceding_modules)
        super(RedirectsForward, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        super(RedirectsForward, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict()
        table['dst'] = self.pb.dst
        table['legacy_rstrip'] = \
            self.pb.legacy_rstrip.value if self.pb.HasField('legacy_rstrip') else self.DEFAULT_LEGACY_RSTRIP
        table['dst_rewrites'] = Config(array=[dst_rewrite.to_config(ctx=ctx, preceding_modules=preceding_modules)
                                              for dst_rewrite in self.dst_rewrites])
        config = Config(table)
        nested_config = self.nested.to_config(
            ctx=ctx, preceding_modules=preceding_modules)
        config.extend(nested_config)
        return config


class RedirectsAction(ConfigWrapperBase):
    __protobuf__ = proto.RedirectsModule.Action

    forward = None  # type: RedirectsForward | None
    redirect = None  # type: RedirectsRedirect | None

    REQUIRED = ['src']
    REQUIRED_PB_ONEOFS = ['kind']

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=()):
        kind = self.pb.WhichOneof('kind')
        with validate(kind):
            getattr(self, kind).validate(ctx=ctx, preceding_modules=preceding_modules)
        super(RedirectsAction, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        super(RedirectsAction, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict()
        table['src'] = self.pb.src
        field = self.pb.WhichOneof('kind')
        table[field] = getattr(self, field).to_config(ctx=ctx, preceding_modules=preceding_modules)
        return Config(table)


class RedirectsModule(ChainableModuleWrapperBase):
    __protobuf__ = proto.RedirectsModule

    actions = []  # type: list[RedirectsAction]

    REQUIRED = ['actions']

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        new_preceding_modules = add_module(preceding_modules, self)
        sources = set()
        for i, action in enumerate(self.actions):
            with validate('actions[{}]'.format(i)):
                if action.pb.src in sources:
                    raise ValidationError('"actions" must have unique "src" fields')
                action.validate(ctx=ctx, preceding_modules=new_preceding_modules)
            sources.add(action.pb.src)

        super(RedirectsModule, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules,
                                                               chained_modules=chained_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()
        super(RedirectsModule, self).validate(ctx=ctx, preceding_modules=preceding_modules,
                                              chained_modules=chained_modules)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        action_configs = Config(array=[action.to_config(ctx=ctx, preceding_modules=preceding_modules)
                                       for action in self.actions])
        table = OrderedDict([('actions', action_configs)])
        return Config(table)


class ExpGetter(ChainableModuleWrapperBase):
    __protobuf__ = proto.ExpGetterModule

    uaas = None  # type: Holder | None

    REQUIRED = ['uaas']

    DEFAULT_FILE_SWITCH = './controls/expgetter.switch'

    ALLOWED_KNOBS = {
        'file_switch': model_pb2.KnobSpec.BOOLEAN,
    }
    DEFAULT_KNOB_IDS = {
        'file_switch': 'balancer_expgetter_switch',
    }

    DEFAULTS = {
        'trusted': False,
        'file_switch': DEFAULT_FILE_SWITCH,
    }

    def _validate_file_switch(self, ctx):
        self.validate_knob('file_switch', ctx=ctx)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()
        self._validate_file_switch(ctx)
        if self.pb.service_name and not self.pb.service_name_header:
            raise ValidationError('"service_name_header" must be specified if you specify "service_name"')
        if self.pb.HasField('headers_size_limit'):
            with validate('headers_size_limit'):
                validate_range(self.pb.headers_size_limit.value, 1024, 512 * 1024)
        super(ExpGetter, self).validate(ctx=ctx, preceding_modules=preceding_modules, chained_modules=chained_modules)

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        new_preceding_modules = add_module(preceding_modules, self)
        with validate('uaas'):
            self.uaas.validate(ctx=ctx, preceding_modules=new_preceding_modules)
        super(ExpGetter, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules,
                                                         chained_modules=chained_modules)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict([
            ('trusted', self.pb.trusted),
            ('file_switch', self.get('file_switch', self.DEFAULT_FILE_SWITCH).to_config(ctx)),
        ])
        if self.pb.service_name:
            table['service_name'] = self.pb.service_name
        if self.pb.service_name_header:
            table['service_name_header'] = self.pb.service_name_header
        if self.pb.exp_headers:
            table['exp_headers'] = self.pb.exp_headers
        if self.pb.HasField('headers_size_limit'):
            table['headers_size_limit'] = self.pb.headers_size_limit.value
        table['uaas'] = self.uaas.to_config(ctx=ctx, preceding_modules=add_module(preceding_modules, self))
        return Config(table)

    def get_branches(self):
        if self.uaas:
            yield self.uaas

    def get_named_branches(self):
        rv = {}
        if self.uaas:
            rv['uaas'] = self.uaas
        return rv


class Shared(ChainableModuleWrapperBase):
    __protobuf__ = proto.SharedModule

    REQUIRED = ['uuid']
    SHAREABLE = True

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()
        # TODO True is not the best value for chained_modules param, need to figure out a better way
        super(Shared, self).validate(ctx=ctx, preceding_modules=preceding_modules,
                                     chained_modules=True)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = {'uuid': self.pb.uuid}
        return Config(table, shared_uuid=self.pb.uuid)


class StatsEater(ChainableModuleWrapperBase):
    __protobuf__ = proto.StatsEaterModule


class ClientCertCheck(ConfigWrapperBase):
    __protobuf__ = proto.ClientCertCheck

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        if self.pb.verify_depth < 0:
            raise ValidationError('must be non-negative', 'verify_depth')

        if not self.pb.verify_peer and any([self.pb.verify_once, self.pb.fail_if_no_peer_cert]):
            raise ValidationError('enabling "fail_if_no_peer_cert" and/or "verify_once" options '
                                  'with disabled "verify_peer" option is prohibited')

        super(ClientCertCheck, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict([
            ('verify_peer', self.pb.verify_peer),
            ('verify_depth', self.pb.verify_depth),
            ('verify_once', self.pb.verify_once),
            ('fail_if_no_peer_cert', self.pb.fail_if_no_peer_cert),
        ])
        if self.pb.crl:
            table['crl'] = self.pb.crl

        return Config(table)


class SecondaryCert(ConfigWrapperBase, CertMixin):
    __protobuf__ = proto.SecondaryCert

    REQUIRED = ['cert', ]
    ALLOWED_CALLS = {
        'cert': [defs.get_str_var.name, defs.get_public_cert_path.name],
        'priv': [defs.get_str_var.name, defs.get_private_cert_path.name],
    }
    ALLOWED_CERTS = ('cert',)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        self._validate_certs(ctx)
        super(SecondaryCert, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = {
            'cert': self.get('cert').to_config(ctx=ctx),
            'priv': self.get('priv').to_config(ctx=ctx),
        }
        if self.pb.ocsp:
            table['ocsp'] = self.pb.ocsp
        return Config(table)


class Servername(ConfigWrapperBase):
    __protobuf__ = proto.Servername

    REQUIRED = ['servername_regexp']

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        with validate('servername_regexp'):
            validate_pire_regexp(self.pb.servername_regexp)
        super(Servername, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        return Config({
            'servername_regexp': self.pb.servername_regexp,
            'case_insensitive': self.pb.case_insensitive,
            'surround': self.pb.surround,
        })


class TicketKey(ConfigWrapperBase):
    __protobuf__ = proto.TicketKey

    REQUIRED = ['keyfile']
    ALLOWED_CALLS = {
        'keyfile': [defs.get_private_cert_path.name],
    }

    @validate('keyfile')
    def _validate_keyfile(self):
        keyfile = self.get('keyfile')
        if keyfile.is_func():
            validate_func_name_one_of(keyfile.value, (defs.get_private_cert_path.name,))
            keyfile.value.validate()

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        self._validate_keyfile()
        super(TicketKey, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, priority, ctx=DEFAULT_CTX, preceding_modules=()):
        return Config(OrderedDict([
            ('priority', priority),
            ('keyfile', self.get('keyfile').to_config(ctx=ctx)),
        ]))


class SslSniContext(ConfigWrapperBase, CertMixin):
    __protobuf__ = proto.SslSniContext

    servername = None  # type: Servername | None
    secondary = None  # type: SecondaryCert | None
    ticket_keys = []  # type: list[TicketKey]
    client = None  # type: ClientCertCheck | None

    REQUIRED = ['cert', ]

    DEFAULT_CONTEXT_KEY = 'default'
    DEFAULT_TIMEOUT = '100800s'
    DEFAULT_OCSP_FILE_SWITCH = './controls/disable_ocsp'

    ALLOWED_KNOBS = {
        'ocsp_file_switch': model_pb2.KnobSpec.BOOLEAN,
    }
    DEFAULT_KNOB_IDS = {
        'ocsp_file_switch': 'balancer_ocsp_switch',
    }

    DEFAULTS = {
        'timeout': DEFAULT_TIMEOUT,
        'ocsp_file_switch': DEFAULT_OCSP_FILE_SWITCH
    }
    ALLOWED_CALLS = {
        'log': (defs.get_str_var.name, defs.get_log_path.name),
        'cert': [defs.get_str_var.name, defs.get_public_cert_path.name],
        'priv': [defs.get_str_var.name, defs.get_private_cert_path.name],
        'ca': [defs.get_str_var.name, defs.get_ca_cert_path.name],
        'secrets_log': (defs.get_str_var.name, defs.get_log_path.name),
    }
    ALLOWED_CERTS = ('cert',)

    # https://a.yandex-team.ru/arc/trunk/arcadia/balancer/modules/ssl/sslitem.cpp
    SSL_PROTOCOLS = {u'sslv2', u'sslv3', u'tlsv1', u'tlsv1.1', u'tlsv1.2', u'tlsv1.3'}

    def _get_ciphers(self, is_ecdsa):
        if self.pb.HasField('disable_rc4_sha_cipher') and self.pb.disable_rc4_sha_cipher.value:
            if is_ecdsa:
                return CipherSuite.DEFAULT_WITH_ECDSA_AND_WITHOUT_RC4
            else:
                return CipherSuite.DEFAULT_WITHOUT_RC4
        else:
            if is_ecdsa:
                return CipherSuite.DEFAULT_WITH_ECDSA
            else:
                return CipherSuite.DEFAULT

    def include_certs(self, namespace_id, cert_spec_pbs, ctx):
        """
        :type namespace_id: six.text_type
        :type cert_spec_pbs: dict[(six.text_type, six.text_type), model_pb2.CertificateSpec]
        :type ctx: ValidationCtx
        :rtype: set[(six.text_type, six.text_type)]
        """
        rv = super(SslSniContext, self).include_certs(namespace_id, cert_spec_pbs, ctx)
        if self.pb.ciphers:
            return rv
        for full_cert_id in rv:
            cert_spec_pb = cert_spec_pbs[full_cert_id]
            if cert_spec_pb.fields.public_key_info.algorithm_id == 'ec':
                self.pb.ciphers = self._get_ciphers(is_ecdsa=True)
                break
        return rv

    def validate_composite_fields(self, ctx=DEFAULT_CTX, key=None, preceding_modules=()):
        with validate('servername'):
            if key == self.DEFAULT_CONTEXT_KEY:
                if self.servername:
                    raise ValidationError('must not be specified for default context')
            else:
                self.require_value(self.servername)
                self.servername.validate(ctx=ctx, preceding_modules=preceding_modules)

        for i, ticket_key in enumerate(self.ticket_keys):
            with validate('ticket_keys[{}]'.format(i)):
                ticket_key.validate(ctx=ctx, preceding_modules=preceding_modules)

        if self.secondary:
            with validate('secondary'):
                self.secondary.validate(ctx=ctx, preceding_modules=preceding_modules)

        if self.client:
            with validate('client'):
                self.client.validate(ctx=ctx, preceding_modules=preceding_modules)
                if not self.pb.ca:
                    raise ValidationError('using "client" opt without "ca" opt is prohibited')

        super(SslSniContext, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules)

    @validate('log')
    def _validate_log(self):
        log = self.get('log')
        if log.value and log.is_func():
            validate_log_call(log.value)

    @validate('ca')
    def _validate_ca(self):
        ca = self.get('ca')
        if ca.is_func():
            validate_func_name_one_of(ca.value, (defs.get_str_var.name, defs.get_ca_cert_path.name))
            ca.value.validate()

    @validate('secrets_log')
    def _validate_secrets_log(self):
        secrets_log = self.get('secrets_log')
        if secrets_log.value and secrets_log.is_func():
            validate_log_call(secrets_log.value)

    def _validate_ocsp_file_switch(self, ctx):
        self.validate_knob('ocsp_file_switch', ctx=ctx)

    def _validate_ssl_protocols(self):
        with validate(u'ssl_protocols'):
            for protocol in self.pb.ssl_protocols:
                if protocol not in self.SSL_PROTOCOLS:
                    raise ValidationError(u'unsupported protocol "{}"'.format(protocol))
            if self.pb.disable_sslv3 and u'sslv3' in self.pb.ssl_protocols:
                raise ValidationError(u'protocol "sslv3" is disabled by config option "disable_sslv3"')
            if self.pb.disable_tlsv1_3.value and u'tlsv1.3' in self.pb.ssl_protocols:
                raise ValidationError(u'protocol "tlsv1.3" is disabled by config option "disable_tlsv1_3"')

    def validate(self, ctx=DEFAULT_CTX, key=None, preceding_modules=()):
        self.auto_validate_required()

        self._validate_certs(ctx=ctx)
        self._validate_ca()
        self._validate_log()
        self._validate_secrets_log()
        self._validate_ocsp_file_switch(ctx)
        if self.pb.timeout:
            with validate('timeout'):
                validate_timedelta(self.pb.timeout)
        if self.pb.ssl_protocols:
            self._validate_ssl_protocols()

        # normally we would call super() here, but we need to check the key, which super doesn't
        # so we manually call composite field validation and call it's super there
        self.validate_composite_fields(ctx=ctx, key=key, preceding_modules=preceding_modules)

    def to_config(self, priority, ctx=DEFAULT_CTX, preceding_modules=()):
        table = {
            'cert': self.get('cert').to_config(ctx=ctx),
            'priv': self.get('priv').to_config(ctx=ctx),
            'timeout': self.pb.timeout or self.DEFAULT_TIMEOUT,
            'priority': priority,
        }
        global_vars = {}

        if self.pb.ciphers:
            table['ciphers'] = self.pb.ciphers
        elif self.secondary:
            # at the moment (11.2020) secondary certs are used for a primary EC cert + fallback RSA cert.
            # There are no situations with a single EC cert, or a primary RSA + secondary EC cert,
            # so we can make this assumption about ciphers set
            varname = 'default_ciphers_ecdsa'
            global_vars[varname] = self._get_ciphers(is_ecdsa=True)
            table['ciphers'] = ConfigCall('get_str_var', (varname,))
        else:
            varname = 'default_ciphers'
            global_vars[varname] = self._get_ciphers(is_ecdsa=False)
            table['ciphers'] = ConfigCall('get_str_var', (varname,))

        if self.servername:
            table['servername'] = self.servername.to_config(ctx=ctx, preceding_modules=preceding_modules)
        ca = self.get('ca')
        if ca.value:
            table['ca'] = ca.to_config(ctx=ctx)
        log = self.get('log')
        if log.value:
            table['log'] = log.to_config(ctx=ctx)
        secrets_log = self.get('secrets_log')
        if secrets_log.value:
            table['secrets_log'] = secrets_log.to_config(ctx=ctx)
        if self.pb.ocsp:
            table['ocsp'] = self.pb.ocsp
            table['ocsp_file_switch'] = self.get('ocsp_file_switch', self.DEFAULT_OCSP_FILE_SWITCH).to_config(ctx)
        if self.ticket_keys:
            keys_array = []
            n = len(self.ticket_keys)
            for i, key in enumerate(self.ticket_keys):
                keys_array.append(key.to_config(priority=n - i, ctx=ctx, preceding_modules=preceding_modules))
            table['ticket_keys_list'] = Config(array=keys_array)
        if self.pb.disable_sslv3:
            table['disable_sslv3'] = self.pb.disable_sslv3
        if self.pb.HasField('disable_tlsv1_3'):
            table['disable_tlsv1_3'] = self.pb.disable_tlsv1_3.value
        if self.pb.ssl_protocols:
            table['ssl_protocols'] = Config(array=list(self.pb.ssl_protocols))
        if self.secondary:
            table['secondary'] = self.secondary.to_config(ctx=ctx, preceding_modules=preceding_modules)
        if self.client:
            table['client'] = self.client.to_config(ctx=ctx, preceding_modules=preceding_modules)

        return Config(table, global_vars=global_vars)


class SslSni(ChainableModuleWrapperBase, CertMixin):
    __protobuf__ = proto.SslSniModule

    __slots__ = ('contexts',)

    context_items = []  # type: list[(basestring, SslSniContext)]
    # contexts = {}  # type: dict[basestring, SslSniContext]

    PB_FIELD_TO_CLS_ATTR_MAPPING = {'contexts': 'context_items'}

    DEFAULT_EVENTS = dict(DEFAULT_EVENTS,
                          reload_ticket_keys='reload_ticket',
                          reload_ocsp_response='reload_ocsp')
    DEFAULT_FORCE_SSL = True
    DEFAULT_HTTP2_ALPN_FILE = './controls/http2_enable.ratefile'
    DEFAULT_HTTP2_ALPN_FREQ = 0

    DEFAULTS = {
        'force_ssl': DEFAULT_FORCE_SSL,
        'http2_alpn_file': DEFAULT_HTTP2_ALPN_FILE,
        'http2_alpn_freq': DEFAULT_HTTP2_ALPN_FREQ,
    }
    ALLOWED_KNOBS = {
        'http2_alpn_file': model_pb2.KnobSpec.RATE,
    }
    DEFAULT_KNOB_IDS = {
        'http2_alpn_file': 'http2_request_rate',
    }

    def wrap_composite_fields(self):
        super(SslSni, self).wrap_composite_fields()
        self.contexts = OrderedDict(self.context_items)

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        validate_key_uniqueness(self.context_items)
        new_preceding_modules = add_module(preceding_modules, self)
        for key, context in six.iteritems(self.contexts):
            with validate('contexts[{}]'.format(key)):
                context.validate(key=key, ctx=ctx, preceding_modules=new_preceding_modules)
        super(SslSni, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules,
                                                      chained_modules=chained_modules)

    def _validate_http2_alpn_file(self, ctx):
        self.validate_knob('http2_alpn_file', ctx=ctx)

    @validate('max_send_fragment')
    def _validate_max_send_fragment(self):
        if self.pb.max_send_fragment:
            validate_range(self.pb.max_send_fragment, 512, 16384)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self._validate_http2_alpn_file(ctx=ctx)
        with validate('http2_alpn_freq'):
            validate_range(self.pb.http2_alpn_freq, 0, 1)
        self._validate_max_send_fragment()
        super(SslSni, self).validate(ctx=ctx, preceding_modules=preceding_modules, chained_modules=chained_modules)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        events = self.DEFAULT_EVENTS
        if self.pb.events:
            events = OrderedDict(sorted(self.pb.events.items()))
        # we want events and contexts go first, thus make table an ordered dict
        force_ssl = self.DEFAULT_FORCE_SSL
        if self.pb.HasField('force_ssl'):
            force_ssl = self.pb.force_ssl.value
        table = OrderedDict([
            ('force_ssl', force_ssl),
            # TODO Default value will be enabled a bit later
            ('events', Config(events)),
        ])
        http2_alpn_file = self.get('http2_alpn_file')
        if http2_alpn_file.value:
            table['http2_alpn_file'] = http2_alpn_file.to_config(ctx)
        if self.pb.http2_alpn_freq:
            table['http2_alpn_freq'] = self.pb.http2_alpn_freq
        if self.pb.max_send_fragment:
            table['max_send_fragment'] = self.pb.max_send_fragment
        if self.pb.validate_cert_date:
            table['validate_cert_date'] = self.pb.validate_cert_date
        if self.pb.ja3_enabled:
            table['ja3_enabled'] = self.pb.ja3_enabled

        n = len(self.contexts)
        contexts_table = OrderedDict()
        for key, context in six.iteritems(self.contexts):
            contexts_table[key] = context.to_config(priority=n, ctx=ctx)
            n -= 1
        table['contexts'] = Config(contexts_table)

        return Config(table, shareable=True)


class Geobase(ChainableModuleWrapperBase):
    __protobuf__ = proto.GeobaseModule

    geo = None  # type: Holder | None

    ALLOWED_KNOBS = {
        'file_switch': model_pb2.KnobSpec.BOOLEAN,
    }

    REQUIRED = ['geo']

    DEFAULT_GEO_HOST = 'laas.yandex.ru'
    DEFAULT_GEO_PATH = '/region?response_format=header&version=1&service=balancer'
    DEFAULT_TAKE_IP_FROM = 'X-Forwarded-For-Y'
    DEFAULT_LAAS_ANSWER_HEADER = 'X-LaaS-Answered'

    DEFAULTS = {
        'laas_answer_header': DEFAULT_LAAS_ANSWER_HEADER,
        'geo_host': DEFAULT_GEO_HOST,
        'geo_path': DEFAULT_GEO_PATH,
        'take_ip_from': DEFAULT_TAKE_IP_FROM,
        'trusted': False,
    }

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        with validate('geo'):
            self.geo.validate(ctx=ctx, preceding_modules=add_module(preceding_modules, self))
        super(Geobase, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules,
                                                       chained_modules=chained_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()
        super(Geobase, self).validate(ctx=ctx, preceding_modules=preceding_modules, chained_modules=chained_modules)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = {
            'geo': self.geo.to_config(ctx=ctx, preceding_modules=add_module(preceding_modules, self)),
            'laas_answer_header': self.pb.laas_answer_header or self.DEFAULT_LAAS_ANSWER_HEADER,
            'geo_host': self.pb.geo_host or self.DEFAULT_GEO_HOST,
            'geo_path': self.pb.geo_path or self.DEFAULT_GEO_PATH,
            'take_ip_from': self.pb.take_ip_from or self.DEFAULT_TAKE_IP_FROM,
            'trusted': self.pb.trusted,
        }
        file_switch = self.get('file_switch')
        if file_switch.value:
            table['file_switch'] = file_switch.to_config(ctx)
        return Config(table)

    def get_branches(self):
        if self.geo:
            yield self.geo

    def get_named_branches(self):
        rv = {}
        if self.geo:
            rv['geo'] = self.geo
        return rv


class InstanceMacro(ModuleWrapperBase, MacroBase):
    __protobuf__ = proto.InstanceMacro

    __slots__ = ('sections',)

    # sections = OrderedDict()  # type: dict[basestring, IpdispatchSection]
    section_items = []  # type: list[(basestring, IpdispatchSection)]
    include_upstreams = None  # type: IncludeUpstreams | None
    unistat = None  # type: Unistat | None
    cpu_limiter = None  # type: CpuLimiter | None
    sd = None  # type: ServiceDiscovery | None
    config_check = None  # type: ConfigCheck | None

    REQUIRED_ONEOFS = [('include_upstreams', 'sections')]

    PB_FIELD_TO_CLS_ATTR_MAPPING = {'sections': 'section_items'}

    DEFAULTS = {
        'log_dir': DEFAULT_LOG_DIR,
        'public_cert_dir': DEFAULT_PUBLIC_CERT_DIR,
        'private_cert_dir': DEFAULT_PRIVATE_CERT_DIR,
        'ca_cert_dir': DEFAULT_CA_CERT_DIR,
    }
    ALLOWED_CALLS = {
        'dns_ttl': (defs.get_random_timedelta.name,),
        'dynamic_balancing_log': (defs.get_str_var.name, defs.get_log_path.name),
        'pinger_log': (defs.get_str_var.name, defs.get_log_path.name),
    }

    VERSION_0_0_1 = semantic_version.Version(u'0.0.1')
    VERSION_0_0_2 = semantic_version.Version(u'0.0.2')
    INITIAL_VERSION = VERSION_0_0_1
    VALID_VERSIONS = frozenset((VERSION_0_0_1, VERSION_0_0_2))

    def get_version(self):
        if self.pb.version:
            return semantic_version.Version(self.pb.version)
        return self.INITIAL_VERSION

    def get_would_be_included_full_knob_ids(self, namespace_id, ctx):
        rv = super(InstanceMacro, self).get_would_be_included_full_knob_ids(namespace_id, ctx)
        if ctx.are_knobs_enabled():
            rv.add((namespace_id, Main.DEFAULT_KNOB_IDS['reset_dns_cache_file']))
        return rv

    def wrap_composite_fields(self):
        super(InstanceMacro, self).wrap_composite_fields()
        self.sections = OrderedDict(self.section_items)

    @validate('maxconn')
    def _validate_maxconn(self):
        if self.pb.maxconn < 0:
            raise ValidationError('must be non-negative')

    @validate('buffer')
    def _validate_buffer(self):
        if self.pb.buffer < 0:
            raise ValidationError('must be non-negative')

    @validate('workers')
    def _validate_workers(self):
        workers = self.get('workers')
        if not workers.value:
            return

        if workers.is_func():
            validate_workers_call(workers.value)
        else:
            if self.pb.workers < 0:
                raise ValidationError('must be positive or zero')

    @validate('private_address')
    def _validate_private_address(self):
        if self.pb.private_address:
            if not is_addr_local(self.pb.private_address):
                raise ValidationError('must be local v4 address')

    @validate('tcp_fastopen')
    def _validate_tcp_fastopen(self):
        if self.pb.tcp_fastopen < 0:
            raise ValidationError('must be non-negative')

    @validate('dns_ttl')
    def _validate_dns_ttl(self):
        dns_ttl = self.get('dns_ttl')
        if not dns_ttl.value:
            return

        if dns_ttl.is_func():
            validate_dns_ttl_call(dns_ttl.value)
        else:
            validate_timedelta(self.pb.dns_ttl)

    @validate('dynamic_balancing_log')
    def _validate_dynamic_balancing_log(self):
        dynamic_balancing_log = self.get('dynamic_balancing_log')
        if not dynamic_balancing_log.value:
            return

        if dynamic_balancing_log.is_func():
            validate_log_call(dynamic_balancing_log.value)

    @validate('pinger_log')
    def _validate_pinger_log(self):
        pinger_log = self.get('pinger_log')
        if not pinger_log.value:
            return

        if pinger_log.is_func():
            validate_log_call(pinger_log.value)

    @validate('tcp_listen_queue')
    def _validate_tcp_listen_queue(self):
        if self.pb.tcp_listen_queue < 0:
            raise ValidationError('must be non-negative')

    @validate('state_directory')
    def _validate_state_directory(self):
        if self.pb.pinger_required and not self.pb.state_directory:
            raise ValidationError('must be set if pinger_required is set')

    @validate('worker_start_delay')
    def _validate_worker_start_delay(self):
        if self.pb.worker_start_delay:
            validate_timedelta(self.pb.worker_start_delay)

    @validate('worker_start_duration')
    def _validate_worker_start_duration(self):
        if self.pb.worker_start_duration:
            validate_timedelta(self.pb.worker_start_duration)

    @validate('coro_stack_size')
    def _validate_coro_stack_size(self):
        if self.pb.coro_stack_size:
            min_16_kb = 16384
            max_64_mb = 64 * 2 ** 20
            validate_range(self.pb.coro_stack_size, min_16_kb, max_64_mb)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        self._validate_maxconn()
        self._validate_buffer()
        self._validate_workers()
        self._validate_private_address()
        self._validate_tcp_fastopen()
        self._validate_dns_ttl()
        self._validate_dynamic_balancing_log()
        self._validate_pinger_log()
        self._validate_state_directory()
        self._validate_tcp_listen_queue()
        self._validate_worker_start_delay()
        self._validate_worker_start_duration()
        self._validate_coro_stack_size()
        if self.pb.version:
            validate_version(self.pb.version, self.VALID_VERSIONS, field_name='version')
        super(InstanceMacro, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=()):
        if preceding_modules:
            raise ValidationError('must go first in a config tree')

        with validate('sections'):
            validate_key_uniqueness(self.section_items)

        preceding_modules = add_module(preceding_modules, self)
        for key, section in six.iteritems(self.sections):
            with validate('sections[{}]'.format(key)):
                section.validate(ctx=ctx, preceding_modules=preceding_modules)

        for key_1, key_2 in itertools.combinations(list(self.sections.keys()), 2):
            section_1 = self.sections[key_1]
            section_2 = self.sections[key_2]
            common_addrs = intersect_addrs(section_1.list_addrs(), section_2.list_addrs())
            if common_addrs:
                common_addrs_str = ', '.join(format_ip_port(ip.value, port.value)
                                             for ip, port in sorted(common_addrs))
                raise ValidationError('{} and {} sections intersect by '
                                      'addresses: {}'.format(key_1, key_2, common_addrs_str))

        if self.unistat:
            unistat_addrs = self.unistat.list_addrs()
            for key, section in six.iteritems(self.sections):
                section_addrs = section.list_addrs()
                common_unistat_addrs = intersect_addrs(section_addrs, unistat_addrs)
                if common_unistat_addrs:
                    common_unistat_addrs_str = ', '.join(format_addr(addr) for addr in common_unistat_addrs)
                    raise ValidationError(
                        'unistat addrs and section "{}" addrs intersect: {}'.format(key, common_unistat_addrs_str))

            with validate('unistat'):
                self.unistat.validate(ctx=ctx, preceding_modules=preceding_modules)

        if self.cpu_limiter:
            with validate('cpu_limiter'):
                self.cpu_limiter.validate(ctx=ctx, preceding_modules=preceding_modules)

        if self.sd:
            with validate('sd'):
                if not self.unistat:
                    raise ValidationError('can not be enabled without unistat')
                self.sd.validate(ctx=ctx, preceding_modules=preceding_modules)

        if self.config_check:
            with validate('config_check'):
                self.config_check.validate(ctx=ctx, preceding_modules=preceding_modules)

    def expand(self, ctx=DEFAULT_CTX, preceding_modules=()):
        h_pb = proto.Holder()

        addrs = OrderedDict()
        admin_addrs = OrderedDict()
        for section in six.itervalues(self.sections):
            for addr in section.list_addrs():
                if section.is_admin():
                    admin_addrs[addr] = 1
                else:
                    addrs[addr] = 1
        addrs = list(addrs.keys())
        admin_addrs = list(admin_addrs.keys())

        m = h_pb.main
        if self.pb.HasField('unistat'):
            m.unistat.CopyFrom(self.pb.unistat)
        if self.pb.HasField('sd'):
            m.sd.CopyFrom(self.pb.sd)
            if self.get_version() >= self.VERSION_0_0_2:
                m.sd.f_cache_dir.type = m.sd.f_cache_dir.GET_STR_VAR
                get_str_var_params_pb = m.sd.f_cache_dir.get_str_var_params
                get_str_var_params_pb.var = 'sd_cache_dir'
                get_str_var_params_pb.default = ServiceDiscovery.DEFAULT_CACHE_DIR
        if self.pb.HasField('cpu_limiter'):
            m.cpu_limiter.CopyFrom(self.pb.cpu_limiter)
        if self.pb.HasField('config_check'):
            m.config_check.CopyFrom(self.pb.config_check)
        m.storage_gc_required = self.pb.storage_gc_required
        m.enable_matcher_map_fix = self.pb.enable_matcher_map_fix
        m.enable_reuse_port = self.pb.enable_reuse_port
        m.disable_reuse_port = self.pb.disable_reuse_port
        m.coro_stack_size = self.pb.coro_stack_size
        if self.pb.HasField('coro_stack_guard'):
            m.coro_stack_guard.CopyFrom(self.pb.coro_stack_guard)
        if self.pb.HasField('shutdown_accept_connections'):
            m.shutdown_accept_connections.CopyFrom(self.pb.shutdown_accept_connections)
        if self.pb.HasField('shutdown_close_using_bpf'):
            m.shutdown_close_using_bpf.CopyFrom(self.pb.shutdown_close_using_bpf)

        ip, port = admin_addrs[0]
        call_pb = m.f_log
        call_pb.type = proto.Call.GET_LOG_PATH
        params_pb = call_pb.get_log_path_params
        params_pb.name = 'childs_log'
        if port.is_func():
            params_pb.f_port.CopyFrom(port.value.pb)
        else:
            params_pb.port = port.value
        params_pb.default_log_dir = self.pb.log_dir or DEFAULT_LOG_DIR

        m.maxconn = self.pb.maxconn
        m.buffer = self.pb.buffer

        workers = self.get('workers')
        if workers.is_func():
            m.f_workers.MergeFrom(workers.value.pb)
        else:
            m.workers = workers.value
        m.thread_mode = self.pb.thread_mode

        m.private_address = self.pb.private_address
        m.tcp_fastopen = self.pb.tcp_fastopen
        m.tcp_congestion_control = self.pb.tcp_congestion_control

        if self.pb.tcp_listen_queue:
            m.tcp_listen_queue = self.pb.tcp_listen_queue

        dns_ttl = self.get('dns_ttl')
        if dns_ttl.value:
            if dns_ttl.is_func():
                m.f_dns_ttl.MergeFrom(dns_ttl.value.pb)
            else:
                m.dns_ttl = dns_ttl.value
        else:
            call_pb_dttl = m.f_dns_ttl
            call_pb_dttl.type = proto.Call.GET_RANDOM_TIMEDELTA
            params_pb_dttl = call_pb_dttl.get_random_timedelta_params
            params_pb_dttl.start = 600
            params_pb_dttl.end = 900
            params_pb_dttl.unit = 's'

        m.pinger_required = self.pb.pinger_required
        m.state_directory = self.pb.state_directory
        m.dynamic_balancing_log = self.pb.dynamic_balancing_log
        if self.pb.HasField('f_dynamic_balancing_log'):
            m.f_dynamic_balancing_log.CopyFrom(self.pb.f_dynamic_balancing_log)
        m.pinger_log = self.pb.pinger_log
        if self.pb.HasField('f_pinger_log'):
            m.f_pinger_log.CopyFrom(self.pb.f_pinger_log)

        if self.pb.HasField('default_tcp_rst_on_error'):
            m.default_tcp_rst_on_error.CopyFrom(self.pb.default_tcp_rst_on_error)

        if self.pb.worker_start_delay:
            m.worker_start_delay = self.pb.worker_start_delay
        if self.pb.worker_start_duration:
            m.worker_start_duration = self.pb.worker_start_duration

        def assign_addrs(ip_port_pairs, to):
            """
            :type ip_port_pairs: list[(FieldValue, FieldValue)]
            """
            for ip, port in ip_port_pairs:
                addr = to.add()
                if ip.is_func():
                    addr.f_ip.CopyFrom(ip.value.pb)
                else:
                    addr.ip = ip.value
                if port.is_func():
                    addr.f_port.CopyFrom(port.value.pb)
                else:
                    addr.port = port.value

        assign_addrs(addrs, m.addrs)
        assign_addrs(admin_addrs, m.admin_addrs)

        if self.pb.HasField('include_upstreams'):
            m.nested.ipdispatch.include_upstreams.CopyFrom(self.pb.include_upstreams)
        m.nested.ipdispatch.sections.extend(self.pb.sections)

        return [h_pb]

    def includes_upstreams(self):
        return self.include_upstreams

    def get_branches(self):
        return six.itervalues(self.sections)

    def get_named_branches(self):
        return self.sections


class ExtendedHttpMacroSslSniContext(ConfigWrapperBase):
    __protobuf__ = proto.ExtendedHttpMacroSslSniContext

    client = None  # type: ClientCertCheck | None

    REQUIRED = ['servername_regexp']

    DEFAULT_DISABLE_OCSP = True

    DEFAULTS = {
        'disable_ocsp': DEFAULT_DISABLE_OCSP,
    }
    ALLOWED_CERTS = ('cert', 'secondary_cert',)

    def validate_composite_fields(self, ctx=DEFAULT_CTX, key=None, preceding_modules=()):
        if self.client:
            with validate('client'):
                self.client.validate(ctx=ctx, preceding_modules=preceding_modules)
                if not self.pb.ca:
                    raise ValidationError('using "client" opt without "ca" opt is prohibited')
        if self.pb.ssl_protocols:
            with validate(u'ssl_protocols'):
                for protocol in self.pb.ssl_protocols:
                    if protocol not in SslSniContext.SSL_PROTOCOLS:
                        raise ValidationError(u'unsupported protocol "{}"'.format(protocol))

        super(ExtendedHttpMacroSslSniContext, self).validate_composite_fields(
            ctx=ctx, preceding_modules=preceding_modules)

    @validate('cert')
    def _validate_cert(self):
        cert = self.get('cert')
        if not cert.value:
            return
        if not cert.is_cert():
            raise ValidationError('in ExtendedHttpMacro only supports !c-values')

    @validate('secondary_cert')
    def _validate_secondary_cert(self):
        cert = self.get('secondary_cert')
        if not cert.value:
            return
        if not cert.is_cert():
            raise ValidationError('in ExtendedHttpMacro only supports !c-values')
        if self.pb.secondary_cert_postfix:
            raise ValidationError('in ExtendedHttpMacro cannot be used together with secondary_cert_postfix')

    def validate(self, ctx=DEFAULT_CTX, key=None, preceding_modules=()):
        self.auto_validate_required()
        with validate('servername_regexp'):
            validate_pire_regexp(self.pb.servername_regexp)
        self._validate_cert()
        self._validate_secondary_cert()
        super(ExtendedHttpMacroSslSniContext, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        disable_ocsp = self.DEFAULT_DISABLE_OCSP
        if self.pb.HasField('disable_ocsp'):
            disable_ocsp = self.pb.keepalive.value

        table = {
            'servername_regexp': self.pb.servername_regexp,
            'disable_ocsp': disable_ocsp,
        }
        if self.pb.secondary_cert_postfix:
            table['secondary_cert_postfix'] = self.pb.secondary_cert_postfix
        if self.pb.ca:
            table['ca'] = self.pb.ca

        if self.client:
            table['client'] = self.client.to_config(ctx=ctx, preceding_modules=preceding_modules)
        secrets_log = self.pb.get('secrets_log')
        if secrets_log.value:
            table['secrets_log'] = secrets_log.to_config(ctx=ctx)
        return Config(table)


class ExtendedHttpMacro(ChainableModuleWrapperBase, MacroBase, HttpMixin):
    __protobuf__ = proto.ExtendedHttpMacro

    ssl_sni_context_items = []  # type: list[(basestring, ExtendedHttpMacroSslSniContext)]
    ssl_sni_contexts = OrderedDict()  # type: dict[basestring, ExtendedHttpMacroSslSniContext]

    DEFAULT_MAXLEN = 65536
    DEFAULT_MAXREQ = 65536
    DEFAULT_KEEPALIVE = True
    DEFAULT_REPORT_UUID = 'service_total'
    DEFAULT_REPORT_REFERS = 'service_total'
    DEFAULT_HTTP2_ALPN_FREQ = 1.0

    YANDEX_COOKIE_POLICY_UUID = 'service_total'
    DEFAULT_YANDEX_POLICIES = 'stable'

    PB_FIELD_TO_CLS_ATTR_MAPPING = {'ssl_sni_contexts': 'ssl_sni_context_items'}

    DEFAULTS = {
        'maxlen': DEFAULT_MAXLEN,
        'maxreq': DEFAULT_MAXREQ,
        'keepalive': DEFAULT_KEEPALIVE,
        'report_uuid': DEFAULT_REPORT_UUID,
        'report_refers': DEFAULT_REPORT_REFERS,
        'force_ssl': SslSni.DEFAULT_FORCE_SSL,
        'http2_alpn_freq': DEFAULT_HTTP2_ALPN_FREQ,
    }
    ALLOWED_CALLS = {
        'port': [defs.get_int_var.name, defs.get_port_var.name]
    }
    ALLOWED_KNOBS = {
        'disable_client_hints_restore_file': model_pb2.KnobSpec.BOOLEAN,
    }

    def is_chainable(self):
        return True

    def wrap_composite_fields(self):
        super(ExtendedHttpMacro, self).wrap_composite_fields()
        self.ssl_sni_contexts = OrderedDict(self.ssl_sni_context_items)

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        if self.pb.report_outgoing_codes:
            with validate('report_outgoing_codes'):
                validate_status_codes(self.pb.report_outgoing_codes, allow_families=False)
        new_preceding_modules = add_module(preceding_modules, self)
        for key, context in six.iteritems(self.ssl_sni_contexts):
            with validate('ssl_sni_contexts[{}]'.format(key)):
                context.validate(key=key, preceding_modules=new_preceding_modules)
        super(ExtendedHttpMacro, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules,
                                                                 chained_modules=chained_modules)

    @validate('port')
    def _validate_port(self):
        port = self.get('port')
        if port.value:
            if port.is_func():
                validate_port_call(port.value)
            else:
                validate_port(self.pb.port)

    @validate('ssl_sni_max_send_fragment')
    def _validate_ssl_sni_max_send_fragment(self):
        if self.pb.ssl_sni_max_send_fragment:
            validate_range(self.pb.ssl_sni_max_send_fragment, 512, 16384)

    def _get_port(self, preceding_modules=()):
        port = self.get('port')
        if port.value:
            return port
        else:
            ipdispatch_section = find_module(preceding_modules, IpdispatchSection)
            if not ipdispatch_section:
                raise ValidationError('must be preceded by ipdispatch module if port is not specified')
            ip, port = next(ipdispatch_section.list_addrs())
            return port

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        if not preceding_modules:
            raise ValidationError('can not go first in a config tree')

        if not (self.pb.disable_error_log and self.pb.disable_access_log):
            self._validate_port()
            if self.pb.error_log_level and self.pb.error_log_level not in Errorlog.LOG_LEVELS:
                raise ValidationError('must be one of the {}'.format(', '.join(Errorlog.LOG_LEVELS)), 'error_log_level')

        if self.pb.enable_ssl:
            self.require_value(self.pb.ssl_sni_contexts, 'ssl_sni_contexts')

        if self.pb.HasField('http2_alpn_freq'):
            with validate('http2_alpn_freq'):
                if not self.pb.enable_http2:
                    raise ValidationError('is not allowed without enable_http2')
                if not self.pb.enable_ssl:
                    raise ValidationError('is not allowed without enable_ssl')
                validate_range(self.pb.http2_alpn_freq.value, 0, 1)
        if self.pb.http2_allow_without_ssl.value:
            if not self.pb.enable_http2:
                raise ValidationError('is not allowed without enable_http2', 'http2_allow_without_ssl')
            if self.pb.enable_ssl:
                raise ValidationError('is not allowed with enable_ssl', 'http2_allow_without_ssl')
        if self.pb.http2_allow_sending_trailers.value:
            if not self.pb.enable_http2:
                raise ValidationError('is not allowed without enable_http2', 'http2_allow_sending_trailers')

        if self.pb.maxlen < 0:
            raise ValidationError('must be non-negative', 'maxlen')
        if self.pb.maxreq < 0:
            raise ValidationError('must be non-negative', 'maxreq')
        if self.pb.HasField('maxheaders'):
            if self.pb.maxheaders.value <= 0:
                raise ValidationError('must be positive', 'maxheaders')
        if self.pb.report_input_size_ranges:
            with validate('report_input_size_ranges'):
                validate_comma_separated_ints(self.pb.report_input_size_ranges)
        if self.pb.report_output_size_ranges:
            with validate('report_output_size_ranges'):
                validate_comma_separated_ints(self.pb.report_output_size_ranges)

        self._validate_keepalive_params(ctx)
        self._validate_ssl_sni_max_send_fragment()
        if self.pb.client_hints_ua_header:
            with validate('client_hints_ua_header'):
                validate_header_name(self.pb.client_hints_ua_header)
        if self.pb.client_hints_ua_proto_header:
            with validate('client_hints_ua_proto_header'):
                validate_header_name(self.pb.client_hints_ua_proto_header)

        super(ExtendedHttpMacro, self).validate(ctx=ctx,
                                                preceding_modules=preceding_modules,
                                                chained_modules=chained_modules)

    @staticmethod
    def _fill_get_log_path(call_pb, name, port, default_log_dir=None):
        """
        :type call_pb: awacs.proto.modules_pb2.Call
        """
        call_pb.type = proto.Call.GET_LOG_PATH
        params_pb = call_pb.get_log_path_params
        params_pb.name = name
        if port.is_func():
            params_pb.f_port.CopyFrom(port.value.pb)
        else:
            params_pb.port = port.value
        if default_log_dir:
            params_pb.default_log_dir = default_log_dir

    @staticmethod
    def _fill_get_ca_cert_path(call_pb, name, default_ca_cert_dir=None):
        """
        :type call_pb: awacs.proto.modules_pb2.Call
        """
        call_pb.type = proto.Call.GET_CA_CERT_PATH
        params_pb = call_pb.get_ca_cert_path_params
        params_pb.name = name
        if default_ca_cert_dir:
            params_pb.default_ca_cert_dir = default_ca_cert_dir

    def expand(self, ctx=DEFAULT_CTX, preceding_modules=()):
        instance_macro = find_module(preceding_modules, InstanceMacro)
        if instance_macro:
            log_dir = instance_macro.pb.log_dir or DEFAULT_LOG_DIR
            public_cert_dir = instance_macro.pb.public_cert_dir or DEFAULT_PUBLIC_CERT_DIR
            private_cert_dir = instance_macro.pb.private_cert_dir or DEFAULT_PRIVATE_CERT_DIR
            ca_cert_dir = instance_macro.pb.ca_cert_dir or DEFAULT_CA_CERT_DIR
        else:
            log_dir = None
            public_cert_dir = None
            private_cert_dir = None
            ca_cert_dir = None
        holder_pbs = []

        port = self._get_port(preceding_modules=preceding_modules)

        if not self.pb.disable_error_log:
            h = proto.Holder()
            h.errorlog.log_level = self.pb.error_log_level or 'ERROR'
            self._fill_get_log_path(h.errorlog.f_log, 'error_log', port, default_log_dir=log_dir)
            holder_pbs.append(h)

        if self.pb.enable_ssl:
            h = proto.Holder()
            h.ssl_sni.max_send_fragment = self.pb.ssl_sni_max_send_fragment
            h.ssl_sni.validate_cert_date = self.pb.ssl_sni_validate_cert_date
            h.ssl_sni.ja3_enabled = self.pb.ssl_sni_ja3_enabled
            for key, context in six.iteritems(self.ssl_sni_contexts):
                entry = h.ssl_sni.contexts.add()
                entry.key = 'default' if context.pb.servername_regexp == 'default' else key
                context_pb = entry.value

                cert = context.get('cert')
                if cert.value:
                    context_pb.c_cert.id = cert.value.id
                    # NOTE: !c-cert is used, let's override user-specified key by certificate id
                    # to meet the logic of awacs.lib.certs:pack_certs
                    key = cert.value.id
                else:
                    fill_get_public_cert_path(context_pb.f_cert, 'allCAs-{}.pem'.format(key),
                                              default_public_cert_dir=public_cert_dir)
                    fill_get_private_cert_path(context_pb.f_priv, '{}.pem'.format(key),
                                               default_private_cert_dir=private_cert_dir)

                secondary_cert = context.get('secondary_cert')
                if secondary_cert.value:
                    context_pb.secondary.c_cert.id = secondary_cert.value.id
                elif context.pb.secondary_cert_postfix:
                    fill_get_public_cert_path(context_pb.secondary.f_cert,
                                              'allCAs-{}_{}.pem'.format(key, context.pb.secondary_cert_postfix),
                                              default_public_cert_dir=public_cert_dir)
                    fill_get_private_cert_path(context_pb.secondary.f_priv,
                                               '{}_{}.pem'.format(key, context.pb.secondary_cert_postfix),
                                               default_private_cert_dir=private_cert_dir)

                if context.pb.HasField('disable_rc4_sha_cipher'):
                    context_pb.disable_rc4_sha_cipher.CopyFrom(context.pb.disable_rc4_sha_cipher)

                if context.pb.ciphers:
                    context_pb.ciphers = context.pb.ciphers
                if context.pb.ssl_protocols:
                    context_pb.ssl_protocols.extend(context.pb.ssl_protocols)

                self._fill_get_log_path(context_pb.f_log, 'ssl_sni', port, default_log_dir=log_dir)

                if context.pb.secrets_log:
                    self._fill_get_log_path(
                        context_pb.f_secrets_log,
                        context.pb.secrets_log,
                        port,
                        default_log_dir=ca_cert_dir
                    )

                if context.pb.ca:
                    self._fill_get_ca_cert_path(
                        context_pb.f_ca,
                        '{}'.format(context.pb.ca),
                        default_ca_cert_dir=ca_cert_dir
                    )

                if self.pb.disable_sslv3:
                    context_pb.disable_sslv3 = self.pb.disable_sslv3
                if self.pb.HasField('disable_tlsv1_3'):
                    context_pb.disable_tlsv1_3.CopyFrom(self.pb.disable_tlsv1_3)

                if context.pb.servername_regexp != 'default':
                    context_pb.servername.servername_regexp = context.pb.servername_regexp

                if not context.pb.disable_ocsp:
                    context_pb.ocsp = './ocsp/allCAs-{}.der'.format(key)
                    if context.pb.secondary_cert_postfix:
                        context_pb.secondary.ocsp = './ocsp/allCAs-{}_{}.der'.format(
                            key, context.pb.secondary_cert_postfix)

                for item in ('1st', '2nd', '3rd'):
                    ticket_key = context_pb.ticket_keys.add()
                    fill_get_private_cert_path(ticket_key.f_keyfile, '{}.{}.key'.format(item, key),
                                               default_private_cert_dir=private_cert_dir)

                if context.pb.HasField('client'):
                    context_pb.client.CopyFrom(context.pb.client)

            if self.pb.HasField('force_ssl'):
                h.ssl_sni.force_ssl.value = self.pb.force_ssl.value
            else:
                h.ssl_sni.force_ssl.value = SslSni.DEFAULT_FORCE_SSL

            if self.pb.enable_http2:
                h.ssl_sni.http2_alpn_file = SslSni.DEFAULT_HTTP2_ALPN_FILE
                if self.pb.HasField('http2_alpn_freq'):
                    h.ssl_sni.http2_alpn_freq = self.pb.http2_alpn_freq.value
                else:
                    h.ssl_sni.http2_alpn_freq = self.DEFAULT_HTTP2_ALPN_FREQ

            holder_pbs.append(h)

        if self.pb.enable_http2:
            h = proto.Holder()
            h.http2.goaway_debug_data_enabled = Http2.DEFAULT_GOAWAY_DEBUG_DATA_ENABLED
            h.http2.debug_log_enabled = Http2.DEFAULT_DEBUG_LOG_ENABLED
            self._fill_get_log_path(
                h.http2.f_debug_log_name,
                'http2_debug_log',
                port,
                default_log_dir=log_dir
            )
            if self.pb.HasField('http2_allow_without_ssl'):
                h.http2.allow_http2_without_ssl.value = self.pb.http2_allow_without_ssl.value
            if self.pb.HasField('http2_allow_sending_trailers'):
                h.http2.allow_sending_trailers.value = self.pb.http2_allow_sending_trailers.value

            holder_pbs.append(h)

        h = proto.Holder()
        h.http.maxlen = self.pb.maxlen or self.DEFAULT_MAXLEN
        h.http.maxreq = self.pb.maxreq or self.DEFAULT_MAXREQ
        h.http.events.update(self.pb.events)
        if self.pb.HasField('keepalive'):
            h.http.keepalive.value = self.pb.keepalive.value
        else:
            h.http.keepalive.value = self.DEFAULT_KEEPALIVE
        if self.pb.keepalive_timeout:
            h.http.keepalive_timeout = self.pb.keepalive_timeout
        if self.pb.HasField('keepalive_requests'):
            h.http.keepalive_requests.value = self.pb.keepalive_requests.value
        if self.pb.HasField('keepalive_drop_probability'):
            h.http.keepalive_drop_probability.value = self.pb.keepalive_drop_probability.value
        if self.pb.HasField('maxheaders'):
            h.http.maxheaders.value = self.pb.maxheaders.value
        h.http.allow_webdav = self.pb.allow_webdav
        h.http.allow_client_hints_restore = self.pb.allow_client_hints_restore
        h.http.client_hints_ua_header = self.pb.client_hints_ua_header
        h.http.client_hints_ua_proto_header = self.pb.client_hints_ua_proto_header
        h.http.disable_client_hints_restore_file = self.pb.disable_client_hints_restore_file
        if self.pb.HasField('k_disable_client_hints_restore_file'):
            h.http.k_disable_client_hints_restore_file.CopyFrom(self.pb.k_disable_client_hints_restore_file)

        holder_pbs.append(h)

        if not self.pb.disable_access_log:
            h = proto.Holder()
            if self.pb.additional_ip_header:
                h.accesslog.additional_ip_header = self.pb.additional_ip_header
            if self.pb.additional_port_header:
                h.accesslog.additional_port_header = self.pb.additional_port_header
            self._fill_get_log_path(h.accesslog.f_log, 'access_log', port, default_log_dir=log_dir)
            holder_pbs.append(h)

        if not self.pb.disable_report:
            h = proto.Holder()

            h.report.uuid = self.pb.report_uuid or self.DEFAULT_REPORT_UUID
            if h.report.uuid != self.DEFAULT_REPORT_UUID:
                refers_uuids = []
                if self.pb.report_refers:
                    refers_uuids.append(self.pb.report_refers)
                refers_uuids.append(self.DEFAULT_REPORT_REFERS)
                h.report.refers = ','.join(refers_uuids)

            h.report.ranges = Report.DEFAULT_RANGES_ALIAS

            if self.pb.report_input_size_ranges:
                h.report.input_size_ranges = self.pb.report_input_size_ranges
            if self.pb.report_output_size_ranges:
                h.report.output_size_ranges = self.pb.report_output_size_ranges

            if self.pb.report_enable_molly_uuid:
                matcher_map_entry_pb = h.report.matcher_map.add()
                matcher_map_entry_pb.key = u'molly'
                matcher_pb = matcher_map_entry_pb.value
                matcher_pb.match_fsm.cgi = u'.*everybodybecoolthisis=(crasher|molly).*'

            h.report.outgoing_codes.extend(self.pb.report_outgoing_codes)

            holder_pbs.append(h)

        if self.pb.yandex_cookie_policy == self.pb.YCP_STABLE:
            h = proto.Holder()
            h.cookie_policy.uuid = self.YANDEX_COOKIE_POLICY_UUID
            h.cookie_policy.default_yandex_policies = self.DEFAULT_YANDEX_POLICIES
            holder_pbs.append(h)

        return holder_pbs


class Webauth(ChainableModuleWrapperBase):
    __protobuf__ = proto.WebauthModule

    checker = None  # type: Holder | None
    on_forbidden = None  # type: Holder | None
    on_error = None  # type: Holder | None

    REQUIRED = ['auth_path', 'checker', 'on_forbidden']

    MODULE_NAME = 'webauth'

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        new_preceding_modules = add_module(preceding_modules, self)
        with validate('checker'):
            self.checker.validate(ctx=ctx, preceding_modules=new_preceding_modules)
        with validate('on_forbidden'):
            self.on_forbidden.validate(ctx=ctx, preceding_modules=new_preceding_modules)
        super(Webauth, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules, chained_modules=chained_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_raise_if_blacklisted(ctx)
        self.auto_validate_required()
        if self.pb.allow_options_passthrough:
            with validate(u'allow_options_passthrough'):
                ctx.ensure_component_version(
                    model_pb2.ComponentMeta.PGINX_BINARY,
                    min=min_component_versions.WEBAUTH_PASS_OPTIONS_REQUESTS_THROUGH)
        super(Webauth, self).validate(ctx=ctx, preceding_modules=preceding_modules, chained_modules=chained_modules)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        module_config = Config()
        self._add_nested_module_to_config(module_config, preceding_modules=preceding_modules)
        table = OrderedDict([
            ('auth_path', self.pb.auth_path),
            ('checker', self.checker.to_config(ctx=ctx, preceding_modules=add_module(preceding_modules, self))),
            ('on_forbidden', self.on_forbidden.to_config(ctx=ctx, preceding_modules=add_module(preceding_modules, self))),
            ('unauthorized_set_cookie', self.pb.unauthorized_set_cookie),
            ('unauthorized_redirect', self.pb.unauthorized_redirect),
        ])
        if self.pb.role:
            table['role'] = self.pb.role
        if self.pb.allow_options_passthrough:
            table['allow_options_passthrough'] = self.pb.allow_options_passthrough
        if self.on_error:
            table['on_error'] = self.on_error.to_config(ctx=ctx, preceding_modules=add_module(preceding_modules, self))
        return Config(table)

    def get_branches(self):
        if self.checker:
            yield self.checker
        if self.on_forbidden:
            yield self.on_forbidden
        if self.on_error:
            yield self.on_error

    def get_named_branches(self):
        rv = {}
        if self.checker:
            rv['checker'] = self.checker
        if self.on_forbidden:
            rv['on_forbidden'] = self.on_forbidden
        if self.on_error:
            rv['on_error'] = self.on_error
        return rv


class Antirobot(ChainableModuleWrapperBase):
    __protobuf__ = proto.AntirobotModule

    checker = None  # type: Holder | None

    REQUIRED = ['checker']

    DEFAULT_CUT_REQUEST = True
    DEFAULT_NO_CUT_REQUEST_FILE = './controls/no_cut_request_file'
    DEFAULT_FILE_SWITCH = './controls/disable_antirobot_module'
    DEFAULT_CUT_REQUEST_BYTES = 512

    ALLOWED_KNOBS = {
        'no_cut_request_file': model_pb2.KnobSpec.BOOLEAN,
        'file_switch': model_pb2.KnobSpec.BOOLEAN,
    }
    DEFAULT_KNOB_IDS = {
        'no_cut_request_file': 'no_cut_request_file',
        'file_switch': 'balancer_antirobot_module_switch',
    }

    # info attrs:
    DEFAULTS = {
        'cut_request': DEFAULT_CUT_REQUEST,
        'no_cut_request_file': DEFAULT_NO_CUT_REQUEST_FILE,
        'file_switch': DEFAULT_FILE_SWITCH,
        'cut_request_bytes': DEFAULT_CUT_REQUEST_BYTES,
    }

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        new_preceding_modules = add_module(preceding_modules, self)
        with validate('checker'):
            self.checker.validate(ctx=ctx, preceding_modules=new_preceding_modules)
        super(Antirobot, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules,
                                                         chained_modules=chained_modules)

    def _validate_no_cut_request_file(self, ctx):
        self.validate_knob('no_cut_request_file', ctx=ctx)

    def _validate_file_switch(self, ctx):
        self.validate_knob('file_switch', ctx=ctx)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()
        self._validate_no_cut_request_file(ctx)
        self._validate_file_switch(ctx)
        super(Antirobot, self).validate(ctx=ctx, preceding_modules=preceding_modules, chained_modules=chained_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        module_config = Config()
        self._add_nested_module_to_config(module_config, preceding_modules=preceding_modules)
        table = OrderedDict([
            ('cut_request', self.pb.cut_request or self.DEFAULT_CUT_REQUEST),
            ('no_cut_request_file', self.get('no_cut_request_file', self.DEFAULT_NO_CUT_REQUEST_FILE).to_config(ctx)),
            ('file_switch', self.get('file_switch', self.DEFAULT_FILE_SWITCH).to_config(ctx)),
            ('cut_request_bytes', self.pb.cut_request_bytes or self.DEFAULT_CUT_REQUEST_BYTES),
            ('checker', self.checker.to_config(ctx=ctx, preceding_modules=add_module(preceding_modules, self))),
            ('module', module_config),
        ])
        return Config(table, outlets=(module_config,), shareable=True)

    def get_branches(self):
        if self.checker:
            yield self.checker

    def get_named_branches(self):
        rv = {}
        if self.checker:
            rv['checker'] = self.checker
        return rv


class AntirobotMacro(ChainableModuleWrapperBase, MacroBase):
    __protobuf__ = proto.AntirobotMacro

    instances = []  # type:  list[GeneratedProxyBackendsInstance]
    nanny_snapshots = []  # type:  list[GeneratedProxyBackendsNannySnapshot]
    gencfg_groups = []  # type:  list[GeneratedProxyBackendsGencfgGroup]
    endpoint_sets = []  # type:  list[GeneratedProxyBackendsEndpointSet]
    include_backends = None  # type:  IncludeBackends | None

    VERSION_0_0_1 = semantic_version.Version(u'0.0.1')
    VERSION_0_0_2 = semantic_version.Version(u'0.0.2')  # use SD backends
    VERSION_0_0_3 = semantic_version.Version(u'0.0.3')  # automatically set X-Forwarded-For-Y header
    VERSION_0_0_4 = semantic_version.Version(u'0.0.4')  # autoset XFFY, add flag to disable it
    VERSION_0_0_5 = semantic_version.Version(u'0.0.5')  # autoset icookie, add flag to disable it
    VERSION_0_0_6 = semantic_version.Version(u'0.0.6')  # automatically set X-Yandex-Ja3 and ..-Ja4 headers
    VERSION_0_0_7 = semantic_version.Version(u'0.0.7')  # subnet_v4_mask=32, subnet_v6_mask=64
    VERSION_0_0_8 = semantic_version.Version(u'0.0.8')  # take_ip_from='X-Forwarded-For-Y' for hasher, add X-Forwarded-For-Y before hasher module
    VERSION_0_0_9 = semantic_version.Version(u'0.0.9')  # choose a random location for each attempt in myt, iva

    INITIAL_VERSION = VERSION_0_0_1
    VALID_VERSIONS = frozenset((
        VERSION_0_0_1, VERSION_0_0_2, VERSION_0_0_3, VERSION_0_0_4, VERSION_0_0_5, VERSION_0_0_6, VERSION_0_0_7,
        VERSION_0_0_8, VERSION_0_0_9))

    ANTIROBOT_LOCATIONS = ('man', 'sas', 'vla')
    ANTIROBOT_DUMMY_LOCATIONS = ('iva', 'myt') # see https://st.yandex-team.ru/AWACS-1296
    ANTIROBOT_FULL_BACKEND_IDS = {
        location: ('common-antirobot', 'antirobot_{}'.format(location))
        for location in ANTIROBOT_LOCATIONS
    }
    ANTIROBOT_FULL_SD_BACKEND_IDS = {
        location: ('common-antirobot', 'antirobot_{}_yp'.format(location))
        for location in ANTIROBOT_LOCATIONS
    }

    DEFAULT_REPORT_UUID = 'antirobot'
    HASHER_SUBNET_V4_MASK = 32
    HASHER_SUBNET_V6_MASK = 64
    DEFAULT_ANTIROBOT_CONNECT_TIMEOUT = '30ms'
    DEFAULT_ANTIROBOT_BACKEND_TIMEOUT = '100ms'
    DEFAULT_ATTEMPTS_NUMBER = 2

    ALLOWED_KNOBS = {
        'file_switch': model_pb2.KnobSpec.BOOLEAN,
    }
    DEFAULT_KNOB_IDS = {
        'file_switch': 'balancer_antirobot_module_switch',
    }

    DEFAULTS = {
        'report_uuid': DEFAULT_REPORT_UUID,
        'file_switch': Antirobot.DEFAULT_FILE_SWITCH,
        'attempts': DEFAULT_ATTEMPTS_NUMBER,
    }
    MAX_CUT_REQUEST_BYTES = 4096

    def get_version(self):
        if self.pb.version:
            return semantic_version.Version(self.pb.version)
        return self.INITIAL_VERSION

    def is_chainable(self):
        return True

    def would_include_backends(self):
        return (self.include_backends or  # if macro explicitly includes backends
                (not self.instances and  # or implicitly (via defaults)
                 not self.nanny_snapshots and
                 not self.gencfg_groups and
                 not self.endpoint_sets and
                 not self.include_backends))

    def get_would_be_included_full_backend_ids(self, current_namespace_id):
        if (not self.instances and
                not self.nanny_snapshots and
                not self.gencfg_groups and
                not self.endpoint_sets and
                not self.include_backends):
            if self.get_version() == self.VERSION_0_0_1:
                return six.itervalues(self.ANTIROBOT_FULL_BACKEND_IDS)
            else:
                return six.itervalues(self.ANTIROBOT_FULL_SD_BACKEND_IDS)

        elif self.include_backends:
            return self.include_backends.get_included_full_backend_ids(current_namespace_id)
        else:
            return set()

    def get_would_be_included_full_knob_ids(self, namespace_id, ctx):
        rv = super(AntirobotMacro, self).get_would_be_included_full_knob_ids(namespace_id, ctx)
        if ctx.are_knobs_enabled():
            for field_name, knob_id in six.iteritems(Antirobot.DEFAULT_KNOB_IDS):
                if knob_id != 'file_switch':  # explicitly set by expand
                    rv.add((namespace_id, knob_id))
            for field_name, knob_id in six.iteritems(Balancer2.DEFAULT_KNOB_IDS):
                rv.add((namespace_id, knob_id))
        return rv

    def _validate_file_switch(self, ctx):
        self.validate_knob('file_switch', ctx=ctx)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()
        self._validate_file_switch(ctx)
        if self.pb.version:
            validate_version(self.pb.version, self.VALID_VERSIONS, field_name='version')
        if self.get_version() != self.VERSION_0_0_1 and ctx.config_type == ctx.CONFIG_TYPE_FULL:
            require_sd(preceding_modules, field_name='sd')
        with validate('cut_request_bytes'):
            validate_range(self.pb.cut_request_bytes, min_=0, max_=self.MAX_CUT_REQUEST_BYTES)

        if self.pb.cut_request_bytes > AntirobotMacro.MAX_CUT_REQUEST_BYTES:
            with validate('cut_request_bytes'):
                raise ValidationError(
                    'Max allowed value is {}'.format(AntirobotMacro.MAX_CUT_REQUEST_BYTES))

        with validate('trust_x_yandex_ja_x'):
            if self.get_version() < self.VERSION_0_0_6 and self.pb.trust_x_yandex_ja_x:
                raise ValidationError('not allowed in versions prior to 0.0.6')

        super(AntirobotMacro, self).validate(ctx=ctx,
                                             preceding_modules=preceding_modules,
                                             chained_modules=chained_modules)

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        if len([v for v in
                (self.instances, self.nanny_snapshots, self.gencfg_groups, self.endpoint_sets, self.include_backends) if
                v]) > 1:
            raise ValidationError('at most one of the "instances", "nanny_snapshots", '
                                  '"gencfg_groups", "endpoint_sets" or "include_backends" must be specified')

        if self.include_backends:
            with validate('include_backends'):
                self.include_backends.validate(ctx=ctx, preceding_modules=preceding_modules)

        new_preceding_modules = add_module(preceding_modules, self)

        instances_by_spec = set()
        for i, instance in enumerate(self.instances):
            with validate('instances[{}]'.format(i)):
                instance.validate(ctx=ctx, preceding_modules=new_preceding_modules)
                spec = (instance.pb.host, instance.pb.port)
                if spec in instances_by_spec:
                    raise ValidationError('duplicate host and port: {}'.format(spec))
                instances_by_spec.add(spec)

        snapshots_by_spec = set()
        for i, snapshot in enumerate(self.nanny_snapshots):
            with validate('nanny_snapshots[{}]'.format(i)):
                snapshot.validate(ctx=ctx, preceding_modules=new_preceding_modules)
                spec = (snapshot.pb.service_id, snapshot.pb.snapshot_id)
                if spec in snapshots_by_spec:
                    raise ValidationError('duplicate service_id and snapshot_id: {}'.format(spec))
                snapshots_by_spec.add(spec)

        gencfg_groups_by_spec = set()
        for i, group in enumerate(self.gencfg_groups):
            with validate('gencfg_groups[{}]'.format(i)):
                group.validate(ctx=ctx, preceding_modules=new_preceding_modules)
                spec = (group.pb.name, group.pb.version)
                if spec in gencfg_groups_by_spec:
                    raise ValidationError('duplicate name and version: {}'.format(spec))
                gencfg_groups_by_spec.add(spec)

        endpoint_sets_by_spec = set()
        for i, yp_service in enumerate(self.endpoint_sets):
            with validate('endpoint_sets[{}]'.format(i)):
                yp_service.validate(ctx=ctx, preceding_modules=preceding_modules)
                spec = (yp_service.pb.cluster_id, yp_service.pb.service_id)
                if spec in endpoint_sets_by_spec:
                    raise ValidationError('duplicate cluster_id and service_id: {}:{}'.format(*spec))
                endpoint_sets_by_spec.add(spec)

        super(AntirobotMacro, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules,
                                                              chained_modules=chained_modules)

    def expand(self, ctx=DEFAULT_CTX, preceding_modules=()):
        holder_pbs = []
        if self.get_version() < self.VERSION_0_0_8:
            self.add_hasher(holder_pbs)
            self.add_h100(holder_pbs)
            self.add_cutter(holder_pbs)
            self.add_antirobot_headers(holder_pbs)
            self.add_xffy(holder_pbs)
        else:
            self.add_xffy(holder_pbs)
            self.add_hasher(holder_pbs)
            self.add_h100(holder_pbs)
            self.add_cutter(holder_pbs)
            self.add_antirobot_headers(holder_pbs)

        self.add_icookie(holder_pbs)
        self.add_ja_x(holder_pbs)

        antirobot_h = proto.Holder()
        file_switch = self.get('file_switch')
        if file_switch.type == Value.KNOB:
            antirobot_h.antirobot.k_file_switch.CopyFrom(self.pb.k_file_switch)
        elif file_switch.type == Value.CALL:
            antirobot_h.antirobot.f_file_switch.CopyFrom(self.pb.f_file_switch)
        else:
            if ctx.are_knobs_enabled():
                antirobot_h.antirobot.k_file_switch.id = self.DEFAULT_KNOB_IDS['file_switch']
                antirobot_h.antirobot.k_file_switch.optional = True
            else:
                antirobot_h.antirobot.file_switch = self.pb.file_switch or Antirobot.DEFAULT_FILE_SWITCH
        antirobot_h.antirobot.cut_request_bytes = self.pb.cut_request_bytes
        holder_pbs.append(antirobot_h)

        report_pb = antirobot_h.antirobot.checker.report
        report_pb.uuid = self.pb.report_uuid or self.DEFAULT_REPORT_UUID
        report_pb.ranges = Report.DEFAULT_RANGES_ALIAS

        if (not self.instances and
                not self.nanny_snapshots and
                not self.gencfg_groups and
                not self.endpoint_sets and
                not self.include_backends):
            balancer2_pb = report_pb.nested.balancer2
            balancer2_pb.attempts = 1
            balancer2_pb.rr.SetInParent()
            call_pb_bnp_pb = balancer2_pb.balancing_policy.by_name_policy.f_name
            call_pb_bnp_pb.type = proto.Call.GET_GEO
            params_pb_bnp_pb = call_pb_bnp_pb.get_geo_params
            params_pb_bnp_pb.name = 'antirobot_'
            params_pb_bnp_pb.default_geo = 'random'
            balancer2_pb.balancing_policy.by_name_policy.balancing_policy.simple_policy.SetInParent()

            for location in self.ANTIROBOT_LOCATIONS:
                antirobot_backend_pb = balancer2_pb.backends.add()
                antirobot_backend_pb.name = 'antirobot_{}'.format(location)
                antirobot_backend_pb.weight = 1
                antirobot_balancer2_pb = antirobot_backend_pb.nested.balancer2
                antirobot_balancer2_pb.attempts = self.pb.attempts or self.DEFAULT_ATTEMPTS_NUMBER
                antirobot_balancer2_pb.hashing.SetInParent()
                if self.get_version() == self.VERSION_0_0_1:
                    full_antirobot_backend_id = self.ANTIROBOT_FULL_BACKEND_IDS[location]
                else:
                    full_antirobot_backend_id = self.ANTIROBOT_FULL_SD_BACKEND_IDS[location]
                gpb_pb = antirobot_balancer2_pb.generated_proxy_backends
                gpb_pb.include_backends.type = proto.IncludeBackends.BY_ID
                gpb_pb.include_backends.ids.append('/'.join(full_antirobot_backend_id))
                proxy_options_pb = gpb_pb.proxy_options
                proxy_options_pb.connect_timeout = self.DEFAULT_ANTIROBOT_CONNECT_TIMEOUT
                proxy_options_pb.backend_timeout = self.DEFAULT_ANTIROBOT_BACKEND_TIMEOUT

            if self.get_version() >= self.VERSION_0_0_9:
                antirobot_backends_pb = balancer2_pb.backends[:]
                for location in self.ANTIROBOT_DUMMY_LOCATIONS:
                    antirobot_backend_pb = balancer2_pb.backends.add()
                    antirobot_backend_pb.name = 'antirobot_{}'.format(location)
                    antirobot_backend_pb.weight = 1
                    inner_balancer2_pb = antirobot_backend_pb.nested.balancer2
                    inner_balancer2_pb.attempts = self.pb.attempts or self.DEFAULT_ATTEMPTS_NUMBER
                    inner_balancer2_pb.rr.SetInParent()

                    for outer_backend_pb in antirobot_backends_pb:
                        inner_backend_pb = inner_balancer2_pb.backends.add()
                        inner_backend_pb.MergeFrom(outer_backend_pb)
                        inner_backend_pb.nested.balancer2.attempts = 1

        else:
            balancer2_pb = report_pb.nested.stats_eater.nested.balancer2
            balancer2_pb.attempts = self.pb.attempts or self.DEFAULT_ATTEMPTS_NUMBER
            balancer2_pb.hashing.SetInParent()
            balancer2_pb.generated_proxy_backends.instances.extend(self.pb.instances)
            balancer2_pb.generated_proxy_backends.nanny_snapshots.extend(self.pb.nanny_snapshots)
            balancer2_pb.generated_proxy_backends.gencfg_groups.extend(self.pb.gencfg_groups)
            balancer2_pb.generated_proxy_backends.endpoint_sets.extend(self.pb.endpoint_sets)
            if self.include_backends:
                balancer2_pb.generated_proxy_backends.include_backends.CopyFrom(self.pb.include_backends)

            proxy_options_pb = balancer2_pb.generated_proxy_backends.proxy_options
            proxy_options_pb.connect_timeout = self.DEFAULT_ANTIROBOT_CONNECT_TIMEOUT
            proxy_options_pb.backend_timeout = self.DEFAULT_ANTIROBOT_BACKEND_TIMEOUT

        return holder_pbs

    def add_hasher(self, holder_pbs):
        hasher_h = proto.Holder()
        hasher_h.hasher.mode = 'subnet'
        if self.get_version() >= self.VERSION_0_0_7:
            hasher_h.hasher.subnet_v4_mask = self.HASHER_SUBNET_V4_MASK
            hasher_h.hasher.subnet_v6_mask = self.HASHER_SUBNET_V6_MASK

        if self.get_version() >= self.VERSION_0_0_8:
            hasher_h.hasher.take_ip_from = self.pb.hasher_take_ip_from or 'X-Forwarded-For-Y'
        else:
            hasher_h.hasher.take_ip_from = self.pb.hasher_take_ip_from or 'X-Real-IP'
        holder_pbs.append(hasher_h)

    def add_h100(self, holder_pbs):
        h100_h = proto.Holder()
        h100_h.h100.SetInParent()
        holder_pbs.append(h100_h)

    def add_cutter(self, holder_pbs):
        cutter_h = proto.Holder()
        cutter_h.cutter.SetInParent()
        holder_pbs.append(cutter_h)

    def add_antirobot_headers(self, holder_pbs):
        if self.pb.service or self.pb.req_group:
            headers_h_pb = proto.Holder()
            if self.pb.service:
                headers_h_pb.headers.create.add(key='X-Antirobot-Service-Y', value=self.pb.service)
            if self.pb.req_group:
                headers_h_pb.headers.create.add(key='X-Antirobot-Req-Group', value=self.pb.req_group)
            holder_pbs.append(headers_h_pb)

    def add_xffy(self, holder_pbs):
        version = self.get_version()
        if version < AntirobotMacro.VERSION_0_0_3:
            return
        headers_h_pb = proto.Holder()
        if version == AntirobotMacro.VERSION_0_0_3:
            headers_h_pb.headers.create_func_weak['X-Forwarded-For-Y'] = 'realip'
        if version >= AntirobotMacro.VERSION_0_0_4:
            if self.pb.trust_x_forwarded_for_y:
                headers_h_pb.headers.create_func_weak['X-Forwarded-For-Y'] = 'realip'
            else:
                headers_h_pb.headers.create_func['X-Forwarded-For-Y'] = 'realip'
        holder_pbs.append(headers_h_pb)

    def add_icookie(self, holder_pbs):
        version = self.get_version()
        if version >= self.VERSION_0_0_5:
            holder_pb = proto.Holder()
            holder_pb.icookie.use_default_keys = True
            holder_pb.icookie.trust_parent.value = self.pb.trust_icookie
            holder_pb.icookie.domains.extend(ICOOKIE_DOMAINS)
            holder_pbs.append(holder_pb)

    def add_ja_x(self, holder_pbs):
        if self.get_version() >= self.VERSION_0_0_6:
            headers_h_pb = proto.Holder()
            if self.pb.trust_x_yandex_ja_x:
                headers_h_pb.headers.create_func_weak['X-Yandex-Ja3'] = 'ja3'
                headers_h_pb.headers.create_func_weak['X-Yandex-Ja4'] = 'ja4'
            else:
                headers_h_pb.headers.create_func['X-Yandex-Ja3'] = 'ja3'
                headers_h_pb.headers.create_func['X-Yandex-Ja4'] = 'ja4'
            holder_pbs.append(headers_h_pb)

    def includes_backends(self):
        return bool(self.include_backends)


class HttpToHttpsMacro(ModuleWrapperBase, MacroBase):
    __protobuf__ = proto.HttpToHttpsMacro

    def expand(self, ctx=DEFAULT_CTX, preceding_modules=()):
        holder_pbs = []

        h = proto.Holder()
        headers_pb = h.headers
        headers_pb.create.add(key='Location', value='1')
        holder_pbs.append(h)

        h = proto.Holder()
        rewrite_pb = h.rewrite
        rewrite_pb.actions.add(
            header_name='Location',
            regexp='.*',
            rewrite='https://%{host}%{url}'
        )
        holder_pbs.append(h)

        h = proto.Holder()
        regexp_pb = h.regexp
        sections_entry = regexp_pb.sections.add()
        sections_entry.key = 'unsafe_methods'
        section_pb = sections_entry.value
        section_pb.matcher.match_fsm.match = '(DELETE|PATCH|POST|PUT).*'
        section_pb.nested.errordocument.status = httplib.TEMPORARY_REDIRECT
        section_pb.nested.errordocument.remain_headers = 'Location'

        sections_entry = regexp_pb.sections.add()
        section_pb = sections_entry.value
        section_pb.matcher.SetInParent()
        section_pb.nested.errordocument.status = httplib.MOVED_PERMANENTLY if self.pb.permanent else httplib.FOUND
        section_pb.nested.errordocument.remain_headers = 'Location'
        holder_pbs.append(h)

        return holder_pbs


class SlbPingMacroActiveCheckReply(ConfigWrapperBase):
    __protobuf__ = proto.SlbPingMacro.ActiveCheckReply

    DEFAULT_DEFAULT_WEIGHT = 1
    DEFAULT_WEIGHT_FILE = u'./controls/l3_dynamic_weight'

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()

        if self.pb.HasField('default_weight'):
            with validate(u'default_weight'):
                validate_range(self.pb.default_weight.value, 0, 1000, exclusive_min=False)

    def fill_holder(self, holder_pb):
        """
        :type holder_pb: modules_pb2.Holder
        """
        active_check_reply_pb = holder_pb.active_check_reply
        active_check_reply_pb.use_header.value = True
        active_check_reply_pb.use_body.value = True
        active_check_reply_pb.weight_file = self.DEFAULT_WEIGHT_FILE
        if self.pb.HasField('zero_weight_at_shutdown'):
            active_check_reply_pb.zero_weight_at_shutdown.value = self.pb.zero_weight_at_shutdown.value
        if self.pb.HasField('default_weight'):
            default_weight = self.pb.default_weight.value
        else:
            default_weight = self.DEFAULT_DEFAULT_WEIGHT
        active_check_reply_pb.default_weight = default_weight


class SlbPingMacro(ModuleWrapperBase, MacroBase):
    __protobuf__ = proto.SlbPingMacro

    active_check_reply = None  # type: SlbPingMacroActiveCheckReply | None
    generated_proxy_backends = None  # type: GeneratedProxyBackends | None

    REQUIRED_ONEOFS = [('errordoc', 'active_check_reply', 'generated_proxy_backends', 'use_shared_backends')]

    DEFAULT_ATTEMPTS_NUMBER = 5
    DEFAULT_CONNECT_TIMEOUT = ProxyOptionsMixin.DEFAULT_CONNECT_TIMEOUT
    DEFAULT_BACKEND_TIMEOUT = ProxyOptionsMixin.DEFAULT_BACKEND_TIMEOUT
    DEFAULT_SLB_CHECK_WEIGHTS_FILE = './controls/slb_check.weights'
    DEFAULT_SLB_CHECK_WEIGHTS_FILE_KNOB_ID = 'service_balancer_off'

    DEFAULTS = {
        'attempts': DEFAULT_ATTEMPTS_NUMBER,
        'switch_off_status_code': httplib.SERVICE_UNAVAILABLE,
    }

    def get_would_be_included_full_knob_ids(self, namespace_id, ctx):
        rv = super(SlbPingMacro, self).get_would_be_included_full_knob_ids(namespace_id, ctx)
        if ctx.are_knobs_enabled():
            rv.add((namespace_id, self.DEFAULT_SLB_CHECK_WEIGHTS_FILE_KNOB_ID))
            for field_name, knob_id in six.iteritems(Balancer2.DEFAULT_KNOB_IDS):
                rv.add((namespace_id, knob_id))
        return rv

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=()):
        new_preceding_modules = add_module(preceding_modules, self)
        if self.active_check_reply:
            with validate('active_check_reply'):
                self.active_check_reply.validate(ctx=ctx, preceding_modules=new_preceding_modules)
        if self.generated_proxy_backends:
            with validate('generated_proxy_backends'):
                self.generated_proxy_backends.validate(ctx=ctx, preceding_modules=new_preceding_modules)
        super(SlbPingMacro, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()

        if self.pb.switch_off_status_code:
            with validate('switch_off_status_code'):
                # str() is used because of checker operates with string
                validate_status_code_or_family(str(self.pb.switch_off_status_code))
        super(SlbPingMacro, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def expand(self, ctx=DEFAULT_CTX, preceding_modules=()):
        holder_pbs = []
        h = proto.Holder()
        stats_eater_pb = h.stats_eater
        stats_eater_pb.SetInParent()
        holder_pbs.append(h)

        h = proto.Holder()
        top_balancer2_pb = h.balancer2
        if ctx.are_knobs_enabled():
            top_balancer2_pb.rr.k_weights_file.id = self.DEFAULT_SLB_CHECK_WEIGHTS_FILE_KNOB_ID
            top_balancer2_pb.rr.k_weights_file.optional = True
        else:
            top_balancer2_pb.rr.weights_file = self.DEFAULT_SLB_CHECK_WEIGHTS_FILE
        top_balancer2_pb.attempts = 1

        backends_entry = top_balancer2_pb.backends.add()
        backends_entry.name = 'to_upstream'
        backends_entry.weight = 1

        if self.pb.errordoc:
            backends_entry.nested.errordocument.status = httplib.OK
        elif self.active_check_reply:
            self.active_check_reply.fill_holder(backends_entry.nested)
        elif self.pb.use_shared_backends:
            backends_entry.nested.shared.uuid = 'backends'
        else:
            bottom_balancer2_pb = backends_entry.nested.balancer2
            bottom_balancer2_pb.weighted2.SetInParent()
            bottom_balancer2_pb.attempts = self.pb.attempts or self.DEFAULT_ATTEMPTS_NUMBER
            bottom_balancer2_pb.generated_proxy_backends.MergeFrom(self.generated_proxy_backends.pb)

            proxy_options_pb = bottom_balancer2_pb.generated_proxy_backends.proxy_options
            proxy_options_pb.connect_timeout = proxy_options_pb.connect_timeout or self.DEFAULT_CONNECT_TIMEOUT
            proxy_options_pb.backend_timeout = proxy_options_pb.backend_timeout or self.DEFAULT_BACKEND_TIMEOUT

        backends_entry = top_balancer2_pb.backends.add()
        backends_entry.name = 'switch_off'
        backends_entry.weight = -1
        backends_entry.nested.errordocument.status = self.pb.switch_off_status_code or httplib.SERVICE_UNAVAILABLE
        holder_pbs.append(h)

        return holder_pbs

    def get_branches(self):
        if self.generated_proxy_backends:
            yield self.generated_proxy_backends

    def get_named_branches(self):
        rv = {}
        if self.generated_proxy_backends:
            rv['generated_proxy_backends'] = self.generated_proxy_backends
        return rv


class ExpGetterMacroTestingModeParams(ConfigWrapperBase):
    __protobuf__ = proto.ExpGetterMacro.TestingModeParams

    matcher = None  # type: Matcher | None

    DEFAULT_TESTING_HEADER_NAME = 'X-L7-EXP-Testing'
    DEFAULT_MATCHER = proto.Matcher(
        match_fsm=proto.MatchFsm(
            cgi='(exp-testing=da|exp_confs=testing)',
            surround=True
        )
    )
    DEFAULTS = {
        'testing_header_name': DEFAULT_TESTING_HEADER_NAME,
    }

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=()):
        if self.matcher:
            self.matcher.validate(ctx=ctx, preceding_modules=add_module(preceding_modules, self))
        super(ExpGetterMacroTestingModeParams, self).validate_composite_fields(ctx=ctx,
                                                                               preceding_modules=preceding_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        if self.pb.testing_header_name:
            with validate('testing_header_name'):
                validate_header_name(self.pb.testing_header_name)
        super(ExpGetterMacroTestingModeParams, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        return Config()


class ExpGetterMacro(ChainableModuleWrapperBase, MacroBase):
    __protobuf__ = proto.ExpGetterMacro

    testing_mode = None  # type: ExpGetterMacroTestingModeParams | None

    DEFAULT_VERSION = 2
    MIN_VERSION = 1
    MAX_VERSION = 3

    DEFAULT_SERVICE_NAME_HEADER = 'Y-Service'
    DEFAULT_HEADER_NAME = 'X-L7-EXP'
    DEFAULT_EXP_HEADERS = 'X-Yandex-ExpConfigVersion(-Pre)?|X-Yandex-ExpBoxes(-Pre)?|X-Yandex-ExpFlags(-Pre)?|{}'
    DEFAULT_CONNECT_TIMEOUT = '5ms'
    DEFAULT_BACKEND_TIMEOUT = '10ms'
    GEO_LOCATIONS = ('man', 'sas', 'vla')
    ALL_GEO_LOCATIONS = ('man', 'sas', 'vla', 'myt', 'iva')

    DEFAULTS = {
        'header_name': DEFAULT_HEADER_NAME,
        'service_name_header': DEFAULT_SERVICE_NAME_HEADER,
    }

    UAAS_FULL_BACKEND_IDS = {
        location: ('uaas.search.yandex.net', 'production_balancer_uaas_{}'.format(location))
        for location in GEO_LOCATIONS
    }
    UAAS_FULL_BACKEND_IDS_VERSION_2 = {
        location: ('uaas.search.yandex.net', 'usersplit_{}'.format(location))
        for location in GEO_LOCATIONS
    }
    UAAS_FULL_BACKEND_IDS_VERSION_3 = {
        location: ('common-uaas', 'yp_production_uaas_{}'.format(location))
        for location in ALL_GEO_LOCATIONS
    }

    @classmethod
    def _get_uaas_full_backend_ids_by_location(cls, version=DEFAULT_VERSION):
        """
        :type version: int
        :rtype: dict[str, (str, str)]
        """
        if version == 1:
            return cls.UAAS_FULL_BACKEND_IDS
        elif version == 2:
            return cls.UAAS_FULL_BACKEND_IDS_VERSION_2
        elif version == 3:
            return cls.UAAS_FULL_BACKEND_IDS_VERSION_3
        else:
            raise AssertionError('unknown expgetter_macro version')

    @classmethod
    def get_uaas_full_backend_ids(self, version=DEFAULT_VERSION):
        full_backend_ids = self._get_uaas_full_backend_ids_by_location(version=version)
        return set(six.itervalues(full_backend_ids))

    def _get_version(self):
        if self.pb._version == 0:
            return self.DEFAULT_VERSION
        else:
            return self.pb._version

    def is_chainable(self):
        return True

    @classmethod
    def would_include_backends(cls):
        return True

    def get_would_be_included_full_backend_ids(self, current_namespace_id):
        full_backend_ids = self._get_uaas_full_backend_ids_by_location(version=self._get_version())
        return set(six.itervalues(full_backend_ids))

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        if self.pb._version:
            with validate('_version'):
                validate_range(self.pb._version, self.MIN_VERSION, self.MAX_VERSION)
        if self._get_version() >= 3 and ctx.config_type == ctx.CONFIG_TYPE_FULL:
            require_sd(preceding_modules, field_name='_version')
        if self.pb.HasField('headers_size_limit'):
            with validate('headers_size_limit'):
                validate_range(self.pb.headers_size_limit.value, 1024, 512 * 1024)
        super(ExpGetterMacro, self).validate(ctx=ctx,
                                             preceding_modules=preceding_modules,
                                             chained_modules=chained_modules)

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        if self.testing_mode:
            self.testing_mode.validate(ctx=ctx, preceding_modules=add_module(preceding_modules, self))
        super(ExpGetterMacro, self).validate_composite_fields(ctx=ctx,
                                                              preceding_modules=preceding_modules,
                                                              chained_modules=chained_modules)

    def expand(self, ctx=DEFAULT_CTX, preceding_modules=()):
        if self.testing_mode:
            matcher_pb = self.testing_mode.matcher and self.testing_mode.matcher.pb or self.testing_mode.DEFAULT_MATCHER

            h = proto.Holder()

            entry = h.regexp.sections.add()
            entry.key = 'exp_testing'
            testing_section_pb = entry.value
            testing_section_pb.matcher.CopyFrom(matcher_pb)
            testing_exp_getter_macro_pb = testing_section_pb.nested.exp_getter_macro
            testing_exp_getter_macro_pb.CopyFrom(self.pb)
            testing_exp_getter_macro_pb.ClearField('testing_mode')
            testing_exp_getter_macro_pb.header_name = (self.testing_mode.pb.testing_header_name or
                                                       self.testing_mode.DEFAULT_TESTING_HEADER_NAME)
            if self.pb.HasField('headers_size_limit'):
                testing_exp_getter_macro_pb.headers_size_limit.CopyFrom(self.pb.headers_size_limit)

            entry = h.regexp.sections.add()
            entry.key = 'default'
            default_section_pb = entry.value
            default_section_pb.matcher.SetInParent()
            default_exp_getter_macro_pb = default_section_pb.nested.exp_getter_macro
            default_exp_getter_macro_pb.CopyFrom(self.pb)
            default_exp_getter_macro_pb.ClearField('testing_mode')
            if self.pb.HasField('headers_size_limit'):
                default_exp_getter_macro_pb.headers_size_limit.CopyFrom(self.pb.headers_size_limit)

            return [h]
        else:
            holder_pbs = self._expand(ctx=ctx)
            return holder_pbs

    def _expand(self, ctx=DEFAULT_CTX):
        holder_pbs = []

        h = proto.Holder()
        headers_pb = h.headers
        header_name = self.pb.header_name or self.DEFAULT_HEADER_NAME
        headers_pb.create.add(key=header_name, value='true')
        holder_pbs.append(h)

        h = proto.Holder()
        exp_getter_pb = h.exp_getter
        exp_getter_pb.file_switch = ExpGetter.DEFAULT_FILE_SWITCH
        exp_getter_pb.trusted = self.pb.trusted
        if self.pb.exp_headers:
            exp_getter_pb.exp_headers = self.DEFAULT_EXP_HEADERS.format(self.pb.exp_headers)
        if self.pb.service_name:
            exp_getter_pb.service_name = self.pb.service_name
            exp_getter_pb.service_name_header = self.pb.service_name_header or self.DEFAULT_SERVICE_NAME_HEADER
        if self.pb.HasField('headers_size_limit'):
            exp_getter_pb.headers_size_limit.CopyFrom(self.pb.headers_size_limit)
        holder_pbs.append(h)

        report_pb = exp_getter_pb.uaas.report
        report_pb.uuid = 'expgetter'
        report_pb.ranges = Report.DEFAULT_RANGES_ALIAS

        balancer2_pb = report_pb.nested.stats_eater.nested.balancer2
        balancer2_pb.attempts = 1
        balancer2_pb.rr.SetInParent()
        call_pb_bnp_pb = balancer2_pb.balancing_policy.by_name_policy.f_name
        call_pb_bnp_pb.type = proto.Call.GET_GEO
        params_pb_bnp_pb = call_pb_bnp_pb.get_geo_params
        params_pb_bnp_pb.name = 'bygeo_'
        params_pb_bnp_pb.default_geo = 'random'
        balancer2_pb.balancing_policy.by_name_policy.balancing_policy.simple_policy.SetInParent()

        uaas_full_backend_ids = self._get_uaas_full_backend_ids_by_location(version=self._get_version())
        for location in sorted(uaas_full_backend_ids):
            expgetter_backend_pb = balancer2_pb.backends.add()
            expgetter_backend_pb.name = 'bygeo_{}'.format(location)
            expgetter_backend_pb.weight = 1
            expgetter_balancer2_pb = expgetter_backend_pb.nested.balancer2
            expgetter_balancer2_pb.attempts = 1
            expgetter_balancer2_pb.connection_attempts = 5
            expgetter_balancer2_pb.rr.SetInParent()
            gpb_pb = expgetter_balancer2_pb.generated_proxy_backends
            gpb_pb.include_backends.type = proto.IncludeBackends.BY_ID
            full_uaas_backend_id = uaas_full_backend_ids[location]
            gpb_pb.include_backends.ids.append('/'.join(full_uaas_backend_id))
            proxy_options_pb = gpb_pb.proxy_options
            proxy_options_pb.connect_timeout = self.DEFAULT_CONNECT_TIMEOUT
            proxy_options_pb.backend_timeout = self.DEFAULT_BACKEND_TIMEOUT
            proxy_options_pb.keepalive_count = 1

        on_error_pb = balancer2_pb.on_error.balancer2
        on_error_pb.attempts = 1
        on_error_pb.rr.SetInParent()
        on_error_pb.generated_proxy_backends.instances.add(
            weight=1.000,
            host='uaas.search.yandex.net',
            port=80
        )
        proxy_options_pb = on_error_pb.generated_proxy_backends.proxy_options
        proxy_options_pb.connect_timeout = '20ms'
        proxy_options_pb.backend_timeout = '30ms'
        proxy_options_pb.keepalive_count = 1

        return holder_pbs


class CaptchaMacro(ModuleWrapperBase, MacroBase):
    __protobuf__ = proto.CaptchaMacro

    generated_proxy_backends = None  # type: GeneratedProxyBackends | None
    include_backends = None  # type:  IncludeBackends | None

    VERSION_0_0_1 = semantic_version.Version(u'0.0.1')
    VERSION_0_0_2 = semantic_version.Version(u'0.0.2')
    VERSION_0_0_3 = semantic_version.Version(u'0.0.3')
    VERSION_0_0_4 = semantic_version.Version(u'0.0.4')
    VERSION_0_0_5 = semantic_version.Version(u'0.0.5')
    INITIAL_VERSION = VERSION_0_0_1
    VALID_VERSIONS = frozenset((VERSION_0_0_1, VERSION_0_0_2, VERSION_0_0_3, VERSION_0_0_4, VERSION_0_0_5))

    DEFAULT_ATTEMPTS_NUMBER = 2
    DEFAULT_CONNECT_TIMEOUT = '30ms'
    DEFAULT_BACKEND_TIMEOUT = '10s'

    DEFAULTS = {
        'attempts': DEFAULT_ATTEMPTS_NUMBER,
    }

    def get_version(self):
        if self.pb.version:
            return semantic_version.Version(self.pb.version)
        return self.INITIAL_VERSION

    @staticmethod
    def get_default_cut_request_bytes(version):
        if version >= CaptchaMacro.VERSION_0_0_4:
            return 65536
        else:
            return 0

    def would_include_backends(self):
        return not self.generated_proxy_backends

    def get_would_be_included_full_backend_ids(self, current_namespace_id):
        rv = set()
        if not self.generated_proxy_backends:
            if self.get_version() == self.VERSION_0_0_1:
                rv.update(six.itervalues(AntirobotMacro.ANTIROBOT_FULL_BACKEND_IDS))
            else:
                rv.update(six.itervalues(AntirobotMacro.ANTIROBOT_FULL_SD_BACKEND_IDS))
        return rv

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        if self.generated_proxy_backends:
            new_preceding_modules = add_module(preceding_modules, self)
            self.generated_proxy_backends.validate(ctx=ctx, preceding_modules=new_preceding_modules)
        super(CaptchaMacro, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        if self.pb.version:
            validate_version(self.pb.version, self.VALID_VERSIONS, field_name='version')
        if self.get_version() != self.VERSION_0_0_1 and ctx.config_type == ctx.CONFIG_TYPE_FULL:
            require_sd(preceding_modules, field_name='version')
        super(CaptchaMacro, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def add_xffy(self, holder_pbs):
        version = self.get_version()
        if version < self.VERSION_0_0_3:
            return
        headers_h_pb = proto.Holder()
        if self.pb.trust_x_forwarded_for_y:
            headers_h_pb.headers.create_func_weak['X-Forwarded-For-Y'] = 'realip'
        else:
            headers_h_pb.headers.create_func['X-Forwarded-For-Y'] = 'realip'
        holder_pbs.append(headers_h_pb)

    def add_icookie(self, holder_pbs):
        version = self.get_version()
        if version < self.VERSION_0_0_3:
            return
        holder_pb = proto.Holder()
        holder_pb.icookie.use_default_keys = True
        holder_pb.icookie.trust_parent.value = self.pb.trust_icookie
        holder_pb.icookie.domains.extend(ICOOKIE_DOMAINS)
        holder_pbs.append(holder_pb)

    def add_ja_x(self, holder_pbs):
        if self.get_version() < self.VERSION_0_0_3:
            return
        headers_h_pb = proto.Holder()
        if self.pb.trust_x_yandex_ja_x:
            headers_h_pb.headers.create_func_weak['X-Yandex-Ja3'] = 'ja3'
            headers_h_pb.headers.create_func_weak['X-Yandex-Ja4'] = 'ja4'
        else:
            headers_h_pb.headers.create_func['X-Yandex-Ja3'] = 'ja3'
            headers_h_pb.headers.create_func['X-Yandex-Ja4'] = 'ja4'
        holder_pbs.append(headers_h_pb)

    def expand(self, ctx=DEFAULT_CTX, preceding_modules=()):
        holder_pbs = []

        h = proto.Holder()
        h.report.uuid = 'captchasearch'
        h.report.ranges = Report.DEFAULT_RANGES_ALIAS
        holder_pbs.append(h)

        h = proto.Holder()
        h.h100.SetInParent()
        holder_pbs.append(h)

        h = proto.Holder()
        h.cutter.SetInParent()
        holder_pbs.append(h)

        if self.pb.service or self.pb.req_group:
            headers_h_pb = proto.Holder()
            if self.pb.service:
                headers_h_pb.headers.create.add(key='X-Antirobot-Service-Y', value=self.pb.service)
            if self.pb.req_group:
                headers_h_pb.headers.create.add(key='X-Antirobot-Req-Group', value=self.pb.req_group)
            holder_pbs.append(headers_h_pb)

        self.add_xffy(holder_pbs)
        self.add_icookie(holder_pbs)
        self.add_ja_x(holder_pbs)

        h = proto.Holder()
        h.antirobot.cut_request_bytes = self.get_default_cut_request_bytes(self.get_version())
        h.antirobot.file_switch = './controls/do.not.use.it'
        holder_pbs.append(h)

        if not self.generated_proxy_backends:
            balancer2_pb = h.antirobot.checker.balancer2
            balancer2_pb.attempts = 1
            balancer2_pb.rr.SetInParent()
            call_pb_bnp_pb = balancer2_pb.balancing_policy.by_name_policy.f_name
            call_pb_bnp_pb.type = proto.Call.GET_GEO
            params_pb_bnp_pb = call_pb_bnp_pb.get_geo_params
            params_pb_bnp_pb.name = 'antirobot_'
            params_pb_bnp_pb.default_geo = 'random'
            balancer2_pb.balancing_policy.by_name_policy.balancing_policy.simple_policy.SetInParent()

            for location in AntirobotMacro.ANTIROBOT_LOCATIONS:
                antirobot_backend_pb = balancer2_pb.backends.add()
                antirobot_backend_pb.name = 'antirobot_{}'.format(location)
                antirobot_backend_pb.weight = 1
                antirobot_balancer2_pb = antirobot_backend_pb.nested.balancer2
                antirobot_balancer2_pb.attempts = self.DEFAULT_ATTEMPTS_NUMBER
                antirobot_balancer2_pb.weighted2.SetInParent()
                if self.get_version() == self.VERSION_0_0_1:
                    full_antirobot_backend_id = AntirobotMacro.ANTIROBOT_FULL_BACKEND_IDS[location]
                else:
                    full_antirobot_backend_id = AntirobotMacro.ANTIROBOT_FULL_SD_BACKEND_IDS[location]
                gpb_pb = antirobot_balancer2_pb.generated_proxy_backends
                gpb_pb.include_backends.type = proto.IncludeBackends.BY_ID
                gpb_pb.include_backends.ids.append('/'.join(full_antirobot_backend_id))
                proxy_options_pb = gpb_pb.proxy_options
                proxy_options_pb.connect_timeout = self.DEFAULT_CONNECT_TIMEOUT
                proxy_options_pb.backend_timeout = self.DEFAULT_BACKEND_TIMEOUT

            if self.get_version() >= self.VERSION_0_0_5:
                antirobot_backends_pb = balancer2_pb.backends[:]
                for location in AntirobotMacro.ANTIROBOT_DUMMY_LOCATIONS: # see https://st.yandex-team.ru/AWACS-1340
                    antirobot_backend_pb = balancer2_pb.backends.add()
                    antirobot_backend_pb.name = 'antirobot_{}'.format(location)
                    antirobot_backend_pb.weight = 1
                    inner_balancer2_pb = antirobot_backend_pb.nested.balancer2
                    inner_balancer2_pb.attempts = self.DEFAULT_ATTEMPTS_NUMBER
                    inner_balancer2_pb.rr.SetInParent()

                    for outer_backend_pb in antirobot_backends_pb:
                        inner_backend_pb = inner_balancer2_pb.backends.add()
                        inner_backend_pb.MergeFrom(outer_backend_pb)
                        inner_backend_pb.nested.balancer2.attempts = 1
        else:
            balancer2_pb = h.antirobot.checker.stats_eater.nested.balancer2
            balancer2_pb.attempts = self.pb.attempts or self.DEFAULT_ATTEMPTS_NUMBER
            balancer2_pb.weighted2.SetInParent()
            balancer2_pb.generated_proxy_backends.MergeFrom(self.generated_proxy_backends.pb)

            proxy_options_pb = balancer2_pb.generated_proxy_backends.proxy_options
            proxy_options_pb.connect_timeout = proxy_options_pb.connect_timeout or self.DEFAULT_CONNECT_TIMEOUT
            proxy_options_pb.backend_timeout = proxy_options_pb.backend_timeout or self.DEFAULT_BACKEND_TIMEOUT

        h = proto.Holder()
        h.errordocument.status = httplib.FORBIDDEN
        holder_pbs.append(h)

        return holder_pbs

    def get_branches(self):
        if self.generated_proxy_backends:
            yield self.generated_proxy_backends

    def get_named_branches(self):
        rv = {}
        if self.generated_proxy_backends:
            rv['generated_proxy_backends'] = self.generated_proxy_backends
        return rv


class GeobaseMacro(ChainableModuleWrapperBase, MacroBase):
    __protobuf__ = proto.GeobaseMacro

    generated_proxy_backends = None  # type: GeneratedProxyBackends | None

    DEFAULT_REPORT_UUID = 'geobasemodule'
    DEFAULT_FILE_SWITCH = './controls/disable_geobase.switch'
    DEFAULT_ATTEMPTS_NUMBER = 2
    DEFAULT_KEEPALIVE_COUNT = 10
    DEFAULT_CONNECT_TIMEOUT = '15ms'
    DEFAULT_BACKEND_TIMEOUT = '20ms'
    # https://st.yandex-team.ru/AWACS-883#60f7e719a644a05bfde6c223:
    DEFAULT_ATTEMPTS_NUMBER_0_0_2 = 1
    DEFAULT_CONNECT_TIMEOUT_0_0_2 = '25ms'
    DEFAULT_BACKEND_TIMEOUT_0_0_2 = '45ms'

    ALLOWED_KNOBS = {
        'file_switch': model_pb2.KnobSpec.BOOLEAN,
    }
    DEFAULT_KNOB_IDS = {
        'file_switch': 'balancer_geolib_switch',
    }

    DEFAULTS = {
        'report_uuid': DEFAULT_REPORT_UUID,
        'file_switch': DEFAULT_FILE_SWITCH,
        'attempts': DEFAULT_ATTEMPTS_NUMBER,
        'geo_host': Geobase.DEFAULT_GEO_HOST,
    }

    VERSION_0_0_1 = semantic_version.Version(u'0.0.1')
    VERSION_0_0_2 = semantic_version.Version(u'0.0.2')
    VERSION_0_0_3 = semantic_version.Version(u'0.0.3')
    INITIAL_VERSION = VERSION_0_0_1
    VALID_VERSIONS = frozenset((VERSION_0_0_1, VERSION_0_0_2, VERSION_0_0_3))

    def get_version(self):
        if self.pb.version:
            return semantic_version.Version(self.pb.version)
        return self.INITIAL_VERSION

    def is_chainable(self):
        return True

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        new_preceding_modules = add_module(preceding_modules, self)
        if self.generated_proxy_backends:
            self.generated_proxy_backends.validate(ctx=ctx, preceding_modules=new_preceding_modules)
        super(GeobaseMacro, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules)

    def _validate_file_switch(self, ctx):
        self.validate_knob('file_switch', ctx=ctx)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()
        if self.pb.version:
            validate_version(self.pb.version, self.VALID_VERSIONS, field_name='version')
        self._validate_file_switch(ctx)
        super(GeobaseMacro, self).validate(ctx=ctx,
                                           preceding_modules=preceding_modules,
                                           chained_modules=chained_modules)

    def expand(self, ctx=DEFAULT_CTX, preceding_modules=()):
        version = self.get_version()
        holder_pbs = []

        h_pb = proto.Holder()
        geobase_pb = h_pb.geobase
        file_switch = self.get('file_switch')
        if file_switch == Value.KNOB:
            geobase_pb.k_file_switch.CopyFrom(self.pb.k_file_switch)
        elif file_switch.type == Value.CALL:
            geobase_pb.f_file_switch.CopyFrom(self.pb.f_file_switch)
        else:
            if ctx.are_knobs_enabled():
                geobase_pb.k_file_switch.id = self.DEFAULT_KNOB_IDS['file_switch']
                geobase_pb.k_file_switch.optional = True
            else:
                geobase_pb.file_switch = self.pb.file_switch or self.DEFAULT_FILE_SWITCH
        geobase_pb.trusted = self.pb.trusted
        if version >= self.VERSION_0_0_3:
            geobase_pb.geo_path = Geobase.DEFAULT_GEO_PATH + '&service=' + ctx.namespace_id
        holder_pbs.append(h_pb)

        report_pb = geobase_pb.geo.report
        report_pb.uuid = self.pb.report_uuid or self.DEFAULT_REPORT_UUID
        report_pb.ranges = Report.DEFAULT_RANGES_ALIAS

        balancer2_pb = report_pb.nested.stats_eater.nested.balancer2
        if version == self.VERSION_0_0_1:
            default_attempts = self.DEFAULT_ATTEMPTS_NUMBER
            default_connect_timeout = self.DEFAULT_CONNECT_TIMEOUT
            default_backend_timeout = self.DEFAULT_BACKEND_TIMEOUT
        else:
            default_attempts = self.DEFAULT_ATTEMPTS_NUMBER_0_0_2
            default_connect_timeout = self.DEFAULT_CONNECT_TIMEOUT_0_0_2
            default_backend_timeout = self.DEFAULT_BACKEND_TIMEOUT_0_0_2
        balancer2_pb.attempts = self.pb.attempts or default_attempts
        balancer2_pb.balancing_policy.simple_policy.SetInParent()
        balancer2_pb.rr.SetInParent()

        generated_proxy_backends = balancer2_pb.generated_proxy_backends
        if self.generated_proxy_backends:
            generated_proxy_backends.CopyFrom(self.generated_proxy_backends.pb)

        proxy_options_pb = balancer2_pb.generated_proxy_backends.proxy_options
        proxy_options_pb.connect_timeout = proxy_options_pb.connect_timeout or default_connect_timeout
        proxy_options_pb.backend_timeout = proxy_options_pb.backend_timeout or default_backend_timeout
        proxy_options_pb.keepalive_count = proxy_options_pb.keepalive_count or self.DEFAULT_KEEPALIVE_COUNT

        if not self.generated_proxy_backends:
            generated_proxy_backends.instances.add(weight=1.000,
                                                   host=self.pb.geo_host or Geobase.DEFAULT_GEO_HOST,
                                                   port=80)

        return holder_pbs

    def get_branches(self):
        if self.generated_proxy_backends:
            yield self.generated_proxy_backends

    def get_named_branches(self):
        rv = {}
        if self.generated_proxy_backends:
            rv['generated_proxy_backends'] = self.generated_proxy_backends
        return rv


class RpcRewrite(ChainableModuleWrapperBase):
    __protobuf__ = proto.RpcRewriteModule

    rpc = None  # type: Holder | None

    REQUIRED = ['rpc']

    DEFAULT_RPC_REWRITE_HOST = 'bolver.yandex-team.ru'
    DEFAULT_RPC_REWRITE_URL = '/proxy'
    DEFAULT_RPC_SUCCESS_HEADER = 'X-Metabalancer-Answered'
    DEFAULT_FILE_SWITCH = './controls/disable_rpcrewrite_module'

    ALLOWED_KNOBS = {
        'file_switch': model_pb2.KnobSpec.BOOLEAN,
    }
    DEFAULT_KNOB_IDS = {
        'file_switch': 'balancer_disable_rpcrewrite_module',
    }

    DEFAULTS = {
        'host': DEFAULT_RPC_REWRITE_HOST,
        'url': DEFAULT_RPC_REWRITE_URL,
        'rpc_success_header': DEFAULT_RPC_SUCCESS_HEADER,
        'file_switch': DEFAULT_FILE_SWITCH,
    }

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        new_preceding_modules = add_module(preceding_modules, self)
        with validate('rpc'):
            self.rpc.validate(ctx=ctx, preceding_modules=new_preceding_modules)
        if self.on_rpc_error:
            with validate('on_rpc_error'):
                self.on_rpc_error.validate(ctx=ctx, preceding_modules=new_preceding_modules)
        super(RpcRewrite, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules,
                                                          chained_modules=chained_modules)

    def _validate_file_switch(self, ctx):
        self.validate_knob('file_switch', ctx=ctx)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()
        self._validate_file_switch(ctx)
        super(RpcRewrite, self).validate(ctx=ctx, preceding_modules=preceding_modules, chained_modules=chained_modules)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        new_preceding_modules = add_module(preceding_modules, self)
        table = {
            'rpc': self.rpc.to_config(ctx=ctx, preceding_modules=new_preceding_modules),
            'host': self.pb.host or self.DEFAULT_RPC_REWRITE_HOST,
            'url': self.pb.url or self.DEFAULT_RPC_REWRITE_URL,
            'dry_run': self.pb.dry_run,
            'rpc_success_header': self.pb.rpc_success_header or self.DEFAULT_RPC_SUCCESS_HEADER,
            'file_switch': self.get('file_switch', self.DEFAULT_FILE_SWITCH).to_config(ctx),
        }
        if self.on_rpc_error:
            table['on_rpc_error'] = self.on_rpc_error.to_config(ctx=ctx, preceding_modules=new_preceding_modules)
        return Config(table)

    def get_branches(self):
        if self.rpc:
            yield self.rpc
        if self.on_rpc_error:
            yield self.on_rpc_error

    def get_named_branches(self):
        rv = {}
        if self.rpc:
            rv['rpc'] = self.rpc
        if self.on_rpc_error:
            rv['on_rpc_error'] = self.on_rpc_error
        return rv


class RpcRewriteMacro(ChainableModuleWrapperBase, MacroBase):
    __protobuf__ = proto.RpcRewriteMacro

    generated_proxy_backends = None  # type: GeneratedProxyBackends | None

    DEFAULT_ATTEMPTS_NUMBER = 3
    DEFAULT_CONNECT_TIMEOUT = '150ms'
    DEFAULT_BACKEND_TIMEOUT = '10s'
    DEFAULT_REPORT_UUID = 'rpcrewrite-backend'

    DEFAULTS = {
        'attempts': DEFAULT_ATTEMPTS_NUMBER,
        'file_switch': RpcRewrite.DEFAULT_FILE_SWITCH,
    }

    ALLOWED_KNOBS = {
        'file_switch': model_pb2.KnobSpec.BOOLEAN,
    }
    DEFAULT_KNOB_IDS = {
        'file_switch': 'balancer_disable_rpcrewrite_module',
    }

    def get_would_be_included_full_knob_ids(self, namespace_id, ctx):
        rv = super(RpcRewriteMacro, self).get_would_be_included_full_knob_ids(namespace_id, ctx)
        if ctx.are_knobs_enabled():
            for field_name, knob_id in six.iteritems(RpcRewrite.DEFAULT_KNOB_IDS):
                if knob_id != 'file_switch':  # explicitly set by expand
                    rv.add((namespace_id, knob_id))
            for field_name, knob_id in six.iteritems(Balancer2.DEFAULT_KNOB_IDS):
                rv.add((namespace_id, knob_id))
        return rv

    def is_chainable(self):
        return True

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        new_preceding_modules = add_module(preceding_modules, self)
        if self.generated_proxy_backends:
            self.generated_proxy_backends.validate(ctx=ctx, preceding_modules=new_preceding_modules)
        super(RpcRewriteMacro, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules)

    def _validate_file_switch(self, ctx):
        self.validate_knob('file_switch', ctx=ctx)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()
        self._validate_file_switch(ctx)
        super(RpcRewriteMacro, self).validate(ctx=ctx, preceding_modules=preceding_modules,
                                              chained_modules=chained_modules)

    def expand(self, ctx=DEFAULT_CTX, preceding_modules=()):
        holder_pbs = []

        h = proto.Holder()
        file_switch = self.get('file_switch')
        if file_switch == Value.KNOB:
            h.rpcrewrite.k_file_switch.CopyFrom(self.pb.k_file_switch)
        elif file_switch.type == Value.CALL:
            h.rpcrewrite.f_file_switch.CopyFrom(self.pb.f_file_switch)
        else:
            if ctx.are_knobs_enabled():
                h.rpcrewrite.k_file_switch.id = self.DEFAULT_KNOB_IDS['file_switch']
                h.rpcrewrite.k_file_switch.optional = True
            else:
                h.rpcrewrite.file_switch = self.pb.file_switch or RpcRewrite.DEFAULT_FILE_SWITCH
        holder_pbs.append(h)

        report_pb = h.rpcrewrite.rpc.report
        report_pb.uuid = self.DEFAULT_REPORT_UUID
        report_pb.ranges = Report.DEFAULT_RANGES_ALIAS

        balancer2_pb = report_pb.nested.stats_eater.nested.balancer2
        balancer2_pb.attempts = self.pb.attempts or self.DEFAULT_ATTEMPTS_NUMBER
        balancer2_pb.balancing_policy.simple_policy.SetInParent()
        balancer2_pb.rr.SetInParent()

        generated_proxy_backends = balancer2_pb.generated_proxy_backends
        if self.generated_proxy_backends:
            generated_proxy_backends.CopyFrom(self.generated_proxy_backends.pb)

        proxy_options_pb = balancer2_pb.generated_proxy_backends.proxy_options
        proxy_options_pb.connect_timeout = proxy_options_pb.connect_timeout or self.DEFAULT_CONNECT_TIMEOUT
        proxy_options_pb.backend_timeout = proxy_options_pb.backend_timeout or self.DEFAULT_BACKEND_TIMEOUT
        proxy_options_pb.keepalive_count = proxy_options_pb.keepalive_count

        if not self.generated_proxy_backends:
            generated_proxy_backends.instances.add(
                weight=1.000,
                host=RpcRewrite.DEFAULT_RPC_REWRITE_HOST,
                port=80
            )

        if self.pb.enable_on_rpc_error:
            h.rpcrewrite.on_rpc_error.errordocument.status = 500
            h.rpcrewrite.on_rpc_error.errordocument.content = 'Failed to rewrite request using RPC'

        return holder_pbs

    def get_branches(self):
        if self.generated_proxy_backends:
            yield self.generated_proxy_backends

    def get_named_branches(self):
        rv = {}
        if self.generated_proxy_backends:
            rv['generated_proxy_backends'] = self.generated_proxy_backends
        return rv


class Click(ChainableModuleWrapperBase):
    __protobuf__ = proto.ClickModule

    DEFAULT_KEYS_FILE = './data/clickdaemon.keys'

    DEFAULTS = {
        'keys': DEFAULT_KEYS_FILE,
    }

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        super(Click, self).validate(ctx=ctx, preceding_modules=preceding_modules, chained_modules=chained_modules)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict([
            ('keys', self.pb.keys or self.DEFAULT_KEYS_FILE),
        ])
        if self.pb.json_keys:
            table['json_keys'] = self.pb.json_keys
        return Config(table)


class ClickMacro(ModuleWrapperBase, MacroBase):
    __protobuf__ = proto.ClickMacro

    generated_proxy_backends = None  # type: GeneratedProxyBackends | None

    DEFAULT_REPORT_UUID = 'clcksearch'
    DEFAULT_ATTEMPTS_NUMBER = 5
    DEFAULT_CONNECT_TIMEOUT = '30ms'
    DEFAULT_BACKEND_TIMEOUT = '1s'

    VERSION_0_0_1 = semantic_version.Version(u'0.0.1')
    VERSION_0_0_2 = semantic_version.Version(u'0.0.2')  # use SD backends
    INITIAL_VERSION = VERSION_0_0_1
    VALID_VERSIONS = frozenset((VERSION_0_0_1, VERSION_0_0_2))

    CLCK_LOCATIONS = ('man', 'sas', 'vla')
    CLCK_FULL_BACKEND_IDS = {
        location: ('common-clck', 'clck_misc_{}'.format(location))
        for location in CLCK_LOCATIONS
    }
    CLCK_FULL_SD_BACKEND_IDS = {
        location: ('common-clck', 'clck_misc_{}-sd'.format(location))
        for location in CLCK_LOCATIONS
    }

    DEFAULTS = {
        'attempts': DEFAULT_ATTEMPTS_NUMBER,
    }

    def get_version(self):
        if self.pb.version:
            return semantic_version.Version(self.pb.version)
        return self.INITIAL_VERSION

    def would_include_backends(self):
        return not self.generated_proxy_backends

    def get_would_be_included_full_backend_ids(self, current_namespace_id):
        if not self.generated_proxy_backends:
            if self.get_version() == self.VERSION_0_0_1:
                return six.itervalues(self.CLCK_FULL_BACKEND_IDS)
            else:
                return six.itervalues(self.CLCK_FULL_SD_BACKEND_IDS)
        else:
            return set()

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        if self.generated_proxy_backends:
            new_preceding_modules = add_module(preceding_modules, self)
            self.generated_proxy_backends.validate(ctx=ctx, preceding_modules=new_preceding_modules)
        super(ClickMacro, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        if self.get_version() != self.VERSION_0_0_1 and ctx.config_type == ctx.CONFIG_TYPE_FULL:
            require_sd(preceding_modules, field_name='sd')
        super(ClickMacro, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def expand(self, ctx=DEFAULT_CTX, preceding_modules=()):
        holder_pbs = []

        h = proto.Holder()
        rewrite_pb = h.rewrite
        rewrite_pb.actions.add(
            rewrite='%1',
            regexp='/clck(/.*)'
        )
        holder_pbs.append(h)

        h = proto.Holder()
        h.click.SetInParent()
        holder_pbs.append(h)

        h = proto.Holder()
        h.report.uuid = self.DEFAULT_REPORT_UUID
        h.report.ranges = Report.DEFAULT_RANGES_ALIAS
        holder_pbs.append(h)

        if not self.generated_proxy_backends:
            balancer2_pb = h.report.nested.balancer2
            balancer2_pb.attempts = 1
            balancer2_pb.rr.SetInParent()
            call_pb_bnp_pb = balancer2_pb.balancing_policy.by_name_policy.f_name
            call_pb_bnp_pb.type = proto.Call.GET_GEO
            params_pb_bnp_pb = call_pb_bnp_pb.get_geo_params
            params_pb_bnp_pb.name = 'clck_'
            params_pb_bnp_pb.default_geo = 'random'
            balancer2_pb.balancing_policy.by_name_policy.balancing_policy.simple_policy.SetInParent()

            for location in self.CLCK_LOCATIONS:
                clck_backend_pb = balancer2_pb.backends.add()
                clck_backend_pb.name = 'clck_{}'.format(location)
                clck_backend_pb.weight = 1
                clck_balancer2_pb = clck_backend_pb.nested.stats_eater.nested.balancer2
                clck_balancer2_pb.attempts = self.DEFAULT_ATTEMPTS_NUMBER
                clck_balancer2_pb.weighted2.SetInParent()
                if self.get_version() == self.VERSION_0_0_1:
                    full_clck_backend_id = self.CLCK_FULL_BACKEND_IDS[location]
                else:
                    full_clck_backend_id = self.CLCK_FULL_SD_BACKEND_IDS[location]
                gpb_pb = clck_balancer2_pb.generated_proxy_backends
                gpb_pb.include_backends.type = proto.IncludeBackends.BY_ID
                gpb_pb.include_backends.ids.append('/'.join(full_clck_backend_id))
                proxy_options_pb = gpb_pb.proxy_options
                proxy_options_pb.connect_timeout = self.DEFAULT_CONNECT_TIMEOUT
                proxy_options_pb.backend_timeout = self.DEFAULT_BACKEND_TIMEOUT
        else:
            balancer2_pb = h.report.nested.stats_eater.nested.balancer2
            balancer2_pb.attempts = self.pb.attempts or self.DEFAULT_ATTEMPTS_NUMBER
            balancer2_pb.weighted2.SetInParent()
            balancer2_pb.generated_proxy_backends.MergeFrom(self.generated_proxy_backends.pb)

            prx_opts_pb = balancer2_pb.generated_proxy_backends.proxy_options
            prx_opts_pb.connect_timeout = prx_opts_pb.connect_timeout or self.DEFAULT_CONNECT_TIMEOUT
            prx_opts_pb.backend_timeout = prx_opts_pb.backend_timeout or self.DEFAULT_BACKEND_TIMEOUT
            prx_opts_pb.keepalive_count = prx_opts_pb.keepalive_count

        return holder_pbs

    def get_branches(self):
        if self.generated_proxy_backends:
            yield self.generated_proxy_backends

    def get_named_branches(self):
        rv = {}
        if self.generated_proxy_backends:
            rv['generated_proxy_backends'] = self.generated_proxy_backends
        return rv


class RequestReplier(ChainableModuleWrapperBase):
    __protobuf__ = proto.RequestReplierModule

    sink = None  # type: Holder | None

    REQUIRED = ['sink']

    DEFAULT_RATE_FILE = './controls/request_repl.ratefile'

    DEFAULTS = {
        'rate_file': DEFAULT_RATE_FILE,
    }

    ALLOWED_KNOBS = {
        'rate_file': model_pb2.KnobSpec.RATE,
    }
    DEFAULT_KNOB_IDS = {
        'rate_file': 'common_request_replier_rate',
    }

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        with validate('sink'):
            self.sink.validate(ctx=ctx, preceding_modules=add_module(preceding_modules, self))
        super(RequestReplier, self).validate_composite_fields(ctx=ctx,
                                                              preceding_modules=preceding_modules,
                                                              chained_modules=chained_modules)

    def _validate_rate_file(self, ctx):
        self.validate_knob('rate_file', ctx=ctx)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()
        self._validate_rate_file(ctx)
        super(RequestReplier, self).validate(ctx=ctx,
                                             preceding_modules=preceding_modules,
                                             chained_modules=chained_modules)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict([
            ('sink', self.sink.to_config(ctx=ctx, preceding_modules=add_module(preceding_modules, self))),
            ('enable_failed_requests_replication', self.pb.enable_failed_requests_replication),
            ('rate', self.pb.rate),
            ('rate_file', self.get('rate_file', self.DEFAULT_RATE_FILE).to_config(ctx)),
        ])
        return Config(table)

    def get_branches(self):
        if self.sink:
            yield self.sink

    def get_named_branches(self):
        rv = {}
        if self.sink:
            rv['sink'] = self.sink
        return rv


class HeadersHasher(ChainableModuleWrapperBase):
    __protobuf__ = proto.HeadersHasherModule

    DEFAULTS = {
        'surround': False,
        'randomize_empty_match': False,
    }

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()

        with validate('header_name'):
            validate_pire_regexp(self.pb.header_name)

        super(HeadersHasher, self).validate(ctx=ctx, preceding_modules=preceding_modules,
                                            chained_modules=chained_modules)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict([
            ('header_name', self.pb.header_name),
            ('surround', self.pb.surround),
            ('randomize_empty_match', self.pb.randomize_empty_match),
        ])
        return Config(table)


class ResponseHeadersIf(ChainableModuleWrapperBase):
    __protobuf__ = proto.ResponseHeadersIfModule

    create_header = []  # type: list[HeaderMapEntry]
    create_weak = []  # type: list[HeaderMapEntry]
    matcher = None  # type: ResponseMatcherMatcher

    REQUIRED_ONEOFS = [
        ('if_has_header', 'matcher'),
    ]
    REQUIRED_ANYOFS = [
        ('create_header', 'delete_header'),
    ]

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()

        if self.pb.if_has_header:
            with validate('if_has_header'):
                validate_pire_regexp(self.pb.if_has_header)
        elif self.matcher:
            with validate('matcher'):
                self.matcher.validate(ctx=ctx, preceding_modules=preceding_modules)
        else:
            raise AssertionError()
        if self.create_header:
            for i, header_entry in enumerate(self.create_header):
                with validate('create_header[{}]'.format(i)):
                    header_entry.validate()
        if self.pb.delete_header:
            with validate('delete_header'):
                validate_pire_regexp(self.pb.delete_header)

        super(ResponseHeadersIf, self).validate(ctx=ctx,
                                                preceding_modules=preceding_modules,
                                                chained_modules=chained_modules)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict()
        if self.pb.if_has_header:
            table['if_has_header'] = self.pb.if_has_header
            table['erase_if_has_header'] = self.pb.erase_if_has_header
        elif self.matcher:
            table['matcher'] = self.matcher.to_config(ctx=ctx, preceding_modules=preceding_modules)
        else:
            raise AssertionError()

        if self.pb.create_header:
            create_header_table = OrderedDict(sorted([entry.to_config_item(ctx=ctx) for entry in self.create_header]))
            table['create_header'] = Config(create_header_table)
        if self.pb.delete_header:
            table['delete_header'] = self.pb.delete_header
        return Config(table)


class TcpRstOnError(ChainableModuleWrapperBase):
    __protobuf__ = proto.TcpRstOnErrorModule

    DEFAULT_SEND_RST = True

    DEFAULTS = {
        'send_rst': DEFAULT_SEND_RST,
    }

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        send_rst = self.DEFAULT_SEND_RST
        if self.pb.HasField('send_rst'):
            send_rst = self.pb.send_rst.value

        table = {'send_rst': send_rst}
        return Config(table)


class HeadersForwarderAction(ConfigWrapperBase):
    __protobuf__ = proto.HeadersForwarderAction

    REQUIRED = ['request_header', 'response_header']

    DEFAULTS = {
        'erase_from_request': False,
        'erase_from_response': False,
        'weak': False,
    }

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        if self.pb.weak and self.pb.erase_from_response:
            raise ValidationError(
                'simultaneously using of "erase_from_response" and "weak" is prohibited')
        super(HeadersForwarderAction, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict([
            ('request_header', self.pb.request_header),
            ('response_header', self.pb.response_header),
            ('erase_from_request', self.pb.erase_from_request),
            ('erase_from_response', self.pb.erase_from_response),
            ('weak', self.pb.weak)
        ])
        return Config(table)


class HeadersForwarder(ChainableModuleWrapperBase):
    __protobuf__ = proto.HeadersForwarderModule

    actions = []  # type: list[HeadersForwarderAction]

    REQUIRED = ['actions']

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        for i, action in enumerate(self.actions):
            with validate('actions[{}]'.format(i)):
                action.validate(ctx=ctx, preceding_modules=add_module(preceding_modules, self))
        super(HeadersForwarder, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules,
                                                                chained_modules=chained_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()
        super(HeadersForwarder, self).validate(ctx=ctx, preceding_modules=preceding_modules,
                                               chained_modules=chained_modules)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        action_configs = [action.to_config(ctx=ctx, preceding_modules=preceding_modules)
                          for action in self.actions]
        table = OrderedDict([
            ('actions', Config(array=action_configs)),
        ])
        return Config(table)


class CookieHasher(ChainableModuleWrapperBase):
    __protobuf__ = proto.CookieHasherModule

    REQUIRED = ['cookie']

    DEFAULT_FILE_SWITCH = './controls/disable_cookie_hasher'

    DEFAULTS = {
        'file_switch': DEFAULT_FILE_SWITCH,
    }

    ALLOWED_KNOBS = {
        'file_switch': model_pb2.KnobSpec.BOOLEAN,
    }
    DEFAULT_KNOB_IDS = {
        'file_switch': 'balancer_disable_cookie_hasher',
    }

    def _validate_file_switch(self, ctx):
        self.validate_knob('file_switch', ctx=ctx)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()
        self._validate_file_switch(ctx)
        with validate('cookie'):
            validate_pire_regexp(self.pb.cookie)
        super(CookieHasher, self).validate(ctx=ctx, preceding_modules=preceding_modules,
                                           chained_modules=chained_modules)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict([
            ('cookie', self.pb.cookie),
            ('file_switch', self.get('file_switch', self.DEFAULT_FILE_SWITCH).to_config(ctx)),
        ])
        return Config(table)


class GgiParameter(ConfigWrapperBase):
    __protobuf__ = proto.GgiParameter

    REQUIRED = ['value']

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        validate_pire_regexp(self.pb.value)
        super(GgiParameter, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        return self.pb.value


class CgiHasher(ChainableModuleWrapperBase):
    __protobuf__ = proto.CgiHasherModule

    parameters = []  # type: list[GgiParameter]

    REQUIRED = ['parameters']

    DEFAULT_RANDOMIZE_EMPTY_MATCH = True

    # info attrs:
    DEFAULTS = {
        'randomize_empty_match': DEFAULT_RANDOMIZE_EMPTY_MATCH,
        'case_insensitive': False,
    }
    ALLOWED_MODES = ('concatenated', 'priority')

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()
        if self.pb.mode and self.pb.mode not in self.ALLOWED_MODES:
            raise ValidationError('must be one of the "{}"'.format('", "'.join(self.ALLOWED_MODES)), 'mode')
        super(CgiHasher, self).validate(ctx=ctx, preceding_modules=preceding_modules, chained_modules=chained_modules)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        randomize_empty_match = self.DEFAULT_RANDOMIZE_EMPTY_MATCH
        if self.pb.HasField('randomize_empty_match'):
            randomize_empty_match = self.pb.randomize_empty_match.value

        new_preceding_modules = add_module(preceding_modules, self)
        parameter_configs = [parameter.to_config(ctx=ctx, preceding_modules=new_preceding_modules)
                             for parameter in self.parameters]
        table = OrderedDict([
            ('parameters', Config(array=parameter_configs)),
            ('randomize_empty_match', randomize_empty_match),
            ('case_insensitive', self.pb.case_insensitive),
        ])
        if self.pb.mode:
            table['mode'] = self.pb.mode
        return Config(table)


class Pinger(ChainableModuleWrapperBase):
    __protobuf__ = proto.PingerModule

    admin_error_replier = None  # type: Holder | None

    REQUIRED = ['ping_request_data', 'admin_request_uri', 'admin_error_replier']

    DEFAULT_LO = 0.5
    DEFAULT_HI = 0.7
    DEFAULT_DELAY = '1s'
    DEFAULT_HISTTIME = '4s'
    # https://ro.racktables.yandex.net/export/expand-fw-macro.php?macro=_SLB_LO_NETS_
    # NOCDEV-191
    # Update 03-03-2018
    DEFAULT_ADMIN_IPS = (
        '5.45.193.144/29,'
        '5.45.208.88/30,'
        '5.45.228.160/30,'
        '5.45.229.168/30,'
        '5.45.232.160/30,'
        '5.45.240.168/30,'
        '5.45.243.16/30,'
        '5.45.247.144/29,'
        '5.45.247.176/29,'
        '5.255.192.64/30,'
        '5.255.194.12/30,'
        '5.255.195.144/30,'
        '5.255.196.168/30,'
        '5.255.197.168/30,'
        '5.255.200.160/30,'
        '5.255.252.160/30,'
        '37.9.73.160/30,'
        '37.140.136.88/29,'
        '37.140.137.104/29,'
        '77.88.35.168/30,'
        '77.88.54.168/30,'
        '84.201.159.176/28,'
        '87.250.226.0/25,'
        '87.250.226.128/25,'
        '87.250.228.0/24,'
        '87.250.234.0/24,'
        '95.108.180.40/29,'
        '95.108.237.0/25,'
        '95.108.237.128/25,'
        '100.43.92.144/28,'
        '141.8.136.128/25,'
        '141.8.154.200/29,'
        '141.8.155.104/29,'
        '185.32.186.8/29,'
        '213.180.202.160/30,'
        '213.180.223.16/30,'
        '2001:678:384:100::/64,'
        '2620:10f:d000:100::/64,'
        '2a02:6b8:0:300::/64,'
        '2a02:6b8:0:400::/64,'
        '2a02:6b8:0:800::/64,'
        '2a02:6b8:0:900::/64,'
        '2a02:6b8:0:d00::/64,'
        '2a02:6b8:0:e00::/64,'
        '2a02:6b8:0:1000::/64,'
        '2a02:6b8:0:1100::/64,'
        '2a02:6b8:0:1200::/64,'
        '2a02:6b8:0:1300::/64,'
        '2a02:6b8:0:1400::/64,'
        '2a02:6b8:0:1500::/64,'
        '2a02:6b8:0:1600::/64,'
        '2a02:6b8:0:1700::/64,'
        '2a02:6b8:0:1800::/64,'
        '2a02:6b8:0:1900::/64,'
        '2a02:6b8:0:1a00::/64,'
        '2a02:6b8:0:1b00::/64,'
        '2a02:6b8:0:1d00::/64,'
        '2a02:6b8:0:1e00::/64,'
        '2a02:6b8:0:1f00::/64,'
        '2a02:6b8:0:2000::/64,'
        '2a02:6b8:0:2200::/64,'
        '2a02:6b8:0:2c00::/64,'
        '2a02:6b8:0:3000::/64,'
        '2a02:6b8:0:3100::/64,'
        '2a02:6b8:0:3401::/64,'
        '2a02:6b8:0:3c00::/64,'
        '2a02:6b8:0:3d00::/64,'
        '2a02:6b8:0:3e00::/64,'
        '2a02:6b8:0:3f00::/64,'
        '2a02:6b8:0:4000::/64,'
        '2a02:6b8:0:4200::/64,'
        '2a02:6b8:0:4700::/64,'
        '2a02:6b8:b010:b000::/64'
    )
    DEFAULT_ENABLE_TCP_CHECK_FILE = './controls/tcp_check_on'
    DEFAULT_SWITCH_OFF_FILE = './controls/slb_check.weights'
    DEFAULT_SWITCH_OFF_KEY = 'switch_off'

    ALLOWED_KNOBS = {
        'enable_tcp_check_file': model_pb2.KnobSpec.BOOLEAN,
        'switch_off_file': model_pb2.KnobSpec.YB_BACKEND_WEIGHTS,
    }
    DEFAULT_KNOB_IDS = {
        'enable_tcp_check_file': 'balancer_tcp_check_on',
        'switch_off_file': 'service_balancer_off',
    }

    DEFAULTS = {
        'lo': DEFAULT_LO,
        'hi': DEFAULT_HI,
        'delay': DEFAULT_DELAY,
        'histtime': DEFAULT_HISTTIME,
        # 'admin_ips': DEFAULT_ADMIN_IPS,  # too long for docs
        'enable_tcp_check_file': DEFAULT_ENABLE_TCP_CHECK_FILE,
        'switch_off_file': DEFAULT_SWITCH_OFF_FILE,
        'switch_off_key': DEFAULT_SWITCH_OFF_KEY,
    }

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        new_preceding_modules = add_module(preceding_modules, self)
        with validate('admin_error_replier'):
            self.admin_error_replier.validate(ctx=ctx, preceding_modules=new_preceding_modules)
        super(Pinger, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules,
                                                      chained_modules=chained_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()

        with validate('ping_request_data'):
            validate_request_line(self.pb.ping_request_data)

        if self.pb.lo and not (0 <= self.pb.lo <= 1):
            raise ValidationError('must be between 0 and 1', 'lo')

        if self.pb.hi and not (0 <= self.pb.hi <= 1):
            raise ValidationError('must be between 0 and 1', 'hi')

        if self.pb.delay:
            with validate('delay'):
                validate_timedelta(self.pb.delay)

        if self.pb.histtime:
            with validate('histtime'):
                validate_timedelta(self.pb.histtime)

        if self.pb.status_codes:
            with validate('status_codes'):
                validate_status_codes(self.pb.status_codes)

        if self.pb.status_codes_exceptions:
            with validate('status_codes_exceptions'):
                validate_status_codes(self.pb.status_codes_exceptions)
        super(Pinger, self).validate(ctx=ctx, preceding_modules=preceding_modules, chained_modules=chained_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        module_config = Config()
        self._add_nested_module_to_config(module_config, preceding_modules=preceding_modules)
        admin_error_replier_config = self.admin_error_replier.to_config(
            ctx=ctx, preceding_modules=add_module(preceding_modules, self))
        table = OrderedDict([
            ('lo', self.pb.lo or self.DEFAULT_LO),
            ('hi', self.pb.hi or self.DEFAULT_HI),
            ('delay', self.pb.delay or self.DEFAULT_DELAY),
            ('histtime', self.pb.histtime or self.DEFAULT_HISTTIME),
            ('ping_request_data', self.pb.ping_request_data),
            ('admin_request_uri', self.pb.admin_request_uri),
            ('admin_ips', self.pb.admin_ips or self.DEFAULT_ADMIN_IPS),
            ('enable_tcp_check_file',
             self.get('enable_tcp_check_file', self.DEFAULT_ENABLE_TCP_CHECK_FILE).to_config(ctx)),
            ('switch_off_file', self.get('switch_off_file', self.DEFAULT_SWITCH_OFF_FILE).to_config(ctx)),
            ('switch_off_key', self.pb.switch_off_key or self.DEFAULT_SWITCH_OFF_KEY),
            ('admin_error_replier', admin_error_replier_config),
        ])

        if self.pb.status_codes:
            table['status_codes'] = Config(array=list(self.pb.status_codes))
        if self.pb.status_codes_exceptions:
            table['status_codes_exceptions'] = Config(array=list(self.pb.status_codes_exceptions))

        table['module'] = module_config

        return Config(table, outlets=(module_config,), shareable=True)

    def get_branches(self):
        if self.admin_error_replier:
            yield self.admin_error_replier

    def get_named_branches(self):
        rv = {}
        if self.admin_error_replier:
            rv['admin_error_replier'] = self.admin_error_replier
        return rv


class LogHeaders(ChainableModuleWrapperBase):
    __protobuf__ = proto.LogHeadersModule

    REQUIRED_ANYOFS = [('name_re', 'response_name_re')]

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()

        if self.pb.name_re:
            with validate('name_re'):
                validate_pire_regexp(self.pb.name_re)

        if self.pb.response_name_re:
            with validate('response_name_re'):
                validate_pire_regexp(self.pb.response_name_re)

        seen_cookie_fields = set()
        for i, field in enumerate(self.pb.cookie_fields):
            with validate('cookie_fields[{}]'.format(i)):
                if not field:
                    raise ValidationError('must not be empty')
                if field in seen_cookie_fields:
                    raise ValidationError('duplicate field "{}"'.format(field))
            seen_cookie_fields.add(field)

        super(LogHeaders, self).validate(ctx=ctx, preceding_modules=preceding_modules, chained_modules=chained_modules)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict()
        if self.pb.name_re:
            table['name_re'] = self.pb.name_re
        if self.pb.cookie_fields:
            table['cookie_fields'] = ','.join(self.pb.cookie_fields)
        if self.pb.response_name_re:
            table['response_name_re'] = self.pb.response_name_re
        if self.pb.log_response_body_md5:
            table['log_response_body_md5'] = self.pb.log_response_body_md5
        return Config(table)


class Icookie(ChainableModuleWrapperBase):
    __protobuf__ = proto.IcookieModule

    REQUIRED = ['domains']
    REQUIRED_ONEOFS = [('use_default_keys', 'keys_file')]

    DEFAULT_DECRYPTED_UID_HEADER = 'X-Yandex-ICookie'
    DEFAULT_ERROR_HEADER = 'X-Yandex-ICookie-Error'
    DEFAULT_TRUST_PARENT = False
    DEFAULT_TRUST_CHILDREN = False
    DEFAULT_ENABLE_SET_COOKIE = True
    DEFAULT_ENABLE_DECRYPTING = True

    DEFAULTS = {
        'trust_parent': DEFAULT_TRUST_PARENT,
        'trust_children': DEFAULT_TRUST_CHILDREN,
        'enable_set_cookie': DEFAULT_ENABLE_SET_COOKIE,
        'enable_decrypting': DEFAULT_ENABLE_DECRYPTING,
        'decrypted_uid_header': DEFAULT_DECRYPTED_UID_HEADER,
        'error_header': DEFAULT_ERROR_HEADER,
        'use_default_keys': False,
    }

    ALLOWED_KNOBS = {
        'file_switch': model_pb2.KnobSpec.BOOLEAN,
    }

    def _validate_file_switch(self, ctx):
        self.validate_knob('file_switch', ctx=ctx)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()
        self._validate_file_switch(ctx)
        if self.pb.HasField('scheme_bitmask'):
            if self.pb.scheme_bitmask.value not in (0, 1, 2, 3):
                raise ValidationError(u'must be 0, 1, 2 or 3', u'scheme_bitmask')
        if self.pb.HasField('max_transport_age'):
            if self.pb.max_transport_age.value <= 0:
                raise ValidationError(u'must be positive', u'max_transport_age')
        super(Icookie, self).validate(ctx=ctx, preceding_modules=preceding_modules, chained_modules=chained_modules)

    def _get_bool_value(self, name, default):
        rv = default
        if self.pb.HasField(name):
            rv = getattr(self.pb, name).value
        return rv

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict()
        if self.pb.use_default_keys:
            table['use_default_keys'] = self.pb.use_default_keys
        else:
            table['keys_file'] = self.pb.keys_file
        table.update([
            ('domains', ','.join(self.pb.domains)),
            ('trust_parent', self._get_bool_value('trust_parent', self.DEFAULT_TRUST_PARENT)),
            ('trust_children', self._get_bool_value('trust_children', self.DEFAULT_TRUST_CHILDREN)),
            ('enable_set_cookie', self._get_bool_value('enable_set_cookie', self.DEFAULT_ENABLE_SET_COOKIE)),
            ('enable_decrypting', self._get_bool_value('enable_decrypting', self.DEFAULT_ENABLE_DECRYPTING)),
            ('decrypted_uid_header', self.pb.decrypted_uid_header or self.DEFAULT_DECRYPTED_UID_HEADER),
            ('error_header', self.pb.error_header or self.DEFAULT_ERROR_HEADER),
        ])
        file_switch = self.get('file_switch')
        if file_switch.value:
            table['file_switch'] = file_switch.to_config(ctx)
        if self.pb.take_randomuid_from:
            table['take_randomuid_from'] = self.pb.take_randomuid_from
        if self.pb.HasField('force_equal_to_yandexuid'):
            table['force_equal_to_yandexuid'] = self.pb.force_equal_to_yandexuid.value
        if self.pb.HasField('force_generate_from_searchapp_uuid'):
            table['force_generate_from_searchapp_uuid'] = self.pb.force_generate_from_searchapp_uuid.value
        if self.pb.HasField('enable_parse_searchapp_uuid'):
            table['enable_parse_searchapp_uuid'] = self.pb.enable_parse_searchapp_uuid.value
        if self.pb.HasField('scheme_bitmask'):
            table['scheme_bitmask'] = self.pb.scheme_bitmask.value
        if self.pb.encrypted_header:
            table['encrypted_header'] = self.pb.encrypted_header
        if self.pb.HasField('max_transport_age'):
            table['max_transport_age'] = self.pb.max_transport_age.value
        return Config(table)


class AntirobotWrapper(ChainableModuleWrapperBase):
    __protobuf__ = proto.AntirobotWrapperModule

    ALLOWED_KNOBS = {
        'no_cut_request_file': model_pb2.KnobSpec.BOOLEAN,
    }
    DEFAULT_KNOB_IDS = {
        'no_cut_request_file': 'no_cut_request_file',
    }

    def _validate_no_cut_request_file(self, ctx):
        self.validate_knob('no_cut_request_file', ctx=ctx)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        if self.pb.cut_request_bytes < 0:
            raise ValidationError('must be non-negative', 'cut_request_bytes')
        self._validate_no_cut_request_file(ctx)
        super(AntirobotWrapper, self).validate(ctx=ctx,
                                               preceding_modules=preceding_modules,
                                               chained_modules=chained_modules)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        cut_request = self.pb.cut_request.value if self.pb.HasField('cut_request') else Antirobot.DEFAULT_CUT_REQUEST
        table = OrderedDict([
            ('cut_request', cut_request),
            ('no_cut_request_file',
             self.get('no_cut_request_file', Antirobot.DEFAULT_NO_CUT_REQUEST_FILE).to_config(ctx)),
            ('cut_request_bytes', self.pb.cut_request_bytes or Antirobot.DEFAULT_CUT_REQUEST_BYTES),
        ])
        return Config(table)


class RemoteLog(ChainableModuleWrapperBase):
    __protobuf__ = proto.RemoteLogModule

    remote_log_storage = None  # type: Holder | None

    DEFAULT_NO_REMOTE_LOG_FILE = './controls/remote_log.switch'

    REQUIRED = ['remote_log_storage']

    DEFAULTS = {
        'no_remote_log_file': DEFAULT_NO_REMOTE_LOG_FILE,
    }

    ALLOWED_KNOBS = {
        'no_remote_log_file': model_pb2.KnobSpec.BOOLEAN,
    }
    DEFAULT_KNOB_IDS = {
        'no_remote_log_file': 'balancer_remote_log_switch',
    }

    def _validate_no_remote_log_file(self, ctx):
        self.validate_knob('no_remote_log_file', ctx=ctx)

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        with validate('remote_log_storage'):
            self.remote_log_storage.validate(ctx=ctx, preceding_modules=add_module(preceding_modules, self))
        super(RemoteLog, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules,
                                                         chained_modules=chained_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()
        self._validate_no_remote_log_file(ctx)
        super(RemoteLog, self).validate(ctx=ctx, preceding_modules=preceding_modules,
                                        chained_modules=chained_modules)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        remote_log_storage_config = self.remote_log_storage.to_config(
            ctx=ctx, preceding_modules=add_module(preceding_modules, self))
        table = OrderedDict([
            ('uaas_mode', self.pb.uaas_mode),
            ('no_remote_log_file', self.get('no_remote_log_file', self.DEFAULT_NO_REMOTE_LOG_FILE).to_config(ctx)),
            ('remote_log_storage', remote_log_storage_config),
        ])
        return Config(table)

    def get_branches(self):
        if self.remote_log_storage:
            yield self.remote_log_storage

    def get_named_branches(self):
        rv = {}
        if self.remote_log_storage:
            rv['remote_log_storage'] = self.remote_log_storage
        return rv


class Hdrcgi(ChainableModuleWrapperBase):
    __protobuf__ = proto.HdrcgiModule

    REQUIRED_ANYOFS = [('cgi_from_hdr', 'hdr_from_cgi')]

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()

        if self.pb.hdr_from_cgi:
            with validate('hdr_from_cgi'):
                for header_name in six.iterkeys(self.pb.hdr_from_cgi):
                    validate_header_name(header_name)

        if self.pb.cgi_from_hdr:
            with validate('cgi_from_hdr'):
                for header_name in six.itervalues(self.pb.cgi_from_hdr):
                    validate_header_name(header_name)

        if self.pb.body_scan_limit < 0:
            raise ValidationError('must be non-negative')

        super(Hdrcgi, self).validate(ctx=ctx, preceding_modules=preceding_modules, chained_modules=chained_modules)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict()
        if self.pb.hdr_from_cgi:
            table['hdr_from_cgi'] = Config(OrderedDict(sorted(self.pb.hdr_from_cgi.items())))
        if self.pb.cgi_from_hdr:
            table['cgi_from_hdr'] = Config(OrderedDict(sorted(self.pb.cgi_from_hdr.items())))
        if self.pb.body_scan_limit:
            table['body_scan_limit'] = self.pb.body_scan_limit
        return Config(table)


class RpsLimiter(ChainableModuleWrapperBase):
    __protobuf__ = proto.RpsLimiterModule

    checker = None  # type: Holder | None
    on_error = None  # type: Holder | None

    REQUIRED = ['checker']

    ALLOWED_KNOBS = {
        'disable_file': model_pb2.KnobSpec.BOOLEAN,
    }

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        new_preceding_modules = add_module(preceding_modules, self)
        with validate('checker'):
            self.checker.validate(ctx=ctx, preceding_modules=new_preceding_modules)
        if self.on_error:
            with validate('on_error'):
                if self.pb.skip_on_error:
                    raise ValidationError('must not be set together with skip_on_error')
                self.on_error.validate(ctx=ctx, preceding_modules=new_preceding_modules)
        super(RpsLimiter, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules,
                                                          chained_modules=chained_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()
        super(RpsLimiter, self).validate(ctx=ctx, preceding_modules=preceding_modules,
                                         chained_modules=chained_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        module_config = Config()
        self._add_nested_module_to_config(module_config, preceding_modules=preceding_modules)
        checker_config = self.checker.to_config(
            ctx=ctx, preceding_modules=add_module(preceding_modules, self))
        table = OrderedDict([
            ('skip_on_error', self.pb.skip_on_error),
        ])
        disable_file = self.get('disable_file')
        if disable_file.value:
            table['disable_file'] = disable_file.to_config(ctx)
        if self.pb.record_name:
            table['namespace'] = self.pb.record_name
        if self.on_error:
            on_error_config = self.on_error.to_config(
                ctx=ctx, preceding_modules=add_module(preceding_modules, self))
            table['on_error'] = on_error_config
        table['checker'] = checker_config
        table['module'] = module_config

        return Config(table, outlets=(module_config,), shareable=True)

    def get_branches(self):
        if self.checker:
            yield self.checker
        if self.on_error:
            yield self.on_error

    def get_named_branches(self):
        rv = {}
        if self.checker:
            rv['checker'] = self.checker
        if self.on_error:
            rv['on_error'] = self.on_error
        return rv


class RpsLimiterMacro(ChainableModuleWrapperBase, MacroBase):
    __protobuf__ = proto.RpsLimiterMacro

    REQUIRED = ['record_name']

    VERSION_0_0_1 = semantic_version.Version(u'0.0.1')
    VERSION_0_0_2 = semantic_version.Version(u'0.0.2')
    VERSION_0_0_3 = semantic_version.Version(u'0.0.3')
    VERSION_0_0_4 = semantic_version.Version(u'0.0.4')
    INITIAL_VERSION = VERSION_0_0_1
    VALID_VERSIONS = frozenset((VERSION_0_0_1, VERSION_0_0_2, VERSION_0_0_3, VERSION_0_0_4))

    DEFAULT_ATTEMPTS_NUMBER = 2
    DEFAULT_ATTEMPTS_RATE_LIMITER_LIMIT = 0.15
    DEFAULT_CONNECT_TIMEOUT = '50ms'
    DEFAULT_BACKEND_TIMEOUT = '200ms'

    def get_version(self):
        if self.pb.version:
            return semantic_version.Version(self.pb.version)
        return self.INITIAL_VERSION

    def is_chainable(self):
        return True

    def would_include_backends(self):
        return True

    def get_would_be_included_full_backend_ids(self, current_namespace_id):
        inst = rps_limiter_settings.get_installation(self.pb.installation)
        return six.itervalues(inst.get_full_backend_ids())

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        if ctx.config_type == ctx.CONFIG_TYPE_FULL:
            require_sd(preceding_modules)
        super(RpsLimiterMacro, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules,
                                                               chained_modules=chained_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()
        if self.pb.version:
            validate_version(self.pb.version, self.VALID_VERSIONS, field_name='version')

        rps_limiter_settings.validate_installation(self.pb.installation, ctx.rps_limiter_allowed_installations)
        super(RpsLimiterMacro, self).validate(ctx=ctx, preceding_modules=preceding_modules,
                                              chained_modules=chained_modules)

    def expand(self, ctx=DEFAULT_CTX, preceding_modules=()):
        holder_pbs = []

        h_pb = proto.Holder()
        headers_pb = h_pb.headers
        if self.get_version() == self.VERSION_0_0_1:
            headers_pb.create.add(key='x-rpslimiter-balancer', value=self.pb.record_name)
        entry_pb = headers_pb.create.add(key='x-rpslimiter-geo')
        f_value_pb = entry_pb.f_value
        f_value_pb.type = f_value_pb.GET_GEO
        f_value_pb.get_geo_params.name = ''
        f_value_pb.get_geo_params.default_geo = 'unknown'
        holder_pbs.append(h_pb)

        h_pb = proto.Holder()
        rps_limiter_pb = h_pb.rps_limiter
        rps_limiter_pb.skip_on_error = True
        if self.get_version() > self.VERSION_0_0_1:
            rps_limiter_pb.record_name = self.pb.record_name
        if self.get_version() >= self.VERSION_0_0_3:
            rps_limiter_pb.disable_file = './controls/rps_limiter_disabled'

        checker_pb = rps_limiter_pb.checker

        checker_balancer2_pb = checker_pb.balancer2
        checker_balancer2_pb.attempts = 1
        checker_balancer2_pb.rr.SetInParent()
        by_name_policy_pb = checker_balancer2_pb.balancing_policy.by_name_policy
        f_name_pb = by_name_policy_pb.f_name
        f_name_pb.type = f_name_pb.GET_GEO
        f_name_pb.get_geo_params.name = 'rpslimiter_'
        f_name_pb.get_geo_params.default_geo = 'random'
        by_name_policy_pb.balancing_policy.unique_policy.SetInParent()

        installation = rps_limiter_settings.get_installation(self.pb.installation)

        for location in installation.locations:
            backend_pb = checker_balancer2_pb.backends.add()
            backend_pb.name = 'rpslimiter_{}'.format(location)
            backend_pb.weight = 1

            report_pb = backend_pb.nested.report
            report_pb.uuid = 'rpslimiter-{}'.format(location)
            report_pb.ranges = Report.DEFAULT_RANGES_ALIAS

            if self.get_version() >= self.VERSION_0_0_3:
                report_pb.outgoing_codes.append('429')

            backend_balancer2_pb = report_pb.nested.balancer2
            backend_balancer2_pb.attempts = self.DEFAULT_ATTEMPTS_NUMBER
            backend_balancer2_pb.attempts_rate_limiter.limit = self.DEFAULT_ATTEMPTS_RATE_LIMITER_LIMIT
            backend_balancer2_pb.rr.SetInParent()

            full_rps_limiter_backend_id = installation.get_full_backend_id(location)

            gpb_pb = backend_balancer2_pb.generated_proxy_backends
            gpb_pb.include_backends.type = proto.IncludeBackends.BY_ID
            gpb_pb.include_backends.ids.append('/'.join(full_rps_limiter_backend_id))
            proxy_options_pb = gpb_pb.proxy_options
            proxy_options_pb.connect_timeout = self.pb.connect_timeout or self.DEFAULT_CONNECT_TIMEOUT
            proxy_options_pb.backend_timeout = self.pb.backend_timeout or self.DEFAULT_BACKEND_TIMEOUT

        if self.get_version() >= self.VERSION_0_0_4:
            # see https://st.yandex-team.ru/AWACS-1296
            dummy_locations = {u'man', u'sas', u'vla', u'iva', u'myt'} - set(installation.locations)
            # XXX romanovich@ py2 compat:
            dummy_locations = sorted(dummy_locations, key=[u'myt', u'iva', u'man', u'sas', u'vla'].index)
            checker_backends_pb = checker_balancer2_pb.backends[:]
            for location in dummy_locations:
                backend_pb = checker_balancer2_pb.backends.add()
                backend_pb.name = 'rpslimiter_{}'.format(location)
                backend_pb.weight = 1

                backend_balancer2_pb = backend_pb.nested.balancer2
                backend_balancer2_pb.attempts = self.DEFAULT_ATTEMPTS_NUMBER
                backend_balancer2_pb.rr.SetInParent()

                for checker_backend_pb in checker_backends_pb:
                    backend_pb = backend_balancer2_pb.backends.add()
                    backend_pb.MergeFrom(checker_backend_pb)
                    backend_pb.nested.report.nested.balancer2.attempts = 1

        holder_pbs.append(h_pb)
        return holder_pbs


class RateLimiter(ChainableModuleWrapperBase):
    __protobuf__ = proto.RateLimiterModule

    REQUIRED = ['max_requests', 'interval']

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()
        with validate(u'max_requests'):
            if self.pb.max_requests <= 0:
                raise ValidationError('must be greater than 0')
        if self.pb.HasField('max_requests_in_queue'):
            with validate(u'max_requests_in_queue'):
                validate_range(self.pb.max_requests_in_queue.value, 0, 1000)
        with validate(u'interval'):
            validate_timedelta_range(self.pb.interval, '1s', '60m')
        super(RateLimiter, self).validate(ctx=ctx, preceding_modules=preceding_modules,
                                          chained_modules=chained_modules)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict([
            ('max_requests', self.pb.max_requests),
            ('interval', self.pb.interval),
        ])
        if self.pb.HasField('max_requests_in_queue'):
            table['max_requests_in_queue'] = self.pb.max_requests_in_queue.value
        return Config(table)


class Cryprox(ChainableModuleWrapperBase):
    __protobuf__ = proto.CryproxModule

    MODULE_NAME = 'cryprox'

    cryprox_backend = None  # type: Holder | None
    use_cryprox_matcher = None  # type: Matcher | None

    REQUIRED = ['partner_token', 'use_cryprox_matcher', 'secrets_file',
                'disable_file', 'cryprox_backend']

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        new_preceding_modules = add_module(preceding_modules, self)
        with validate('cryprox_backend'):
            self.cryprox_backend.validate(ctx=ctx, preceding_modules=new_preceding_modules)
        with validate('use_cryprox_matcher'):
            self.use_cryprox_matcher.validate(ctx=ctx, preceding_modules=new_preceding_modules)
        super(Cryprox, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules,
                                                       chained_modules=chained_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_raise_if_blacklisted(ctx)
        self.auto_validate_required()
        super(Cryprox, self).validate(ctx=ctx, preceding_modules=preceding_modules,
                                      chained_modules=chained_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        new_preceding_modules = add_module(preceding_modules, self)

        service_backend_config = Config()
        self._add_nested_module_to_config(service_backend_config, preceding_modules=new_preceding_modules)

        cryprox_backend_config = self.cryprox_backend.to_config(ctx=ctx, preceding_modules=new_preceding_modules)

        table = OrderedDict([
            ('partner_token', self.get('partner_token').to_config(ctx)),
            ('use_cryprox_matcher', self.use_cryprox_matcher.to_config(ctx, preceding_modules=new_preceding_modules)),
            ('secrets_file', self.pb.secrets_file),
            ('disable_file', self.get('disable_file').to_config(ctx)),
            ('cryprox_backend', cryprox_backend_config),
            ('service_backend', service_backend_config),
        ])

        return Config(table, outlets=(service_backend_config,), shareable=True)

    def get_branches(self):
        if self.cryprox_backend:
            yield self.cryprox_backend

    def get_named_branches(self):
        rv = {}
        if self.cryprox_backend:
            rv['cryprox_backend'] = self.cryprox_backend
        return rv


class ResponseMatcherMatcherResponseHeader(ConfigWrapperBase):
    __protobuf__ = proto.ResponseMatcherModule.Matcher.ResponseHeader

    REQUIRED = ['name', 'value']

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        super(ResponseMatcherMatcherResponseHeader, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        t = OrderedDict()
        t['name'] = self.pb.name
        t['value'] = self.pb.value
        if self.pb.HasField('value_case_insensitive'):
            t['value_case_insensitive'] = self.pb.value_case_insensitive.value
        return Config(t)


class ResponseMatcherMatcherResponseCodes(ConfigWrapperBase):
    __protobuf__ = proto.ResponseMatcherModule.Matcher.ResponseCodes

    REQUIRED = ['codes']

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        with validate('codes'):
            for code in self.pb.codes:
                validate_range(code, min_=100, max_=600, exclusive_max=True)
        super(ResponseMatcherMatcherResponseCodes, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        return Config({
            'codes': Config(array=list(self.pb.codes), compact=True),
        })


class ResponseMatcherMatcher(ConfigWrapperBase):  # ¯\_(ツ)_/¯
    __protobuf__ = proto.ResponseMatcherModule.Matcher

    match_header = None  # type: ResponseMatcherMatcherResponseHeader | None
    match_response_codes = None  # type: ResponseMatcherMatcherResponseCodes | None
    match_not = None  # type: ResponseMatcherMatcher | None
    match_and = []  # type: list[ResponseMatcherMatcher]
    match_or = []  # type: list[ResponseMatcherMatcher]

    def _count_set_fields(self):
        return len([f for f in (self.match_header,
                                self.match_response_codes,
                                self.match_not,
                                self.match_and,
                                self.match_or) if f])

    def is_empty(self):
        return self._count_set_fields() == 0

    def validate(self, ctx=DEFAULT_CTX, is_nested=False, preceding_modules=()):
        self.auto_validate_required()
        self.validate_composite_fields(ctx=ctx, is_nested=is_nested, preceding_modules=preceding_modules)

    def validate_composite_fields(self, ctx=DEFAULT_CTX, is_nested=False, preceding_modules=()):
        c = self._count_set_fields()
        if is_nested and c == 0:
            raise ValidationError('either "match_header", "match_response_codes", '
                                  '"match_not", "match_and" or "match_or" must be specified')
        if c > 1:
            raise ValidationError('at most one of the "match_header", "match_response_codes", '
                                  '"match_not", "match_and" or "match_or" must be specified')

        if self.match_header:
            with validate('match_header'):
                self.match_header.validate(ctx=ctx, preceding_modules=preceding_modules)
        if self.match_response_codes:
            with validate('match_response_codes'):
                self.match_response_codes.validate(ctx=ctx, preceding_modules=preceding_modules)
        if self.match_not:
            with validate('match_not'):
                self.match_not.validate(is_nested=True, ctx=ctx, preceding_modules=preceding_modules)
        for i, matcher in enumerate(self.match_or):
            with validate('match_or[{}]'.format(i)):
                matcher.validate(is_nested=True, ctx=ctx, preceding_modules=preceding_modules)
        for i, matcher in enumerate(self.match_and):
            with validate('match_and[{}]'.format(i)):
                matcher.validate(is_nested=True, ctx=ctx, preceding_modules=preceding_modules)

        super(ResponseMatcherMatcher, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = {}
        if self.match_header:
            table['match_header'] = self.match_header.to_config(
                ctx=ctx, preceding_modules=preceding_modules)
        if self.match_response_codes:
            table['match_response_codes'] = self.match_response_codes.to_config(
                ctx=ctx, preceding_modules=preceding_modules)
        if self.match_not:
            table['match_not'] = self.match_not.to_config(ctx=ctx, preceding_modules=preceding_modules)
        if self.match_or:
            table['match_or'] = Config(array=[m.to_config(ctx=ctx, preceding_modules=preceding_modules)
                                              for m in self.match_or])
        if self.match_and:
            table['match_and'] = Config(array=[m.to_config(ctx=ctx, preceding_modules=preceding_modules)
                                               for m in self.match_and])
        return Config(table)


class ResponseMatcherOnResponseSection(ChainableModuleWrapperBase):
    __protobuf__ = proto.ResponseMatcherModule.OnResponseSection

    matcher = None  # type: Matcher | None

    REQUIRED = ['matcher']

    # since it's not a "real" module, it can't be shared
    SHAREABLE = False

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=(), key=None):
        self.auto_validate_required()
        if not preceding_modules or not isinstance(preceding_modules[-1], ResponseMatcher):
            raise ValidationError('must be a child of "response_matcher" module')
        self.require_nested(chained_modules)
        self.validate_composite_fields(ctx=ctx,
                                       preceding_modules=preceding_modules,
                                       chained_modules=chained_modules)

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        with validate('matcher'):
            self.matcher.validate(ctx=ctx, preceding_modules=add_module(preceding_modules, self))
        super(ResponseMatcherOnResponseSection, self).validate_composite_fields(ctx=ctx,
                                                                                preceding_modules=preceding_modules,
                                                                                chained_modules=chained_modules)

    def _to_params_config(self, priority, ctx=DEFAULT_CTX, preceding_modules=()):
        # we want priority and matcher go first, thus make table an ordered dict
        config = Config(OrderedDict())
        config.table['priority'] = priority
        config.extend(self.matcher.to_config(ctx=ctx, preceding_modules=add_module(preceding_modules, self)))
        return config


class ResponseMatcher(ChainableModuleWrapperBase):
    __protobuf__ = proto.ResponseMatcherModule

    __slots__ = ('on_response',)

    # on_response = OrderedDict()  # type: dict[six.text_type, ResponseMatcherOnResponseSection]
    on_response_items = []  # type: list[(six.text_type, ResponseMatcherOnResponseSection)]

    REQUIRED = ['buffer_size', 'on_response']

    PB_FIELD_TO_CLS_ATTR_MAPPING = {'on_response': 'on_response_items'}

    ALLOWED_KNOBS = {
        'disable_file': model_pb2.KnobSpec.BOOLEAN,
    }

    def wrap_composite_fields(self):
        super(ResponseMatcher, self).wrap_composite_fields()
        self.on_response = OrderedDict(self.on_response_items)

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        new_preceding_modules = add_module(preceding_modules, self)

        with validate('on_response'):
            validate_key_uniqueness(self.on_response_items)

        for key, section in six.iteritems(self.on_response):
            with validate('on_response[{}]'.format(key)):
                section.validate(ctx=ctx, preceding_modules=new_preceding_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()
        super(ResponseMatcher, self).validate(ctx=ctx, preceding_modules=preceding_modules,
                                              chained_modules=chained_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        module_config = Config()
        self._add_nested_module_to_config(module_config, preceding_modules=preceding_modules)

        preceding_modules = add_module(preceding_modules, self)
        table = OrderedDict()
        n = len(self.on_response)
        for i, (key, section) in enumerate(six.iteritems(self.on_response)):
            section_config = section.to_config(priority=n - i, ctx=ctx, preceding_modules=preceding_modules)
            table[key] = section_config
        on_response_config = Config(table, shareable=False)

        table = OrderedDict()
        if self.pb.forward_headers:
            table['forward_headers'] = self.pb.forward_headers
        table.update([
            ('buffer_size', self.pb.buffer_size),
            ('on_response', on_response_config),
            ('module', module_config),
        ])
        return Config(table, outlets=(module_config,), shareable=True)

    def get_branches(self):
        return six.itervalues(self.on_response)

    def get_named_branches(self):
        return self.on_response


class AabCookieVerify(ChainableModuleWrapperBase):
    __protobuf__ = proto.AabCookieVerifyModule

    antiadblock = None  # type: Holder | None

    REQUIRED = ['antiadblock', 'aes_key_path']

    ALLOWED_KNOBS = {
        'disable_antiadblock_file': model_pb2.KnobSpec.BOOLEAN,
    }

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        new_preceding_modules = add_module(preceding_modules, self)
        with validate('antiadblock'):
            self.antiadblock.validate(ctx=ctx, preceding_modules=new_preceding_modules)
        if self.pb.cookie:
            with validate('cookie'):
                validate_cookie_name(self.pb.cookie)
        if self.pb.cookie_lifetime:
            with validate('cookie_lifetime'):
                validate_long_timedelta(self.pb.cookie_lifetime)
        if self.pb.ip_header:
            with validate('ip_header'):
                validate_header_name(self.pb.ip_header)
        super(AabCookieVerify, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules,
                                                               chained_modules=chained_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()
        super(AabCookieVerify, self).validate(ctx=ctx, preceding_modules=preceding_modules,
                                              chained_modules=chained_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        module_config = Config()
        self._add_nested_module_to_config(module_config, preceding_modules=preceding_modules)
        antiadblock_config = self.antiadblock.to_config(
            ctx=ctx, preceding_modules=add_module(preceding_modules, self))
        table = OrderedDict([
            ('aes_key_path', self.pb.aes_key_path)
        ])
        disable_antiadblock_file = self.get('disable_antiadblock_file')
        if disable_antiadblock_file.value:
            table['disable_antiadblock_file'] = disable_antiadblock_file.to_config(ctx)
        if self.pb.cookie:
            table['cookie'] = self.pb.cookie
        if self.pb.cookie_lifetime:
            table['cookie_lifetime'] = self.pb.cookie_lifetime
        if self.pb.ip_header:
            table['ip_header'] = self.pb.ip_header
        table['antiadblock'] = antiadblock_config
        table['default'] = module_config

        return Config(table, outlets=(module_config,), shareable=True)


class Cachalot(ChainableModuleWrapperBase):
    __protobuf__ = proto.CachalotModule

    cacher = None  # type: Holder | None

    REQUIRED = ['collection', 'cacher']

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        new_preceding_modules = add_module(preceding_modules, self)
        with validate('cacher'):
            self.cacher.validate(ctx=ctx, preceding_modules=new_preceding_modules)
        super(Cachalot, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules,
                                                        chained_modules=chained_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()
        super(Cachalot, self).validate(ctx=ctx, preceding_modules=preceding_modules,
                                       chained_modules=chained_modules)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        new_preceding_modules = add_module(preceding_modules, self)
        cacher_config = self.cacher.to_config(ctx=ctx, preceding_modules=new_preceding_modules)
        table = OrderedDict([
            ('collection', self.pb.collection),
            ('cacher', cacher_config),
        ])
        return Config(table)


class Compressor(ChainableModuleWrapperBase):
    __protobuf__ = proto.CompressorModule

    DEFAULT_ENABLE_COMPRESSION = True
    DEFAULT_ENABLE_DECOMPRESSION = False
    DEFAULT_COMPRESSION_CODECS = 'gzip,deflate,br,x-gzip,x-deflate'
    DEFAULTS = {
        'enable_compression': DEFAULT_ENABLE_COMPRESSION,
        'enable_decompression': DEFAULT_ENABLE_DECOMPRESSION
    }

    COMPRESSION_CODECS_NAMES = frozenset(('gzip', 'deflate', 'br', 'x-gzip', 'x-deflate'))

    def _compression_enabled(self):
        return self.pb.enable_compression.value if self.pb.HasField('enable_compression') else self.DEFAULT_ENABLE_COMPRESSION

    def _decompression_enabled(self):
        return self.pb.enable_decompression.value if self.pb.HasField('enable_decompression') else self.DEFAULT_ENABLE_DECOMPRESSION

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()
        if not self._compression_enabled() and not self._decompression_enabled():
            raise ValidationError('one of "enable_compression" and "enable_decompression" must be enabled')
        with validate('compression_codecs'):
            validate_item_uniqueness(self.pb.compression_codecs)
        for i, compression_codec in enumerate(self.pb.compression_codecs):
            if compression_codec not in self.COMPRESSION_CODECS_NAMES:
                raise ValidationError('unknown compression codec name: "{}"'.format(compression_codec),
                                      field_name='compression_codecs[{}]'.format(i))
        super(Compressor, self).validate(ctx=ctx, preceding_modules=preceding_modules, chained_modules=chained_modules)

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict([
            ('enable_compression', self._compression_enabled()),
            ('enable_decompression', self._decompression_enabled())
        ])
        table['compression_codecs'] = ','.join(self.pb.compression_codecs) if self.pb.compression_codecs else self.DEFAULT_COMPRESSION_CODECS
        return Config(table)


class ActiveCheckReply(ModuleWrapperBase):
    __protobuf__ = proto.ActiveCheckReplyModule

    REQUIRED = ['default_weight']

    ALLOWED_KNOBS = {
        'weight_file': model_pb2.KnobSpec.INTEGER,
    }
    DEFAULT_USE_HEADER = False
    DEFAULT_USE_BODY = True
    DEFAULTS = {
        'use_header': DEFAULT_USE_HEADER,
        'use_body': DEFAULT_USE_BODY,
    }

    def _validate_weight_file(self, ctx):
        self.validate_knob('weight_file', ctx=ctx)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        with validate('default_weight'):
            validate_range(self.pb.default_weight, 1, 1000)
        self._validate_weight_file(ctx=ctx)
        if (self.pb.HasField('use_header') and not self.pb.use_header.value and
                self.pb.HasField('use_body') and not self.pb.use_body.value):
            raise ValidationError('at least one of use_header or use_body must be set')
        super(ActiveCheckReply, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def to_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        table = OrderedDict([
            ('default_weight', self.pb.default_weight),
            ('use_header', self.pb.use_header.value if self.pb.HasField('use_header') else self.DEFAULT_USE_HEADER),
            ('use_body', self.pb.use_body.value if self.pb.HasField('use_body') else self.DEFAULT_USE_BODY),
            ('use_dynamic_weight', False),  # https://st.yandex-team.ru/SWAT-6615
        ])
        weight_file = self.get('weight_file')
        if weight_file.value:
            table['weight_file'] = weight_file.to_config(ctx)
        if self.pb.HasField('zero_weight_at_shutdown'):
            table['zero_weight_at_shutdown'] = self.pb.zero_weight_at_shutdown.value
        return Config(table)


class FlagsGetter(ChainableModuleWrapperBase):
    __protobuf__ = proto.FlagsGetterModule

    flags = None  # type: Holder | None

    REQUIRED = ['service_name', 'flags']

    ALLOWED_KNOBS = {
        'file_switch': model_pb2.KnobSpec.BOOLEAN,
    }

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        new_preceding_modules = add_module(preceding_modules, self)
        with validate('flags'):
            self.flags.validate(ctx=ctx, preceding_modules=new_preceding_modules)
        super(FlagsGetter, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules,
                                                           chained_modules=chained_modules)

    def _validate_file_switch(self, ctx):
        self.validate_knob('file_switch', ctx=ctx)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        self.auto_validate_required()
        self._validate_file_switch(ctx=ctx)
        super(FlagsGetter, self).validate(ctx=ctx, preceding_modules=preceding_modules,
                                          chained_modules=chained_modules)

    def get_branches(self):
        if self.flags:
            yield self.flags

    def get_named_branches(self):
        rv = {}
        if self.flags:
            rv['flags'] = self.flags
        return rv

    def _to_params_config(self, ctx=DEFAULT_CTX, preceding_modules=()):
        new_preceding_modules = add_module(preceding_modules, self)
        flags_config = self.flags.to_config(ctx=ctx, preceding_modules=new_preceding_modules)
        table = OrderedDict([
            ('service_name', self.pb.service_name),
        ])
        if self.pb.flags_path:
            table['flags_path'] = self.pb.flags_path
        if self.pb.flags_host:
            table['flags_host'] = self.pb.flags_host
        file_switch = self.get('file_switch')
        if file_switch.value:
            table['file_switch'] = file_switch.to_config(ctx)
        table['flags'] = flags_config
        return Config(table)


class L7FastUpstreamMacroOuterBalancingOptions(ConfigWrapperBase):
    __protobuf__ = proto.L7FastUpstreamMacro.OuterBalancingOptions

    MIN_ATTEMPTS = 1
    MAX_ATTEMPTS = 10

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        with validate('attempts'):
            validate_range(value=self.pb.attempts,
                           min_=self.MIN_ATTEMPTS,
                           max_=self.MAX_ATTEMPTS)
        super(L7FastUpstreamMacroOuterBalancingOptions, self).validate(ctx=ctx, preceding_modules=preceding_modules)


class L7FastUpstreamMacroInnerBalancingOptions(ConfigWrapperBase):
    __protobuf__ = proto.L7FastUpstreamMacro.InnerBalancingOptions

    MIN_ATTEMPTS = 1
    MAX_ATTEMPTS = 10
    MIN_CONNECT_TIMEOUT = '1ms'
    MAX_CONNECT_TIMEOUT = '5s'
    MIN_BACKEND_TIMEOUT = '10ms'
    MAX_BACKEND_TIMEOUT = '86400s'

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        with validate('attempts'):
            validate_range(value=self.pb.attempts,
                           min_=self.MIN_ATTEMPTS,
                           max_=self.MAX_ATTEMPTS)
        with validate('connect_timeout'):
            validate_timedelta(self.pb.connect_timeout)
            validate_timedelta_range(value=self.pb.connect_timeout,
                                     min_=self.MIN_CONNECT_TIMEOUT,
                                     max_=self.MAX_CONNECT_TIMEOUT)
        with validate('backend_timeout'):
            validate_timedelta(self.pb.backend_timeout)
            validate_timedelta_range(value=self.pb.backend_timeout,
                                     min_=self.MIN_BACKEND_TIMEOUT,
                                     max_=self.MAX_BACKEND_TIMEOUT)
        if self.pb.keepalive_count < 0:
            raise ValidationError('must be non-negative', 'keepalive_count')
        super(L7FastUpstreamMacroInnerBalancingOptions, self).validate(ctx=ctx, preceding_modules=preceding_modules)


class L7FastUpstreamMacroDestination(ConfigWrapperBase):
    __protobuf__ = proto.L7FastUpstreamMacro.Destination

    include_backends = None  # type: IncludeBackends | None

    REQUIRED = ['id', 'include_backends']
    ID_PATTERN = '^[a-z][a-z0-9-]*$'

    def get_would_be_included_full_backend_ids(self, current_namespace_id):
        return self.include_backends.get_included_full_backend_ids(current_namespace_id)

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        with validate('include_backends'):
            self.include_backends.validate()
            if self.include_backends.pb.type != self.include_backends.pb.BY_ID:
                raise ValidationError('must be BY_ID', 'type')

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        if not re.match(self.ID_PATTERN, self.pb.id):
            raise ValidationError('must match {}'.format(self.ID_PATTERN), 'id')
        super(L7FastUpstreamMacroDestination, self).validate(ctx=ctx, preceding_modules=preceding_modules)


class L7FastUpstreamMacro(ModuleWrapperBase, MacroBase):
    __protobuf__ = proto.L7FastUpstreamMacro

    outer_balancing_options = None  # type: L7FastUpstreamMacroOuterBalancingOptions | None
    inner_balancing_options = None  # type: L7FastUpstreamMacroInnerBalancingOptions | None
    destinations = []  # type: list[L7FastUpstreamMacroDestination]

    ID_PATTERN = '^[a-z][a-z0-9-]*$'
    REQUIRED = ['id', 'outer_balancing_options', 'inner_balancing_options', 'destinations']
    DEFAULT_WEIGHTS_FILE = './controls/traffic_control.weights'
    BACKEND_CONNECTION_ATTEMPTS = 5  # https://st.yandex-team.ru/SWAT-5529
    ATTEMPTS_RATE_LIMITER_LIMIT = 0.1
    ATTEMPTS_RATE_LIMITER_COEFF = 0.99

    def would_include_backends(self):
        return True

    def get_would_be_included_full_backend_ids(self, current_namespace_id):
        rv = set()
        for destination in self.destinations:
            rv.update(destination.get_would_be_included_full_backend_ids(current_namespace_id))
        return rv

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        new_preceding_modules = add_module(preceding_modules, self)
        with validate('outer_balancing_options'):
            self.outer_balancing_options.validate(ctx=ctx, preceding_modules=new_preceding_modules)
        with validate('inner_balancing_options'):
            self.inner_balancing_options.validate(ctx=ctx, preceding_modules=new_preceding_modules)
        for i, destination in enumerate(self.destinations):
            with validate('destinations[{}]'.format(i)):
                destination.validate(ctx=ctx, preceding_modules=new_preceding_modules)
        super(L7FastUpstreamMacro, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        if not re.match(self.ID_PATTERN, self.pb.id):
            raise ValidationError('must match {}'.format(self.ID_PATTERN), 'id')
        super(L7FastUpstreamMacro, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def _fill_balancer2_with_destinations(self, balancer2_pb):
        """
        :type balancer2_pb: awacs.proto.modules_pb2.Balancer2Module
        """
        balancer2_pb.balancing_policy.retry_policy.balancing_policy.unique_policy.SetInParent()
        balancer2_pb.attempts = self.outer_balancing_options.pb.attempts
        balancer2_pb.rr.weights_file = self.DEFAULT_WEIGHTS_FILE
        self._configure_5xx_retries_for_outer_balancer2(balancer2_pb)

        for destination in self.destinations:
            destination_full_id = '{}_{}'.format(self.pb.id, destination.pb.id)
            backend_pb = balancer2_pb.backends.add(name=destination.pb.id, weight=1)
            report_pb = backend_pb.nested.report
            report_pb.uuid = destination_full_id
            report_pb.ranges = 'default'

            destination_balancer2_pb = report_pb.nested.balancer2
            destination_balancer2_pb.balancing_policy.retry_policy.balancing_policy.unique_policy.SetInParent()
            destination_balancer2_pb.attempts = self.inner_balancing_options.pb.attempts
            destination_balancer2_pb.connection_attempts = self.BACKEND_CONNECTION_ATTEMPTS
            destination_balancer2_pb.attempts_rate_limiter.limit = self.ATTEMPTS_RATE_LIMITER_LIMIT
            destination_balancer2_pb.attempts_rate_limiter.coeff = self.ATTEMPTS_RATE_LIMITER_COEFF

            gpb_pb = destination_balancer2_pb.generated_proxy_backends
            gpb_pb.proxy_options.connect_timeout = self.inner_balancing_options.pb.connect_timeout
            gpb_pb.proxy_options.backend_timeout = self.inner_balancing_options.pb.backend_timeout
            gpb_pb.proxy_options.keepalive_count = self.inner_balancing_options.pb.keepalive_count
            gpb_pb.proxy_options.fail_on_5xx.CopyFrom(self.inner_balancing_options.pb.fail_on_5xx)
            gpb_pb.include_backends.CopyFrom(destination.pb.include_backends)

            self._configure_5xx_retries_for_inner_balancer2(destination_balancer2_pb)

            yield destination_full_id, destination_balancer2_pb

    def _configure_5xx_retries_for_outer_balancer2(self, balancer2_pb):
        """
        :type balancer2_pb: modules_pb2.Balancer2
        """
        on_5xx = self.inner_balancing_options.pb.on_5xx
        if on_5xx == self.inner_balancing_options.pb.ON_5XX_NONE:
            return
        elif on_5xx == self.inner_balancing_options.pb.PROXY_FIRST_5XX:
            pass
        elif on_5xx == self.inner_balancing_options.pb.RST_LAST_5XX:
            balancer2_pb.status_code_blacklist.append('5xx')
        elif on_5xx == self.inner_balancing_options.pb.PROXY_LAST_5XX:
            balancer2_pb.return_last_5xx = True
            balancer2_pb.status_code_blacklist.append('5xx')
        else:
            raise AssertionError()

    def _configure_5xx_retries_for_inner_balancer2(self, balancer2_pb):
        """
        :type balancer2_pb: modules_pb2.Balancer2
        """
        on_5xx = self.inner_balancing_options.pb.on_5xx
        if on_5xx == self.inner_balancing_options.pb.ON_5XX_NONE:
            return
        elif on_5xx == self.inner_balancing_options.pb.PROXY_FIRST_5XX:
            balancer2_pb.generated_proxy_backends.proxy_options.fail_on_5xx.value = False
        elif on_5xx == self.inner_balancing_options.pb.RST_LAST_5XX:
            balancer2_pb.status_code_blacklist.append('5xx')
            balancer2_pb.generated_proxy_backends.proxy_options.fail_on_5xx.value = False
        elif on_5xx == self.inner_balancing_options.pb.PROXY_LAST_5XX:
            balancer2_pb.return_last_5xx = True
            balancer2_pb.status_code_blacklist.append('5xx')
            balancer2_pb.generated_proxy_backends.proxy_options.fail_on_5xx.value = False
        else:
            raise AssertionError()

    def expand(self, ctx=DEFAULT_CTX, preceding_modules=()):
        holder_pbs = []

        h_pb = proto.Holder()
        holder_pbs.append(h_pb)
        h_pb.report.uuid = self.pb.id
        h_pb.report.ranges = Report.DEFAULT_RANGES_ALIAS

        h_pb = proto.Holder()
        holder_pbs.append(h_pb)
        decision_balancer2_pb = h_pb.balancer2
        decision_balancer2_pb.rr.weights_file = './controls/dynamic_balancing_switch'
        decision_balancer2_pb.attempts = 1
        self._configure_5xx_retries_for_outer_balancer2(decision_balancer2_pb)

        dynamic_balancing_enabled_h_pb = decision_balancer2_pb.backends.add(
            name='dynamic_balancing_enabled', weight=-1).nested
        for destination_full_id, destination_balancer2_pb in \
                self._fill_balancer2_with_destinations(dynamic_balancing_enabled_h_pb.balancer2):
            destination_balancer2_pb.dynamic.max_pessimized_share = .1
            destination_balancer2_pb.dynamic.min_pessimization_coeff = .1
            destination_balancer2_pb.dynamic.weight_increase_step = .1
            destination_balancer2_pb.dynamic.history_interval = '10s'
            destination_balancer2_pb.dynamic.backends_name = destination_full_id

        dynamic_balancing_disabled_h_pb = decision_balancer2_pb.backends.add(
            name='dynamic_balancing_disabled', weight=1).nested
        for _, destination_balancer2_pb in \
                self._fill_balancer2_with_destinations(dynamic_balancing_disabled_h_pb.balancer2):
            destination_balancer2_pb.rr.SetInParent()

        return holder_pbs


class L7FastSitemapUpstreamMacroS3Options(ConfigWrapperBase):
    __protobuf__ = proto.L7FastSitemapUpstreamMacro.S3Options

    REQUIRED = ['bucket_name']

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()


class L7FastSitemapUpstreamMacro(ModuleWrapperBase, MacroBase):
    __protobuf__ = proto.L7FastSitemapUpstreamMacro

    s3 = None  # type: L7FastSitemapUpstreamMacroS3Options | None

    ID_PATTERN = '^[a-z][a-z0-9-]*_sitemap$'
    REQUIRED = ['id', 's3']

    S3_DOMAIN = 's3.yandex.net'
    S3_SHARED_UUID = 's3'

    def would_include_backends(self):
        return False

    def validate_composite_fields(self, ctx=DEFAULT_CTX, preceding_modules=(), chained_modules=()):
        new_preceding_modules = add_module(preceding_modules, self)
        with validate('s3'):
            self.s3.validate(ctx=ctx, preceding_modules=new_preceding_modules)
        super(L7FastSitemapUpstreamMacro, self).validate_composite_fields(ctx=ctx, preceding_modules=preceding_modules)

    def validate(self, ctx=DEFAULT_CTX, preceding_modules=()):
        self.auto_validate_required()
        if not re.match(self.ID_PATTERN, self.pb.id):
            raise ValidationError('must match {}'.format(self.ID_PATTERN), 'id')
        super(L7FastSitemapUpstreamMacro, self).validate(ctx=ctx, preceding_modules=preceding_modules)

    def expand(self, ctx=DEFAULT_CTX, preceding_modules=()):
        holder_pbs = []

        h_pb = proto.Holder()
        h_pb.report.uuid = self.pb.id
        h_pb.report.ranges = Report.DEFAULT_RANGES_ALIAS
        holder_pbs.append(h_pb)

        h_pb = proto.Holder()
        headers_pb = h_pb.headers
        headers_pb.create.add(key='Host', value='{}.{}'.format(self.s3.pb.bucket_name, self.S3_DOMAIN))
        holder_pbs.append(h_pb)

        h_pb = proto.Holder()
        h_pb.shared.uuid = self.S3_SHARED_UUID
        holder_pbs.append(h_pb)

        return holder_pbs


class Knob(base.WrapperBase):
    __protobuf__ = proto.KnobRef
    __slots__ = ('id', 'optional')

    def validate(self):
        if '/' in self.id:
            raise ValidationError('specifying namespace id in knob id is not allowed')

    def update_pb(self, pb):
        super(Knob, self).update_pb(pb)
        self.id = pb.id
        self.optional = pb.optional


class Cert(base.WrapperBase):
    __protobuf__ = proto.CertRef
    __slots__ = ('id',)

    def validate(self):
        if '/' in self.id:
            raise ValidationError('specifying namespace id in certificate id is not allowed')

    def update_pb(self, pb):
        super(Cert, self).update_pb(pb)
        self.id = pb.id


class Call(base.WrapperBase):
    __protobuf__ = proto.Call

    __slots__ = ('func_name', 'func_params')

    def update_pb(self, pb):
        super(Call, self).update_pb(pb)
        self.func_name = proto.Call.FuncType.Name(self.pb.type).lower()
        self.func_params = None
        params_name = pb.WhichOneof('params')
        if params_name:
            setattr(self, params_name, base.wrap(getattr(pb, params_name)))
            self.func_params = getattr(self, params_name, None)

    def validate(self):
        if not self.func_params:
            raise errors.ValidationError('params for {} function are not specified'.format(self.func_name))
        args = self.func_params.get_args()
        validate_args(self.func_params.__func__, args)

    def to_config(self, ctx=DEFAULT_CTX):
        args = []
        for arg in self.func_params.get_args():
            if arg.is_func():
                args.append(arg.value.to_config(ctx=ctx))
            else:
                args.append(arg.value)
        return ConfigCall(self.func_name, args)

    def __str__(self):
        return self.to_config().to_lua()

    def __hash__(self):
        return hash(str(self))

    def __eq__(self, other):
        if not isinstance(other, Call):
            return False
        return self.pb == other.pb


class ParamsWrapper(base.WrapperBase):
    __func__ = None

    def get_args(self):
        args = []
        for arg_name in itertools.chain(six.iterkeys(self.__func__.args),
                                        six.iterkeys(self.__func__.optional_args)):
            arg_value = self.get(arg_name)
            if arg_value.is_func():
                args.append(arg_value)
            else:
                field_desc = self.pb.DESCRIPTOR.fields_by_name[arg_name]
                if field_desc.message_type and field_desc.message_type.name in ('StringValue',
                                                                                'BoolValue',
                                                                                'FloatValue',
                                                                                'Int32Value'):
                    if self.pb.HasField(arg_name):
                        args.append(Value(Value.VALUE, arg_value.value.value))
                else:
                    args.append(arg_value)
        return args


class GetIpByIprouteParams(ParamsWrapper):
    __protobuf__ = proto.Call.GetIpByIprouteParams
    __func__ = defs.get_ip_by_iproute


class GetStrVarParams(ParamsWrapper):
    __protobuf__ = proto.Call.GetStrVarParams
    __func__ = defs.get_str_var


class GetStrEnvVarParams(ParamsWrapper):
    __protobuf__ = proto.Call.GetStrEnvVarParams
    __func__ = defs.get_str_env_var


class GetIntVarParams(ParamsWrapper):
    __protobuf__ = proto.Call.GetIntVarParams
    __func__ = defs.get_int_var


class GetPortVarParams(ParamsWrapper):
    __protobuf__ = proto.Call.GetPortVarParams
    __func__ = defs.get_port_var


class GetLogPathParams(ParamsWrapper):
    __protobuf__ = proto.Call.GetLogPathParams
    __func__ = defs.get_log_path


class CountBackendsParams(ParamsWrapper):
    __protobuf__ = proto.Call.CountBackendsParams
    __func__ = defs.count_backends


class CountBackendsSdParams(ParamsWrapper):
    __protobuf__ = proto.Call.CountBackendsSdParams
    __func__ = defs.count_backends_sd


class GetPublicCertPath(ParamsWrapper):
    __protobuf__ = proto.Call.GetPublicCertPathParams
    __func__ = defs.get_public_cert_path


class GetPrivateCertPath(ParamsWrapper):
    __protobuf__ = proto.Call.GetPrivateCertPathParams
    __func__ = defs.get_private_cert_path


class GetGeo(ParamsWrapper):
    __protobuf__ = proto.Call.GetGeoParams
    __func__ = defs.get_geo


class PrefixWithDc(ParamsWrapper):
    __protobuf__ = proto.Call.PrefixWithDcParams
    __func__ = defs.prefix_with_dc


class SuffixWithDc(ParamsWrapper):
    __protobuf__ = proto.Call.SuffixWithDcParams
    __func__ = defs.suffix_with_dc


class GetCaCertPath(ParamsWrapper):
    __protobuf__ = proto.Call.GetCaCertPathParams
    __func__ = defs.get_ca_cert_path


class GetRandomTimedelta(ParamsWrapper):
    __protobuf__ = proto.Call.GetRandomTimedeltaParams
    __func__ = defs.get_random_timedelta


class GetTotalWeightPercentParams(ParamsWrapper):
    __protobuf__ = proto.Call.GetTotalWeightPercentParams
    __func__ = defs.get_total_weight_percent


class GetWorkersParams(ParamsWrapper):
    __protobuf__ = proto.Call.GetWorkersParams
    __func__ = defs.get_workers


class GetItsControlPathParams(ParamsWrapper):
    __protobuf__ = proto.Call.GetItsControlPathParams
    __func__ = defs.get_its_control_path


DEFAULT_FILE_TO_ITS_RUCHKA_ID = {
    Antirobot.DEFAULT_FILE_SWITCH: 'balancer_antirobot_module_switch',
    Antirobot.DEFAULT_NO_CUT_REQUEST_FILE: 'no_cut_request_file',
    Main.DEFAULT_RESET_DNS_CACHE_FILE: 'reset_dns_cache',
    Http.DEFAULT_NO_KEEPALIVE_FILE: 'balancer_disable_keepalive',
    WatermarkPolicy.DEFAULT_PARAMS_FILE: 'common_watermark_policy_params_file',
    ExpGetter.DEFAULT_FILE_SWITCH: 'balancer_expgetter_switch',
    SslSniContext.DEFAULT_OCSP_FILE_SWITCH: 'balancer_ocsp_switch',
    SslSni.DEFAULT_HTTP2_ALPN_FILE: 'http2_request_rate',
    SlbPingMacro.DEFAULT_SLB_CHECK_WEIGHTS_FILE: 'service_balancer_off',
    GeobaseMacro.DEFAULT_FILE_SWITCH: 'balancer_geolib_switch',
    RpcRewrite.DEFAULT_FILE_SWITCH: 'balancer_disable_rpcrewrite_module',
    RequestReplier.DEFAULT_RATE_FILE: 'common_request_replier_rate',
    CookieHasher.DEFAULT_FILE_SWITCH: 'balancer_disable_cookie_hasher',
    Pinger.DEFAULT_ENABLE_TCP_CHECK_FILE: 'balancer_tcp_check_on',
    Pinger.DEFAULT_SWITCH_OFF_FILE: 'service_balancer_off',
    RemoteLog.DEFAULT_NO_REMOTE_LOG_FILE: 'balancer_remote_log_switch',
}

DEFAULT_KNOB_TYPES = {}


def fill_default_knob_types():
    for cls in six.itervalues(base.REGISTRY):
        if hasattr(cls, 'DEFAULT_KNOB_IDS') and hasattr(cls, 'ALLOWED_KNOBS'):
            for field_name, default_knob_id in six.iteritems(cls.DEFAULT_KNOB_IDS):
                default_knob_type = cls.ALLOWED_KNOBS[field_name]
                if default_knob_id not in DEFAULT_KNOB_TYPES:
                    DEFAULT_KNOB_TYPES[default_knob_id] = default_knob_type
                else:
                    assert DEFAULT_KNOB_TYPES[default_knob_id] == default_knob_type


fill_default_knob_types()
assert len(DEFAULT_KNOB_TYPES.keys()) == len(set(DEFAULT_FILE_TO_ITS_RUCHKA_ID.values()))

from .l7upstreammacro import L7UpstreamMacro  # noqa
