import attr
import six
from typing import Iterable, Set, Tuple, Dict

from awacs.lib.strutils import quote_join_sorted
from .errors import ValidationError


@attr.s(slots=True, weakref_slot=True, frozen=True)
class RpsLimiterInstallation(object):
    namespace_id = attr.ib(type=six.text_type)
    backend_id_template = attr.ib(type=six.text_type)
    locations = attr.ib(type=Iterable[six.text_type])
    deprecated = attr.ib(default=False, type=bool)

    def get_full_backend_id(self, location):
        """
        :type location: six.text_type
        :rtype: Tuple[six.text_type, six.text_type]
        """
        return self.namespace_id, self.backend_id_template.format(location)

    def get_full_backend_ids(self):
        """
        :rtype: Dict[six.text_type, Tuple[six.text_type, six.text_type]]
        """
        return {location: self.get_full_backend_id(location) for location in self.locations}


_available_installations = {
    u'COMMON': RpsLimiterInstallation(
        namespace_id=u'common-rpslimiter',
        backend_id_template=u'rpslimiter-serval-{}-sd',
        locations=(u'man', u'sas', u'vla'),
        deprecated=True,
    ),
    u'COMMON_V2': RpsLimiterInstallation(
        namespace_id=u'common-rpslimiter-balancer',
        backend_id_template=u'rpslimiter-balancer-{}-sd',
        locations=(u'man', u'sas', u'vla'),
    ),
    u'MAPS_FRONT': RpsLimiterInstallation(
        namespace_id=u'common-rpslimiter-maps-front',
        backend_id_template=u'rpslimiter-maps-front-{}-sd',
        locations=(u'man', u'sas', u'vla'),
    ),
    u'PDB': RpsLimiterInstallation(
        namespace_id=u'common-rpslimiter-pdb',
        backend_id_template=u'rpslimiter-pdb-{}-sd',
        locations=(u'man', u'sas', u'vla'),
    ),
    u'TRANSLATE': RpsLimiterInstallation(
        namespace_id=u'common-rpslimiter-translate',
        backend_id_template=u'rpslimiter-translate-{}-sd',
        locations=(u'man', u'sas', u'vla'),
    ),
    u'MUSIC': RpsLimiterInstallation(
        namespace_id=u'common-rpslimiter-music',
        backend_id_template=u'rpslimiter-music-{}-sd',
        locations=(u'man', u'sas', u'vla'),
    ),
    u'EDUCATION': RpsLimiterInstallation(
        namespace_id=u'common-rpslimiter-education',
        backend_id_template=u'rpslimiter-education-{}-sd',
        locations=(u'iva', u'sas', u'vla'),
    ),
    u'MEDIABILLING': RpsLimiterInstallation(
        namespace_id=u'common-rpslimiter-mediabilling',
        backend_id_template=u'rpslimiter-mediabilling-{}-sd',
        locations=(u'man', u'sas', u'myt'),
        deprecated=True,
    ),
    u'AFISHA': RpsLimiterInstallation(
        namespace_id=u'common-rpslimiter-afisha',
        backend_id_template=u'rpslimiter-afisha-{}-sd',
        locations=(u'man', u'sas', u'vla', u'myt'),
        deprecated=True,
    ),
    u'NEWS': RpsLimiterInstallation(
        namespace_id=u'common-rpslimiter-news',
        backend_id_template=u'rpslimiter-news-{}-sd',
        locations=(u'man', u'sas', u'vla'),
    ),
    u'ALICE': RpsLimiterInstallation(
        namespace_id=u'common-rpslimiter-alice',
        backend_id_template=u'rpslimiter-alice-{}-sd',
        locations=(u'man', u'sas', u'vla'),
    ),
    u'KINOPOISK': RpsLimiterInstallation(
        namespace_id=u'common-rpslimiter-kinopoisk',
        backend_id_template=u'rpslimiter-kinopoisk-{}-sd',
        locations=(u'man', u'sas', u'vla', u'iva'),
    ),
    u'MESSENGER': RpsLimiterInstallation(
        namespace_id=u'common-rpslimiter-messenger',
        backend_id_template=u'rpslimiter-messenger-{}-sd',
        locations=(u'man', u'sas', u'vla'),
    ),
    u'YABS': RpsLimiterInstallation(
        namespace_id=u'common-rpslimiter-yabs',
        backend_id_template=u'rpslimiter-yabs-{}-sd',
        locations=(u'man', u'sas', u'vla'),
    ),
    u'WEB': RpsLimiterInstallation(
        namespace_id=u'common-rpslimiter-web',
        backend_id_template=u'rpslimiter-web-{}-sd',
        locations=(u'man', u'sas', u'vla'),
    ),
    u'ZAPRAVKI': RpsLimiterInstallation(
        namespace_id=u'common-rpslimiter-zapravki',
        backend_id_template=u'rpslimiter-zapravki-{}-sd',
        locations=(u'man', u'sas', u'vla', u'iva')
    ),
    u'WEATHER': RpsLimiterInstallation(
        namespace_id=u'common-rpslimiter-weather',
        backend_id_template=u'rpslimiter-weather-{}-sd',
        locations=(u'man', u'sas', u'vla')
    ),
    u'MEDIABILLING2': RpsLimiterInstallation(
        namespace_id=u'common-rpslimiter-mediabilling',
        backend_id_template=u'rpslimiter-mediabilling-{}-sd',
        locations=(u'man', u'sas', u'vla', u'myt'),
        deprecated=True,
    ),
    u'ANSWERS': RpsLimiterInstallation(
        namespace_id=u'common-rpslimiter-answers',
        backend_id_template=u'rpslimiter-answers-{}-sd',
        locations=(u'sas', u'vla'),
    ),
    u'PLUS_EXTERNAL': RpsLimiterInstallation(
        namespace_id=u'common-rpslimiter-mediabilling',
        backend_id_template=u'rpslimiter-mediabilling-{}-sd',
        locations=(u'man', u'sas', u'vla', u'myt'),
    ),
    u'AFISHA_EXTERNAL': RpsLimiterInstallation(
        namespace_id=u'common-rpslimiter-afisha',
        backend_id_template=u'rpslimiter-afisha-{}-sd',
        locations=(u'man', u'sas', u'vla', u'myt'),
    ),
    u'PLUS_INTERNAL': RpsLimiterInstallation(
        namespace_id=u'common-rpslimiter-internal-plus',
        backend_id_template=u'rpslimiter-internal-plus-{}-sd',
        locations=(u'sas', u'vla', u'myt'),
    ),
    u'AFISHA_INTERNAL': RpsLimiterInstallation(
        namespace_id=u'common-rpslimiter-internal-afisha',
        backend_id_template=u'rpslimiter-internal-afisha-{}-sd',
        locations=(u'sas', u'vla', u'myt'),
    ),
}

_default_installation_name = u'COMMON'
_default_installation = _available_installations[_default_installation_name]

_public_installation_names = (u'COMMON', u'COMMON_V2')
_available_installation_names = set(_available_installations.keys())


def get_available_installation_names():
    """
    :rtype: Set[six.text_type]
    """
    return _available_installation_names


def get_non_public_installation_names():
    """
    :rtype: Set[six.text_type]
    """
    return _available_installation_names - set(_public_installation_names)


def get_installation(installation_name):
    """
    :type installation_name: six.text_type
    :rtype: RpsLimiterInstallation
    """
    if installation_name == u'':
        return _default_installation
    return _available_installations[installation_name]


def validate_installation(installation_name, allowed_installation_names):
    """
    :type installation_name: six.text_type
    :type allowed_installation_names: Iterable[six.text_type]
    :raises: ValidationError
    """
    if installation_name == u'':
        return
    rv = _available_installations.get(installation_name)
    if rv is None:
        available_names = quote_join_sorted(_available_installation_names)
        raise ValidationError(u'installation "{}" doesn\'t exist. Available installations: {}'
                              .format(installation_name, available_names))

    if installation_name not in _public_installation_names and installation_name not in allowed_installation_names:
        raise ValidationError((u'using installation "{}" is not allowed in this namespace. '
                               u'Please create a ticket in st/BALANCERSUPPORT if you need to use it.')
                              .format(installation_name))


__all__ = [
    u'get_installation',
    u'validate_installation',
    u'get_available_installation_names',
    u'get_non_public_installation_names',
]
