import copy

from infra.awacs.proto import modules_pb2


# Based on
# https://wiki.yandex-team.ru/security/ssl/bestpractices/ciphersuites/
# https://wiki.mozilla.org/Security/Server_Side_TLS
# https://a.yandex-team.ru/arc/trunk/arcadia/gencfg/custom_generators/balancer_gencfg/src/constants.py?rev=r8492469#L206

# order of ciphers matters - we need to preserve the same generated lua if the ciphers list didn't change
_default_ciphers = [
    u'kEECDH+AESGCM+AES128',
    u'kEECDH+AES128',
    u'kEECDH+AESGCM+AES256',
    u'kRSA+AESGCM+AES128',
    u'kRSA+AES128',
    u'RC4-SHA',
    u'DES-CBC3-SHA',
]

# order of default excluded ciphers also matters
_default_excluded_ciphers = [
    u'aNULL',
    u'eNULL',
    u'MD5',
    u'EXPORT',
    u'LOW',
    u'SEED',
    u'CAMELLIA',
    u'IDEA',
    u'PSK',
    u'SRP',
    u'SSLv2',
]

_ecdsa_ciphers = [
    u'ECDHE-ECDSA-AES128-GCM-SHA256'
]

_ciphers_without_rc4_sha = copy.copy(_default_ciphers)
_ciphers_without_rc4_sha.remove(u'RC4-SHA')
_ciphers_with_ecdsa = _ecdsa_ciphers + _default_ciphers
_ciphers_with_ecdsa_and_without_rc4_sha = copy.copy(_ciphers_with_ecdsa)
_ciphers_with_ecdsa_and_without_rc4_sha.remove(u'RC4-SHA')

_strong_ciphers = [
    # TLSv1.3 (yes, they have underscores in their names)
    u'TLS_AES_128_GCM_SHA256',
    u'TLS_CHACHA20_POLY1305_SHA256',

    # TLSv1.2
    u'ECDHE-RSA-AES128-GCM-SHA256',
    u'ECDHE-RSA-CHACHA20-POLY1305',
    u'kEECDH+AESGCM+AES128',
    u'kEECDH+AESGCM+AES256',
    u'kRSA+AESGCM+AES128',

    # TLSv1.1
    u'kEECDH+AES128',
    u'kRSA+AES128',
]

_strong_ecdsa_ciphers = _ecdsa_ciphers + [
    u'ECDHE-ECDSA-CHACHA20-POLY1305',
]

_strong_excluded_ciphers = _default_excluded_ciphers + [
    u'RC4',
    u'3DES',
    u'DSS',
]

_strong_ciphers_with_ecdsa = _strong_ecdsa_ciphers + _strong_ciphers


def make_cipher_suite(allowed_ciphers, excluded_ciphers):
    return u':'.join(allowed_ciphers + [u'!{}'.format(excluded) for excluded in excluded_ciphers])


# construct common cipher suites during import, they don't change
class CipherSuite(object):
    # legacy suites
    DEFAULT = make_cipher_suite(_default_ciphers, _default_excluded_ciphers)
    DEFAULT_WITHOUT_RC4 = make_cipher_suite(_ciphers_without_rc4_sha, _default_excluded_ciphers)
    DEFAULT_WITH_ECDSA = make_cipher_suite(_ciphers_with_ecdsa, _default_excluded_ciphers)
    DEFAULT_WITH_ECDSA_AND_WITHOUT_RC4 = make_cipher_suite(_ciphers_with_ecdsa_and_without_rc4_sha,
                                                           _default_excluded_ciphers)

    # suites for easy-mode TLS presets
    STRONG = make_cipher_suite(_strong_ciphers, _strong_excluded_ciphers)
    STRONG_WITH_ECDSA = make_cipher_suite(_strong_ciphers_with_ecdsa, _strong_excluded_ciphers)


def get_cipher_suite_for_preset(preset, enable_ecdsa_cipher):
    """
    :type preset: modules_pb2.L7Macro.HttpsSettings.TlsSettings.Preset
    :type enable_ecdsa_cipher: bool
    :rtype: six.text_type
    """
    if preset in (modules_pb2.L7Macro.HttpsSettings.TlsSettings.STRONG,
                  modules_pb2.L7Macro.HttpsSettings.TlsSettings.INTERMEDIATE):
        if enable_ecdsa_cipher:
            return CipherSuite.STRONG_WITH_ECDSA
        else:
            return CipherSuite.STRONG

    elif preset == modules_pb2.L7Macro.HttpsSettings.TlsSettings.DEFAULT:
        if enable_ecdsa_cipher:
            return CipherSuite.DEFAULT_WITH_ECDSA
        else:
            return CipherSuite.DEFAULT

    else:
        raise ValueError(u'Unknown TLS preset "{}"'.format(preset))


def get_ssl_protocols_for_preset(preset):
    """
    :type preset: modules_pb2.L7Macro.HttpsSettings.TlsSettings.Preset
    :rtype: list[six.text_type] | None
    """
    if preset == modules_pb2.L7Macro.HttpsSettings.TlsSettings.STRONG:
        return [u'tlsv1.1', u'tlsv1.2', u'tlsv1.3']
    elif preset == modules_pb2.L7Macro.HttpsSettings.TlsSettings.INTERMEDIATE:
        return [u'tlsv1', u'tlsv1.1', u'tlsv1.2', u'tlsv1.3']
    return None


__all__ = [u'CipherSuite', u'get_cipher_suite_for_preset', u'get_ssl_protocols_for_preset']
