# coding: utf-8
import string
import itertools

import six
from six.moves import map, zip

import sympy
from sympy.utilities.lambdify import lambdastr


class Filter(object):
    """
    :param expr: result of :func:`its.parser.parse`
    :type expr: :class:`boolalg.BooleanFunction`

    Пример использования::

        >>> from blinovmatcher import parse, Filter, Tag, TagType
        >>> I = lambda name: Tag(TagType.INSTANCE, name)
        >>> rule = Filter(parse('[I@a . I@b] - [I@c I@d]'))
        >>> rule.apply([I('a')])
        False
        >>> rule.apply([I('a'), I('b')])
        True
        >>> rule.apply([I('a'), I('b'), I('c')])
        False
    """

    def __init__(self, expr):
        self._source_expr = expr

        # expr может содержать символы (:class:`sympy.Symbol`) с именами,
        # содержащими знаки минуса, равно и так далее. позвать lambdify на такое
        # выражение невозможно:
        # >>> lambdify(['a-b-c'], Not(Symbol('a-b=c')))
        # Traceback (most recent call last):
        # File "<stdin>", line 1, in <module>
        #  File ".../lib/python2.7/site-packages/sympy/utilities/lambdify.py", line 312, in lambdify
        #    return eval(lstr, namespace)
        #  File "<string>", line 1
        #    lambda a-b-c: ((not (a-b=c)))
        # поэтому мы отображаем переменные expr в новые переменные с "хорошими" именами
        self._old_args_to_new_args = self._create_mapping(expr.atoms())
        self._old_args = set(six.iterkeys(self._old_args_to_new_args))
        self._new_args = set(six.itervalues(self._old_args_to_new_args))
        prepared_expr = expr.subs({
            old_arg: sympy.Symbol(new_arg) for old_arg, new_arg
            in six.iteritems(self._old_args_to_new_args)
        })
        """
        XXX does not work for some reason
        # приводим наше булево выражение в дизъюнктивную нормальную форму,
        # чтобы ускорить вычисления его значения. также следует указывать
        # simplify=True, чтобы получать минимальную ДНФ, но по причине
        # https://github.com/sympy/sympy/issues/7597 мы этого пока не делаем
        self._compiled_expr = sympy.lambdify(self._new_args, sympy.to_dnf(prepared_expr),
                                             modules='math')
        """
        self._compiled_expr = eval(lambdastr(self._new_args, sympy.to_dnf(prepared_expr)))

    @staticmethod
    def _create_mapping(args):
        # такого количества имён должно быть достаточно на все случаи жизни
        assert len(args) < 26 + 26 ** 2 + 26 ** 3
        # итератор по вначале кортежам букв (сначала кортежи из одной буквы, затем
        # двойки букв, затем тройки)
        letter_tuples_it = itertools.chain(*[
            itertools.product(string.ascii_uppercase, repeat=n) for n in (1, 2, 3)
        ])
        # итератор по склеенным кортежам
        names_it = map(''.join, letter_tuples_it)

        def filtered_names_it(names_it):
            # избегаем совпадения старых и сгенерированных имён
            arg_names = {arg.name.lower() for arg in args}
            for name in names_it:
                if name.lower() not in arg_names:
                    yield name

        return dict(zip(args, filtered_names_it(names_it)))

    def apply(self, args):
        """
        :param args: теги
        :type args: iterable[Tag]
        :returns: значение выражения
        :rtype: boolean
        """
        kwargs = dict.fromkeys(self._new_args, False)
        for arg in self._old_args & set(args):
            kwargs[self._old_args_to_new_args[arg]] = True
        return self._compiled_expr(**kwargs)
