# coding: utf-8
import functools
import re
import operator

from funcparserlib.lexer import LexerError
from funcparserlib.parser import NoParseError
from funcparserlib import lexer, parser as p
from sympy.logic.boolalg import And, Not, Or

from .tags import ALL_PREFIXES, Tag


class ExpressionSyntaxError(Exception):
    """Ошибка при разборе выражения."""
    pass


prefix_tokens = [lexer.Token('id', prefix) for prefix in ALL_PREFIXES]

_tokenizer = lexer.make_tokenizer([
    ('space', (r'[ \t\r\n]+', re.MULTILINE)),
    ('at', (r'@',)),
    ('op', (r'[\[\]\.\-]',)),
    ('id', (r'[a-zA-Z0-9_\-=\.]+',)),
])


def _tokenize(expression):
    return list(_tokenizer(expression))


# stateless helpers
getval = lambda tok: tok.value
const = lambda x: lambda _: x

# stateless parsers
maybe_space = p.skip(p.maybe(p.some(lambda tok: tok.type == 'space')))
space = p.skip(p.some(lambda tok: tok.type == 'space'))
prefix = functools.reduce(operator.or_, [p.a(t) for t in prefix_tokens]) >> getval
at = p.some(lambda tok: tok.type == 'at')
op = lambda s: p.a(lexer.Token('op', s)) >> getval

intersection_op = space + op('.') + space >> const(lambda x, y: And(x, y))
subtraction_op = space + op('-') + space >> const(lambda x, y: And(x, Not(y)))
union_op = space >> const(lambda x, y: Or(x, y))
bracketed = lambda expr: p.skip(op('[')) + maybe_space + expr + maybe_space + p.skip(op(']'))


def eval_expr(args):
    value, ops = args
    for f, x in ops:
        value = f(value, x)
    return value


def noop_eval_expr(args):
    return


class Parser(object):
    __slots__ = ('eval_expr', 'curr_prefix', '_parser')

    def __init__(self, eval_expr=noop_eval_expr):
        self.eval_expr = eval_expr
        self.curr_prefix = [None]
        self._parser = None
        self._create_parser()

    # stateful helpers
    def get_curr_prefix(self):
        return self.curr_prefix[0]

    def set_curr_prefix(self, prefix_name):
        self.curr_prefix.pop()
        self.curr_prefix.append(prefix_name)

    def create_tag(self, name):
        return Tag.from_prefix_and_name(prefix=self.get_curr_prefix(), name=name)

    def _create_parser(self):
        # stateful parsers
        name = p.some(lambda tok: tok.type == 'id') >> getval >> self.create_tag

        @p.with_forward_decls
        def primary():
            return name | bracketed(expr)

        intersection = primary + p.many(intersection_op + primary) >> self.eval_expr
        expr = intersection + p.many((subtraction_op | union_op) + intersection) >> self.eval_expr

        @p.with_forward_decls
        def prefixed_primary():
            return (
                # once we met prefix, we descend into the "unprefixed" expr parser, which
                # does not allow prefixes
                (p.skip(prefix + p.skip(at) >> self.set_curr_prefix) + (name | bracketed(expr))) |
                # if we didn't meet prefix, continue parsing using `prefixed_expr` parser
                bracketed(prefixed_expr)
            )

        prefixed_intersection = prefixed_primary + p.many(intersection_op + prefixed_primary) >> self.eval_expr
        prefixed_expr = prefixed_intersection + p.many((subtraction_op | union_op) + prefixed_intersection) >> self.eval_expr
        self._parser = maybe_space + prefixed_expr + maybe_space + p.skip(p.finished)

    def parse(self, expression):
        try:
            tokens = _tokenize(expression)
            return self._parser.parse(tokens)
        except (NoParseError, LexerError) as e:
            raise ExpressionSyntaxError(e)
        finally:
            self.curr_prefix = [None]


def parse(expression):
    """
    :param str expression: выражение на языке Блинова
    :raises: ExpressionSyntaxError
    :returns: sympy.logic.boolalg.BooleanFunction
    """
    return Parser(eval_expr).parse(expression)


def validate(expression):
    """
    :param str expression: выражение на языке Блинова
    :raises: ExpressionSyntaxError
    """
    return Parser().parse(expression)
