# coding: utf-8
import sympy

from .tags import Tag, TagType


class Resolver(object):
    def __init__(self, tag_resolvers):
        """
        :param tag_resolvers:
            словарь, ключами которого являются поля blinovmatcher.TagType, а значениями —
            резолверы тегов. Резолвер — функция, принимающая на вход имя тега и возвращающая
            некоторое множество (будь то множество хостов или инстансов).
        """
        if callable(tag_resolvers):
            # maintain backwards compatibility
            self._tag_resolvers = {TagType.INSTANCE: tag_resolvers}
        else:
            self._tag_resolvers = tag_resolvers

    def resolve_tag(self, tag):
        """
        :type tag: blinovmatcher.Tag
        :rtype: set
        """
        if tag.type not in self._tag_resolvers:
            raise ValueError('Resolver for {} tags is not specified'.format(tag.type))
        tag_resolver = self._tag_resolvers[tag.type]
        return tag_resolver(tag.name)

    def resolve_or(self, ast):
        """
        :type ast: sympy.logic.boolalg.Or
        :rtype: set
        """
        rv = set()
        for arg in ast.args:
            rv |= self._resolve(arg)
        return rv

    def resolve_and(self, ast):
        """
        :type ast: sympy.logic.boolalg.And
        :rtype: set
        """
        plus_args = []
        minus_args = []
        for arg in ast.args:
            if isinstance(arg, sympy.Not):
                minus_args.append(arg.args[0])
            else:
                plus_args.append(arg)
        plus_args = iter(plus_args)
        rv = set(self._resolve(next(plus_args)))  # important: do not change, clone resolved set
        for arg in plus_args:
            rv &= self._resolve(arg)
        for arg in minus_args:
            rv -= self._resolve(arg)
        return rv

    def _resolve(self, ast):
        if isinstance(ast, sympy.Or):
            rv = self.resolve_or(ast)
        elif isinstance(ast, sympy.And):
            rv = self.resolve_and(ast)
        elif isinstance(ast, Tag):
            rv = self.resolve_tag(ast)
        else:
            raise ValueError('Unresolvable AST ({}): {!r}'.format(type(ast), ast))
        return rv

    def resolve(self, ast):
        dnf_ast = sympy.to_dnf(ast)
        return self._resolve(dnf_ast)
