# coding: utf-8
import enum
import six
import sympy


class TagType(enum.Enum):
    HOSTGROUP = 0
    HOST = 1
    SHARDTAG = 2
    SHARD = 3
    CONF = 4
    INSTANCE = 5
    DC = 6
    LINE = 7
    FEDERATED = 8


class Tag(sympy.Symbol):
    __slots__ = ['name', 'type']

    def __new__(cls, type, name, **assumptions):
        # __xnew__ does not cache created symbols
        obj = super(Tag, cls).__xnew__(cls, name, **assumptions)
        if type not in TagType:
            raise ValueError('Unknown tag type: {!r}'.format(type))
        obj.type = type
        return obj

    @classmethod
    def from_prefix_and_name(cls, prefix, name):
        if prefix not in TAG_TYPE_BY_PREFIXES:
            raise ValueError('Unknown tag prefix: {!r}'.format(prefix))
        type = TAG_TYPE_BY_PREFIXES[prefix]
        return cls.__new__(cls, type, name)

    def __getnewargs__(self):
        return self.type, self.name

    def __eq__(self, other):
        if isinstance(other, Tag):
            return self.type == other.type and self.name == other.name
        return False

    def __ne__(self, other):
        return not self.__eq__(other)

    def __hash__(self):
        return (self.type, self.name).__hash__()

    def __repr__(self):
        return self.type.name + '@' + super(Tag, self).__repr__()

    def __str__(self):
        return self.type.name + '@' + super(Tag, self).__str__()


PREFIXES = {
    TagType.HOSTGROUP: ['HOSTGROUP', 'H', 'GROUP', 'hostgroup', 'group'],
    TagType.HOST: ['host', 'h'],
    TagType.SHARDTAG: ['SHARDTAG', 'STAG', 'S', 'shardtag', 'stag'],
    TagType.SHARD: ['shard', 's'],
    TagType.CONF: ['CONF', 'conf', 'C'],
    TagType.INSTANCE: ['INSTANCETAG', 'instancetag', 'ITAG', 'itag', 'I'],
    TagType.DC: ['dc', 'd'],
    TagType.LINE: ['line', 'l'],
    TagType.FEDERATED: ['f'],
}

ALL_PREFIXES = []
TAG_TYPE_BY_PREFIXES = {}
for tag_type, prefixes in six.iteritems(PREFIXES):
    ALL_PREFIXES.extend(prefixes)
    TAG_TYPE_BY_PREFIXES.update({prefix: tag_type for prefix in prefixes})

TAG_TYPES = set(six.iterkeys(PREFIXES))
