# coding: utf-8
import collections
import sys
import textwrap

from google.protobuf.descriptor import FieldDescriptor as d

from infra.awacs.proto import modules_pb2
from awacs.wrappers import main  # to register our wrappers  # noqa
from awacs.yamlparser.core import Builder
from awacs.yamlparser.schemautil import JsonSchemaBuilder
from .ext import AbsolutifyHrefs


URL = 'https://bb.yandex-team.ru/projects/NANNY/repos/nanny/raw/awacs/docs/build/'
LOGIN = 'nanny-robot'
PASSWORD = 'XXX'
HIDDEN_MODULES = [
    'cachalot',
    'response_matcher',
    'l7_fast_upstream_macro',
    'l7_fast_sitemap_upstream_macro',
    'l7_macro',
    'srcrwr',
    'srcrwr_ext',
    'l7_upstream_macro',
    'cookie_policy',
    'cache2',
    'aab_cookie_verify',
    'webauth',
    'rate_limiter'
]

TYPES = {
    d.TYPE_DOUBLE: 'double',
    d.TYPE_FLOAT: 'float',
    d.TYPE_INT64: 'int64',
    d.TYPE_UINT64: 'uint64',
    d.TYPE_INT32: 'int32',
    d.TYPE_FIXED64: 'fixed64',
    d.TYPE_FIXED32: 'fixed32',
    d.TYPE_BOOL: 'bool',
    d.TYPE_STRING: 'string',
    d.TYPE_GROUP: 'group',
    d.TYPE_MESSAGE: 'message',
    d.TYPE_BYTES: 'bytes',
    d.TYPE_UINT32: 'uint32',
    d.TYPE_ENUM: 'enum',
    d.TYPE_SFIXED32: 'sfixed32',
    d.TYPE_SFIXED64: 'sfixed64',
    d.TYPE_SINT32: 'sint32',
    d.TYPE_SINT64: 'sint64',
}

VISITING = object()


def unprefix(value):
    prefix = 'awacs.modules.'
    if value.startswith(prefix):
        return value[len(prefix):]
    else:
        return value


def process_message(desc, name=None, aux_messages=None):
    name = name or desc.name

    data = JsonSchemaBuilder.get_message_docs(desc)
    allowed_calls = data.get('allowed_calls', {})

    if not data.get('docs'):
        sys.stderr.write('No doc source for {}\n'.format(name))

    docs = data.get('docs', {})
    field_docs = data.get('field_docs', {})

    description = docs.get('desc', '')
    notes = docs.get('notes', '')

    for item in data.get('required_oneofs', []):
        if description != '':
            description += u'  \n'
        description += (
            u'В точности одно из следующих полей должно быть указано: `{}`.'
        ).format('`, `'.join(item))

    for item in data.get('required_anyofs', []):
        description += (
            u'  \n'
            u'Хотя бы одно из следующих полей должно быть указано: `{}`.'
        ).format('`, `'.join(item))

    rv = Message(
        name,
        examples=docs.get('examples', ''),
        notes=notes,
        desc=description
    )

    for field in desc.fields:
        if field.name == 'nested':
            continue
        repeated = field.label == field.LABEL_REPEATED
        embedded_field_desc = None
        field_allowed_calls = allowed_calls.get(field.name, [])
        if field.type == d.TYPE_MESSAGE:
            message_desc = field.message_type

            embedded_field_name = Builder.get_embedded_field_name(message_desc)
            if embedded_field_name:
                embedded_field_desc = message_desc.fields_by_name[embedded_field_name]

            if message_desc.name == 'Call' and field.name.startswith('f_'):
                continue
            elif message_desc.name == 'KnobRef' and field.name.startswith('k_'):
                continue
            elif embedded_field_desc:
                type_ = Type(name=TYPES[embedded_field_desc.type])
                message_data = JsonSchemaBuilder.get_message_docs(message_desc)
                field_allowed_calls = message_data.get('allowed_calls', {}).get(embedded_field_name, [])
            elif message_desc.full_name == 'awacs.modules.Holder':
                type_ = Type('another module')
            elif Builder.is_map_entry(message_desc) or Builder.is_ordered_map_entry(message_desc):
                key_field = message_desc.fields_by_name['key']
                value_field = message_desc.fields_by_name['value']
                value_message_desc = value_field.message_type
                if value_message_desc and value_message_desc.name != 'Holder':
                    ref = unprefix(value_message_desc.full_name)
                    if ref not in aux_messages:
                        aux_messages[ref] = VISITING
                        field_message = process_message(value_message_desc, name=None, aux_messages=aux_messages)
                        aux_messages[ref] = field_message
                    value_type = Type(name=TYPES[field.type], ref=ref)
                else:
                    value_type = Type(name=TYPES[value_field.type])
                name = 'map[{}, {}]'.format(
                    Type(name=TYPES[key_field.type]),
                    value_type,
                )
                type_ = Type(name=name)
                repeated = False
            else:
                ref = unprefix(message_desc.full_name)
                if ref not in aux_messages:
                    aux_messages[ref] = VISITING
                    field_message = process_message(message_desc, name=None, aux_messages=aux_messages)
                    aux_messages[ref] = field_message
                type_ = Type(name=TYPES[field.type], ref=ref)
        else:
            type_ = Type(name=TYPES[field.type])
        type_.repeated = repeated

        field_doc = docs.get('fields', {}).get(field.name, {})

        field_description = field_doc.get('desc')
        field_notes = field_doc.get('notes')
        if not field_description:
            field_description = ''
        if field_doc.get('markdown_desc'):
            field_description = field_description
        if field_doc.get('deprecated'):
            continue

        data_defaults = data.get('defaults', {})
        if field.name in data_defaults:
            default = data_defaults[field.name]
        else:
            default = field_docs.get(field.name, {}).get('default', '')
        field = Field(
            name=field.name,
            type=type_,
            required=field.name in docs.get('required', ()) or field.name in data.get('required_props', ()),
            default=default,
            desc=field_description,
            notes=field_notes,
            allowed_calls=field_allowed_calls
        )
        rv.add_field(field)
    return rv


class Type(object):
    def __init__(self, name, ref=None):
        self.name = name
        self.ref = ref
        self.repeated = False

    def __str__(self):
        if self.ref:
            rv = '[{0}](#{0})'.format(self.ref)
        else:
            rv = self.name
        if self.repeated:
            rv = 'list of {}'.format(rv)
        return rv


class Field(object):
    def __init__(self, name, type, required, default, desc, notes, allowed_calls=()):
        self.name = name
        self.type = type
        self.required = required
        self.default = default
        self.desc = desc
        self.allowed_calls = allowed_calls
        self.notes = notes

    def to_md_desc(self):
        rv = self.desc
        if self.allowed_calls:
            rv += '  \n'
            rv += u'Допускает использование следующих функций: `{}`.'.format('`, `'.join(self.allowed_calls))
        return rv


MD_TEMPLATE = '''
{desc}
{options}
'''.strip()

OPTIONS_TEMPLATE = '''|| Имя | Тип | Обязательность | Дефолт | Описание ||'''.strip()

OPTIONS_ROW = '''|| {name} | {type} | {required} | {default} | {desc} ||'''

BASE_WIKI_URL = 'https://wiki.yandex-team.ru/cplb/awacs/modules-reference/'
ext = AbsolutifyHrefs(BASE_WIKI_URL)


class Message(object):
    def __init__(self, name, examples, notes, desc=''):
        self.name = name
        self.desc = desc
        self._fields = collections.OrderedDict()
        self.examples = examples
        self.notes = notes

    def add_field(self, field):
        self._fields[field.name] = field

    def to_md_markup(self):
        parts = []
        if self._fields:
            parts.extend([
                '#|',
                OPTIONS_TEMPLATE,
            ])
            for field in self._fields.values():
                row = OPTIONS_ROW.format(
                    name=field.name,
                    type=str(field.type),
                    required='*' if field.required else '',
                    default=(
                        '\n'.join(map('`{}`'.format, field.default)) if isinstance(field.default, list) else
                        '`{}`'.format(field.default) if field.default not in (None, '') else ''
                    ),
                    desc=field.to_md_desc())
                parts.append(row)
            parts.append('|#')
        else:
            parts.append('Модуль не имеет параметров.')
        if self.notes:
            parts.append(self.notes)
        if self.examples:
            examples_header = '  \n  \n##### Примеры использования'
            parts.append(examples_header)
            examples = ''
            for n, e in enumerate(self.examples, 1):
                examples += '* [Пример #{}]({})\n'.format(n, self.examples[e])
            parts.append(examples)
        options = '\n'.join(parts)

        rv = MD_TEMPLATE.format(name=self.name, desc=self.desc, options=options)
        return rv


class Chunk(object):
    def __init__(self, type, name, field_names):
        self.type = type
        self.name = name
        self.field_names = field_names


def is_top_level(name):
    return name in {'main', 'ipdispatch', 'instance_macro'} or name.endswith('_section')


def is_router(name):
    return name in {'ipdispatch', 'instance_macro', 'regexp', 'regexp_path', 'prefix_path_router', 'regexp_host'}


def is_router_section(name):
    return name in {'ipdispatch_section', 'regexp_section', 'regexp_path_section', 'prefix_path_router_section',
                    'regexp_host_section'}


def is_macro(name):
    return name.endswith('_macro')


def classify(name):
    if is_top_level(name) and not is_router_section(name):
        rv = (0, 'Top-level модули и макросы')
    elif is_router(name):
        rv = (1, 'Роутеры запросов и их секции')
    elif is_router_section(name):
        rv = (1, 'Роутеры запросов и их секции')
    elif is_macro(name):
        rv = (4, 'Макросы')
    else:
        rv = (3, 'Модули')
    return rv


def iter_chunks(seq):
    """
    :type seq: list[FieldDescriptor]
    """
    # XXX Quick and very dirty solution for https://st.yandex-team.ru/SWAT-4546
    chunk_items = []
    n = 0
    prev_chunk_type = None
    for item_type, item in sorted([(classify(item), item) for item in seq]):
        if prev_chunk_type is not None and prev_chunk_type != item_type:
            if chunk_items:
                yield Chunk(prev_chunk_type[0], prev_chunk_type[1], chunk_items)
                n = 0
                chunk_items = []
        chunk_items.append(item)
        n += len(item)
        if n > 100:
            yield Chunk(item_type[0], item_type[1], chunk_items)
            chunk_items = []
            n = 0
        prev_chunk_type = item_type
    if chunk_items:
        yield Chunk(item_type[0], item_type[1], chunk_items)


def main():
    holder_desc = modules_pb2.Holder.DESCRIPTOR.oneofs_by_name['module']

    toc = []
    holder_field_names = [field_desc.name for field_desc in holder_desc.fields]
    prev_chunk = None
    for chunk in iter_chunks(holder_field_names):
        if not prev_chunk or prev_chunk.type != chunk.type:
            toc.append('\n{}:\n'.format(chunk.name))
        i = 0
        for field_name in chunk.field_names:
            if field_name in HIDDEN_MODULES:
                continue
            if i > 0:
                toc.append(' &nbsp;&nbsp;&nbsp; ')
            toc.append('[{0}](#{0})'.format(field_name))
            i += 1
        toc.append('\n')
        prev_chunk = chunk

    assert toc.pop() == '\n'

    warning = textwrap.dedent("""
    %%(wacko wrapper=shade border="2px dashed red")
    **!!Эта документация относится к full-mode конфигам авакса, они сложны в написании и поддержке, и не рекомендуются к использованию. Продолжайте на свой страх и риск.
    Доки по актуальным и рекомендуемым подходам: https://wiki.yandex-team.ru/awacs/#osnovnoe!!**
    %%
    """)
    top_level_parts = [warning, '===Оглавление', ''.join(toc)]

    with open('./src/_intro.wiki', 'r') as f:
        top_level_parts.append(f.read())

    aux_messages = {}
    for field_desc in holder_desc.fields:
        field_name = field_desc.name
        if field_name in HIDDEN_MODULES:
            continue
        mod = process_message(field_desc.message_type, name=field_name, aux_messages=aux_messages)
        md = mod.to_md_markup()
        with open('./build/{}.md'.format(field_name), 'w') as f:
            f.seek(0)
            f.truncate()
            f.write(md)

            top_level_parts.append('{{{{a name="{}"}}}}'.format(field_name))
            top_level_parts.append('==={0}'.format(field_name))
            top_level_parts.append(md)

    for name, message in sorted(aux_messages.items()):
        md = message.to_md_markup()
        with open('./build/{}.md'.format(name), 'w') as f:
            f.seek(0)
            f.truncate()
            f.write(md)

            top_level_parts.append('{{{{a name="{}"}}}}'.format(name))
            top_level_parts.append('===={0}'.format(name))
            top_level_parts.append(md)

    top_level = '\n'.join(top_level_parts)
    with open('./build/_top_level.wiki', 'w') as f:
        f.write(top_level)


if __name__ == '__main__':
    main()
