package config

import (
	"context"
	"strings"

	"a.yandex-team.ru/library/go/core/log"
	"github.com/heetch/confita"
	"github.com/heetch/confita/backend/file"
	uzap "go.uber.org/zap"
	"go.uber.org/zap/zapcore"
)

type Config struct {
	LBGateway      string    `yaml:"lb_gateway"`
	LBIface        string    `yaml:"lb_iface"`
	BBIface        string    `yaml:"bb_iface"`
	LBSubnet       string    `yaml:"lb_subnet"`
	LBTableID      int       `yaml:"lb_table_id"`
	SubscriptionID string    `yaml:"subscription_id"`
	ResourceGroup  string    `yaml:"resource_group"`
	StatePath      string    `yaml:"state_path"`
	PoolName       string    `yaml:"pool_name"`
	Log            LogConfig `yaml:"log"`
	Mode           string    `yaml:"mode"`
	SNATSubnet     string    `yaml:"snat_subnet"`
	SNATInterface  string    `yaml:"snat_interface"`
	SNATTableID    int       `yaml:"snat_table_id"`
	SNATGateway    string    `yaml:"snat_gateway"`
}

type LogConfig struct {
	Level  string `yaml:"level"`
	Output string `yaml:"output"`
}

func (l *LogConfig) ZapConfig() uzap.Config {
	var lvl zapcore.Level
	switch strings.ToLower(l.Level) {
	case log.TraceString:
		lvl = zapcore.DebugLevel
	case log.DebugString:
		lvl = zapcore.DebugLevel
	case log.InfoString:
		lvl = zapcore.InfoLevel
	case log.WarnString:
		lvl = zapcore.WarnLevel
	case log.ErrorString:
		lvl = zapcore.ErrorLevel
	case log.FatalString:
		lvl = zapcore.FatalLevel
	default:
		lvl = zapcore.InfoLevel
	}

	return uzap.Config{
		Level:            uzap.NewAtomicLevelAt(lvl),
		Encoding:         "kv",
		OutputPaths:      []string{l.Output},
		ErrorOutputPaths: []string{"stderr"},
		EncoderConfig: zapcore.EncoderConfig{
			MessageKey:     "msg",
			LevelKey:       "level",
			TimeKey:        "ts",
			NameKey:        "logger",
			LineEnding:     zapcore.DefaultLineEnding,
			EncodeLevel:    zapcore.CapitalLevelEncoder,
			EncodeTime:     zapcore.ISO8601TimeEncoder,
			EncodeDuration: zapcore.SecondsDurationEncoder,
			EncodeCaller:   zapcore.ShortCallerEncoder,
		},
	}
}

func DefaultConfig() *Config {
	return &Config{
		LBGateway:      "",
		LBIface:        "",
		BBIface:        "",
		LBSubnet:       "",
		LBTableID:      104,
		SubscriptionID: "",
		ResourceGroup:  "",
		StatePath:      "/var/lib/azure-porto/lb-helper.json",
		PoolName:       "backend",
		Log: LogConfig{
			Level:  "info",
			Output: "/var/log/azure-porto-lb-helper.log",
		},
	}
}

func FromFile(path string) (*Config, error) {
	conf := DefaultConfig()
	loader := confita.NewLoader(file.NewBackend(path))
	if err := loader.Load(context.Background(), conf); err != nil {
		return nil, err
	}
	return conf, nil
}
