package controller

import (
	"context"
	"errors"
	"fmt"
	"io/ioutil"
	"net"
	"os"
	"path"

	"a.yandex-team.ru/infra/azure/lb-conf-porto-helper/internal/config"
	"a.yandex-team.ru/infra/azure/lb-conf-porto-helper/internal/rtslots"
	"a.yandex-team.ru/infra/azure/lb-conf-porto-helper/pkg/azurehelper"
	"a.yandex-team.ru/infra/azure/lb-conf-porto-helper/pkg/portohelper"
	"a.yandex-team.ru/library/go/core/log/zap"
	"github.com/Azure/azure-sdk-for-go/profiles/latest/network/mgmt/network"
)

func LoadState(c *config.Config) (*LoadedState, error) {
	f, err := os.Open(c.StatePath)
	if err != nil && errors.Is(err, os.ErrNotExist) {
		return &LoadedState{s: rtslots.EmptyState(), c: c}, nil
	}
	if err != nil {
		return nil, err
	}
	defer f.Close()
	s, err := rtslots.StateFromReader(f)
	if err != nil {
		return nil, err
	}
	return &LoadedState{
		s: s,
		c: c,
	}, nil
}

type LoadedState struct {
	s *rtslots.State
	c *config.Config
}

func (s *LoadedState) ActualizeState() (*ActualizedState, error) {
	containers, err := portohelper.GetContainerNamesMap()
	if err != nil {
		return nil, err
	}
	missingSlots := map[string]*rtslots.Slot{}
	var usedSlots []*rtslots.Slot
	for _, s := range s.s.Slots {
		if _, ok := containers[s.Container]; !ok {
			missingSlots[s.Address] = s
		} else {
			usedSlots = append(usedSlots, s)
		}
	}
	s.s.Slots = usedSlots
	return &ActualizedState{LoadedState: s, missingSlots: missingSlots}, nil
}

type ActualizedState struct {
	*LoadedState
	missingSlots map[string]*rtslots.Slot
}

func (s *ActualizedState) getAvailableSlotCount() int {
	_, n, _ := net.ParseCIDR(s.c.LBSubnet)
	maskLen, _ := n.Mask.Size()
	return (2 << (32 - maskLen)) - len(s.s.Slots) + len(s.missingSlots)
}

func (s *ActualizedState) findMissingForLB(lb string) *rtslots.Slot {
	for _, s := range s.missingSlots {
		if s.LBName == lb {
			return s
		}
	}
	return nil
}

func (s *ActualizedState) allocateFromMissing(lb, container, rg string) *rtslots.Slot {
	if slot := s.findMissingForLB(lb); slot != nil {
		slot.Container = container
		delete(s.missingSlots, slot.Address)
		return slot
	}
	if rg == "" {
		rg = s.c.ResourceGroup
	}
	for _, slot := range s.missingSlots {
		newSlot := slot.Copy()
		newSlot.PoolName = s.c.PoolName
		newSlot.LBName = lb
		newSlot.Container = container
		newSlot.ResourceGroup = rg
		return newSlot
	}
	return nil
}

func (s *ActualizedState) allocateFromFree(lb, container, rg string) (*rtslots.Slot, error) {
	usedIps := map[string]struct{}{}
	for _, s := range s.s.Slots {
		usedIps[s.Address] = struct{}{}
	}
	candidateIP, candidateNet, err := net.ParseCIDR(s.c.LBSubnet)
	if err != nil {
		return nil, err
	}
	maskLen, _ := candidateNet.Mask.Size()
	startAddrLowByte := candidateIP[15]
	for i := byte(0); i < startAddrLowByte+2<<(32-maskLen)-1; i++ {
		candidateIP[15] = startAddrLowByte + i
		if _, ok := usedIps[candidateIP.String()]; !ok {
			break
		}
	}
	if rg == "" {
		rg = s.c.ResourceGroup
	}
	return &rtslots.Slot{
		Container:     container,
		Address:       candidateIP.String(),
		PoolName:      s.c.PoolName,
		LBName:        lb,
		ResourceGroup: rg,
	}, nil
}

func (s *ActualizedState) allocateSlot(lb, container, rg string) (*rtslots.Slot, error) {
	// TODO: make slot allocation more reliable
	if s.getAvailableSlotCount() < 1 {
		return nil, fmt.Errorf("no slots available on subnet %s", s.c.LBSubnet)
	}
	slot := s.allocateFromMissing(lb, container, rg)
	if slot == nil {
		fs, err := s.allocateFromFree(lb, container, rg)
		if err != nil {
			return nil, err
		}
		slot = fs
	}
	s.s.Slots = append(s.s.Slots, slot)
	return slot, nil
}

func (s *ActualizedState) isLBConfigured(lb, container string) bool {
	for _, slot := range s.s.Slots {
		if slot.Container == container && slot.LBName == lb {
			return true
		}
	}
	return false
}

func (s *ActualizedState) UpdateState(lb, container, rg string) (*UpdatedState, error) {
	if lb == "" || container == "" {
		return &UpdatedState{ActualizedState: s}, nil
	}
	ns, err := s.allocateSlot(lb, container, rg)
	if err != nil {
		return nil, err
	}
	return &UpdatedState{
		ActualizedState: s,
		newSlot:         ns,
	}, nil
}

type UpdatedState struct {
	*ActualizedState
	newSlot *rtslots.Slot
}

func (s *UpdatedState) getMutations() azurehelper.LBBackendPoolMutations {
	rv := make([]azurehelper.LBBackendPoolMutation, 0, len(s.missingSlots)+1)
	for _, s := range s.missingSlots {
		rv = append(rv, (azurehelper.LBBackendPoolMutation)(&azurehelper.RemoveBackend{
			LBAddrDesc: azurehelper.LBAddrDesc{
				PoolDesc: azurehelper.PoolDesc{
					RG:   s.ResourceGroup,
					LB:   s.LBName,
					Name: s.PoolName,
				},
				Addr: s.Address,
				Slot: s.Container,
			},
		}))
	}
	if s.newSlot != nil {
		rv = append(rv, (azurehelper.LBBackendPoolMutation)(&azurehelper.UpsertBackend{
			LBAddrDesc: azurehelper.LBAddrDesc{
				PoolDesc: azurehelper.PoolDesc{
					RG:   s.newSlot.ResourceGroup,
					LB:   s.newSlot.LBName,
					Name: s.newSlot.PoolName,
				},
				Addr: s.newSlot.Address,
				Slot: s.newSlot.Container,
			},
		}))
	}
	return rv
}

func (s *UpdatedState) ApplyMutations(ctx context.Context, logger *zap.Logger, client network.LoadBalancerBackendAddressPoolsClient) (*AppliedState, error) {
	mutations := s.getMutations()
	err := mutations.Apply(ctx, logger, client)
	if err != nil {
		return nil, err
	}
	return &AppliedState{UpdatedState: s}, nil
}

func (s *UpdatedState) GetNewSlot() *rtslots.Slot {
	return s.newSlot
}

type AppliedState struct {
	*UpdatedState
}

func (s *AppliedState) PersistState() error {
	stateDir := path.Dir(s.c.StatePath)
	f, err := ioutil.TempFile(stateDir, "new_state")
	if err != nil {
		return err
	}
	defer f.Close()
	if err := s.s.PersistToWriter(f); err != nil {
		return err
	}
	return os.Rename(f.Name(), s.c.StatePath)
}
