package azurehelper

import (
	"context"
	"encoding/json"
	"fmt"

	"a.yandex-team.ru/library/go/core/log/zap"
	"github.com/Azure/azure-sdk-for-go/profiles/latest/network/mgmt/network"
)

type LBBackendPoolMutation interface {
	Apply(logger *zap.Logger, address *network.BackendAddressPool) error
	PoolDesc() *PoolDesc
}

type LBBackendPoolMutations []LBBackendPoolMutation

type PoolDesc struct {
	RG   string
	LB   string
	Name string
}

type LBAddrDesc struct {
	PoolDesc
	Addr string
	Slot string
}

type RemoveBackend struct {
	LBAddrDesc
}

func (b *RemoveBackend) Apply(logger *zap.Logger, pool *network.BackendAddressPool) error {
	logger = logger.WithName(fmt.Sprintf("remove backend[%s] -> %s", b.LB, b.Addr)).(*zap.Logger)
	var addresses []network.LoadBalancerBackendAddress
	for _, a := range *pool.LoadBalancerBackendAddresses {
		if *a.IPAddress != b.Addr {
			addresses = append(addresses, a)
		} else {
			logger.Debugf("removed address %s with name %s", *a.IPAddress, *a.Name)
		}
	}
	pool.LoadBalancerBackendAddresses = &addresses
	return nil
}

func (b *RemoveBackend) PoolDesc() *PoolDesc {
	return &b.LBAddrDesc.PoolDesc
}

type UpsertBackend struct {
	LBAddrDesc
}

func (b *UpsertBackend) Apply(logger *zap.Logger, pool *network.BackendAddressPool) error {
	logger = logger.WithName(fmt.Sprintf("upsert backend[%s] -> %s", b.LB, b.Addr)).(*zap.Logger)
	addresses := *pool.LoadBalancerBackendAddresses
	// skip-or-update
	for _, addr := range addresses {
		buf, err := json.Marshal(addr)
		logger.Debugf("examining address %s, %v", buf, err)
		if *addr.Name == b.Slot || *addr.IPAddress == b.Addr {
			logger.Debugf("found existing address %s, name: %s", *addr.IPAddress, *addr.Name)
			if *addr.IPAddress != b.Addr {
				logger.Debugf("updating address %s -> %s", *addr.IPAddress, b.Addr)
				addr.IPAddress = &b.Addr
			}
			if *addr.Name != b.Slot {
				logger.Debugf("updating name %s -> %s", *addr.Name, b.Slot)
				addr.Name = &b.Slot
			}
			return nil
		}
	}
	// add new address
	newAddress := network.LoadBalancerBackendAddress{
		Name: &b.Slot,
		LoadBalancerBackendAddressPropertiesFormat: &network.LoadBalancerBackendAddressPropertiesFormat{IPAddress: &b.Addr, VirtualNetwork: addresses[0].VirtualNetwork},
	}
	addresses = append(addresses, newAddress)
	buf, err := json.Marshal(newAddress)
	logger.Debugf("appending new address %s, %v", buf, err)
	pool.LoadBalancerBackendAddresses = &addresses
	return nil
}

func (b *UpsertBackend) PoolDesc() *PoolDesc {
	return &b.LBAddrDesc.PoolDesc
}

func GetPool(ctx context.Context, d *PoolDesc, client network.LoadBalancerBackendAddressPoolsClient) (*network.BackendAddressPool, error) {
	pool, err := client.Get(ctx, d.RG, d.LB, d.Name)
	if err != nil {
		return nil, err
	}
	return &pool, nil
}

func SavePool(ctx context.Context, client network.LoadBalancerBackendAddressPoolsClient, d *PoolDesc, pool *network.BackendAddressPool) error {
	_, err := client.CreateOrUpdate(ctx, d.RG, d.LB, d.Name, *pool)
	return err
}

type change struct {
	p *network.BackendAddressPool
	d *PoolDesc
}

func (m LBBackendPoolMutations) Apply(ctx context.Context, logger *zap.Logger, client network.LoadBalancerBackendAddressPoolsClient) error {
	changes := map[string]change{}
	for _, mut := range m {
		var pool *network.BackendAddressPool
		if p, ok := changes[mut.PoolDesc().LB]; ok {
			pool = p.p
		} else {
			p, err := GetPool(ctx, mut.PoolDesc(), client)
			if err != nil {
				return err
			}
			pool = p
			changes[mut.PoolDesc().LB] = change{p: pool, d: mut.PoolDesc()}
		}
		if err := mut.Apply(logger, pool); err != nil {
			return err
		}
	}
	for _, ch := range changes {
		if err := SavePool(ctx, client, ch.d, ch.p); err != nil {
			return err
		}
	}
	return nil
}
