package isshelper

import (
	"context"
	"encoding/base64"
	"io"
	"strings"

	"github.com/go-resty/resty/v2"
)

const (
	DefaultIssURL = "http://[::1]:25536"
	podsInfoPath  = "/pods/info"
)

type PodsInfo struct {
	Pods []*PodInfo `json:"pod_info"`
}

type PodInfo struct {
	ID                string `json:"id"`
	ContainerName     string `json:"container_name"`
	DynamicAttributes *struct {
		Labels *struct {
			Attributes []*Attribute `json:"attributes"`
		} `json:"labels"`
	} `json:"dynamic_attributes"`
	IP6AddressAllocations []struct {
		Address string `json:"address"`
		VlanID  string `json:"vlan_id"`
	} `json:"ip6_address_allocations"`
}

type Attribute struct {
	Key   string `json:"key"`
	Value string `json:"value"`
}

func (a *Attribute) GetValue() ([]byte, error) {
	return io.ReadAll(base64.NewDecoder(base64.StdEncoding, strings.NewReader(a.Value)))
}

type Client interface {
	GetPodsInfo(ctx context.Context) (*PodsInfo, error)
}

type client struct {
	c *resty.Client
}

func NewClient(issURL string) Client {
	c := resty.New().
		SetBaseURL(issURL).
		SetRetryCount(2).
		SetDoNotParseResponse(false).
		SetRedirectPolicy(resty.NoRedirectPolicy())
	return &client{c: c}
}

func NewDefaultClient() Client {
	return NewClient(DefaultIssURL)
}

func (c *client) GetPodsInfo(ctx context.Context) (*PodsInfo, error) {
	rv := &PodsInfo{}
	_, err := c.c.R().SetContext(ctx).SetResult(rv).Get(podsInfoPath)
	if err != nil {
		return nil, err
	}
	return rv, nil
}
