locals {
  bootstrap_data = base64gzip(templatefile("${path.module}/bootstrap.sh.tftpl", {
    hostname       = var.hostname
    host_num       = var.host_num
    ipv6_addr      = var.ipv6_addr
    bb_ipv6_addr   = var.bb_ipv6_addr
    mtn_prefix     = var.mtn_prefix
    lb_subnet      = var.lb_subnet
    lb_gateway     = var.lb_gateway
    resource_group = var.resource_group_name
  }))
}

resource "azurerm_network_interface" "backbone_nic" {
  resource_group_name  = var.resource_group_name
  name                 = "${var.hostname}-nic"
  location             = var.location
  enable_ip_forwarding = true

  ip_configuration {
    name                          = "ipconfig1"
    subnet_id                     = var.rtc_net_id
    private_ip_address_allocation = "Dynamic"
    private_ip_address_version    = "IPv4"
    primary                       = true
  }

  ip_configuration {
    name                          = "ipv6"
    subnet_id                     = var.rtc_net_id
    private_ip_address_allocation = "Static"
    private_ip_address_version    = "IPv6"
    private_ip_address            = var.ipv6_addr
  }

}

resource "azurerm_network_interface" "lb_nic" {
  resource_group_name  = var.resource_group_name
  name                 = "${var.hostname}-lb-nic"
  location             = var.location
  enable_ip_forwarding = false

  dynamic "ip_configuration" {
    for_each = { for idx, addr in var.lb_pool : idx => addr }
    content {
      name                          = "slot${ip_configuration.key + 1}"
      subnet_id                     = var.lb_net_id
      private_ip_address_allocation = "Static"
      private_ip_address_version    = "IPv4"
      private_ip_address            = ip_configuration.value
      primary                       = true ? ip_configuration.key == "0" : false
    }
  }
}

# resource "azurerm_managed_disk" "place_disk" {
#   resource_group_name  = var.resource_group_name
#   location             = var.location
#   name                 = "${var.hostname}_place"
#   disk_size_gb         = var.place_size
#   storage_account_type = "StandardSSD_LRS"
#   create_option        = "Empty"
#   # disk_iops_read_write = 500
#   # disk_mbps_read_write = 60
# }

resource "azurerm_managed_disk" "place_part0" {
  resource_group_name  = var.resource_group_name
  location             = var.location
  name                 = "${var.hostname}_place0"
  disk_size_gb         = 512
  storage_account_type = "Premium_LRS"
  tier = "P20"
  create_option        = "Empty"
  # disk_iops_read_write = 500
  # disk_mbps_read_write = 60
  zones = [ var.zone ]
}

resource "azurerm_managed_disk" "place_part1" {
  resource_group_name  = var.resource_group_name
  location             = var.location
  name                 = "${var.hostname}_place1"
  disk_size_gb         = 512
  storage_account_type = "Premium_LRS"
  tier = "P20"
  create_option        = "Empty"
  # disk_iops_read_write = 500
  # disk_mbps_read_write = 60
  zones = [ var.zone ]
}

resource "azurerm_managed_disk" "place_part2" {
  resource_group_name  = var.resource_group_name
  location             = var.location
  name                 = "${var.hostname}_place2"
  disk_size_gb         = 512
  storage_account_type = "Premium_LRS"
  tier = "P20"
  create_option        = "Empty"
  # disk_iops_read_write = 500
  # disk_mbps_read_write = 60
  zones = [ var.zone ]
}

resource "azurerm_managed_disk" "place_part3" {
  resource_group_name  = var.resource_group_name
  location             = var.location
  name                 = "${var.hostname}_place3"
  disk_size_gb         = 512
  storage_account_type = "Premium_LRS"
  tier = "P20"
  create_option        = "Empty"
  # disk_iops_read_write = 500
  # disk_mbps_read_write = 60
  zones = [ var.zone ]
}

resource "azurerm_linux_virtual_machine" "vm" {
  depends_on = [azurerm_managed_disk.place_part0, azurerm_managed_disk.place_part1, azurerm_managed_disk.place_part2, azurerm_managed_disk.place_part3]
  name                  = var.hostname
  resource_group_name   = var.resource_group_name
  location              = var.location
  network_interface_ids = [azurerm_network_interface.backbone_nic.id, azurerm_network_interface.lb_nic.id]
  size                  = var.vm_size
  boot_diagnostics {
    storage_account_uri = var.diag_endpoint
  }

  source_image_id = var.vm_image_id

  os_disk {
    name                 = "${var.hostname}-os"
    storage_account_type = "StandardSSD_LRS"
    disk_size_gb         = 32
    caching              = "ReadWrite"
  }

  identity {
    type = "SystemAssigned"
  }
  custom_data                     = local.bootstrap_data
  admin_username                  = var.admin_ssh_login
  computer_name                   = "${var.hostname}.rtc-oc.yandex.net"
  disable_password_authentication = true
  zone = var.zone

  tags = {
    "ipv6_addr" = var.ipv6_addr
  }

  admin_ssh_key {
    public_key = var.admin_ssh_key
    username   = var.admin_ssh_login
  }
  provisioner "local-exec" {
    command    = "dns-monkey.pl --zone-update --debug --expression 'add-a ${var.hostname}.rtc-oc.yandex.net ${var.ipv6_addr}'"
    on_failure = continue
  }
  provisioner "local-exec" {
    command    = "dns-monkey.pl --zone-update --debug --expression 'add-ptr ${var.ipv6_addr} ${var.hostname}.rtc-oc.yandex.net'"
    on_failure = continue
  }
  provisioner "local-exec" {
    command    = "curl -s -X POST -d '{\"check\": false, \"disable_admin_requests\": true, \"name\": \"${var.hostname}.rtc-oc.yandex.net\", \"project\": \"${var.walle_project}\" }' https://api.wall-e.yandex-team.ru/v1/hosts -H \"Authorization: OAuth $WALLE_TOKEN\" -H \"Content-Type: application/json\" && curl -s -X PUT -H \"Authorization: OAuth $WALLE_TOKEN\" -H 'Content-type: application/json' -d'{\"city\": \"VLADIMIR\", \"country\": \"RU\", \"datacenter\": \"BETA\", \"network_source\": \"lldp\", \"physical_timestamp\": \"1620811495\", \"port\": \"${var.host_num}\", \"queue\": \"VLA-09\", \"rack\": \"azure-${var.walle_project}-z${var.zone}\", \"short_datacenter_name\": \"vla\", \"short_queue_name\": \"vla-09\", \"switch\": \"azure-virtual-${var.walle_project}-z${var.zone}\", \"unit\": \"${var.host_num}\"}' 'https://api.wall-e.yandex-team.ru/v1/hosts/${var.hostname}.rtc-oc.yandex.net/location'"
    on_failure = continue
  }
  provisioner "local-exec" {
    command = "sleep 120; ssh -A -o UserKnownHostsFile=/dev/null -o StrictHostKeyChecking=no ${var.admin_ssh_login}@${var.hostname}.rtc-oc.yandex.net 'TOKEN=$(sudo certctl gen-token --user ${var.admin_ssh_login} 2>/dev/null | awk \"{print \\$2}\"); sudo certctl issue --token $TOKEN'"
  }


  provisioner "local-exec" {
    when       = destroy
    command    = "curl -s -X DELETE -H \"Authorization: OAuth $WALLE_TOKEN\" 'https://api.wall-e.yandex-team.ru/v1/hosts/${self.name}.rtc-oc.yandex.net?disable_admin_requests=true&ignore_cms=true&instant=true'"
    on_failure = continue
  }
  provisioner "local-exec" {
    command    = "dns-monkey.pl --zone-update --debug --expression 'delete-a ${self.name}.rtc-oc.yandex.net ${self.tags.ipv6_addr}'"
    on_failure = continue
    when       = destroy
  }
  provisioner "local-exec" {
    command    = "dns-monkey.pl --zone-update --debug --expression 'delete-ptr ${self.tags.ipv6_addr} ${self.name}.rtc-oc.yandex.net'"
    on_failure = continue
    when       = destroy
  }
}

resource "azurerm_virtual_machine_data_disk_attachment" "place0" {
  managed_disk_id    = azurerm_managed_disk.place_part0.id
  virtual_machine_id = azurerm_linux_virtual_machine.vm.id
  lun                = "0"
  caching            = "ReadWrite"
}

resource "azurerm_virtual_machine_data_disk_attachment" "place1" {
  managed_disk_id    = azurerm_managed_disk.place_part1.id
  virtual_machine_id = azurerm_linux_virtual_machine.vm.id
  lun                = "1"
  caching            = "ReadWrite"
}

resource "azurerm_virtual_machine_data_disk_attachment" "place2" {
  managed_disk_id    = azurerm_managed_disk.place_part2.id
  virtual_machine_id = azurerm_linux_virtual_machine.vm.id
  lun                = "2"
  caching            = "ReadWrite"
}

resource "azurerm_virtual_machine_data_disk_attachment" "place3" {
  managed_disk_id    = azurerm_managed_disk.place_part3.id
  virtual_machine_id = azurerm_linux_virtual_machine.vm.id
  lun                = "3"
  caching            = "ReadWrite"
}

resource "azurerm_route" "containers_route" {
  name                   = "${var.hostname}-vm-bb"
  route_table_name       = var.route_table_name
  resource_group_name    = var.shared_rg_name
  address_prefix         = var.vm_subnet
  next_hop_type          = "VirtualAppliance"
  next_hop_in_ip_address = var.ipv6_addr
}

resource "azurerm_network_interface_security_group_association" "lb_nsg_assoc" {
  network_interface_id      = azurerm_network_interface.lb_nic.id
  network_security_group_id = var.lb_nsg_id
}
