locals {
  bootstrap_data = base64gzip(templatefile("${path.module}/bootstrap.sh.tftpl", {
    hostname       = var.hostname
    host_num       = var.host_num
    ipv6_addr      = var.ipv6_addr
    bb_ipv6_addr   = var.bb_ipv6_addr
    mtn_prefix     = var.mtn_prefix
    lb_subnet      = var.lb_subnet
    lb_gateway     = var.lb_gateway
    resource_group = var.resource_group_name
  }))
}

resource "azurerm_network_interface" "backbone_nic" {
  resource_group_name  = var.resource_group_name
  name                 = "${var.hostname}-nic"
  location             = var.location
  enable_ip_forwarding = true

  ip_configuration {
    name                          = "ipconfig1"
    subnet_id                     = var.rtc_net_id
    private_ip_address_allocation = "Dynamic"
    private_ip_address_version    = "IPv4"
    primary                       = true
  }

  ip_configuration {
    name                          = "ipv6"
    subnet_id                     = var.rtc_net_id
    private_ip_address_allocation = "Static"
    private_ip_address_version    = "IPv6"
    private_ip_address            = var.ipv6_addr
  }

}

resource "azurerm_network_interface" "lb_nic" {
  resource_group_name  = var.resource_group_name
  name                 = "${var.hostname}-lb-nic"
  location             = var.location
  enable_ip_forwarding = false

  dynamic "ip_configuration" {
    for_each = { for idx, addr in var.lb_pool : idx => addr }
    content {
      name                          = "slot${ip_configuration.key + 1}"
      subnet_id                     = var.lb_net_id
      private_ip_address_allocation = "Static"
      private_ip_address_version    = "IPv4"
      private_ip_address            = ip_configuration.value
      primary                       = true ? ip_configuration.key == "0" : false
    }
  }
}

resource "azurerm_managed_disk" "place_disk" {
  resource_group_name  = var.resource_group_name
  location             = var.location
  name                 = "${var.hostname}_place"
  disk_size_gb         = var.place_size
  storage_account_type = "StandardSSD_LRS"
  create_option        = "Empty"
  # disk_iops_read_write = 500
  # disk_mbps_read_write = 60
}

resource "azurerm_linux_virtual_machine" "vm" {
  name                  = var.hostname
  resource_group_name   = var.resource_group_name
  location              = var.location
  network_interface_ids = [azurerm_network_interface.backbone_nic.id, azurerm_network_interface.lb_nic.id]
  size                  = var.vm_size
  boot_diagnostics {
    storage_account_uri = var.diag_endpoint
  }

  source_image_id = var.vm_image_id

  os_disk {
    name                 = "${var.hostname}-os"
    storage_account_type = "StandardSSD_LRS"
    disk_size_gb         = 32
    caching              = "ReadWrite"
  }

  identity {
    type = "SystemAssigned"
  }
  custom_data                     = local.bootstrap_data
  admin_username                  = "warwish"
  computer_name                   = "${var.hostname}.rtc-oc.yandex.net"
  disable_password_authentication = true
  admin_ssh_key {
    public_key = "ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAAEAQC/jEZapPBPqmANGZvhwheVq1R1vZEccME0sqyeGVODJz3F4s059L0OLWA8kZhf/6I8OPtbMa6y0YMdhOVRwV1q33Pw6qcEBY7KcOBgtXNCO4QHAlH14FIVomi+PC4QSgbNJVbfL+5iO5EwhKaQ5R09uwkk0Z1sg1ZoPffY7S1GKgFyOrIApIcBKqL7X5FZJyGICueWzhebXAUME1hY5iBUthIdkAr/vReXrnPIMq/8XAr+dfgJTYj7lXKxNBLMDHyBwQE07RtLdwcGmGSh3AdRbErDZF7k5xFDzFQm0ewHb67uN0zSGAPinZY/G0GbkfaTLFCmRDZsr0FlT0yCresPd8c7wGyzth/AyWkYcRzTNDboieZ7kPQJMtYGISNyQi8Dgh81mdq8MRkpYa1WnAnHRBk2zq4QaYMGJDy1S58xbc5vozK4t8MeXPANonyHyACSDHl5yU0Yp0piYJ4PK9HRoYsIvAj2Tdymrk3ucHmristeMAvvD7yIzbaPyj2THg0AUCF7bJZ6ovqpzSyJfCdwPckQxenw/tK1Yk5ehJ9iDRrSR3Lc+4NBWNqw49vxEgd8m+kjNSjU+x9ff+qUzPxE5QLKNQnl28JaPtqm2iAQ5Xglsrwx6PeQRoLxBb1jwNcrtMfBi5zTFYwxZ482tua+1s29RMarFfNzgyZyMbcjMY6ie4KwmJYZ+azJVabxLr1nLX8Jw0S1dTxwcjANOiOOq+TzUml+amUDrDiZyXqQF+seBx+fBG46y4hcMgdgEnkVGrDG9Ku+rDU2Iw2qcBuPEA9Db2aIeK6YlmE2PgpnXiak6lEfti6+gR9aDwG98CQE0FgrVdBYzPb7wTjFAQr22+whAmARREhiz3kqU/oPYcqrkrImdoF3HQtWcQfWAxvgowIz4M+sTy+7m6idDASsdXcyIR1rM81P83uGgsMWm177tpH5Kg0+L5fFKJ/Qlw5A5HDOX7g/qrlvvB0KbMZzn2XLkMsF+aNlNL+lx80tDRQOleJCjNQ959QT7P86Qjs09xQjcDmURf93gBm1xJZZDrB3aOEbzVl8aBtbb+7I/FOOxJqKpXS0nJmC9lHDZ2T4Kkp6yJubovSWffDFq8WxPwSN5nIRaV74m45DDZySe6kFU/ZdJUzjebWpA0Kp0OqqNjw9ezk6ygL8iI9C6JygzAKwLBk0l2qOpdo2W9W4GBjx/WNAtIa6eJPgQSesRT7MRtx2VdPQ6AQ2LM8p071s/TjGZOozQZoZq/km9gy6M+UW3sUFcDnJBIjkgDtxmAtHxuAZrJijh2tYhke6ufamxx56ckO5xyNlEhr+vluCEqFTkpJipN4nLkMzO7/BBr9jFnmywT/yUJqYJv2OhK97 warwish@warwish-osx"
    username   = "warwish"
  }
}

resource "azurerm_virtual_machine_data_disk_attachment" "place" {
  managed_disk_id    = azurerm_managed_disk.place_disk.id
  virtual_machine_id = azurerm_linux_virtual_machine.vm.id
  lun                = "0"
  caching            = "ReadWrite"
}

resource "azurerm_route" "containers_route" {
  name                   = "${var.hostname}-vm-bb"
  route_table_name       = var.route_table_name
  resource_group_name    = var.shared_rg_name
  address_prefix         = var.vm_subnet
  next_hop_type          = "VirtualAppliance"
  next_hop_in_ip_address = var.ipv6_addr
}

resource "azurerm_network_interface_security_group_association" "lb_nsg_assoc" {
  network_interface_id      = azurerm_network_interface.lb_nic.id
  network_security_group_id = var.lb_nsg_id
}