#include "controller.h"

#include <infra/box_dns_controller/libs/sensors/sensors.h>

#include <infra/libs/sensors/macros.h>
#include <infra/libs/yp_dns/dns_names/reverse.h>

namespace {

constexpr TStringBuf REQUIRED_BOX_SUBNET_SUFFIX = "/112";

NYP::NClient::TDnsRecordSet CreateBoxRecordSet(const TString& fqdn, const NYpDns::TIP6Address& address, NYP::NClient::NApi::NProto::TDnsRecordSetSpec_TResourceRecord_EType type) {
    NYP::NClient::TDnsRecordSet recordSet;
    recordSet.MutableLabels()->SetValueByPath("is_box", true);

    auto* record = recordSet.MutableSpec()->add_records();
    record->set_class_("IN");
    record->set_type(type);

    switch (type) {
        case NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord::AAAA:
            recordSet.MutableMeta()->set_id(fqdn);
            record->set_data(ToString(address));
            break;
        case NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord::PTR:
            recordSet.MutableMeta()->set_id(NYpDns::BuildIp6PtrDnsAddress(address));
            record->set_data(fqdn);
            break;
        default:
            ythrow yexception() << "Not implemented";
    }

    return recordSet;
}

void NormalizeName(TString& boxFqdn) {
    DNSName validFqdn(boxFqdn);
    validFqdn.makeUsLowerCase();
    boxFqdn = validFqdn.toStringNoDot();
    for (char symbol : NInfra::NBoxDns::INVALID_CHARS) {
        if (boxFqdn.Contains(symbol)) {
            ythrow yexception() << "Builded box fqdn [" << boxFqdn << "] contains invalid char [" << symbol << "]";
        }
    }
}

TString GetPodBoxesSubnet(const NYP::NClient::TPod& pod) {
    TString subnet;
    bool found = false;

    for (const auto& it : pod.Status().ip6_subnet_allocations()) {
        for (const auto& attr: it.labels().attributes()) {
            auto node = NYT::NodeFromYsonString(attr.value());
            if (attr.key() == "id" && node.AsString() == "boxes_subnet") {
                Y_ENSURE(!found, TStringBuilder{} << "Pod with id [" << pod.Meta().id() << "] got several subnets with 'boxes_subnet' label");
                found = true;
                TStringBuf addr = it.subnet();
                if (!addr.ChopSuffix(REQUIRED_BOX_SUBNET_SUFFIX)) {
                    ythrow yexception() << "Pod with id [" << pod.Meta().id() << "] subnet should end with '" << REQUIRED_BOX_SUBNET_SUFFIX << "', got '" << addr << "'";
                }
                subnet = addr;
            }
        }
    }

    if (found) {
        return subnet;
    }

    ythrow yexception() << "Pod with id [" << pod.Meta().id() << "] has no subnet with 'boxes_subnet' label";
}

} //anonymous namespace

namespace NInfra::NBoxDns {

TBoxDnsManager::TBoxDnsManager(
    TVector<NYP::NClient::TPod> pods,
    TVector<NYP::NClient::TDnsRecordSet> existingRecordSets,
    const TString& address,
    const ui32 maxUpdateObjectsNumber
)
    : Address_(address)
    , MaxUpdateObjectsNumber_(maxUpdateObjectsNumber)
    , ExistingRecordSets_(std::move(existingRecordSets))
    , Pods_(std::move(pods))
    , SensorGroup_(NSensors::BOX_DNS_MANAGER_GROUP_NAME)
{
    SensorGroup_.AddLabel("yp_address", Address_);
    TRateSensor(SensorGroup_, NSensors::RECORD_SET_CREATE_NUMBER);
    TRateSensor(SensorGroup_, NSensors::RECORD_SET_CREATE_SUCCESS);
    TRateSensor(SensorGroup_, NSensors::RECORD_SET_CREATE_FAILED);
    TRateSensor(SensorGroup_, NSensors::BOX_FQDN_COLLISIONS_NUMBER);
    TRateSensor(SensorGroup_, NSensors::BOX_SUBNET_FQDN_RESERVED);
    TRateSensor(SensorGroup_, NSensors::BOX_SUBNET_RECORD_SET_CREATE_SUCCESS);
    TRateSensor(SensorGroup_, NSensors::BOX_SUBNET_RECORD_SET_CREATE_FAILED);
    TRateSensor(SensorGroup_, NSensors::BOX_IP_CONFLICTS_WITH_SUBNET_ROOT);
}

void TBoxDnsManager::AddRecords(
    TVector<NYP::NClient::TDnsRecordSet>& actualRecordSets,
    const TString& fqdn,
    const TString& stringAddress,
    const TString& boxSubnetRootAddress
) const {
    static const auto mask = NYpDns::TIP6Address::FromString("ffff:ffff:ffff:ffff:ffff:ffff:ffff:0000");
    const auto subnetRootAddress = NYpDns::TIP6Address::FromString(boxSubnetRootAddress);

    const auto address = NYpDns::TIP6Address::FromString(stringAddress);
    const auto subnet = NYT::NNet::TIP6Network(subnetRootAddress, mask);
    if (!subnet.Contains(address)) {
        NON_STATIC_INFRA_RATE_SENSOR(SensorGroup_, NSensors::BOX_IP_CONFLICTS_WITH_SUBNET_ROOT);
        ythrow yexception() << "Box address [" << stringAddress << "] does not correspond to pod's box subnet address [" << boxSubnetRootAddress << "]";
    }

    NYP::NClient::TDnsRecordSet recordSetAAAA = CreateBoxRecordSet(fqdn, address, NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord::AAAA);
    NYP::NClient::TDnsRecordSet recordSetPTR = CreateBoxRecordSet(fqdn, address, NYP::NClient::NApi::NProto::TDnsRecordSetSpec::TResourceRecord::PTR);

    actualRecordSets.emplace_back(std::move(recordSetAAAA));
    actualRecordSets.emplace_back(std::move(recordSetPTR));
}

TString TBoxDnsManager::GetObjectId() const {
    return Address_;
}

bool TBoxDnsManager::TryAddBoxSubnetRootRecords(
    const NYP::NClient::TPod& pod
    , const TString& boxSubnetRootAddress
    , const THashMap<TString, TString>& boxFqdn2BoxId
    , TVector<NYP::NClient::TDnsRecordSet>& actualRecordSets
    , TLogFramePtr frame
) const {
    size_t boxSubnetRootNumber = 0;

    while (true) { 
        TString boxSubnetRootFqdn = !boxSubnetRootNumber
            ? TString::Join("box_subnet_root", ".", pod.Status().dns().persistent_fqdn())
            : TString::Join("box_subnet_root_", ToString(boxSubnetRootNumber), ".", pod.Status().dns().persistent_fqdn());

        try {
            NormalizeName(boxSubnetRootFqdn);
        } catch (...) {
            NON_STATIC_INFRA_RATE_SENSOR(SensorGroup_, NSensors::RECORD_SET_CREATE_FAILED);
            frame->LogEvent(ELogPriority::TLOG_ERR, TBoxSubnetRootValidateError(pod.Meta().id(), CurrentExceptionMessage()));
            return false;
        }

        if (boxFqdn2BoxId.find(boxSubnetRootFqdn) == boxFqdn2BoxId.end()) {
            try {
                AddRecords(actualRecordSets, boxSubnetRootFqdn, boxSubnetRootAddress, boxSubnetRootAddress);
            } catch (...) {
                NON_STATIC_INFRA_RATE_SENSOR(SensorGroup_, NSensors::RECORD_SET_CREATE_FAILED);
                frame->LogEvent(ELogPriority::TLOG_ERR, TBoxSubnetRootRecordSetCreateError(pod.Meta().id(), boxSubnetRootFqdn, boxSubnetRootAddress, CurrentExceptionMessage()));
                return false;
            }

            NON_STATIC_INFRA_RATE_SENSOR(SensorGroup_, NSensors::RECORD_SET_CREATE_SUCCESS);
            frame->LogEvent(ELogPriority::TLOG_INFO, TBoxSubnetRootDnsRecordSetCreateSuccess(pod.Meta().id(), boxSubnetRootFqdn, boxSubnetRootAddress));
            return true;
        }

        NON_STATIC_INFRA_RATE_SENSOR(SensorGroup_, NSensors::BOX_SUBNET_FQDN_RESERVED);
        frame->LogEvent(ELogPriority::TLOG_ERR, TBoxSubnetRootFqdnIsReserved(pod.Meta().id(), boxSubnetRootFqdn, boxSubnetRootAddress));
        ++boxSubnetRootNumber;
    }

    return true;
}

void TBoxDnsManager::GenerateYpUpdates(
    const ISingleClusterObjectManager::TDependentObjects& dependentObjects,
    TVector<ISingleClusterObjectManager::TRequest>& requests,
    TLogFramePtr frame
) const {
    Y_UNUSED(dependentObjects);
    TVector<NYP::NClient::TDnsRecordSet> actualRecordSets;

    int numberOfSuccesses = 0;
    int numberOfFails = 0;
    int numberOfCollisions = 0;
    int numberOfSubnetCreationFails = 0;
    int numberOfSubnetCreationSuccesses = 0;

    THashMap<TString, TString> boxFqdn2BoxId;

    for (const NYP::NClient::TPod& pod : Pods_) {
        TString boxSubnetRootAddress;
        try {
            boxSubnetRootAddress = GetPodBoxesSubnet(pod);
        } catch (...) {
            frame->LogEvent(ELogPriority::TLOG_ERR, TBoxSubnetRootValidateError(pod.Meta().id(), CurrentExceptionMessage()));
            ++numberOfFails;
            ++numberOfSubnetCreationFails;
            NON_STATIC_INFRA_RATE_SENSOR(SensorGroup_, NSensors::RECORD_SET_CREATE_FAILED);
            NON_STATIC_INFRA_RATE_SENSOR(SensorGroup_, NSensors::BOX_SUBNET_RECORD_SET_CREATE_FAILED);
            continue;
        }

        for (const auto& box : pod.Status().agent().pod_agent_payload().status().boxes()) {
            NON_STATIC_INFRA_RATE_SENSOR(SensorGroup_, NSensors::RECORD_SET_CREATE_NUMBER);
            TString boxFqdn = TString::Join(box.id(), ".", pod.Status().dns().persistent_fqdn());

            try {
                NormalizeName(boxFqdn);
            } catch (...) {
                frame->LogEvent(ELogPriority::TLOG_ERR, TBoxFqdnValidateError(box.id(), pod.Meta().id(), pod.Status().dns().persistent_fqdn(), CurrentExceptionMessage()));
                ++numberOfFails;
                NON_STATIC_INFRA_RATE_SENSOR(SensorGroup_, NSensors::RECORD_SET_CREATE_FAILED);
                continue;
            }

            if (const auto [boxFqdn2BoxIdIt, inserted] = boxFqdn2BoxId.emplace(boxFqdn, box.id()); !inserted) {
                frame->LogEvent(ELogPriority::TLOG_ERR, TBoxFqdnCollisionError(boxFqdn2BoxIdIt->first, boxFqdn2BoxIdIt->second, box.id()));
                ++numberOfCollisions;
                ++numberOfFails;
                NON_STATIC_INFRA_RATE_SENSOR(SensorGroup_, NSensors::BOX_FQDN_COLLISIONS_NUMBER);
                NON_STATIC_INFRA_RATE_SENSOR(SensorGroup_, NSensors::RECORD_SET_CREATE_FAILED);
                continue;
            }

            try {
                AddRecords(actualRecordSets, boxFqdn, box.ip6_address(), boxSubnetRootAddress);

                NON_STATIC_INFRA_RATE_SENSOR(SensorGroup_, NSensors::RECORD_SET_CREATE_SUCCESS);
                ++numberOfSuccesses;
            } catch (...) {
                frame->LogEvent(ELogPriority::TLOG_ERR, TBoxRecordsCreationError(box.id(), box.ip6_address(), pod.Meta().id(), pod.Status().dns().persistent_fqdn(), CurrentExceptionMessage()));
                ++numberOfFails;
                NON_STATIC_INFRA_RATE_SENSOR(SensorGroup_, NSensors::RECORD_SET_CREATE_FAILED);
            }
        }

        if (TryAddBoxSubnetRootRecords(pod, boxSubnetRootAddress, boxFqdn2BoxId, actualRecordSets, frame)) {
            ++numberOfSuccesses;
            ++numberOfSubnetCreationSuccesses;
            NON_STATIC_INFRA_RATE_SENSOR(SensorGroup_, NSensors::RECORD_SET_CREATE_SUCCESS);
            NON_STATIC_INFRA_RATE_SENSOR(SensorGroup_, NSensors::BOX_SUBNET_RECORD_SET_CREATE_SUCCESS);
        } else {
            ++numberOfFails;
            ++numberOfSubnetCreationFails;
            NON_STATIC_INFRA_RATE_SENSOR(SensorGroup_, NSensors::RECORD_SET_CREATE_FAILED);
            NON_STATIC_INFRA_RATE_SENSOR(SensorGroup_, NSensors::BOX_SUBNET_RECORD_SET_CREATE_FAILED);
        }

        NON_STATIC_INFRA_RATE_SENSOR(SensorGroup_, NSensors::RECORD_SET_CREATE_NUMBER);
    }

    if (numberOfFails) {
        frame->LogEvent(ELogPriority::TLOG_ERR, TBoxDnsFqdnNumberOfFails(numberOfFails));
    }

    if (numberOfSubnetCreationFails) {
        frame->LogEvent(ELogPriority::TLOG_ERR, TBoxSubnetFqdnNumberOfFails(numberOfSubnetCreationFails));
    }

    if (numberOfCollisions) {
        frame->LogEvent(ELogPriority::TLOG_ERR, TBoxDnsFqdnNumberOfCollisions(numberOfCollisions));
    }

    frame->LogEvent(ELogPriority::TLOG_INFO, TBoxDnsFqdnNumberOfSuccesses(numberOfSuccesses));
    frame->LogEvent(ELogPriority::TLOG_INFO, TBoxSubnetFqdnNumberOfSuccesses(numberOfSubnetCreationSuccesses));

    Sort(
        actualRecordSets.begin(),
        actualRecordSets.end(),
        [](const NYP::NClient::TDnsRecordSet& lhs, const NYP::NClient::TDnsRecordSet& rhs) {
            return lhs.Meta().id() < rhs.Meta().id();
        }
    );

    auto existingRecordSetIt = ExistingRecordSets_.begin();
    auto actualRecordSetIt = actualRecordSets.begin();

    while (existingRecordSetIt != ExistingRecordSets_.end() || actualRecordSetIt != actualRecordSets.end()) {
        if (existingRecordSetIt == ExistingRecordSets_.end()) {
            requests.emplace_back(NYP::NClient::TCreateObjectRequest(std::move(*actualRecordSetIt)));
            ++actualRecordSetIt;
        } else if (actualRecordSetIt == actualRecordSets.end()) {
            requests.emplace_back(NYP::NClient::TRemoveObjectRequest(NYP::NClient::TDnsRecordSet::ObjectType, existingRecordSetIt->Meta().id()));
            ++existingRecordSetIt;
        } else {
            if (actualRecordSetIt->Meta().id() < existingRecordSetIt->Meta().id()) {
                requests.emplace_back(NYP::NClient::TCreateObjectRequest(std::move(*actualRecordSetIt)));
                ++actualRecordSetIt;
            } else if (existingRecordSetIt->Meta().id() < actualRecordSetIt->Meta().id()) {
                requests.emplace_back(NYP::NClient::TRemoveObjectRequest(NYP::NClient::TDnsRecordSet::ObjectType, existingRecordSetIt->Meta().id()));
                ++existingRecordSetIt;
            } else {
                if (!google::protobuf::util::MessageDifferencer::Equals(existingRecordSetIt->Spec(), actualRecordSetIt->Spec())) {
                    auto records = actualRecordSetIt->Spec().records();
                    requests.emplace_back(NYP::NClient::TUpdateRequest(
                        NYP::NClient::TDnsRecordSet::ObjectType,
                        actualRecordSetIt->Meta().id(),
                        /* set */ {
                            NYP::NClient::TSetRequest("/spec/records", records),
                        },
                        /* remove */ {}
                    ));
                }
                ++actualRecordSetIt;
                ++existingRecordSetIt;
            }
        }

        if (requests.size() >= MaxUpdateObjectsNumber_) {
            break;
        }
    }
}

TBoxDnsManagerFactory::TBoxDnsManagerFactory(
    NUpdatableProtoConfig::TAccessor<TBoxDnsConfig> config
    , NUpdatableProtoConfig::TAccessor<NController::TClientConfig> ypClientConfig
    , NController::TShardPtr shard
)
    : ISingleClusterObjectManagersFactory(TStringBuilder() << "box_dns_manager_factory_" << TStringBuf(CONFIG_SNAPSHOT_VALUE(ypClientConfig, GetAddress())).Before('.'), shard)
    , Config_(std::move(config))
    , ActualConfig_(*Config_)
    , Address_(CONFIG_SNAPSHOT_VALUE(ypClientConfig, GetAddress()))
{
    Config_.SubscribeForUpdate([this](const TBoxDnsConfig& oldConfig, const TBoxDnsConfig& newConfig, const NUpdatableProtoConfig::TWatchContext& context = {}) {
        if (context.Id == GetFactoryName() && !google::protobuf::util::MessageDifferencer::Equivalent(oldConfig, newConfig)) {
            ActualConfig_ = newConfig;
        }
    });
}

TVector<NController::ISingleClusterObjectManager::TSelectArgument> TBoxDnsManagerFactory::GetSelectArguments(const TVector<TVector<NController::TSelectorResultPtr>>& /* aggregateResults */, NInfra::TLogFramePtr) const {
    NYP::NClient::TSelectObjectsOptions optionsPod;
    NYP::NClient::TSelectObjectsOptions optionsRecordSet;
    optionsPod.SetLimit(ActualConfig_.GetSelectChunkSize());
    optionsRecordSet.SetLimit(ActualConfig_.GetSelectChunkSize());

    return {
        NController::ISingleClusterObjectManager::TSelectArgument{
            NYP::NClient::TPod::ObjectType,
            /* selectors = */ {
                "/meta/id",
                "/status/dns/persistent_fqdn",
                "/status/agent/pod_agent_payload/status/boxes",
                "/status/ip6_subnet_allocations"
            },
            /* filter */ "[/status/agent/pod_agent_payload/status/boxes] != # and [/spec/node_id] != ''",
            optionsPod
        },
        NController::ISingleClusterObjectManager::TSelectArgument{
            NYP::NClient::TDnsRecordSet::ObjectType,
            /* selectors = */ {
                "/meta/id",
                "/spec/records"
            },
            /* filter */ "[/labels/is_box] = %true",
           optionsRecordSet
        }
    };
}

TVector<TExpected<NController::TSingleClusterObjectManagerPtr, TBoxDnsManagerFactory::TValidationError>> TBoxDnsManagerFactory::GetSingleClusterObjectManagers(
    const TVector<NController::TSelectObjectsResultPtr>& selectorResults,
    TLogFramePtr frame
) const {
    Y_UNUSED(frame);
    TVector<TExpected<NController::TSingleClusterObjectManagerPtr, TBoxDnsManagerFactory::TValidationError>> result;
    TVector<NYP::NClient::TPod> pods;
    pods.reserve(selectorResults[0]->Results.size());

    for (const auto& selectedPod : selectorResults[0]->Results) {
        NYP::NClient::TPod pod;
        selectedPod->Fill(
            pod.MutableMeta()->mutable_id(),
            pod.MutableStatus()->mutable_dns()->mutable_persistent_fqdn(),
            pod.MutableStatus()->mutable_agent()->mutable_pod_agent_payload()->mutable_status()->mutable_boxes(),
            pod.MutableStatus()->mutable_ip6_subnet_allocations()
        );

        pods.emplace_back(std::move(pod));
    }

    TVector<NYP::NClient::TDnsRecordSet> recordSets;
    recordSets.reserve(selectorResults[1]->Results.size());

    for (const auto& selectedRecordSet : selectorResults[1]->Results) {
        NYP::NClient::TDnsRecordSet recordSet;
        selectedRecordSet->Fill(
            recordSet.MutableMeta()->mutable_id(),
            recordSet.MutableSpec()->mutable_records()
        );
        recordSets.emplace_back(std::move(recordSet));
    }

    result.emplace_back(new TBoxDnsManager(std::move(pods), std::move(recordSets), Address_, ActualConfig_.GetMaxUpdateObjectsNumber()));
    return result;
}

} // namspace NINfra::NBoxDns
