import logging
import os
import requests
import subprocess
import retry

from tempfile import NamedTemporaryFile

from utils import get_hostname, get_port, get_ihost, stop_search_service
import utils.searchconfig

HEATED_FILENAMES = ('indexinv', 'indexkeyinv.wad')
HEATED_ENOUGH_PERCENT = 45


def get_basesearch_status():
    try:
        current_shard = get_basesearch_shard(get_hostname(), get_port())
        is_heated = check_heated(current_shard)
    except (requests.exceptions.ConnectionError, RuntimeError):
        return {}

    return {
        current_shard: {
            'status': 'RUN' if is_heated else 'DOWN',
        }
    }


@retry.retry(exceptions=(RuntimeError, requests.exceptions.ConnectionError), tries=5, delay=1)
def get_basesearch_shard(host, port):
    logging.debug('Ask %s:%s about shard', host, port)
    r = requests.get('http://{}:{}/yandsearch?info=getconfig:proto'.format(host, port))
    if r.ok:
        shard_path = utils.searchconfig.get_shard_path(r.text)
        if shard_path:
            return shard_path
        else:
            logging.debug('Proto config is not parsible')
            return ''
    else:
        raise RuntimeError('get_basesearch_status failed: %s' % r.reason)


def get_shard_symlinked(shard_rootdir):
    for i in os.listdir('./'):
        if os.path.islink(i) and os.path.realpath(i).startswith(shard_rootdir):
            return i


def check_heated(shard):
    check_file = find_file_in_shard(shard)
    if not check_file:
        raise RuntimeError('None of {} was found'.format(HEATED_FILENAMES))

    heat_percent = 0.0
    out = subprocess.check_output(['vmtouch', check_file])
    for s in out.splitlines():
        if 'Resident Pages:' in s and s.endswith('%'):
            heat_percent = float(s.rstrip('%').split()[-1])
    return heat_percent >= HEATED_ENOUGH_PERCENT


def find_file_in_shard(shard):
    for filename in HEATED_FILENAMES:
        file_in_shard = os.path.join(shard, filename)
        if os.path.exists(file_in_shard):
            return file_in_shard


def apply_basesearch_config(settings, config):
    if not config['shard']:
        logging.debug('Have got empty shard in config')
        return False

    current_status = get_basesearch_status()
    logging.debug('Current basesearch status: %s', current_status)

    if config['shard'] in current_status and current_status[config['shard']]['status'] == 'RUN':
        return True  # basesearch already run

    shard_src = config['shard']

    if not os.path.exists(shard_src):
        logging.debug('Shard %s not found'.shard_src)
        return False

    with NamedTemporaryFile(dir='./', delete=False) as ofp:
        with open('{}:{}.cfg.tmpl'.format(get_ihost(), get_port())) as ifp:
            template_text = ifp.read()
            ifp.close()
            logging.debug('Will apply %s shard to config', config['shard'])
            config_text = utils.searchconfig.patch_shard_config(template_text, config['shard'])
        ofp.write(config_text)
        ofp.close()
        os.chmod(ofp.name, 0644)
        os.rename(ofp.name, '{}:{}.cfg'.format(get_ihost(), get_port()))

        stop_search_service()
