import logging
import time

import gevent

import utils.collections


class DownloadQueue(object):
    """
    No resource affinity is considered.
    Resources are downloaded in arbitrary order.
    """
    def __init__(self, storage, callback=None):
        self._storage = storage
        self._callback = callback or (lambda _: None)
        self._resources = utils.collections.CircleQueue()
        self._greenlet = None

    def start(self):
        if not self._greenlet:
            self._greenlet = gevent.spawn(self._download_loop)

    def push(self, rbtorrent):
        _log.debug('Enqueueing [%s]', rbtorrent)
        self._resources.add(rbtorrent)

    def remove(self, rbtorrent):
        _log.debug('Removing [%s]', rbtorrent)
        self._resources.remove(rbtorrent)

    def clear(self):
        self._resources.clear()

    def _download_loop(self):
        while True:
            # noinspection PyBroadException
            try:
                resource = self._get_next_resource()

                if not self._storage.is_ready(resource):
                    _log.debug('Download [%s] spawn', resource)
                    self._storage.get_resource(resource)

                self.remove(resource)
                self._callback(resource)
                _log.debug('Callback for [%s] done', resource)

            except utils.collections.Empty:
                pass
            except Exception:
                _log.exception('Unhandled exception in the download loop')
            finally:
                time.sleep(5)

    def _get_next_resource(self):
        return self._resources.get_next()


_log = logging.getLogger(__name__)
