import collections
import logging
import os

import utils
import infra.callisto.libraries.copier as copier


class Storage(object):
    def __init__(self, root):
        self._root = os.path.abspath(root)

        utils.ensure_dir(os.path.join(self._root))
        utils.ensure_dir(os.path.join(self._root, 'download'))
        utils.ensure_dir(os.path.join(self._root, 'ready'))

    # TODO: method for relative path
    @property
    def resources(self):
        return set(os.listdir(os.path.join(self._root, 'download')))

    def ready_resources(self):
        for resource in self.resources:
            rbtorrent = _read_rbtorrent(self._ready_path(resource))
            if rbtorrent:
                yield Resource(name=resource, ready=True, rbtorrent=rbtorrent)

    def not_ready_resources(self):
        for resource in self.resources:
            rbtorrent = _read_rbtorrent(self._ready_path(resource))
            if not rbtorrent:
                yield Resource(name=resource, ready=False, rbtorrent=None)

    def get_slot(self, resource_id):
        path = self._slot_path(resource_id)
        utils.ensure_dir(path)
        return path

    def rm_resource(self, resource):
        _log.info('Removing [%s]', resource)

        utils.remove_dir(self._slot_path(resource))
        utils.remove_file(self._ready_path(resource))
        utils.remove_file(self._ready_path(resource) + '.rbtorrent')

    def is_ready(self, resource):
        return bool(_read_rbtorrent(self._ready_path(resource)))

    def set_ready(self, resource, rbtorrent=None):
        _log.info('Releasing [%s]', resource)

        utils.remove_file(self._ready_path(resource))
        utils.create_symlink(self._slot_path(resource), self._ready_path(resource))

        if rbtorrent:
            _write_rbtorrent(self._ready_path(resource), rbtorrent)

    def get_resource(self, rbtorrent):
        download_dir = self.get_slot(rbtorrent)

        _sky_get(rbtorrent, download_dir)

        self.set_ready(rbtorrent, rbtorrent=rbtorrent)

        return download_dir

    def _slot_path(self, resource):
        return os.path.join(self._root, 'download', resource)

    def _ready_path(self, resource):
        return os.path.join(self._root, 'ready', resource)

    def path_to_resource(self, resource):
        return self._ready_path(resource)

    def cleanup(self, needed_resources):
        for resource in self.resources:
            if resource not in needed_resources:
                self.rm_resource(resource)


def _sky_get(rbtorrent, destination):
    copier.sky_get(
        resource_url=rbtorrent,
        dst_dir=destination,
        timeout=60*10,
        max_dl_speed='25M',
        hardlink=True,
    )


Resource = collections.namedtuple('Resource', ['name', 'ready', 'rbtorrent'])


def _write_rbtorrent(shard_dir, rbtorrent):
    with open(shard_dir + '.rbtorrent', 'w') as f:
        _log.debug('Write rbtorrent [%s]', rbtorrent)
        f.write(rbtorrent + '\n')


def _read_rbtorrent(shard_dir):
    # noinspection PyBroadException
    try:
        with open(shard_dir + '.rbtorrent', 'r') as f:
            rbtorrent = f.read().strip()
            _log.debug('Read rbtorrent [%s]', rbtorrent)
            return rbtorrent
    except Exception:
        return None


_log = logging.getLogger(__name__)
