import logging
import mimetypes
import os
import psutil
import requests
import shutil
import socket
import subprocess
import time


def timestamp():
    return str(int(time.time()))


def stop_search_service():
    try:
        r = requests.get('http://{ihost}:{iport}/admin?action=shutdown'.format(
            ihost=get_hostname(),
            iport=get_port()
        ))
        logging.debug('Call action shutdown: {} ({})'.format(r.status_code, r.content.strip()))
    except requests.exceptions.ConnectionError:
        logging.debug('Call action shutdown, but got exception')
        pass  # Already stopped?


def get_hostname():
    return socket.gethostname()


def get_node():
    return os.environ.get('NODE_NAME') or os.environ['DEPLOY_NODE_FQDN']


def get_port():
    return os.environ['BSCONFIG_IPORT']


def get_ihost():
    return os.environ['BSCONFIG_IHOST']


def get_tags():
    return os.environ.get('BSCONFIG_ITAGS', '').split()


def get_tag(pattern):
    for t in get_tags():
        if t.startswith(pattern):
            return t


def get_loadaverage():
    la1, la5, la15 = os.getloadavg()
    return la1 / psutil.cpu_count()


def get_freespace(mountpoint):
    if os.path.exists(mountpoint):
        vfs = os.statvfs(mountpoint)
        return vfs.f_bavail * vfs.f_frsize
    else:
        return None


def get_shard_number():
    shard_tag = get_tag('a_shard_')
    if shard_tag:
        return '-'.join(shard_tag.split('-')[-3:-1])


def get_tier():
    shard_tag = get_tag('a_tier_')
    if shard_tag:
        return shard_tag.replace('a_tier_', '')


def get_shard():
    shard_tag = get_tag('a_shard_')
    if shard_tag:
        return shard_tag.replace('a_shard_', '')
    raise RuntimeError('shard tag not found')


def get_topology():
    topo_tag = get_tag('a_topology_')
    if topo_tag:
        return topo_tag.replace('a_topology_', '')


def get_itype():
    return get_a_tag('a_itype_')


def get_metaprj():
    return get_a_tag('a_metaprj_')


def get_a_tag(tag):
    itype = get_tag(tag)
    if itype:
        return itype.replace(tag, '')


def get_shortname(host):
    return host.split('.')[0]


def ls(path):
    return [os.path.join(path, d) for d in os.listdir(path)]


def remove_dir(path):
    if os.path.exists(path):
        shutil.rmtree(path)


def ensure_dir(path):
    if not os.path.exists(path):
        os.makedirs(path)
    return os.path.abspath(path)


def create_symlink(src, dst):
    os.symlink(src, dst)


def remove_file(path):
    try:
        os.remove(path)
    except OSError:
        pass


def extract_file(archive_file_name, file_to_extract, target_dir):
    filetype, encoding = mimetypes.guess_type(archive_file_name)
    if filetype == 'application/x-7z-compressed':
        return extract_7z_file(archive_file_name, file_to_extract, target_dir)
    elif (filetype, encoding) == ('application/x-tar', 'gzip'):
        return extract_targz_file(archive_file_name, file_to_extract, target_dir)
    else:
        raise RuntimeError('unknown format')


def extract_7z_file(archive_file_name, file_to_extract, target_dir):
    path = os.path.join(target_dir, os.path.basename(file_to_extract))
    if not os.path.exists(path):
        logging.debug('will extract %s from %s to %s using 7zr', archive_file_name, archive_file_name, target_dir)
        cmd = ['7zr', 'e', archive_file_name, '-o' + target_dir, file_to_extract, '-y']
        with open(os.devnull, 'w') as dev_null:
            popen = subprocess.Popen(cmd, stdout=dev_null, stderr=dev_null)
            return_code = popen.wait()
        assert return_code == 0, '7z error ({})'.format(return_code)
        logging.debug('extracted to %s', path)

    assert os.path.exists(path), 'file not found'
    return path


def extract_targz_file(archive_file_name, file_to_extract, target_dir):
    path = os.path.join(target_dir, os.path.basename(file_to_extract))

    if not os.path.exists(path):
        logging.debug('will extract %s from %s to %s using tar', file_to_extract, archive_file_name, target_dir)
        cmd = [
            'tar',
            '--extract',
            '--file', archive_file_name,
            '--directory', target_dir,
            '--occurrence=1',
            '--strip-components={}'.format(file_to_extract.count('/')),
            file_to_extract
        ]
        subprocess.check_call(cmd)
        logging.debug('extracted to %s', path)

    return path
