import os
import time
import logging
import logging.handlers


def setup_logging(log_dir, debug):
    if not os.path.exists(log_dir):
        os.makedirs(log_dir)

    logger = logging.getLogger()
    logger.setLevel(logging.DEBUG)

    handler = logging.handlers.RotatingFileHandler(
        os.path.join(log_dir, 'callisto.debug.log'),
        maxBytes=1024**3,
        backupCount=10
    )
    handler.setLevel(logging.DEBUG)
    handler.setFormatter(SkynetishFormatter())

    logger.addHandler(handler)

    handler = logging.handlers.RotatingFileHandler(
        os.path.join(log_dir, 'callisto.info.log'),
        maxBytes=1024**3,
        backupCount=10
    )
    handler.setLevel(logging.INFO)
    handler.setFormatter(SkynetishFormatter())

    logger.addHandler(handler)

    if debug:
        handler = logging.StreamHandler()
        handler.setLevel(logging.DEBUG)
        handler.setFormatter(SkynetishFormatter())
        logger.addHandler(handler)

    logging.getLogger('requests').setLevel(logging.CRITICAL)
    logging.getLogger('httpserver').setLevel(logging.CRITICAL)


class SkynetishFormatter(logging.Formatter):
    def __init__(self):
        super(SkynetishFormatter, self).__init__()

    def formatTime(self, record, **kwargs):
        t = time.strftime('%Y-%m-%d %H:%M:%S', self.converter(record.created))
        return '%s.%03d' % (t, record.msecs)

    def format(self, record):
        levelno = record.levelno
        if levelno > 5:
            level = '[%-4s]' % logging.getLevelName(levelno)
        else:
            level = '(%s)' % (str(levelno) if levelno < 0 else ' %d' % levelno)

        date = self.formatTime(record)
        message = record.getMessage()
        if record.exc_info:
            message += '\n' + self.formatException(record.exc_info)
        header = '{0} {1} [{2}]  '.format(date, level, record.name)

        if '\n' in message:
            # special case for multi-line log messages
            message_lines = message.splitlines()
            line = [header + message_lines[0]]
            prepend = '%s%s' % (' ' * (len(header) - 2), ': ')
            line.extend(['%s%s' % (prepend, l) for l in message_lines[1:]])
            line = '\n'.join(line)
        else:
            line = '{header}{message}'.format(header=header, message=message)
        return line
