from google.protobuf import text_format
import search.config.idl.service_config_pb2 as service_config_pb2
import logging


def get_shard_path(config_text):
    try:
        proto_cfg = _parse_prototext_config(config_text)
        return _get_shard_path_proto(proto_cfg)
    except text_format.ParseError as exc:
        logging.debug('Could not parse protobuf config: %s; falling back to parsing as text', exc.args)

    return _get_shard_path_xml(config_text)


def get_config_version(config_text):
    try:
        proto_cfg = _parse_prototext_config(config_text)
        return _get_cfg_version_proto(proto_cfg)
    except text_format.ParseError as exc:
        logging.warning('Could not parse protobuf config: %s', exc.args)

    return _get_cfg_version_xml(config_text)


def _parse_prototext_config(config_text):
    basesearch_cfg = service_config_pb2.TConfig()
    return text_format.Parse(config_text, basesearch_cfg)


def _get_shard_path_proto(proto_config):
    return proto_config.Collection[0].IndexDir


def _get_cfg_version_proto(proto_config):
    return proto_config.Agent.ConfigVersion


def _get_shard_path_xml(config_text):
    for line in config_text.splitlines():
        if 'IndexDir' in line:
            return line.split()[-1].strip('"')
    raise RuntimeError('Shard not found in xml config')


def _get_cfg_version_xml(config_text):
    for line in config_text.splitlines():
        if '# conf_hash: ' in line:
            return line.split()[-1]
    raise RuntimeError('Conf hash not found in xml config')


def patch_shard_config(config_text, shard):
    try:
        return _patch_shard_proto_config(config_text, shard)
    except text_format.ParseError as exc:
        logging.warning('Could not patch protobuf config: %s', exc.args)
        if '<Server>' in config_text:  # xml like config
            logging.warning('Try to patch config as xml')
            return _patch_shard_xml_config(config_text, shard)
        elif 'Port: ' in config_text:  # template proto text config
            logging.warning('Try to patch config as proto template')
            return _patch_shard_proto_template_config(config_text, shard)

    raise RuntimeError('Could not parse config text')


def _patch_shard_proto_config(config_text, shard):
    basesearch_cfg = service_config_pb2.TConfig()
    text_format.Parse(config_text, basesearch_cfg)
    basesearch_cfg.Collection[0].IndexDir = shard
    return str(basesearch_cfg)


def _patch_shard_xml_config(config_text, shard):
    config_lines = config_text.splitlines()
    for line_num in range(len(config_lines)):
        if 'IndexDir' in config_lines[line_num]:
            config_lines[line_num] = '    IndexDir {}'.format(shard)
    return '\n'.join(config_lines)


def _patch_shard_proto_template_config(config_text, shard):
    config_lines = config_text.splitlines()
    for line_num in range(len(config_lines)):
        if 'IndexDir:' in config_lines[line_num]:
            config_lines[line_num] = '    IndexDir: "{}"'.format(shard)
    return '\n'.join(config_lines)


def patch_version_config(config_text, version):
    try:
        return _patch_version_proto_config(config_text, version)
    except text_format.ParseError as exc:
        logging.warning('Could not patch protobuf config: %s', exc.args)
        if '<Server>' in config_text:  # xml like config
            logging.warning('Try to patch config as xml')
            return _patch_version_xml_config(config_text, version)
        elif 'Port: ' in config_text:  # template proto text config
            logging.warning('Try to patch config as proto template')
            return _patch_version_proto_template_config(config_text, version)

    raise RuntimeError('Could not parse config text')


def _patch_version_proto_config(config_text, version):
    basesearch_cfg = service_config_pb2.TConfig()
    text_format.Parse(config_text, basesearch_cfg)
    basesearch_cfg.Agent.ConfigVersion = version
    return str(basesearch_cfg)


def _patch_version_xml_config(config_text, version):
    return '# conf_hash: {}\n{}'.format(version, config_text)


def _patch_version_proto_template_config(config_text, version):
    config_lines = config_text.splitlines()

    config_lines.extend([
        'Agent {',
        '   ConfigVersion: {}'.format(version),
        '}',
    ])

    return '\n'.join(config_lines)
