import logging
import os
import requests
import tempfile

import v2.plugin
import infra.callisto.deploy.storage.storage as resources_storage
import utils
import utils.searchconfig
import observer


class Plugin(v2.plugin.BasePlugin):
    @classmethod
    def add_args(cls, parser):
        parser.add_argument('--shard-root', help='Shard root dir', required=True)

    def __init__(self, host, port, tags, args):
        super(Plugin, self).__init__(host, port, tags)
        self._shard_root = args.shard_root

        self._observer = observer.Observer(
            host, port,
        )

    def collect_status(self):
        shard_path = self._observer.get_shard_path()
        shard = _shard_name(shard_path)
        return {
            shard: {
                'status': 'RUN',
            }
        }

    def apply_config(self, content):
        config_template = _get_template_config(self.host, self.port)
        if 'resource' in content:
            namespace, name = content['resource']['namespace'], content['resource']['name']
            shard_path = resources_storage.resource_path(self._shard_root, namespace, name)
        elif content['shard'] is not None:
            shard_path = os.path.join(self._shard_root, content['shard'])
        else:
            raise ValueError('Can not apply config - null shard in config')

        patched_config = utils.searchconfig.patch_shard_config(config_template, shard_path)
        _save(patched_config, self.host, self.port)

        running_shard = _shard_name(self._observer.get_shard_path())
        if running_shard != content['shard']:
            logging.info('config changed: %s -> %s', running_shard, content['shard'])
            self.stop()
        else:
            logging.debug('config did not change')

    def stop(self):
        logging.info('stopping')
        r = requests.get(_url(self.host, self.port, 'admin', action='shutdown'), timeout=30)
        r.raise_for_status()


def _get_template_config(host, port):
    config_template = '{}:{}.cfg.template'.format(utils.get_shortname(host), port)
    with open(config_template) as fp:
        return fp.read()


def _save(content, host, port):
    with tempfile.NamedTemporaryFile(dir='./', delete=False) as fp:
        fp.write(content)
        fp.close()
        os.chmod(fp.name, 0644)
        os.rename(fp.name, './{}:{}.cfg'.format(utils.get_shortname(host), port))


def _url(host, port, path, action):
    return 'http://{}:{}/{}?action={}'.format(host, port, path, action)


def _shard_name(path_to_shard):
    if path_to_shard.endswith('@resource'):
        path_to_shard = path_to_shard.replace('/@resource', '')
    if path_to_shard.endswith('/local'):
        path_to_shard = path_to_shard.replace('/local', '')
    return path_to_shard.split('/')[-1]
