import json
import os
import tempfile

import requests
import psutil

import plugin


_REQUESTS_TIMEOUT = 30


class Plugin(plugin.BasePlugin):
    @classmethod
    def add_args(cls, parser):
        parser.add_argument('--shard-root', help='Shard root dir', required=True)

    def __init__(self, host, port, tags, args):
        super(Plugin, self).__init__(host, port, tags)
        self._shard_root = args.shard_root

    def apply_config(self, content):
        # FIXME: _check_config(content)
        _save(json.dumps(content, indent=4))

        r = requests.post(
            url=_url(self.host, self.port, 'reload'),
            data=json.dumps(content),
            timeout=_REQUESTS_TIMEOUT
        )
        r.raise_for_status()

    def collect_status(self):
        status = {
            'topology': _get_topology_version(),
            'shard_number': _get_autotag('shard'),
            'tier': _get_autotag('tier'),
            'freespace': _get_freespace(self._shard_root),
            'la': _get_loadaverage(),
        }

        r = requests.get(_url(self.host, self.port, 'status'), timeout=_REQUESTS_TIMEOUT)
        r.raise_for_status()
        status['shards'] = r.json()

        return status


def _get_autotag(name):
    return _get_tag('a_{}_'.format(name))


def _get_topology_version():
    return _get_tag('a_topology_version-')


def _get_tag(prefix):
    for tag in os.environ.get('BSCONFIG_ITAGS', '').split():
        if tag.startswith(prefix):
            return tag.replace(prefix, '')


def _get_loadaverage():
    la1, la5, la15 = os.getloadavg()
    return la1 / psutil.cpu_count()


def _get_freespace(mount_point):
    if os.path.exists(mount_point):
        vfs = os.statvfs(mount_point)
        return vfs.f_bavail * vfs.f_frsize
    else:
        return None


def _save(content):
    with tempfile.NamedTemporaryFile(dir='./', delete=False) as fp:
        fp.write(content)
        fp.close()
        os.chmod(fp.name, 0644)
        os.rename(fp.name, './callisto.replicamap.json')


def _check_config(content):
    cfg = content  # json.loads(content)
    assert cfg['shards']

    good_shards_count = 0
    for shard_name, shard_cfg in cfg['shards'].iteritems():
        assert shard_name
        assert shard_cfg
        good_shards_count += 1

    assert good_shards_count >= 1


def _url(host, port, action):
    return 'http://{}:{}/{}'.format(host, port, action)
