import gevent
import logging
import os

import infra.callisto.deploy.resource as deploy_resource
import infra.callisto.deploy.tracker.client as tracker_client
import infra.callisto.deploy.storage.storage as storage

import v2.plugin


_REQUESTS_TIMEOUT = 30


class Plugin(v2.plugin.BasePlugin):
    @classmethod
    def add_args(cls, parser):
        parser.add_argument('--storage-root', help='Storage root', required=True)
        parser.add_argument('--tracker-url', help='Cajuper tracker url', required=True)
        parser.add_argument('--cleanup-cooldown', type=int, default=0,
                            help='Sleep for <X> seconds after every resource remove')

    def __init__(self, host, port, tags, args):
        super(Plugin, self).__init__(host, port, tags)
        self._storage_root = args.storage_root
        self._deployer = storage.Deployer(args.storage_root, tracker_client.Client(args.tracker_url),
                                          args.cleanup_cooldown)

        self._deployer.start()
        self._status_thread = gevent.spawn(self._listen_port)

    def apply_config(self, content):
        self._deployer.set_targets({
            deploy_resource.Resource.from_dict(entry['resource']): entry['args']
            for entry in content['resources']
        })

    def collect_status(self):
        if self._status_thread.dead:
            logging.error('Status thread is dead with exception: %s', self._status_thread.exception)
            os._exit(1)

        status = {
            'freespace': _get_freespace(self._storage_root),
            'resources': self._deployer.status(),
        }
        return status

    def _listen_port(self):
        def dummy_handler(env, start_response):
            start_response('200 OK', {})
            return []
        gevent.pywsgi.WSGIServer(('::', int(self.port)), dummy_handler).serve_forever()


def _get_freespace(mount_point):
    if os.path.exists(mount_point):
        vfs = os.statvfs(mount_point)
        return vfs.f_bavail * vfs.f_frsize
    else:
        return None
