import os
import logging
import requests
import tempfile

import v2.plugin
import infra.callisto.deploy.storage.storage as resources_storage


_REQUESTS_TIMEOUT = 30


class Plugin(v2.plugin.BasePlugin):
    @classmethod
    def add_args(cls, parser):
        parser.add_argument('--shard-root', help='Shard root dir', required=True)
        parser.add_argument('--config-template', help='Config template file', required=True)
        parser.add_argument('--config-file', help='Patched config file', required=True)

    def __init__(self, host, port, tags, args):
        super(Plugin, self).__init__(host, port, tags)
        self._shard_root = args.shard_root
        self._config_template = args.config_template
        self._config_file = args.config_file
        self._shard = None

    def start(self):
        pass

    def collect_status(self):
        if self._shard:
            status = 'DOWN'
            try:
                r = requests.get(_url(self.host, self.port, 'unistat'), timeout=_REQUESTS_TIMEOUT)
                r.raise_for_status()
                status = 'RUN'
            except requests.exceptions.RequestException as e:
                logging.warning('Could not collect status: %s', e)
            return {_shard_name(self._shard): {'status': status}}
        return {}

    def apply_config(self, content):
        if 'resource' in content:  # deployer2 mode
            namespace, shard_name = content['resource']['namespace'], content['resource']['name']
            shard_path = resources_storage.resource_path(self._shard_root, namespace, shard_name)
        elif content['shard']:  # deployer mode, obsolete
            shard_name = content['shard']
            shard_path = os.path.join(self._shard_root, shard_name)
        else:
            shard_name = None

        if shard_name == self._shard:
            logging.info('config did not changed')
            return

        logging.info('config changed: %s -> %s', self._shard, shard_name)
        if shard_name:
            patched_config = _patch_config(_load_template_config(self._config_template), shard_path)
            _save_config(patched_config, self._config_file)
        else:
            _drop_config(self._config_file)

        self.stop()
        self._shard = shard_name

    def stop(self):
        logging.info('stopping')
        try:
            r = requests.get(
                _url(self.host, self.port, 'admin', action='shutdown'),
                timeout=_REQUESTS_TIMEOUT
            )
            r.raise_for_status()
        except requests.exceptions.RequestException as e:
            logging.warning('Could not stop: %s', e)


def _patch_config(template, shard):
    return template.replace('_PLACE_HOLDER_', shard)


def _load_template_config(config_template):
    with open(config_template) as fp:
        return fp.read()


def _save_config(content, config_file):
    with tempfile.NamedTemporaryFile(dir='./', delete=False) as fp:
        fp.write(content)
        fp.close()
        os.chmod(fp.name, 0644)
        os.rename(fp.name, config_file)


def _drop_config(config_file):
    if os.path.exists(config_file):
        os.unlink(config_file)


def _shard_name(shard):
    if shard.endswith('/local'):
        return shard.replace('/local', '')
    return shard


def _url(host, port, path, action=None, info=None):
    url = 'http://{}:{}/{}'.format(host, port, path)
    if action:
        url = '{}?action={}'.format(url, action)
    elif info:
        url = '{}?info={}'.format(url, info)
    return url
