import os
import logging
import requests
import retry

import plugin


_REQUESTS_TIMEOUT = 30


class Plugin(plugin.BasePlugin):
    _shard_symlink = 'shard-symlink'

    @classmethod
    def add_args(cls, parser):
        parser.add_argument('--shard-root', help='Shard root dir', required=True)

    def __init__(self, host, port, tags, args):
        super(Plugin, self).__init__(host, port, tags)
        self._shard_root = args.shard_root

    def collect_status(self):
        shard = None
        if _is_running(self.host, self.port):
            shard = _get_active_shard(self.host, self.port)

        if shard:
            return {shard: {'status': 'RUN'}}
        else:
            return {}

    def apply_config(self, content):
        new_shard = content['shard']
        if not new_shard:
            raise RuntimeError('no shard in config')

        _set_symlink(os.path.join(self._shard_root, new_shard), self._shard_symlink)

        if not _is_running(self.host, self.port):
            raise RuntimeError('not running')

        running_shard = _get_active_shard(self.host, self.port)

        if running_shard != new_shard:
            logging.info('shard changed: %s -> %s', running_shard, new_shard)
            self.stop()
        else:
            logging.debug('shard did not change')

    def stop(self):
        logging.info('stopping')
        r = requests.post(_url(self.host, self.port, command='shutdown'), timeout=_REQUESTS_TIMEOUT)
        r.raise_for_status()


@retry.retry(delay=1, tries=3)
def _is_running(host, port):
    r = requests.get(_url(host, port, command='get_status'), timeout=_REQUESTS_TIMEOUT)
    r.raise_for_status()
    return r.json()['status'] == 'Active'


@retry.retry(delay=1, tries=3)
def _get_active_shard(host, port):
    r = requests.get(_url(host, port, command='get_info_server'), timeout=_REQUESTS_TIMEOUT)
    r.raise_for_status()
    indexes = r.json()['result']['indexes'] or {}

    seen_shard_names = set()
    for index_description in indexes.values():
        path_to_shard = os.path.dirname(index_description['dir'])
        if path_to_shard.endswith('/index'):  # FIXME HACK
            path_to_shard = path_to_shard.replace('/index', '')
        shard_name = os.path.basename(path_to_shard)
        seen_shard_names.add(shard_name)

    if not seen_shard_names:
        return None
    if len(seen_shard_names) == 1:
        return seen_shard_names.pop()
    raise RuntimeError('multiple shards')


def _set_symlink(source, link_name):
    if os.path.islink(link_name) and os.readlink(link_name) == source:
        return
    if os.path.islink(link_name):
        os.unlink(link_name)
    os.symlink(source, link_name)


INFO_PORT_SHIFT = 3


def _url(host, port, command=None):
    port = int(port) + INFO_PORT_SHIFT
    url = 'http://{}:{}'.format(host, port)
    if command:
        url = '{}?command={}'.format(url, command)
    return url
